/*$Id: e_storag.cc,v 15.11 1999/10/14 07:46:49 al Exp $ -*- C++ -*-
 * Base class for storage elements of a circuit
 */
#include "e_storag.h"
/*--------------------------------------------------------------------------*/
/* table for selecting local integraton method
 * Determines which one wins in a conflict.
 * "only" wins over non-only.  local (method_u) wins over opt.
 */
//                     OPT::method    method_u
METHOD STORAGE::method_select[meNUM_METHODS][meNUM_METHODS] = {
  /*vv OPT vv*/
  //local>>>EULER,EULERONLY,TRAP,TRAPONLY,GEAR2,GEAR2ONLY,TRAPGEAR,TRAPEULER
  /*meUNKNOWN*/
  {mTRAPGEAR,mEULER,mEULER,mTRAP, mTRAP,mGEAR, mGEAR,mTRAPGEAR,mTRAPEULER},
  /*meEULER*/
  {mEULER,   mEULER,mEULER,mTRAP, mTRAP,mGEAR, mGEAR,mTRAPGEAR,mTRAPEULER},
  /*meEULERONLY*/
  {mEULER,   mEULER,mEULER,mEULER,mTRAP,mEULER,mGEAR,mEULER,   mEULER},
  /*meTRAP*/
  {mTRAP,    mEULER,mEULER,mTRAP, mTRAP,mGEAR, mGEAR,mTRAPGEAR,mTRAPEULER},
  /*meTRAPONLY*/
  {mTRAP,    mTRAP, mEULER,mTRAP, mTRAP,mTRAP, mGEAR,mTRAP,    mTRAP},
  /*meGEAR*/
  {mGEAR,    mEULER,mEULER,mTRAP, mTRAP,mGEAR, mGEAR,mTRAPGEAR,mTRAPEULER},
  /*meGEAR2ONLY*/
  {mGEAR,    mGEAR, mEULER,mGEAR, mTRAP,mGEAR, mGEAR,mGEAR,    mGEAR},
  /*meTRAPGEAR*/
  {mTRAPGEAR,mEULER,mEULER,mTRAP, mTRAP,mGEAR, mGEAR,mTRAPGEAR,mTRAPEULER},
  /*meTRAPEULER*/
  {mTRAPEULER,mEULER,mEULER,mTRAP,mTRAP,mGEAR, mGEAR,mTRAPGEAR,mTRAPEULER}
};
/*--------------------------------------------------------------------------*/
void STORAGE::precalc()
{
  {if (subckt().exists()){
    subckt().precalc();
  }else{
    method_a = method_select[OPT::method][method_u];
    if (value() == 0. && !has_common()){
      error(bPICKY, long_label() + ": short circuit\n");
    }
    y0.f0 = LINEAR;
    y0.f1 = value();
    assert(loss == 0.);
    set_converged();
    assert(!constant()); /* because of integration */
  }}
  /* m0 and acg are frequency/time dependent and cannot be set here.
   * If this is a coupled inductor, there is a subckt, which is expanded
   * by the mutual pseudo-element.
   * Assigning the values here becomes unnecessary, but harmless.
   */
}
/*--------------------------------------------------------------------------*/
void STORAGE::begin_core()
{
  method_a = method_select[OPT::method][method_u];
  if (!has_tr_eval()){
    //assert(y0.f0 == LINEAR); f0 == f1 * x
    assert(y0.f1 == value());
    assert(loss == 0.);
    assert(converged());
    assert(!constant());
  }
}
/*--------------------------------------------------------------------------*/
void STORAGE::dc_advance()
{
  {if (subckt().exists()){
    subckt().dc_advance();
  }else{
    mt1 = m0;
    assert(SIM::time0 == 0.);
    time0 = time1 = time2 = time3 = 0.;
  }}
}
/*--------------------------------------------------------------------------*/
void STORAGE::tr_advance()
{
  {if (subckt().exists()){
    subckt().tr_advance();
  }else{
    {if (time0 < SIM::time0){		// forward
      mt1 = m0;
      yt3 = yt2;
      yt2 = yt1;
      yt1 = y0;
      time3 = time2;
      time2 = time1;
      time1 = time0;
    }else{				// else backward, don't save
      assert(time1 < SIM::time0);
    }}
    time0 = SIM::time0;
  }}
}
/*--------------------------------------------------------------------------*/
double STORAGE::review(double i0, double it1)const
{
  {if (time3 <= 0.){
    return NEVER;
  }else{
    double factor = 1./12.;      /* coefficient of 3rd der, trapezoid rule */
    double dt0 = time0 - time1;
    double dt1 = time1 - time2;		/* BUG: these values should */
    double dt2 = time2 - time3;		/* be stored */
    double ddt0 = time0 - time2;
    double ddt1 = time1 - time3;
    double dddt0 = time0 - time3;

    assert(y0.f0 != LINEAR);
    double q0  = y0.f0;						/* charge */
    double qt1 = yt1.f0;
    double qt2 = yt2.f0;
    double qt3 = yt3.f0;
    double dqdt0 = (q0   - qt1) / dt0;				/* 1st der */
    double dqdt1 = (qt1  - qt2) / dt1;
    double dqdt2 = (qt2  - qt3) / dt2;
    double ddqddt0 = (dqdt0 - dqdt1) / ddt0;			/* 2nd der */
    double ddqddt1 = (dqdt1 - dqdt2) / ddt1;
    double dddqdddt = (ddqddt0 - ddqddt1) / dddt0;		/* 3rd der */

    double currenttol = OPT::abstol + OPT::reltol*std::max(fabs(i0),fabs(it1));
    double chargetol = std::max(OPT::chgtol,std::max(fabs(q0),fabs(qt1)))
      * OPT::reltol / dt0;
    double tol = std::max(currenttol,chargetol);
    double denom = std::max(OPT::abstol,(factor *fabs(dddqdddt)));/*avoid /0*/
    double timestep = OPT::trtol * sqrt(tol / denom);

    if (timestep <= SIM::dtmin){
      untested();
     error(bDANGER,"step control error:%s %g\n",long_label().c_str(),timestep);
      error(bTRACE, "q0=%g i0=%g dq0=%g\n", q0, i0, dqdt0);
      error(bTRACE, "it=%g qt=%g tol=%g\n", currenttol, chargetol, tol);
      timestep = SIM::dtmin;
    }
    {if (timestep < dt0 * OPT::trreject){
      error(bTRACE, "step rejected:" + long_label() + '\n');
      error(bTRACE, "new=%g  old=%g  rej=%g\n",
	    timestep, dt0, dt0 * OPT::trreject);
      return time1 + timestep;
    }else{
      return time0 + timestep;
    }}
  }}
}
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
