/*	$Id: ncurses_redisplay.c,v 1.24 1997/10/31 20:35:54 sandro Exp $	*/

/*
 * Copyright (c) 1997
 *	Sandro Sigala, Brescia, Italy.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/*
 * ncurses redisplay engine.
 *
 * This redisplay engine is simple since ncurses does
 * all the work of finding what is changed on the screen
 * and updating when is needed.
 */

#ifdef __FreeBSD__
/* 
 * XXX a redundant refresh() call fixes a refresh bug under
 * FreeBSD with ncurses 1.8.6 (may be also required under others OSs).
 */
#define NEED_REDUNDANT_REFRESH
#endif

#include "config.h"

#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#ifdef HAVE_NCURSES_H
#include <ncurses.h>
#else
#include <curses.h>
#endif

#include "zile.h"
#include "extern.h"
#include "term_ncurses.h"
 
/*
 * The cached fonts.
 */
static chtype font_character;
static chtype font_character_delimiters;
static chtype font_comment;
static chtype font_directive;
static chtype font_identifier;
static chtype font_keyword;
static chtype font_number;
static chtype font_other;
static chtype font_string;
static chtype font_string_delimiters;

static int cur_tab_width;
static int point_start_column;
static int point_screen_column;

static chtype
strtochtype(char *s)
{
	switch (*s) {
	case 'b':
		if (!strcmp(s, "black"))
			return C_FG_BLACK;
		else if (!strcmp(s, "blue"))
			return C_FG_BLUE;
		return C_FG_WHITE;
	case 'c':
		if (!strcmp(s, "cyan"))
			return C_FG_CYAN;
		return C_FG_WHITE;
	case 'g':
		if (!strcmp(s, "green"))
			return C_FG_GREEN;
		return C_FG_WHITE;
	case 'l':
		if (!strncmp(s, "light-", 6))
			return strtochtype(s + 6)|A_BOLD;
		return C_FG_WHITE;
	case 'm':
		if (!strcmp(s, "magenta"))
			return C_FG_MAGENTA;
		return C_FG_WHITE;
	case 'r':
		if (!strcmp(s, "red"))
			return C_FG_RED;
		return C_FG_WHITE;
	case 'y':
		if (!strcmp(s, "yellow"))
			return C_FG_YELLOW;
		return C_FG_WHITE;
	}

	return C_FG_WHITE;
}

/*
 * Get the font color value from the user specified variables.
 */
static chtype
get_font(char *font)
{
	char *s;

	if ((s = get_variable(font)) == NULL)
		return C_FG_WHITE;

	return strtochtype(s);
}

void
ncurses_refresh_cached_variables(void)
{
	/*
	 * Refresh the font cache.
	 */
	font_character = get_font("font-character");
	font_character_delimiters = get_font("font-character-delimiters");
	font_comment = get_font("font-comment");
	font_directive = get_font("font-directive");
	font_identifier = get_font("font-identifier");
	font_keyword = get_font("font-keyword");
	font_number = get_font("font-number");
	font_other = get_font("font-other");
	font_string = get_font("font-string");
	font_string_delimiters = get_font("font-string-delimiters");
}

static inline int
make_char_printable(char *buf, unsigned int c)
{
	if (c == '\0') {
		strcpy(buf, "^@");
		return 2;
	}

	if (c <= '\32')
		sprintf(buf, "^%c", 'A' + c - 1);
	else
		sprintf(buf, "\\%o", c & 255);

	return strlen(buf);
}

static inline void
outch(int c, chtype font, int *x)
{
	int j, w;
	char buf[16];

	if (*x >= COLS)
		return;

	if (c == '\t')
		for (w = cur_tab_width - *x % cur_tab_width; w > 0 && *x < COLS; w--)
			addch(' '), ++(*x);
	else if (ISPRINT(c))
		addch(c | font), ++(*x);
	else {
		j = make_char_printable(buf, c);
		for (w = 0; w < j && *x < COLS; ++w)
			addch(buf[w] | font), ++(*x);
	}
}

static void
draw_line(int line, int startcol, windowp wp, linep lp)
{
	int x, j;

	move(line, 0);
	for (x = 0, j = startcol; j < lp->size && x < wp->ewidth; ++j)
		outch(lp->text[j], C_FG_WHITE, &x);
	if (x >= COLS)
		mvaddch(line, COLS-1, '>' | C_FG_GREEN | A_BOLD);
}

/*
 * This hack is required for Font Lock because the full line must
 * be parsed always also when is displayed only one truncated fraction.
 */

#define OUTCH(c, font)				\
do { 						\
	if (i >= startcol)			\
		outch(c, font, &x);		\
} while (0)

/*
 * Draw a line on the screen with font lock color.
 */
static void
draw_line_fontlock(int line, int startcol, windowp wp, linep lp, int *lastanchor)
{
	int i, x, c;

	move(line, 0);
	for (x = i = 0; i < lp->size; ++i) {
		c = lp->text[i];

		/* Comment handling. */
		if (lp->anchors[i] == ANCHOR_END_COMMENT) {
			if (*lastanchor == ANCHOR_BEGIN_COMMENT) {
				OUTCH(c, font_comment);
				*lastanchor = ANCHOR_NULL;
			} else
				OUTCH(c, font_other);
		} else if (lp->anchors[i] == ANCHOR_BEGIN_COMMENT) {
			OUTCH(c, font_comment);
			*lastanchor = ANCHOR_BEGIN_COMMENT;
		} else if (*lastanchor == ANCHOR_BEGIN_COMMENT)
			OUTCH(c, font_comment);

		/* String handling. */
		else if (lp->anchors[i] == ANCHOR_END_STRING) {
			OUTCH(c, font_string_delimiters);
			*lastanchor = ANCHOR_NULL;
		} else if (lp->anchors[i] == ANCHOR_BEGIN_STRING) {
			OUTCH(c, font_string_delimiters);
			*lastanchor = ANCHOR_BEGIN_STRING;
		} else if (*lastanchor == ANCHOR_BEGIN_STRING)
			OUTCH(c, font_string);

		/* Other tokens handling. */
		else if (isalpha(c) || c == '_') {
			int j;
			chtype attr;

			j = i;
			while (j < lp->size && (isalnum(lp->text[j]) || lp->text[j] == '_'))
				++j;
			if (is_c_keyword(lp->text + i, j-i))
				attr = font_keyword;
			else
				attr = font_identifier;
			for (; i < j; ++i)
				OUTCH(lp->text[i], attr);
			i--;
		} else if (isdigit(c) || c == '.') {
			/*
			 * XXX
			 * Fix this to support floats correctly.
			 * This stuff is actually a hack but works most the time.
			 */
			if (c == '.' && (i+1 >= lp->size || !isdigit(lp->text[i+1]))) {
			        OUTCH(c, font_other);
				continue;
			}
			do {
				OUTCH(lp->text[i], font_number);
				++i;
			} while (i < lp->size && (isxdigit(lp->text[i]) || strchr("xeE.+-lLfFuU", lp->text[i]) != NULL));
			i--;
		} else if (c == '#') {
			OUTCH(lp->text[i], font_directive);
			++i;
			while (i < lp->size && isspace(lp->text[i])) {
			        OUTCH(lp->text[i], font_directive);
				++i;
			}
			while (i < lp->size && isalpha(lp->text[i])) {
				OUTCH(lp->text[i], font_directive);
				++i;
			}
			i--;
		} else if (c == '\'') {
			OUTCH(lp->text[i], font_character_delimiters);
			while (++i < lp->size && lp->text[i] != '\'')
				if (lp->text[i] == '\\') {
					OUTCH('\\', font_character);
					if (++i < lp->size)
						OUTCH(lp->text[i], font_character);
				} else
					OUTCH(lp->text[i], font_character);
			if (i < lp->size)
				OUTCH(lp->text[i], font_character_delimiters);
		} else
			OUTCH(c, font_other);
	}
	if (x >= COLS)
		mvaddch(line, COLS-1, '>' | C_FG_GREEN | A_BOLD);
}

static void
draw_window(int topline, windowp wp)
{
	int i, lastanchor, startcol;
	linep lp;

	/*
	 * Find the first line to display on the first screen line.
	 */
	for (lp = wp->pointp, i = wp->topdelta;
	     i > 0 && lp->prev != wp->bp->limitp; lp = lp->prev, --i)
		;

	if (wp->bp->flags & BFLAG_FONTLOCK)
		lastanchor = find_last_anchor(wp->bp, lp->prev);

	cur_tab_width = wp->bp->tab_width;

	/*
	 * Draw the window lines.
	 */
	for (i = topline; i < wp->eheight + topline; ++i) {
		move(i, 0);
		clrtoeol();
		if (lp != wp->bp->limitp) {
			if (lp == cur_wp->pointp)
				startcol = point_start_column;
			else
				startcol = 0;
			if (wp->bp->flags & BFLAG_FONTLOCK)
				draw_line_fontlock(i, startcol, wp, lp, &lastanchor);
			else
				draw_line(i, startcol, wp, lp);

			/*
			 * Draw the `[EOB]' end of buffer marker if
			 * the `show-eob-marker' variable is enabled
			 * and the marker is on the current page.
			 */
			if (lp->next == wp->bp->limitp
			    && !(wp->bp->flags & BFLAG_NOEOB)
			    && lookup_bool_variable("show-eob-marker")) {
				static chtype eob_str[] = {
					'['|C_FG_CYAN|A_BOLD,
					'E'|C_FG_CYAN,
					'O'|C_FG_CYAN,
					'B'|C_FG_CYAN,
					']'|C_FG_CYAN|A_BOLD,
					'\0'
				};
				int y, x;

				getyx(stdscr, y, x);

				addchnstr(eob_str, COLS-x > 5 ? 5 : COLS-x);
			}

			lp = lp->next;
		}
	}
}

static char *
make_mode_line_flags(windowp wp)
{
	static char buf[3];

	if ((wp->bp->flags & (BFLAG_MODIFIED | BFLAG_READONLY)) == (BFLAG_MODIFIED | BFLAG_READONLY))
		buf[0] = '%', buf[1] = '*';
	else if (wp->bp->flags & BFLAG_MODIFIED)
		buf[0] = buf[1] = '*';
	else if (wp->bp->flags & BFLAG_READONLY)
		buf[0] = buf[1] = '%';
	else
		buf[0] = buf[1] = '-';

	return buf;
}

static int
get_text_goalc(windowp wp)
{
	int col = 0, t = wp->bp->tab_width;
	char *sp = wp->pointp->text, *p = sp;

	while (p < sp + wp->pointo) {
		if (*p == '\t')
			col |= t - 1;
		++col, ++p;
	}

	return col;
}

/*
 * This function calculates the best start column to draw
 * if the line needs to get truncated.
 * Called only for the line where is the point.
 * XXX require a rewrite because is not so simple to understand.
 */
static void
calculate_start_column(windowp wp)
{
	int col, lastcol, t = wp->bp->tab_width;
	int rpfact, lpfact;
	char buf[16], *rp, *lp, *p;

	rp = wp->pointp->text + wp->pointo;
	rpfact = wp->pointo / (wp->ewidth / 3);

	for (lp = rp; lp >= cur_wp->pointp->text; --lp) {
		for (col = 0, p = lp; p < rp; ++p)
			if (*p == '\t') {
				col |= t - 1;
				++col;
			} else if (ISPRINT(*p))
				++col;
			else
				col += make_char_printable(buf, *p);

		lpfact = (lp - cur_wp->pointp->text) / (wp->ewidth / 3);

		if (col >= wp->ewidth - 1 || lpfact < (rpfact - 2)) {
			point_start_column = lp + 1 - cur_wp->pointp->text;
			point_screen_column = lastcol;
			return;
		}

		lastcol = col;
	}

	point_start_column = 0;
	point_screen_column = col;
}

static char *
make_screen_pos(windowp wp)
{
	static char buf[16];
	if (wp->bp->num_lines <= wp->eheight && wp->topdelta == wp->pointn)
		strcpy(buf, "All");
	else if (wp->pointn == wp->topdelta)
		strcpy(buf, "Top");
	else if (wp->pointn + (wp->eheight - wp->topdelta) > wp->bp->num_lines)
		strcpy(buf, "Bot");
	else
		sprintf(buf, "%2d%%", (int)((float)wp->pointn / wp->bp->num_lines * 100));
	return buf;
}

/*
 * Format the time section of status line.
 */
static char *
make_time_str(char *buf)
{
	char *fmt;
	time_t t;

	if ((fmt = get_variable("display-time-format")) != NULL) {
		time(&t);
		strftime(buf, 80, fmt, localtime(&t)); 
		return buf;
	} else
		return NULL;
}

static void
draw_status_line(int line, windowp wp)
{
	int i;

	attrset(A_REVERSE | get_font("status-line-color"));

	move(line, 0);
	for (i = 0; i < wp->ewidth; ++i)
		addch('-');

	move(line, 0);
	printw("--%2s-Zile: %-18s (%s%s%s%s%s)--L%d/%d,C%d--%s",
	       make_mode_line_flags(wp),
	       wp->bp->name,
	       (wp->bp->flags & BFLAG_CMODE) ? "C" : "Text",
	       (wp->bp->flags & BFLAG_FONTLOCK) ? " Font" : "",
	       (wp->bp->flags & BFLAG_AUTOFILL) ? " Fill" : "",
	       (wp->bp->flags & BFLAG_OVERWRITE) ? " Ovwrt" : "",
	       (thisflag & FLAG_DEFINING_MACRO) ? " Macro" : "",
	       wp->pointn+1, wp->bp->num_lines+1, get_text_goalc(wp)+1,
	       make_screen_pos(wp));

	if (lookup_bool_variable("display-time")) {
		char buf[64], *p = make_time_str(buf);
		if (p != NULL)
			mvaddstr(line, wp->ewidth - strlen(p) - 2, p);
	}

	attrset(0);
}

static void
do_redisplay(void)
{
	int topline, cur_topline;
	windowp wp;

	topline = 0;

	calculate_start_column(cur_wp);

	for (wp = head_wp; wp != NULL; wp = wp->next) {
		if (wp == cur_wp)
			cur_topline = topline;

		draw_window(topline, wp);

#ifdef NEED_REDUNDANT_REFRESH
		refresh();
#endif

		/*
		 * Draw the status line only if there is available space
		 * after the buffer text space.
		 */
		if (wp->fheight - wp->eheight > 0)
			draw_status_line(topline + wp->eheight, wp);

		topline += wp->fheight;
	}

	if (point_start_column > 0)
		mvaddch(cur_topline + cur_wp->topdelta, 0, '<' | C_FG_GREEN | A_BOLD);
	move(cur_topline + cur_wp->topdelta, point_screen_column);
}

void
ncurses_redisplay(void)
{
	do_redisplay();
}

void
ncurses_full_redisplay(void)
{
	clear();
	do_redisplay();
}
