//  UScroll.cpp version 1.5
//  yudit package - Unicode Editor for the X Window System (and Linux) 
//
//  Author: gsinai@iname.com (Gaspar Sinai)
//  GNU Copyright (C) 1997,1998,1999  Gaspar Sinai
// 
//  yudit version 1.5  Copyright(C) 30 November, 1999, Tokyo Japan  Gaspar Sinai
//  yudit version 1.4  Copyright(C) 25 November, 1999, Tokyo Japan  Gaspar Sinai
//  yudit version 1.3  Copyright(C)  5 April,    1999, Tokyo Japan  Gaspar Sinai
//  yudit version 1.2  Copyright(C) 10 December, 1998, Tokyo Japan  Gaspar Sinai
//  yudit version 1.1  Copyright(C) 23 August,   1998, Tokyo Japan  Gaspar Sinai
//  yudit version 1.0  Copyright(C) 17 May,      1998, Tokyo Japan  Gaspar Sinai
//  yudit version 0.99 Copyright(C)  4 April,    1998, Tokyo Japan  Gaspar Sinai
//  yudit version 0.97 Copyright(C)  4 February, 1998, Tokyo Japan  Gaspar Sinai
//  yudit version 0.95 Copyright(C) 10 January,  1998, Tokyo Japan  Gaspar Sinai
//  yudit version 0.94 Copyright(C) 17 December, 1997, Tokyo Japan  Gaspar Sinai
//  yudit version 0.9 Copyright (C)  8 December, 1997, Tokyo Japan  Gaspar Sinai
//  yutex version 0.8 Copyright (C)  5 November, 1997, Tokyo Japan  Gaspar Sinai
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//

#include "UScroll.h"
#include <iostream.h>


UScroll::UScroll  (UFrame* parent_, UScrollStyle style_) 
	: UFrame(parent_, 1) 
{
	int		width=26;
	int		height=26;

	setFrameStyle (IN);
	scrollStyle = style_;
	
	select (ExposureMask | SubstructureNotifyMask | ButtonPressMask);

	switch (scrollStyle)
	{
	case HORIZONTAL:
		inc = new UArrowButton (this, UArrowButton::RIGHT);
		dec = new UArrowButton (this, UArrowButton::LEFT);
		slide = new UArrowButton (this, UArrowButton::BLANK);
		slide->place (height-frameSize, -height+frameSize, height-frameSize, frameSize*2);
		inc->place (-height+frameSize, -height+frameSize,frameSize*2 , frameSize*2);
		dec->place (frameSize, -height+frameSize, -height+frameSize, frameSize*2);
		break;
	case VERTICAL:
		inc = new UArrowButton (this, UArrowButton::DOWN);
		dec = new UArrowButton (this, UArrowButton::UP);
		slide = new UArrowButton (this, UArrowButton::BLANK);
		slide->place (frameSize, width-frameSize, frameSize, width-frameSize);
		inc->place (frameSize, -width+1, frameSize*2, frameSize*2);
		dec->place (frameSize, frameSize, frameSize, -width+frameSize);
	default:
		break;
	}
	inc->setRepeat (UButton::REPEAT);
	dec->setRepeat (UButton::REPEAT);

	setBackground (background);
	setForeground (foreground);

	select (ButtonPressMask |ButtonReleaseMask 
		| Button1MotionMask | LeaveWindowMask | EnterWindowMask);
	place (2, -26, 2, 2);

	bestSize.width=rectangle.width;
	bestSize.height=rectangle.height;
	select (ButtonPressMask |ButtonReleaseMask);

	granuality=NONGRANUAL;
	sliderSize = 0;
	scroll (0, 1, 1, 1);
}

UScroll::~UScroll()
{
	// autodelete from frame
}

void 
UScroll::setBackground (const UColor& color)
{
	unsigned int	i;
	UComponent*	comp;
	UColor		compColor;

	compColor = color;
	background = color;

	// just a little darker
	background.darker(2);

	XSetWindowBackground (background.getDisplay(), window,
		background.getPixel());
	light = compColor;
	light.lighter ();
	XSetForeground (UTOP->display, lightGC, light.getPixel());

	dark = compColor;
	dark.darker ();
	XSetForeground (UTOP->display, darkGC, dark.getPixel());

	XClearArea (UTOP->display, window, 0, 0, rectangle.width,
		rectangle.height, True);

	for (i=0; i< children.getSize(); i++)
	{
		comp=(UComponent*) children.at (i);
		if (comp==0) continue;

		// here is the trick
		comp->setBackground(compColor);
	}
}

void 
UScroll::setBackground (const char* color)
{
	background = color;
	setBackground (background);
}

void
UScroll::setGranuality (UGranuality gr)
{
	granuality=gr;
}

void
UScroll::resize (int width, int height)
{
	switch (scrollStyle)
	{
	case HORIZONTAL:
		bestSize.width=rectangle.width;
		inc->place (-height+frameSize, -height+frameSize,frameSize , frameSize);
		dec->place (frameSize, -height+frameSize, -height+frameSize, frameSize);
		slide->place (height+sliderPosition, frameSize, 
			-(height+sliderPosition+sliderSize), frameSize);
		break;
	case VERTICAL:
		bestSize.height=rectangle.height;
		inc->place (frameSize, -width+1, frameSize, frameSize);
		dec->place (frameSize, frameSize, frameSize, -width+frameSize);
		slide->place (frameSize, width +sliderPosition, 
			frameSize,  -(width +sliderPosition +sliderSize));
		break;
	}
	UFrame::resize (width, height);
	scroll (scrollValue, scrollStep, scrollPage, scrollMax+scrollPage);
}

int
UScroll::isA (UComponent::UType type_)
{
	if (type_ == UComponent::SCROLL) return 1;
	return UFrame::isA (type_);
}

void
UScroll::scroll(int value_, int step_, int page_, int max_)
{
	int		minSize;
	int		moveSize;
	int		step;
	int		page;
	int		max;
	double		scale;
	int		sliderS;
	int		sliderP;
	int		value;
	int		moveFirst;

	if (step_==0 && page_==0 && max_==0)
	{
		scroll (value_);
		return;
	}

	// sanity checks
	step = (step_>0) ? step_ : scrollStep;
	page = (page_>0) ? page_ : scrollPage;
	max = (max_>0) ? max_ : scrollMax-page;
	if (max<0) max=1;
	if (max < page) max=page;

	value =  (value_ < 0) ? scrollValue : value_;
	if (value > max-page) value = max-page;

	moveSize = (scrollStyle==HORIZONTAL) ? 
		(rectangle.width - 2 * rectangle.height)
		: (rectangle.height - 2 * rectangle.width);

	if (moveSize < 1) moveSize=1;

	// calcualte the minimim siz of the slider  - the width / 4
	minSize = (scrollStyle==HORIZONTAL) ? rectangle.height>>2 
		: rectangle.width>>2; 

	// this is the minimum I can handle...
	if (minSize < 10) minSize = 10;

	// see if this is right
	sliderS = moveSize*page/max;

	// we need to modify the slider size. this will affect scale
	if (sliderS < minSize) sliderS = minSize;

	// scale is ideal/real of movement
	scale = 0.0;

	// The maximum value we can take is this.
	max = max-page;
	if (max < 0) max = 0;
	if (moveSize-sliderS > 0 && max > 0) 
	{
		scale = (double (moveSize - sliderS))/ double (max);
	}

	// real postion. double value CAN represent 0.
	sliderP = (scale==0.0) ? 0 : int (double (value)  * scale);

	// avoid flicker
	moveFirst=0;
	if (sliderS > sliderSize) moveFirst = 1;
	if (sliderS < sliderSize) moveFirst = -1;

	// we may do some extar here later
	
	scrollStep = step; scrollPage = page; scrollMax = max;
	scrollValue = value; scaleFactor = scale; sliderSize= sliderS;
	sliderPosition = sliderP;


	// recalculate constraints for slider.
	switch (scrollStyle)
	{
	case VERTICAL:
		slide->place (frameSize, rectangle.width +sliderPosition, 
			frameSize,  -(rectangle.width +sliderPosition +sliderSize));
		if (moveFirst==1)
		{
			slide->move (frameSize, rectangle.width+sliderPosition);
		}
		if (moveFirst!=0)
		{
			slide->resize (rectangle.width-2*frameSize, sliderSize);
		}
		if (moveFirst!=1)
		{
			slide->move (frameSize, rectangle.width+sliderPosition);
		}
		break;
	case HORIZONTAL:
		slide->place (rectangle.height+sliderPosition, frameSize, 
			-(rectangle.height+sliderPosition+sliderSize), frameSize);
		if (moveFirst==1)
		{
			slide->move (rectangle.height+sliderPosition, frameSize);
		}
		if (moveFirst!=0)
		{
			slide->resize (sliderSize, rectangle.height-2*frameSize);
		}
		if (moveFirst!=1)
		{
			slide->move (rectangle.height+sliderPosition, frameSize);
		}
		break;
	};
}

void
UScroll::scroll(int value)
{
	int	newPos;
	int	val;

	val= (value > scrollMax) ? scrollMax : ((value<0) ? 0 : value);

	if (val == scrollValue) return;
	scrollValue = val;

	newPos = int (double(scrollValue) * scaleFactor + 0.5);

	if (newPos == sliderPosition) return;
	sliderPosition = newPos;
	// The constraints will be reset with resize
	switch (scrollStyle)
	{
	case VERTICAL:
		slide->place (frameSize, rectangle.width +sliderPosition, 
			frameSize,  -(rectangle.height +sliderPosition +sliderSize));
		slide->move (frameSize, rectangle.width+sliderPosition);
		break;
	case HORIZONTAL:
		slide->place (rectangle.height+sliderPosition, frameSize, 
			-(rectangle.height+sliderPosition+sliderSize), frameSize);
		slide->move (rectangle.height+sliderPosition, frameSize);
		break;
	}
}

//
// in case you wan to scroll this way
// 
void
UScroll::slideTo(int slider_)
{
	int 	moveSize;
	int	newPosition;
	int	value;

	if (scaleFactor==0) return;

	moveSize = (scrollStyle==HORIZONTAL) ? 
		(rectangle.width - 2 * rectangle.height)
		: (rectangle.height - 2 * rectangle.width);

	moveSize -= sliderSize;
	if (moveSize < 0) return;

	newPosition = slider_;
	if (newPosition > moveSize) newPosition = moveSize;
	if (newPosition < 0) newPosition = 0;

	if (sliderPosition == newPosition) return;
	sliderPosition = newPosition;

	value = int ((double(newPosition))/scaleFactor + 0.5);

	scrollValue = (value > scrollMax) ? scrollMax : value;

	// The constraints will be reset with resize
	switch (scrollStyle)
	{
	case VERTICAL:
		slide->place (frameSize, rectangle.width +sliderPosition, 
			frameSize,  -(rectangle.height +sliderPosition +sliderSize));
		slide->move (frameSize, rectangle.width+sliderPosition);
		break;
	case HORIZONTAL:
		slide->place (rectangle.height+sliderPosition, frameSize, 
			-(rectangle.height+sliderPosition+sliderSize), frameSize);
		slide->move (rectangle.height+sliderPosition, frameSize);
		break;
	}
}

void
UScroll::eventDown (UEvent* event)
{
	int		oldValue;

	switch (event->type)
	{
		case UEvent::TIMER:
			break;
		case UEvent::X:
		switch (event->xevent.type)
		{
		case ButtonPress:
			if (event->xevent.xbutton.button != Button1) return;
			oldValue = scrollValue;
			if (scrollStyle==HORIZONTAL)
			{
				if (event->xevent.xbutton.x <=
					 rectangle.height+sliderPosition)
				{
					scroll (scrollValue-scrollPage);
				}
				if (event->xevent.xbutton.x >=
					 rectangle.height+sliderPosition+sliderSize)
				{
					scroll (scrollValue+scrollPage);
				}
			}
			if (scrollStyle==VERTICAL)
			{
				if (event->xevent.xbutton.y <=
					 rectangle.width+sliderPosition)
				{
					scroll (scrollValue-scrollPage);
				}
				if (event->xevent.xbutton.y >=
					 rectangle.width+sliderPosition+sliderSize)
				{
					scroll (scrollValue+scrollPage);
				}
			}
			if (oldValue != scrollValue)
			{
				event->cardinal = scrollValue;
				event->client = this;
				event->type = UEvent::CARDINAL;
				UFrame::eventUp (event);
			}
			break;
		case ButtonRelease:
			if (event->xevent.xbutton.button != Button1) return;
			break;
		}
	default:
		break;
	}
	UFrame::eventDown(event);
}

void
UScroll::eventUp (UEvent* event)
{
	int		oldValue;
	int		newPos;

	switch (event->type)
	{
		case UEvent::TIMER:
			break;
		case UEvent::PRESSED:
			oldValue = scrollValue;
			if (event->client==dec)
			{
				scroll (scrollValue-scrollStep);
			}
			else if (event->client==inc)
			{
				scroll (scrollValue+scrollStep);
			}
			else if (event->client==slide)
			{
				xRoot=event->xevent.xbutton.x_root;
				yRoot=event->xevent.xbutton.y_root;
				presedPos = int(double(scrollValue) 
					* scaleFactor + 0.5);
			}
			if (oldValue != scrollValue)
			{
				event->cardinal = scrollValue;
				event->client = this;
				event->type = UEvent::CARDINAL;
				UFrame::eventUp (event);
			}
			break;
		case UEvent::X:
		switch (event->xevent.type)
		{
		case ButtonPress:
			if (event->xevent.xbutton.button != Button1) return;
			return;

		case ButtonRelease:
			return;
		case MotionNotify:
			if (event->xevent.xmotion.state & Button1MotionMask==0)
				 return;
			if (event->client != slide) return;
			if (scaleFactor == 0.0 ) return;
			oldValue = scrollValue;
			newPos = (scrollStyle==HORIZONTAL) ?
				presedPos + event->xevent.xmotion.x_root-xRoot:
				presedPos + event->xevent.xmotion.y_root-yRoot;
			if (granuality==GRANUAL)
			{
				scroll (int ( double (newPos) / scaleFactor + 0.5));
			}
			else
			{
				slideTo (newPos);
			}
			if (oldValue != scrollValue)
			{
				event->cardinal = scrollValue;
				event->client = this;
				event->type = UEvent::CARDINAL;
				UFrame::eventUp (event);
			}
			return;
		default:
			break;
		}
	default:
		break;
	}
}

