#   include	<stddef.h>
#   include	<stdlib.h>

#   include	<appWinMeta.h>
#   include	<appPs.h>
#   include	<sioEndian.h>

#   define	y0	math_y0
#   define	y1	math_y1
#   include	<math.h>
#   undef	y0
#   undef	y1

#   include	<debugon.h>

#   ifndef	M_PI
#	define	M_PI	3.14159265358979323846
#   endif

# if 0
#    define	WMFDEB(x)	(x)
# else
#    define	WMFDEB(x)	/*nothing*/
# endif

# if 0
#    define	WMFLOG(x)	(x)
# else
#    define	WMFLOG(x)	/*nothing*/
# endif

/************************************************************************/
/*									*/
/*  Collect a series of points.						*/
/*									*/
/************************************************************************/

static int appMetaGetPointsPs(	DeviceContext *		dc,
				int			count,
				SimpleInputStream *	sis )
    {
    XPoint *		xp;

    int			done;

    xp= (XPoint *)realloc( dc->dcPoints, (count+ 1)* sizeof(XPoint) );
    if  ( ! xp )
	{ LXDEB(count,xp); return -1;	}
    dc->dcPoints= xp;

    for ( done= 0; done < count; xp++, done++ )
	{
	int	x= sioEndianGetLeInt16( sis );
	int	y= sioEndianGetLeInt16( sis );

	xp->x= DC_X( x, dc );
	xp->y= DC_Y( y, dc );
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Play a windows metafile to PostScript.				*/
/*									*/
/************************************************************************/

static void appMetaSetColorPs(		FILE *			f,
					RGB8Color *		rgb )
    {
    if  ( rgb->rgb8Red == rgb->rgb8Green	&&
	  rgb->rgb8Red == rgb->rgb8Blue		)
	{
	fprintf( f, "%g setgray\n", rgb->rgb8Red/255.0 );
	}
    else{
	fprintf( f, "%g %g %g setrgbcolor\n",
				    rgb->rgb8Red/255.0,
				    rgb->rgb8Green/255.0,
				    rgb->rgb8Blue/255.0 );
	}

    return;
    }

static int appMetaBitmapImagePs(	SimpleInputStream *	sis,
					FILE *			f,
					int			expectBytes,
					DeviceContext *		dc,
					int			srcX,
					int			srcY,
					int			srcXExt,
					int			srcYExt,
					int			dstX,
					int			dstY,
					int			dstXExt,
					int			dstYExt )
    {
    BitmapDescription		bd;
    unsigned char *		buffer;

    int				done;

    bmInitDescription( &bd );
    buffer= (unsigned char *)0;

    if  ( srcX != 0 || srcY != 0 )
	{ LLDEB(srcX,srcY); return 0;	}

    done= bmBmpReadDib( &bd, &buffer, sis );
    if  ( done < 0 || done > expectBytes )
	{
	LLDEB(done,expectBytes);

	bmCleanDescription( &bd );
	if  ( buffer )
	    { free( buffer );	}

	return -1;
	}

    while( done < expectBytes )
	{ sioInGetCharacter( sis ); done++;	}

    if  ( srcXExt != bd.bdPixelsWide	||
	  srcYExt != bd.bdPixelsHigh	)
	{
	LLDEB(srcXExt,bd.bdPixelsWide);
	LLDEB(srcYExt,bd.bdPixelsHigh);

	bmCleanDescription( &bd );
	if  ( buffer )
	    { free( buffer );	}

	return 0;
	}

    dstX= DC_X( dstX, dc );
    dstY= DC_Y( dstY, dc );

    dstXExt= DC_W( dstXExt, dc );
    dstYExt= DC_H( dstYExt, dc );

    if  ( bmPsPrintBitmapData( f, 1,
			    (double)dstXExt, (double)-dstYExt,
			    dstX, dstY+ dstYExt, 0, 0,
			    bd.bdPixelsWide, bd.bdPixelsHigh,
			    &bd, buffer ) )
	{ LDEB(1); return -1; }

    bmCleanDescription( &bd );
    if  ( buffer )
	{ free( buffer );	}

    return 0;
    }

static int appMetaSelectPenObjectPs(	DeviceContext *		dc,
					FILE *			f,
					int			ob )
    {
    LogicalPen *	lp= &(dc->dcObjects[ob].mfoLogicalPen);

    WMFDEB(DEBFUN( "SelectObject( ob= %d ) PEN Width= %d ",
							ob, lp->lpWidth ));

    WMFLOG(fprintf( f, "%% SelectObject( ob= %d ) PEN Width= %d ",
							ob, lp->lpWidth ));

    switch( lp->lpStyle )
	{
	case PS_SOLID:
	    WMFDEB(DEBFUN( "SOLID\n" ));
	    WMFLOG(fprintf( f, "SOLID\n" ));
	  solid:
	    dc->dcDrawBorders= 1;

	    fprintf( f, "%d setlinewidth\n", DC_W( lp->lpWidth, dc ) );
	    fprintf( f, "[] 0 setdash\n" );

	    break;

	case PS_INSIDEFRAME:
	    WMFDEB(DEBFUN( "INSIDEFRAME\n" ));
	    WMFLOG(fprintf( f, "INSIDEFRAME\n" ));

	    dc->dcDrawBorders= 1;

	    fprintf( f, "%d setlinewidth\n", DC_W( lp->lpWidth, dc ) );
	    fprintf( f, "[] 0 setdash\n" );

	    break;

	case PS_DASH:
	    WMFDEB(DEBFUN( "DASH\n" ));
	    WMFLOG(fprintf( f, "DASH\n" ));

	    if  ( lp->lpWidth > 1 )
		{ goto solid;	}

	    dc->dcDrawBorders= 1;

	    fprintf( f, "10 setlinewidth\n" );
	    fprintf( f, "[60 40] 0 setdash\n" );

	    break;

	case PS_DOT:
	    WMFDEB(DEBFUN( "DOT\n" ));
	    WMFLOG(fprintf( f, "DOT\n" ));

	    if  ( lp->lpWidth > 1 )
		{ goto solid;	}

	    dc->dcDrawBorders= 1;

	    fprintf( f, "10 setlinewidth\n" );
	    fprintf( f, "[10 20] 0 setdash\n" );

	    break;

	case PS_DASHDOT:
	    WMFDEB(DEBFUN( "DASHDOT\n" ));
	    WMFLOG(fprintf( f, "DASHDOT\n" ));

	    if  ( lp->lpWidth > 1 )
		{ goto solid;	}

	    dc->dcDrawBorders= 1;

	    fprintf( f, "10 setlinewidth\n" );
	    fprintf( f, "[60 20 10 20] 0 setdash\n" );

	    break;

	case PS_DASHDOTDOT:
	    WMFDEB(DEBFUN( "DASHDOTDOT\n" ));
	    WMFLOG(fprintf( f, "DASHDOTDOT\n" ));

	    if  ( lp->lpWidth > 1 )
		{ goto solid;	}

	    dc->dcDrawBorders= 1;

	    fprintf( f, "10 setlinewidth\n" );
	    fprintf( f, "[60 20 10 20 10 20] 0 setdash\n" );

	    break;

	case PS_NULL:
	    WMFDEB(DEBFUN( "NULL\n" ));
	    WMFLOG(fprintf( f, "NULL\n" ));
	    dc->dcDrawBorders= 0;
	    break;

	default:
	    WMFDEB(DEBFUN( "style= %d\n", lp->lpStyle ));
	    WMFLOG(fprintf( f, "style= %d\n", lp->lpStyle ));
	    dc->dcDrawBorders= 0;
	    break;
	}

    if  ( dc->dcDrawBorders )
	{ appMetaSetColorPs( f, &(lp->lpColor) ); }

    dc->dcPen= *lp;

    return 0;
    }

static int appMetaSelectBrushObjectPs(	DeviceContext *		dc,
					FILE *			f,
					int			ob )
    {
    LogicalBrush *		lb= &(dc->dcObjects[ob].mfoLogicalBrush);

    switch( lb->lbStyle )
	{
	case BS_SOLID:
	    dc->dcFillInsides= 1;
	    appMetaSetColorPs( f, &(lb->lbColor) );
	    break;

	case BS_HOLLOW:
	    dc->dcFillInsides= 0;
	    break;

	case BS_HATCHED:
	case BS_PATTERN:
	default:
	    dc->dcFillInsides= 0;
	    LDEB(lb->lbStyle); return -1;
	}

    dc->dcBrush= *lb;

    return 0;
    }

static int appMetaSelectFontObjectPs(	DeviceContext *		dc,
					DocumentFontList *	dfl,
					const char *		afmDirectory,
					FILE *			f,
					int			ob )
    {
    LogicalFont *	lf= &(dc->dcObjects[ob].mfoLogicalFont);

    int			family;
    int			face;

    AppFontFamily *	aff;
    AppFontTypeface *	aft;

    DocumentFont *	df;

    df= dfl->dflFonts+ lf->lfTextAttribute.taFontNumber;

    if  ( appGetPsFont( &family, &face, &aff, &aft,
				    afmDirectory, df, lf->lfTextAttribute ) )
	{
	LDEB(lf->lfTextAttribute.taFontNumber);
	dc->dcAfi= (AfmFontInfo *)0;
	return -1;
	}

    dc->dcAfi= (AfmFontInfo *)aft->aftPrintingData;

    appPsSetFont( f, lf->lfTextAttribute );

    dc->dcFont= *lf;

    return 0;
    }

static int appMetaSelectObjectPs(	DeviceContext *		dc,
					DocumentFontList *	dfl,
					const char *		afmDirectory,
					FILE *			f,
					int			recordSize,
					SimpleInputStream *	sis )
    {
    int			ob;

    ob= sioEndianGetLeInt16( sis );

    if  ( ob < 0 || ob >= dc->dcObjectCount )
	{ LLDEB(ob,dc->dcObjectCount); return -1;	}

    switch( dc->dcObjects[ob].mfoType )
	{
	case MFtypePEN:
	    if  ( appMetaSelectPenObjectPs( dc, f, ob ) )
		{ LDEB(ob); return -1;	}
	    break;
	case MFtypeBRUSH:
	    WMFDEB(DEBFUN( "SelectObject( ob= %d ) ", ob ));
	    WMFDEB(DEBFUN( "BRUSH\n" ));

	    if  ( appMetaSelectBrushObjectPs( dc, f, ob ) )
		{ LDEB(ob); return -1;	}
	    break;
	case MFtypeFONT:
	    WMFDEB(DEBFUN( "SelectObject( ob= %d ) ", ob ));
	    WMFDEB(DEBFUN( "FONT\n" ));
	    if  ( appMetaSelectFontObjectPs( dc, dfl, afmDirectory, f, ob ) )
		{ LDEB(ob); return -1;	}
	    break;
	default:
	    WMFDEB(DEBFUN( "SelectObject( ob= %d ) ", ob ));
	    WMFDEB(DEBFUN( "type= %d\n", dc->dcObjects[ob].mfoType ));
	    break;
	}

    return 0;
    }

static int appMeta_PolyLinePs(	SimpleInputStream *	sis,
				FILE *			f,
				int			recordSize,
				DeviceContext *		dc )
    {
    int		count;
    int		done;

    char *	command= "moveto";

    appMetaSetColorPs( f, &(dc->dcPen.lpColor) );

    count= sioEndianGetLeInt16( sis );

    WMFDEB(DEBFUN("PolyLine( count=%d, ... )\n", count ));

    if  ( 2* count != recordSize- 4 )
	{ LLDEB(2*count,recordSize-4); return -1;	}

    for ( done= 0; done < count; done++ )
	{
	int	x= sioEndianGetLeInt16( sis );
	int	y= sioEndianGetLeInt16( sis );

	x= DC_X( x, dc );
	y= DC_Y( y, dc );

	fprintf( f, "%d %d %s\n", x, y, command );

	command= "lineto";
	}

    fprintf( f, "stroke\n" );

    return 0;
    }

static void appMetaDrawPolygonPs(	FILE *			f,
					DeviceContext *		dc,
					int			count )
    {
    int		done;
    char *	command;

    if  ( dc->dcFillInsides )
	{
	command= "moveto";

	appMetaSetColorPs( f, &(dc->dcBrush.lbColor) );

	for ( done= 0; done < count; done++ )
	    {
	    fprintf( f, "%d %d %s\n",
		    dc->dcPoints[done].x, dc->dcPoints[done].y, command );

	    command= "lineto";
	    }

	fprintf( f, "closepath fill\n" );
	}

    if  ( dc->dcDrawBorders )
	{
	command= "moveto";

	appMetaSetColorPs( f, &(dc->dcPen.lpColor) );

	for ( done= 0; done < count; done++ )
	    {
	    fprintf( f, "%d %d %s\n",
		    dc->dcPoints[done].x, dc->dcPoints[done].y, command );

	    command= "lineto";
	    }

	fprintf( f, "stroke\n" );
	}

    return;
    }

static int appMeta_PolygonPs(	SimpleInputStream *	sis,
				FILE *			f,
				int			recordSize,
				DeviceContext *		dc )
    {
    int		count;

    count= sioEndianGetLeInt16( sis );

    WMFDEB(DEBFUN("Polygon( count=%d, ... )\n", count ));

    if  ( appMetaGetPointsPs( dc, count, sis ) )
	{ LDEB(count); return -1;	}

    appMetaDrawPolygonPs( f, dc, count );

    return 0;
    }

static int appMeta_RectanglePs(	SimpleInputStream *	sis,
				FILE *			f,
				int			recordSize,
				DeviceContext *		dc )
    {
    LogicalPen *	lp= &(dc->dcPen);

    int			y0;
    int			x0;
    int			y1;
    int			x1;

    int			swap;

    y1= sioEndianGetLeInt16( sis );
    x1= sioEndianGetLeInt16( sis );
    y0= sioEndianGetLeInt16( sis );
    x0= sioEndianGetLeInt16( sis );

    WMFDEB(DEBFUN("Rectangle(%d..%d,%d..%d)\n", x0,x1,y0,y1));
    WMFLOG(fprintf( f, "%% Rectangle(%d..%d,%d..%d)\n", x0,x1,y0,y1));

    x0= DC_X( x0, dc );
    y0= DC_Y( y0, dc );
    x1= DC_X( x1, dc );
    y1= DC_Y( y1, dc );

    if  ( x1 < x0 )
	{ swap= x0; x0= x1; x1= swap; }
    if  ( y1 < y0 )
	{ swap= y0; y0= y1; y1= swap; }

    if  ( lp->lpStyle == PS_INSIDEFRAME )
	{
	int	width= 1;

	x0 += ( width+ 1 )/ 2;
	x1 -= ( width+ 1 )/ 2;
	y0 += ( width+ 1 )/ 2;
	y1 -= ( width+ 1 )/ 2;
	}

    if  ( dc->dcFillInsides )
	{
	appMetaSetColorPs( f, &(dc->dcBrush.lbColor) );

	fprintf( f, "%d %d moveto ", x0, y0 );
	fprintf( f, "%d %d lineto ", x1, y0 );
	fprintf( f, "%d %d lineto ", x1, y1 );
	fprintf( f, "%d %d lineto ", x0, y1 );
	fprintf( f, "%d %d lineto closepath fill\n", x0, y0 );
	}

    if  ( dc->dcDrawBorders )
	{
	appMetaSetColorPs( f, &(dc->dcPen.lpColor) );

	fprintf( f, "%d %d moveto ", x0, y0 );
	fprintf( f, "%d %d lineto ", x1, y0 );
	fprintf( f, "%d %d lineto ", x1, y1 );
	fprintf( f, "%d %d lineto ", x0, y1 );
	fprintf( f, "%d %d lineto stroke\n", x0, y0 );
	}

    return 0;
    }

static int appMeta_LineToPs(	SimpleInputStream *	sis,
				FILE *			f,
				int			recordSize,
				DeviceContext *		dc )
    {
    int		x0= dc->dcX;
    int		y0= dc->dcY;
    int		y1;
    int		x1;

    y1= sioEndianGetLeInt16( sis );
    x1= sioEndianGetLeInt16( sis );

    WMFDEB(DEBFUN("LineTo( x:%d-> %d, y:%d-> %d )\n", x0, x1, y0, y1 ));
    WMFLOG(fprintf( f, "%% LineTo( x:%d-> %d, y:%d-> %d )\n", x0, x1, y0, y1 ));

    dc->dcX= x1;
    dc->dcY= y1;

    x0= DC_X( x0, dc );
    y0= DC_Y( y0, dc );
    x1= DC_X( x1, dc );
    y1= DC_Y( y1, dc );

    if  ( dc->dcDrawBorders )
	{
	appMetaSetColorPs( f, &(dc->dcPen.lpColor) );

	fprintf( f, "%d %d moveto ", x0, y0 );
	fprintf( f, "%d %d lineto stroke\n", x1, y1 );
	}

    return 0;
    }

static void appMetaShowStringPs(	FILE *			f,
					int			x0,
					int			y0,
					int			count,
					DeviceContext *		dc )
    {
    AfmFontInfo *	afi= dc->dcAfi;
    LogicalFont *	lf= &(dc->dcFont);
    int			fontSize= 10* lf->lfTextAttribute.taFontSizeHalfPoints;

    if  ( ! afi )
	{ XDEB(afi); return;	}

    y0 += ( fontSize* afi->afiAscender+ 500 )/ 1000;

    appMetaSetColorPs( f, &(dc->dcTextColor) );

    fprintf( f, "(" );
    appPsPrintString( f, dc->dcString, count );
    fprintf( f, ") %d %d mvs\n", x0, y0 );

    if  ( dc->dcFont.lfTextAttribute.taIsUnderlined )
	{
	int		width;
	int		h;

	AfmBBox		abb;

	const int	withKerning= 0;
	int		fontSizeTwips;

	fontSizeTwips= 10* lf->lfTextAttribute.taFontSizeHalfPoints;

	y0 -= (int)( ( fontSize* afi->afiUnderlinePosition+ 500 )/ 1000 );
	h= (int)( ( fontSize* afi->afiUnderlineThickness+ 500 )/ 1000 );

	width= psCalculateStringExtents( &abb, dc->dcString, count,
					fontSizeTwips, withKerning, afi );

	fprintf( f, "%d %d %d %d rectfill\n", x0, y0, width, h );
	}

    return;
    }

static int appMeta_ExtTextOutPs(	SimpleInputStream *	sis,
					FILE *			f,
					int			recordSize,
					DeviceContext *		dc )
    {
    int		x0;
    int		y0;
    int		count;
    int		style;

    int		h1;
    int		w1;
    int		y1;
    int		x1;

    if  ( appMeta_ExtTextOut( sis, recordSize, &x0, &y0, &count, &style,
						&y1, &y1, &w1, &h1, dc ) )
	{ LDEB(1); return -1;	}

    x0= DC_X( x0, dc );
    y0= DC_Y( y0, dc );

    x1= DC_X( x1, dc );
    y1= DC_Y( y1, dc );

    w1= DC_W( w1, dc );
    h1= DC_H( h1, dc );

    appMetaShowStringPs( f, x0, y0, count, dc );

    return 0;
    }

static int appMeta_TextOutPs(	SimpleInputStream *	sis,
				FILE *			f,
				int			recordSize,
				DeviceContext *		dc )
    {
    int		x0;
    int		y0;
    int		count;

    if  ( appMeta_TextOut( sis, recordSize, &x0, &y0, &count, dc ) )
	{ LDEB(1); return -1;	}

    x0= DC_X( x0, dc );
    y0= DC_Y( y0, dc );

    appMetaShowStringPs( f, x0, y0, count, dc );

    return 0;
    }

static int appMeta_PatBltPs(	SimpleInputStream *	sis,
				FILE *			f,
				int			recordSize,
				DeviceContext *		dc )
    {
    long	rop;

    int		h0;
    int		w0;
    int		y0;
    int		x0;

    int		x1;
    int		y1;

    int		swap;

    rop= sioEndianGetLeInt32( sis );
    h0= sioEndianGetLeInt16( sis );
    w0= sioEndianGetLeInt16( sis );
    y0= sioEndianGetLeInt16( sis );
    x0= sioEndianGetLeInt16( sis );

    WMFDEB(DEBFUN("PatBlt([%d+%d,%d+%d], rop=0x%lx )\n", x0,w0,y0,h0, rop ));

    x1= x0+ w0;
    y1= y0+ h0;

    x0= DC_X( x0, dc );
    y0= DC_Y( y0, dc );

    x1= DC_X( x1, dc );
    y1= DC_Y( y1, dc );

    if  ( x1 < x0 )
	{ swap= x0; x0= x1; x1= swap; }
    if  ( y1 < y0 )
	{ swap= y0; y0= y1; y1= swap; }

    if  ( dc->dcFillInsides )
	{
	appMetaSetColorPs( f, &(dc->dcBrush.lbColor) );

	fprintf( f, "%d %d moveto ", x0, y0 );
	fprintf( f, "%d %d lineto ", x1, y0 );
	fprintf( f, "%d %d lineto ", x1, y1 );
	fprintf( f, "%d %d lineto ", x0, y1 );
	fprintf( f, "%d %d lineto ", x0, y0 );
	fprintf( f, "closepath fill\n" );
	}

    return 0;
    }

static void appMetaGetArcPs(	SimpleInputStream *	sis,
				DeviceContext *		dc,
				char *			fun,
				int *			pY0,
				int *			pX0,
				int *			pY1,
				int *			pX1,
				int *			pYs,
				int *			pXs,
				double *		pas,
				double *		pae )

    {
    int		y0;
    int		x0;
    int		y1;
    int		x1;

    int		ys;
    int		xs;
    int		ye;
    int		xe;

    int		ym;
    int		xm;
    int		h;
    int		w;

    double	as;
    double	ae;

    int		swap;

    ye= sioEndianGetLeInt16( sis );
    xe= sioEndianGetLeInt16( sis );
    ys= sioEndianGetLeInt16( sis );
    xs= sioEndianGetLeInt16( sis );

    y1= sioEndianGetLeInt16( sis );
    x1= sioEndianGetLeInt16( sis );
    y0= sioEndianGetLeInt16( sis );
    x0= sioEndianGetLeInt16( sis );

    WMFDEB(DEBFUN("%s(%d..%d,%d..%d, (%d,%d), (%d,%d)\n", fun,
					    x0,x1,y0,y1, xs,ys,xe,ye));
    y0= DC_Y( y0, dc );
    x0= DC_X( x0, dc );
    y1= DC_Y( y1, dc );
    x1= DC_X( x1, dc );

    ys= DC_Y( ys, dc );
    xs= DC_X( xs, dc );
    ye= DC_Y( ye, dc );
    xe= DC_X( xe, dc );

    ym= ( y1+ y0 )/2;
    xm= ( x1+ x0 )/2;

    if  ( x1 < x0 )
	{
	xs= 2* xm- xs;
	xe= 2* xm- xe;
	swap= x0; x0= x1; x1= swap;
	}
    if  ( y1 < y0 )
	{
	ys= 2* ym- ys;
	ye= 2* ym- ye;
	swap= y0; y0= y1; y1= swap;
	}

    h= y1- y0;
    w= x1- x0;

    as= -atan2(	(double) w* ( ys- ym ), (double) h* ( xs- xm ) );
    ae= -atan2(	(double) w* ( ye- ym ), (double) h* ( xe- xm ) );

    xs= (int)( ( ( x1- x0 )/ 2 )* cos( as )+ xm );
    ys= (int)( ( ( x1- x0 )/ 2 )* sin( as ) );

    as= ( 180* as )/ M_PI;
    ae= ( 180* ae )/ M_PI;

    *pY0= y0;
    *pX0= x0;
    *pY1= y1;
    *pX1= x1;
    *pYs= ys;
    *pXs= xs;
    *pas= as;
    *pae= ae;

    return;
    }

static int appMeta_ArcPs(	SimpleInputStream *	sis,
				FILE *			f,
				int			recordSize,
				DeviceContext *		dc )
    {
    int		y0;
    int		x0;
    int		y1;
    int		x1;

    int		ys;
    int		xs;

    double	as;
    double	ae;

    appMetaGetArcPs( sis, dc, "Arc", 
			    &y0, &x0, &y1, &x1, &ys, &xs, &as, &ae );

    fprintf( f, "gsave [1 0 0 %g 0 %d] concat\n",
			    (double)(y0- y1)/(double)(x1- x0), ( y0+ y1 )/ 2 );

    if  ( dc->dcDrawBorders )
	{
	appMetaSetColorPs( f, &(dc->dcPen.lpColor) );

	fprintf( f, "%d %d moveto ", xs, ys );

	fprintf( f, "%d 0 %d %f %f arc stroke\n",
					( x0+ x1 )/ 2, (x1- x0)/2, as, ae );
	}

    fprintf( f, "grestore\n" );

    return 0;
    }

static int appMeta_PiePs(	SimpleInputStream *	sis,
				FILE *			f,
				int			recordSize,
				DeviceContext *		dc )
    {
    int		y0;
    int		x0;
    int		y1;
    int		x1;

    int		ys;
    int		xs;

    double	as;
    double	ae;

    appMetaGetArcPs( sis, dc, "Pie", 
			    &y0, &x0, &y1, &x1, &ys, &xs, &as, &ae );

    fprintf( f, "gsave [1 0 0 %g 0 %d] concat\n",
			    (double)(y0- y1)/(double)(x1- x0), ( y0+ y1 )/ 2 );

    if  ( dc->dcFillInsides )
	{
	appMetaSetColorPs( f, &(dc->dcBrush.lbColor) );

	fprintf( f, "%d %d moveto ", xs, ys );

	fprintf( f, "%d 0 %d %f %f arc\n",
					( x0+ x1 )/ 2, (x1- x0)/2, as, ae );
	fprintf( f, "%d  0 lineto ", ( x0+ x1 )/ 2 );
	fprintf( f, "%d %d lineto closepath fill\n", xs, ys );
	}

    if  ( dc->dcDrawBorders )
	{
	appMetaSetColorPs( f, &(dc->dcPen.lpColor) );

	fprintf( f, "%d %d moveto ", xs, ys );

	fprintf( f, "%d 0 %d %f %f arc ",
					( x0+ x1 )/ 2, (x1- x0)/2, as, ae );
	fprintf( f, "%d  0 lineto ", ( x0+ x1 )/ 2 );
	fprintf( f, "%d %d lineto stroke\n", xs, ys );
	}

    fprintf( f, "grestore\n" );

    return 0;
    }

static int appMeta_EllipsePs(	SimpleInputStream *	sis,
				FILE *			f,
				DeviceContext *		dc )
    {
    int		y0;
    int		x0;
    int		y1;
    int		x1;

    int		swap;

    y1= sioEndianGetLeInt16( sis );
    x1= sioEndianGetLeInt16( sis );
    y0= sioEndianGetLeInt16( sis );
    x0= sioEndianGetLeInt16( sis );

    WMFDEB(DEBFUN("Ellipse(%d..%d,%d..%d)\n", x0,x1,y0,y1));

    x0= DC_X( x0, dc );
    y0= DC_Y( y0, dc );
    x1= DC_X( x1, dc );
    y1= DC_Y( y1, dc );

    if  ( x1 < x0 )
	{ swap= x0; x0= x1; x1= swap; }
    if  ( y1 < y0 )
	{ swap= y0; y0= y1; y1= swap; }

    fprintf( f, "gsave [1 0 0 %g 0 %d] concat\n",
			    (double)(y1- y0)/(double)(x1- x0), ( y0+ y1 )/ 2 );

    if  ( dc->dcFillInsides )
	{
	appMetaSetColorPs( f, &(dc->dcBrush.lbColor) );

	fprintf( f, "%d 0 moveto ", x1 );
	fprintf( f, "%d 0 %d 0 360 arc closepath fill\n",
					    ( x0+ x1 )/ 2, (x1- x0)/2 );
	}

    if  ( dc->dcDrawBorders )
	{
	appMetaSetColorPs( f, &(dc->dcPen.lpColor) );

	fprintf( f, "%d 0 moveto ", x1 );
	fprintf( f, "%d 0 %d 0 360 arc stroke\n",
					    ( x0+ x1 )/ 2, (x1- x0)/2 );
	}

    fprintf( f, "grestore\n" );

    return 0;
    }

static int appMeta_RoundRectPs(	SimpleInputStream *	sis,
				FILE *			f,
				int			recordSize,
				DeviceContext *		dc )
    {
    int		y0;
    int		x0;
    int		y1;
    int		x1;

    int		h;
    int		w;

    int		swap;

    int		top;
    int		bottom;
    int		radius;

    h= sioEndianGetLeInt16( sis );
    w= sioEndianGetLeInt16( sis );

    y1= sioEndianGetLeInt16( sis );
    x1= sioEndianGetLeInt16( sis );
    y0= sioEndianGetLeInt16( sis );
    x0= sioEndianGetLeInt16( sis );

    WMFDEB(DEBFUN("RoundRect(%d..%d,%d..%d, w= %d,h= %d\n",
					    x0,x1,y0,y1, w, h ));

    y0= DC_Y( y0, dc );
    x0= DC_X( x0, dc );
    y1= DC_Y( y1, dc );
    x1= DC_X( x1, dc );

    h= DC_H( h, dc );
    w= DC_W( w, dc );

    if  ( x1 < x0 )
	{ swap= x0; x0= x1; x1= swap; }
    if  ( y1 < y0 )
	{ swap= y0; y0= y1; y1= swap; }

    if  ( h < 0 )
	{ h= -h;	}
    if  ( w < 0 )
	{ w= -w;	}

    top=    w* ( y1- y0 )/ ( 2* h );
    bottom= w* ( y0- y1 )/ ( 2* h );
    radius= w/ 2;

    fprintf( f, "gsave [1 0 0 %g 0 %d] concat\n",
					    (double)h/ w, ( y0+ y1 )/ 2 );

    if  ( dc->dcFillInsides )
	{
	appMetaSetColorPs( f, &(dc->dcBrush.lbColor) );

	fprintf( f, "%d %d moveto\n", x0, bottom+ radius );
	fprintf( f, "%d %d %d %d %d arct\n",
				    x0, top, x0+ radius, top, radius );
	fprintf( f, "%d %d %d %d %d arct\n",
				    x1, top, x1, top- radius, radius );
	fprintf( f, "%d %d %d %d %d arct\n",
				    x1, bottom, x1- radius, bottom, radius );
	fprintf( f, "%d %d %d %d %d arct closepath fill\n",
				    x0, bottom, x0, bottom+ radius, radius );
	}

    if  ( dc->dcDrawBorders )
	{
	appMetaSetColorPs( f, &(dc->dcPen.lpColor) );

	fprintf( f, "%d %d moveto\n", x0, bottom+ radius );
	fprintf( f, "%d %d %d %d %d arct\n",
				    x0, top, x0+ radius, top, radius );
	fprintf( f, "%d %d %d %d %d arct\n",
				    x1, top, x1, top- radius, radius );
	fprintf( f, "%d %d %d %d %d arct\n",
				    x1, bottom, x1- radius, bottom, radius );
	fprintf( f, "%d %d %d %d %d arct stroke\n",
				    x0, bottom, x0, bottom+ radius, radius );
	}

    fprintf( f, "grestore\n" );

    return 0;
    }

static int appMeta_PolyPolygonPs(	SimpleInputStream *	sis,
					FILE *			f,
					int			recordSize,
					DeviceContext *		dc )
    {
    int			count;
    int			i;

    int *		counts;

    if  ( appMeta_GetCounts( sis, recordSize, &count, &counts, dc ) )
	{ LDEB(1); return -1;	}

    WMFDEB(DEBFUN("PolyPolygon( count=%d, ... )\n", count ));

    for ( i= 0; i < count; i++ )
	{
	if  ( appMetaGetPointsPs( dc, counts[i], sis ) )
	    { LDEB(counts[i]); return -1;	}

	appMetaDrawPolygonPs( f, dc, counts[i] );
	}

    return 0;
    }

int appMetaPlayFilePs(	FILE *			f,
			SimpleInputStream *	sis,
			DocumentFontList *	dfl,
			const char *		afmDirectory,
			int			xExt,
			int			yExt,
			int			twipsWide,
			int			twipsHigh )
    {
    int			ob;

    {
    int		fileType;
    int		headerSize;
    int		windowsVersion;
    long		fileSize;
    int		objectCount;
    long		maxRecordSize;
    int		parameterCount;

    DeviceContext	dc;

    fileType= sioEndianGetLeInt16( sis );
    headerSize= sioEndianGetLeInt16( sis );
    windowsVersion= sioEndianGetLeInt16( sis );
    fileSize= sioEndianGetLeInt32( sis );
    objectCount= sioEndianGetLeInt16( sis );
    maxRecordSize= sioEndianGetLeInt32( sis );
    parameterCount= sioEndianGetLeInt16( sis );

    if  ( appMetaInitDeviceContext( &dc, objectCount,
					xExt, yExt, twipsWide, twipsHigh ) )
	{ LDEB(objectCount); return -1;	}

    WMFDEB(DEBFUN( "PS METAFILE ...\n" ));

    for (;;)
	{
	long		rop;

	int		x0;
	int		y0;
	int		w0;
	int		h0;

	int		x1;
	int		y1;
	int		w1;
	int		h1;

	int		done;
	int		ignore;

	long		recordSize;
	int		function;

	recordSize= sioEndianGetLeInt32( sis );
	function= sioEndianGetLeInt16( sis );

	switch( function )
	    {
	    case WINMETA_OffsetWindowOrg:
		y0= sioEndianGetLeInt16( sis );
		x0= sioEndianGetLeInt16( sis );

		WMFDEB(DEBFUN("OffsetWindowOrg(yWinOrg+=%d,xWinOrg+=%d)\n",
							    y0,x0 ));
		dc.dcOrgY += y0;
		dc.dcOrgX += x0;

		continue;
	    case WINMETA_SetWindowOrg:
		y0= sioEndianGetLeInt16( sis );
		x0= sioEndianGetLeInt16( sis );

		WMFDEB(DEBFUN("SetWindowOrg(yWinOrg=%d,xWinOrg=%d)\n",
							    y0,x0 ));
		dc.dcOrgY= y0;
		dc.dcOrgX= x0;

		continue;
	    case WINMETA_SetWindowExt:
		y0= sioEndianGetLeInt16( sis );
		x0= sioEndianGetLeInt16( sis );

		WMFDEB(DEBFUN("SetWindowExt(yExt=%d,xExt=%d)\n", y0, x0 ));

		dc.dcExtY= y0;
		dc.dcExtX= x0;

		continue;
	    case WINMETA_SaveDC:
		WMFDEB(DEBFUN("SaveDC()\n"));
		continue;
	    case WINMETA_RestoreDC:
		ignore= sioEndianGetLeInt16( sis );
		WMFDEB(DEBFUN("RestoreDC(%d)\n",ignore));
		continue;
	    case WINMETA_SetMapperFlags:
		rop= sioEndianGetLeInt32( sis );
		WMFDEB(DEBFUN("WINMETA_SetMapperFlags(0x%lx)\n",rop));
		continue;
	    case WINMETA_IntersectClipRect:
		h0= sioEndianGetLeInt16( sis );
		w0= sioEndianGetLeInt16( sis );
		y0= sioEndianGetLeInt16( sis );
		x0= sioEndianGetLeInt16( sis );
		WMFDEB(DEBFUN("IntersectClipRect([%d+%d,%d+%d])\n",
							    x0,w0,y0,h0 ));
		continue;
	    case WINMETA_StretchBlt:
		rop= sioEndianGetLeInt32( sis );
		h0= sioEndianGetLeInt16( sis );
		w0= sioEndianGetLeInt16( sis );
		y0= sioEndianGetLeInt16( sis );
		x0= sioEndianGetLeInt16( sis );
		h1= sioEndianGetLeInt16( sis );
		w1= sioEndianGetLeInt16( sis );
		y1= sioEndianGetLeInt16( sis );
		x1= sioEndianGetLeInt16( sis );

		WMFDEB(DEBFUN("StretchBlt([%d+%d,%d+%d]->[%d+%d,%d+%d],..)\n",
				x0,w0,y0,h0, x1,w1,y1,h1 ));

		if  ( appMetaBitmapImagePs( sis, f, 2*(recordSize-3-2-8*1),
					&dc, x0, y0, w0, h0, x1, y1, w1, h1 ) )
		    { LDEB(1); return -1; }

		continue;
	    case WINMETA_StretchDIBits:
		rop= sioEndianGetLeInt32( sis );
		ignore= sioEndianGetLeInt16( sis );
		/*  source	*/
		h0= sioEndianGetLeInt16( sis );
		w0= sioEndianGetLeInt16( sis );
		y0= sioEndianGetLeInt16( sis );
		x0= sioEndianGetLeInt16( sis );
		/*  destination	*/
		h1= sioEndianGetLeInt16( sis );
		w1= sioEndianGetLeInt16( sis );
		y1= sioEndianGetLeInt16( sis );
		x1= sioEndianGetLeInt16( sis );

		WMFDEB(DEBFUN(
			"StretchDIBits([%d+%d,%d+%d]->[%d+%d,%d+%d],..)\n",
			x0,w0,y0,h0, x1,w1,y1,h1 ));

		if  ( appMetaBitmapImagePs( sis, f, 2*(recordSize-3-2-9*1),
					&dc, x0, y0, w0, h0, x1, y1, w1, h1 ) )
		    { LDEB(1); return -1; }
		continue;

	    case WINMETA_SelectObject:
		if  ( appMetaSelectObjectPs( &dc, dfl, afmDirectory, f,
							recordSize, sis ) )
		    { LDEB(recordSize); return -1;	}
		continue;

	    case WINMETA_CreateBrushIndirect:
		if  ( appMetaCreateBrushIndirect( &dc, recordSize, sis ) )
		    { LDEB(recordSize); return -1;	}
		continue;

	    case WINMETA_CreatePenIndirect:
		if  ( appMetaCreatePenIndirect( &dc, recordSize, sis ) )
		    { LDEB(recordSize); return -1;	}
		continue;

	    case WINMETA_CreatePatternBrush:
		if  ( appMetaCreatePatternBrush( &dc, recordSize, sis ) )
		    { LDEB(recordSize); return -1;	}
		continue;

	    case WINMETA_CreatePalette:
		if  ( appMetaCreatePalette( &dc, recordSize, sis ) )
		    { LDEB(recordSize); return -1;	}
		continue;

	    case WINMETA_CreateFontIndirect:
		if  ( appMetaCreateFontIndirect( &dc, dfl, recordSize, sis ) )
		    { LDEB(1); return -1;	}
		continue;

	    case WINMETA_DeleteObject:

		ob= sioEndianGetLeInt16( sis );

		if  ( appMetaDeleteObject( &dc, ob ) )
		    { LDEB(1); return -1;	}
		continue;

	    case WINMETA_SetBkColor:
		appMetaGetColor( sis, &(dc.dcBkColor) );

		WMFDEB(DEBFUN("SetBkColor( r=%d, g=%d, b= %d )\n",
			    dc.dcBkColor.rgb8Red,
			    dc.dcBkColor.rgb8Green,
			    dc.dcBkColor.rgb8Blue ));
		continue;

	    case WINMETA_SetTextColor:
		appMetaGetColor( sis, &(dc.dcTextColor) );

		WMFDEB(DEBFUN("SetTextColor( r=%d, g=%d, b= %d )\n",
			    dc.dcBkColor.rgb8Red,
			    dc.dcBkColor.rgb8Green,
			    dc.dcBkColor.rgb8Blue ));
		continue;

	    case WINMETA_SetBkMode:
		WMFDEB(DEBFUN("SetBkMode(...)\n"));
		if  ( recordSize == 4 )
		    {
		    dc.dcBkMode= sioEndianGetLeInt16( sis );
		    continue;
		    }
		if  ( recordSize == 5 )
		    {
		    dc.dcBkMode= sioEndianGetLeInt32( sis );
		    continue;
		    }
		XLDEB(function,recordSize); return -1;

	    case WINMETA_SetTextCharExtra:
		WMFDEB(DEBFUN("SetTextCharExtra(...)\n"));
		dc.dcExtraTextSpacing= sioEndianGetLeInt16( sis );
		continue;

	    case WINMETA_SetTextAlign:
		WMFDEB(DEBFUN("SetTextAlign(...)\n"));
		if  ( recordSize == 4 )
		    {
		    dc.dcTextAlignment= sioEndianGetLeInt16( sis );
		    continue;
		    }
		if  ( recordSize == 5 )
		    {
		    dc.dcTextAlignment= sioEndianGetLeInt32( sis );
		    continue;
		    }
		XLDEB(function,recordSize); return -1;

	    case WINMETA_SetTextJustification:
		WMFDEB(DEBFUN("SetTextJustification(...)\n"));

		dc.dcJustificationAmount= sioEndianGetLeInt16( sis );
		dc.dcJustificationSpaces= sioEndianGetLeInt16( sis );
		continue;

	    case WINMETA_SetStretchBltMode:
		switch( recordSize )
		    {
		    case 4:
			dc.dcStretchBltMode= sioEndianGetLeInt16( sis );
			break;
		    case 5:
			dc.dcStretchBltMode= sioEndianGetLeInt32( sis );
			break;
		    default:
			LDEB(recordSize); return -1;
		    }
		WMFDEB(DEBFUN("SetStretchBltMode(%ld)\n",dc.dcStretchBltMode));
		continue;

	    case WINMETA_PolyLine:
		if  ( appMeta_PolyLinePs( sis, f, recordSize, &dc ) )
		    { LDEB(1); return -1;	}
		continue;

	    case WINMETA_Polygon:
		if  ( appMeta_PolygonPs( sis, f, recordSize, &dc ) )
		    { LDEB(1); return -1;	}
		continue;

	    case WINMETA_Rectangle:
		if  ( appMeta_RectanglePs( sis, f, recordSize, &dc ) )
		    { LDEB(1); return -1;	}
		continue;

	    case WINMETA_LineTo:
		if  ( appMeta_LineToPs( sis, f, recordSize, &dc ) )
		    { LDEB(1); return -1;	}
		continue;

	    case WINMETA_MoveTo:
		y0= sioEndianGetLeInt16( sis );
		x0= sioEndianGetLeInt16( sis );

		WMFDEB(DEBFUN("MoveTo( x=%d, y=%d )\n", x0, y0 ));

		dc.dcX= x0;
		dc.dcY= y0;

		continue;
	    case WINMETA_ExtTextOut:
		if  ( appMeta_ExtTextOutPs( sis, f, recordSize, &dc ) )
		    { LDEB(recordSize); return -1;	}
		continue;

	    case WINMETA_TextOut:
		if  ( appMeta_TextOutPs( sis, f, recordSize, &dc ) )
		    { LDEB(recordSize); return -1;	}
		continue;

	    case WINMETA_Escape:
		WMFDEB(DEBFUN("Escape(...)\n"));
		goto skipArguments;

	    case WINMETA_PatBlt:
		if  ( appMeta_PatBltPs( sis, f, recordSize, &dc ) )
		    { LDEB(recordSize); return -1;	}
		continue;

	    case WINMETA_SetPolyFillMode:
		WMFDEB(DEBFUN("SetPolyFillMode(...)\n"));
		goto skipArguments;

	    case WINMETA_Pie:
		if  ( appMeta_PiePs( sis, f, recordSize, &dc ) )
		    { LDEB(1); return -1;	}
		continue;

	    case WINMETA_Arc:
		if  ( appMeta_ArcPs( sis, f, recordSize, &dc ) )
		    { LDEB(1); return -1;	}
		continue;

	    case WINMETA_Ellipse:
		if  ( appMeta_EllipsePs( sis, f, &dc ) )
		    { LDEB(1); return -1;	}
		continue;

	    case WINMETA_RoundRect:
		if  ( appMeta_RoundRectPs( sis, f, recordSize, &dc ) )
		    { LDEB(1); return -1;	}
		continue;

	    case WINMETA_SetMapMode:
		dc.dcMapMode= sioEndianGetLeInt16( sis );
		WMFDEB(DEBFUN("SetMapMode(%d)\n", dc.dcMapMode ));
		continue;

	    case WINMETA_SetROP2:
		if  ( recordSize == 5 )
		    {
		    dc.dcROP2= sioEndianGetLeInt32( sis );
		    WMFDEB(DEBFUN("SetROP2(%ld)\n", (long)dc.dcROP2 ));
		    continue;
		    }
		if  ( recordSize == 4 )
		    {
		    dc.dcROP2= sioEndianGetLeInt16( sis );
		    WMFDEB(DEBFUN("SetROP2(%ld)\n", (long)dc.dcROP2 ));
		    continue;
		    }
		XLDEB(function,recordSize); return -1;
	    case WINMETA_SelectPalette:
		ob= sioEndianGetLeInt16( sis );
		WMFDEB(DEBFUN("SelectPalette(%d)\n", ob ));
		continue;
	    case WINMETA_RealizePalette:
		WMFDEB(DEBFUN("RealizePalette()\n"));
		continue;
	    case WINMETA_PolyPolygon:
		if  ( appMeta_PolyPolygonPs( sis, f, recordSize, &dc ) )
		    { LDEB(1); return -1;	}
		continue;

	    case WINMETA_ScaleViewportExt:
		x0= sioEndianGetLeInt16( sis );
		y0= sioEndianGetLeInt16( sis );
		x1= sioEndianGetLeInt16( sis );
		y1= sioEndianGetLeInt16( sis );

		WMFDEB(DEBFUN("ScaleViewportExt(%d..%d,%d..%d)\n",
								x0,x1,y0,y1));
		break;
	    case 0:
		if  ( recordSize != 3 )
		    { XLDEB(function,recordSize); return -1;	}
		break;
	    skipArguments:
		for ( done= 3; done < recordSize; done++ )
		    { ignore= sioEndianGetLeInt16( sis ); }
		continue;
	    default:
		XLDEB(function,recordSize);
		/*
		for ( done= 3; done < recordSize; done++ )
		    { ignore= sioEndianGetLeInt16( sis ); }
		continue;
		*/
		return -1;
	    }

	break;
	}

    appMetaCleanDeviceContext( &dc );
    }

    return 0;
    }

/************************************************************************/
/*									*/
/*  List the fonts in a windows metafile.				*/
/*									*/
/*  The geometry parameters are absolutely irrelevant, but required by	*/
/*  appMetaInitDeviceContext().						*/
/*									*/
/************************************************************************/

int appMetaListFontsPs( SimpleInputStream *	sis,
			DocumentFontList *	dfl,
			const char *		afmDirectory,
			PostScriptFont **	pFontList,
			int *			pCount,
			int			xExt,
			int			yExt,
			int			twipsWide,
			int			twipsHigh )
    {
    int		fileType;
    int		headerSize;
    int		windowsVersion;
    long	fileSize;
    int		objectCount;
    long	maxRecordSize;
    int		parameterCount;

    DeviceContext	dc;

    fileType= sioEndianGetLeInt16( sis );
    headerSize= sioEndianGetLeInt16( sis );
    windowsVersion= sioEndianGetLeInt16( sis );
    fileSize= sioEndianGetLeInt32( sis );
    objectCount= sioEndianGetLeInt16( sis );
    maxRecordSize= sioEndianGetLeInt32( sis );
    parameterCount= sioEndianGetLeInt16( sis );

    if  ( appMetaInitDeviceContext( &dc, objectCount,
					xExt, yExt, twipsWide, twipsHigh ) )
	{ LDEB(objectCount); return -1;	}

    WMFDEB(DEBFUN( "PS METAFILE FONTS ...\n" ));

    for (;;)
	{
	int			done;
	int			ignore;

	long			recordSize;
	int			function;

	LogicalFont *		lf;
	int			family;
	int			face;

	AppFontFamily *		aff;
	AppFontTypeface *	aft;

	DocumentFont *		df;
	AfmFontInfo *		afi;

	recordSize= sioEndianGetLeInt32( sis );
	function= sioEndianGetLeInt16( sis );

	switch( function )
	    {
	    case WINMETA_SetWindowExt:
		dc.dcExtY= sioEndianGetLeInt16( sis );
		dc.dcExtX= sioEndianGetLeInt16( sis );
		continue;

	    case WINMETA_CreateFontIndirect:
		if  ( appMetaCreateFontIndirect( &dc, dfl, recordSize, sis ) )
		    { LDEB(1); return -1;	}

		lf= &(dc.dcObjects[0].mfoLogicalFont);

		df= dfl->dflFonts+ lf->lfTextAttribute.taFontNumber;

		if  ( appGetPsFont( &family, &face, &aff, &aft,
				    afmDirectory, df, lf->lfTextAttribute ) )
		    { LDEB(lf->lfTextAttribute.taFontNumber); return -1; }

		afi= (AfmFontInfo *)aft->aftPrintingData;

		if  ( appPsRememberAfi( afi, lf->lfTextAttribute,
						pFontList, pCount ) )
		    { SDEB(afi->afiFontName); return -1;	}

		appMetaDeleteObject( &dc, 0 );

		continue;

	    default:
		for ( done= 3; done < recordSize; done++ )
		    { ignore= sioEndianGetLeInt16( sis ); }
		continue;

	    case 0:
		break;
	    }

	break;
	}

    appMetaCleanDeviceContext( &dc );

    return 0;
    }
