/************************************************************************/
/*									*/
/*  Utility routines for making remarks and posing questions.		*/
/*									*/
/************************************************************************/

#   include	"config.h"

#   include	<stdlib.h>
#   include	<stdio.h>
#   include	<debugon.h>

#   include	<Xm/DialogS.h>
#   include	<X11/Xatom.h>
#   include	<X11/IntrinsicP.h>
#   include	<Xm/Form.h>
#   include	<Xm/PanedWP.h>
#   include	<Xm/Label.h>

#   include	<appFrame.h>
#   include	<appUtil.h>

typedef struct QuestionContext
    {
    Widget	qcQuestionWidget;
    Widget	qcSubjectWidget;

    Widget	qcYesButton;
    Widget	qcNoButton;
    Widget	qcCancelButton;
    Widget	qcOkButton;

    char *	qcYesText;
    char *	qcNoText;
    char *	qcCancelText;
    char *	qcOkText;

    int		qcResponse;
    } QuestionContext;

# define USE_POPUP

/************************************************************************/
/*  Resource table for various button texts.				*/
/************************************************************************/
# define xx(x)	x,x

static XtResource APP_QuestionResourceTable[]=
    {
	{ xx("messageYesText"), XtRString, sizeof(char *),
		    offsetof(QuestionContext,qcYesText),
		    XtRString, "Yes" },
	{ xx("messageNoText"), XtRString, sizeof(char *),
		    offsetof(QuestionContext,qcNoText),
		    XtRString, "No" },
	{ xx("messageCancelText"), XtRString, sizeof(char *),
		    offsetof(QuestionContext,qcCancelText),
		    XtRString, "Cancel" },
	{ xx("messageOkText"), XtRString, sizeof(char *),
		    offsetof(QuestionContext,qcOkText),
		    XtRString, "Ok" },
    };

/************************************************************************/
/*  Make a Ok form with one default button.				*/
/*  Make a Yes/No/Cancel form with three buttons.			*/
/************************************************************************/

static Widget appQuestionMakeOkRow(	Widget		parent,
					Widget		dialog,
					const char *	okText )
    {
    QuestionContext *	qc= (QuestionContext *)0;
    Widget		buttonRow;

    XtVaGetValues( dialog,
			XmNuserData,	&qc,
			NULL );
    if  ( ! qc )
	{ XDEB(qc); return (Widget)0;	}

    buttonRow= appMakeButtonRow( parent, 1 );

    qc->qcOkButton= appMakeRowButton( buttonRow, okText,
				    (XtCallbackProc)0, (void *)0, 0, True );

    XtManageChild( buttonRow );

    return buttonRow;
    }

static Widget appQuestionMakeYesNoCancelRow(	Widget		parent,
						Widget		dialog,
						char *		yesText,
						char *		noText,
						char *		cancelText )
    {
    QuestionContext *	qc= (QuestionContext *)0;

    Widget		buttonRow;

    XtVaGetValues( dialog,
			XmNuserData,	&qc,
			NULL );
    if  ( ! qc )
	{ XDEB(qc); return (Widget)0;	}

    buttonRow= appMakeButtonRow( parent, 3 );

    qc->qcYesButton= appMakeRowButton( buttonRow, yesText,
				    (XtCallbackProc)0, (void *)0, 0, True );

    qc->qcNoButton= appMakeRowButton( buttonRow, noText,
				    (XtCallbackProc)0, (void *)0, 1, False );

    qc->qcCancelButton= appMakeRowButton( buttonRow, cancelText,
				    (XtCallbackProc)0, (void *)0, 2, False );

    XtVaSetValues( dialog,
			XmNdefaultButton,	qc->qcYesButton,
			NULL );
    XtVaSetValues( dialog,
			XmNcancelButton,	qc->qcCancelButton,
			NULL );

    XtManageChild( buttonRow );

    return buttonRow;
    }

static Widget appQuestionMakeOkCancelRow(	Widget		parent,
						Widget		dialog,
						char *		okText,
						char *		cancelText )
    {
    QuestionContext *	qc= (QuestionContext *)0;

    Widget		buttonRow;

    XtVaGetValues( dialog,
			XmNuserData,	&qc,
			NULL );
    if  ( ! qc )
	{ XDEB(qc); return (Widget)0;	}

    buttonRow= appMakeButtonRow( parent, 2 );

    qc->qcOkButton= appMakeRowButton( buttonRow, okText,
				    (XtCallbackProc)0, (void *)0, 0, False );

    qc->qcCancelButton= appMakeRowButton( buttonRow, cancelText,
				    (XtCallbackProc)0, (void *)0, 1, False );

    XtVaSetValues( dialog,
			XmNdefaultButton,	qc->qcOkButton,
			NULL );
    XtVaSetValues( dialog,
			XmNcancelButton,	qc->qcCancelButton,
			NULL );

    XtManageChild( buttonRow );

    return buttonRow;
    }


static Widget appQuestionMakeYesNoRow(	Widget		parent,
					Widget		dialog,
					char *		yesText,
					char *		noText )
    {
    QuestionContext *	qc= (QuestionContext *)0;

    Widget		buttonRow;

    XtVaGetValues( dialog,
			XmNuserData,	&qc,
			NULL );
    if  ( ! qc )
	{ XDEB(qc); return (Widget)0;	}

    buttonRow= appMakeButtonRow( parent, 2 );

    qc->qcYesButton= appMakeRowButton( buttonRow, yesText,
				    (XtCallbackProc)0, (void *)0, 0, False );

    qc->qcNoButton= appMakeRowButton( buttonRow, noText,
				    (XtCallbackProc)0, (void *)0, 1, False );

    XtVaSetValues( dialog,
			XmNdefaultButton,	qc->qcYesButton,
			NULL );
    XtVaSetValues( dialog,
			XmNcancelButton,	qc->qcNoButton,
			NULL );

    XtManageChild( buttonRow );

    return buttonRow;
    }

/************************************************************************/
/*  Make the framework of a Question Dialog.				*/
/************************************************************************/
static void appFreeQuestionContext(	Widget			dialog,
					XtPointer		voidqc,
					XtPointer		callData )
    { free( voidqc ); return;	}


static Widget appMakeQuestionDialog(	EditApplication *	ea,
					Widget			relative,
					Widget *		pPaned,
					Widget *		pTextPart )
    {
    Widget		shell;
    Widget		dialog;
    Widget		paned;
    Widget		textPart;

    Arg			al[20];
    int			ac= 0;

    const int		withSeparator= 1;

    QuestionContext *	qc;

    qc= (QuestionContext *)malloc( sizeof(QuestionContext) );
    if  ( ! qc )
	{ XDEB(qc); return (Widget)0;	}

    qc->qcQuestionWidget= (Widget)0;
    qc->qcSubjectWidget= (Widget)0;
    qc->qcYesButton= (Widget)0;
    qc->qcNoButton= (Widget)0;
    qc->qcCancelButton= (Widget)0;
    qc->qcOkButton= (Widget)0;

    XtGetApplicationResources( ea->eaTopWidget, (void *)qc,
		APP_QuestionResourceTable, XtNumber(APP_QuestionResourceTable),
		NULL, 0 );

#   ifdef USE_POPUP
    appMakeVerticalPopup( &shell, &dialog, &paned, ea,
			(void *)qc, withSeparator, ea->eaMessageDialogName );
#   else
    appMakeVerticalDialog( &shell, &dialog, &paned,
			ea, (XtCallbackProc)0, (void *)qc, (Pixmap)None,
			withSeparator, ea->eaMessageDialogName );
#   endif

    XtAddCallback( dialog, XmNdestroyCallback,
				    appFreeQuestionContext, (void *)qc );

    ac= 0;
    XtSetArg( al[ac],	XmNsashWidth,		1 ); ac++;
    XtSetArg( al[ac],	XmNsashHeight,		1 ); ac++;
    XtSetArg( al[ac],	XmNseparatorOn,		False ); ac++;
    XtSetArg( al[ac],	XmNmarginWidth,		5 ); ac++;
    XtSetArg( al[ac],	XmNmarginHeight,	5 ); ac++;
    XtSetArg( al[ac],	XmNspacing,		5 ); ac++;
    XtSetArg( al[ac],	XmNallowResize,		True ); ac++;
    textPart= XmCreatePanedWindow( paned, WIDGET_NAME, al, ac );

    *pPaned= paned;
    *pTextPart= textPart;

    return dialog;
    }

/************************************************************************/
/*									*/
/*  Insert a subject and a question in a form.				*/
/*									*/
/************************************************************************/

static void appQuestionMakeSubjectQuestion(	Widget		dialog,
						Widget		textPart,
						const char *	subject,
						const char *	question )
    {
    Arg				al[20];
    int				ac= 0;

    QuestionContext *		qc= (QuestionContext *)0;

    XmString			subjectString;
    XmString			questionString;

    XtVaGetValues( dialog,
			XmNuserData,	&qc,
			NULL );

    if  ( ! qc )
	{ XDEB(qc); return;	}

    subjectString= XmStringCreateLocalized( (char *)subject );
    questionString= XmStringCreateLocalized( (char *)question );

    ac= 0;
    XtSetArg( al[ac],	XmNlabelString,		subjectString ); ac++;
    XtSetArg( al[ac],	XmNalignment,		XmALIGNMENT_BEGINNING ); ac++;
    XtSetArg( al[ac],	XmNrecomputeSize,	True ); ac++;
    XtSetArg( al[ac],	XmNallowResize,		True ); ac++;
    XtSetArg( al[ac],	XmNskipAdjust,		True ); ac++;
    qc->qcSubjectWidget= XmCreateLabel( textPart, WIDGET_NAME, al, ac );

    ac= 0;
    XtSetArg( al[ac],	XmNlabelString,		questionString ); ac++;
    XtSetArg( al[ac],	XmNalignment,		XmALIGNMENT_BEGINNING ); ac++;
    XtSetArg( al[ac],	XmNrecomputeSize,	True ); ac++;
    XtSetArg( al[ac],	XmNallowResize,		True ); ac++;
    XtSetArg( al[ac],	XmNskipAdjust,		True ); ac++;
    qc->qcQuestionWidget= XmCreateLabel( textPart, WIDGET_NAME, al, ac );

    XmStringFree( subjectString );
    XmStringFree( questionString );

    XtManageChild( qc->qcSubjectWidget );
    XtManageChild( qc->qcQuestionWidget );

    return;
    }

/************************************************************************/
/*									*/
/*  Insert a subject and a question in a form.				*/
/*									*/
/************************************************************************/
static void appQuestionMakeQuestion(		Widget		dialog,
						Widget		textPart,
						const char *	question )
    {
    Arg				al[20];
    int				ac= 0;

    QuestionContext *		qc= (QuestionContext *)0;

    XmString			questionString;

    XtVaGetValues( dialog,
			XmNuserData,	&qc,
			NULL );
    if  ( ! qc )
	{ XDEB(qc); return;	}

    questionString= XmStringCreateLocalized( (char *)question );

    ac= 0;
    XtSetArg( al[ac],	XmNlabelString,		questionString ); ac++;
    XtSetArg( al[ac],	XmNalignment,		XmALIGNMENT_BEGINNING ); ac++;
    XtSetArg( al[ac],	XmNrecomputeSize,	True ); ac++;
    XtSetArg( al[ac],	XmNallowResize,		True ); ac++;
    XtSetArg( al[ac],	XmNskipAdjust,		True ); ac++;
    qc->qcQuestionWidget= XmCreateLabel( textPart, WIDGET_NAME, al, ac );

    XmStringFree( questionString );

    XtManageChild( qc->qcQuestionWidget );

    return;
    }

/************************************************************************/
/*									*/
/*  Change the question/subject of a question dialog.			*/
/*									*/
/************************************************************************/

static void appQuestionChangeString(	Widget		labelWidget,
					const char *	label )
    {
    XmString			labelString;

    labelString= XmStringCreateLocalized( (char *)label );

    XtVaSetValues( labelWidget,
			XmNlabelString,		labelString,
			NULL );

    XmStringFree( labelString );
    
    return;
    }

static void appQuestionChangeSubject(	Widget		dialog,
					const char *	subject )
    {
    QuestionContext *		qc= (QuestionContext *)0;

    XtVaGetValues( dialog,
			XmNuserData,	&qc,
			NULL );

    if  ( ! qc || ! qc->qcSubjectWidget )
	{ XDEB(qc); return;	}

    appQuestionChangeString( qc->qcSubjectWidget, subject );
    }

static void appQuestionChangeQuestion(	Widget		dialog,
					const char *	question )
    {
    QuestionContext *		qc= (QuestionContext *)0;

    XtVaGetValues( dialog,
			XmNuserData,	&qc,
			NULL );

    if  ( ! qc || ! qc->qcQuestionWidget )
	{ XDEB(qc); return;	}

    appQuestionChangeString( qc->qcQuestionWidget, question );
    }

static void appQuestionChangeOkText(	Widget		dialog,
					const char *	okText )
    {
    XmString			okTextString;

    QuestionContext *		qc= (QuestionContext *)0;

    XtVaGetValues( dialog,
			XmNuserData,	&qc,
			NULL );

    if  ( ! qc || ! qc->qcOkButton )
	{ XDEB(qc); return;	}

    okTextString= XmStringCreateLocalized( (char *)okText );

    XtVaSetValues( qc->qcOkButton,
			XmNlabelString,		okTextString,
			NULL );

    XmStringFree( okTextString );
    }

static void appQuestionChangeYesText(	Widget		dialog,
					const char *	okText )
    {
    XmString			okTextString;

    QuestionContext *		qc= (QuestionContext *)0;

    XtVaGetValues( dialog,
			XmNuserData,	&qc,
			NULL );

    if  ( ! qc || ! qc->qcYesButton )
	{ XDEB(qc); return;	}

    okTextString= XmStringCreateLocalized( (char *)okText );

    XtVaSetValues( qc->qcYesButton,
			XmNlabelString,		okTextString,
			NULL );

    XmStringFree( okTextString );
    }

static void appQuestionChangeNoText(	Widget		dialog,
					const char *	okText )
    {
    XmString			okTextString;

    QuestionContext *		qc= (QuestionContext *)0;

    XtVaGetValues( dialog,
			XmNuserData,	&qc,
			NULL );

    if  ( ! qc || ! qc->qcNoButton )
	{ XDEB(qc); return;	}

    okTextString= XmStringCreateLocalized( (char *)okText );

    XtVaSetValues( qc->qcNoButton,
			XmNlabelString,		okTextString,
			NULL );

    XmStringFree( okTextString );
    }

static void appQuestionChangeCancelText(	Widget		dialog,
					const char *	okText )
    {
    XmString			okTextString;

    QuestionContext *		qc= (QuestionContext *)0;

    XtVaGetValues( dialog,
			XmNuserData,	&qc,
			NULL );

    if  ( ! qc || ! qc->qcCancelButton )
	{ XDEB(qc); return;	}

    okTextString= XmStringCreateLocalized( (char *)okText );

    XtVaSetValues( qc->qcCancelButton,
			XmNlabelString,		okTextString,
			NULL );

    XmStringFree( okTextString );
    }

/************************************************************************/
/*									*/
/*  Run a Yes/No/Cancel dialog.						*/
/*									*/
/************************************************************************/
static void appQuestionButtonPushed(	Widget		button,
					XtPointer	voidqc,
					XtPointer	call_data )
    {
    QuestionContext *		qc= (QuestionContext *)voidqc;

    qc->qcResponse= AQDrespFAILURE;

    if  ( button == qc->qcOkButton )
	{ qc->qcResponse= AQDrespOK; return;	}

    if  ( button == qc->qcYesButton )
	{ qc->qcResponse= AQDrespYES; return;	}

    if  ( button == qc->qcNoButton )
	{ qc->qcResponse= AQDrespNO; return;	}

    if  ( button == qc->qcCancelButton )
	{ qc->qcResponse= AQDrespCANCEL; return;	}

    LDEB(qc->qcResponse); return;
    }

static void appQuestionRunDialog(	EditApplication *	ea,
					QuestionContext *	qc,
					Widget			relative,
					Widget			option,
					Widget			dialog )
    {
    Widget		defaultButton= (Widget)0;

    XtVaGetValues( dialog,
			XmNdefaultButton,	&defaultButton,
			NULL );

    appSetFocusCallback( dialog );

    if  ( relative )
	{
	appDialogRelative( relative, dialog );
	appSetRelativeCallback( relative, dialog );
	}

#   ifdef USE_POPUP
    XtPopup( XtParent( dialog ), XtGrabExclusive );
#   else
    XtManageChild( dialog );
#   endif

    if  ( defaultButton )
	{ XmProcessTraversal( defaultButton, XmTRAVERSE_CURRENT ); }

    appSetShellTitle( XtParent( dialog ), option, ea->eaApplicationName );

    qc->qcResponse= AQDrespNONE;
    while( qc->qcResponse == AQDrespNONE )
	{ XtAppProcessEvent( ea->eaContext, XtIMAll ); }

#   ifdef USE_POPUP
    XtPopdown( XtParent( dialog ) );
#   else
    XtUnmanageChild( dialog );
#   endif

    return;
    }

static int appQuestionRunOkDialog(	EditApplication *	ea,
					Widget			relative,
					Widget			option,
					Widget			dialog )
    {
    QuestionContext *		qc= (QuestionContext *)0;

    XtVaGetValues( dialog,
			XmNuserData,	&qc,
			NULL );

    if  ( ! qc || ! qc->qcOkButton )
	{ XDEB(qc); return -1;	}

    XtAddCallback( qc->qcOkButton, XmNactivateCallback,
				    appQuestionButtonPushed, (void *)qc );

    appQuestionRunDialog( ea, qc, relative, option, dialog );

    XtRemoveCallback( qc->qcOkButton, XmNactivateCallback,
				    appQuestionButtonPushed, (void *)qc );

    return qc->qcResponse;
    }

static int appQuestionRunAnyYesNoCancelDialog(	EditApplication *	ea,
						Widget		dialog,
						Widget		relative,
						Widget		option,
						QuestionContext * qc,
						const char *	yesText,
						const char *	noText,
						const char *	cancelText )
    {
    appQuestionChangeYesText( dialog,
				    yesText?yesText:qc->qcYesText );
    appQuestionChangeNoText( dialog,
				    noText?noText:qc->qcNoText );
    appQuestionChangeCancelText( dialog,
				    cancelText?cancelText:qc->qcCancelText );

    XtAddCallback( qc->qcYesButton, XmNactivateCallback,
				    appQuestionButtonPushed, (void *)qc );
    XtAddCallback( qc->qcNoButton, XmNactivateCallback,
				    appQuestionButtonPushed, (void *)qc );
    XtAddCallback( qc->qcCancelButton, XmNactivateCallback,
				    appQuestionButtonPushed, (void *)qc );

    appQuestionRunDialog( ea, qc, relative, option, dialog );

    XtRemoveCallback( qc->qcYesButton, XmNactivateCallback,
				    appQuestionButtonPushed, (void *)qc );
    XtRemoveCallback( qc->qcNoButton, XmNactivateCallback,
				    appQuestionButtonPushed, (void *)qc );
    XtRemoveCallback( qc->qcCancelButton, XmNactivateCallback,
				    appQuestionButtonPushed, (void *)qc );

    return qc->qcResponse;
    }

static int appQuestionRunAnyOkCancelDialog(	EditApplication *	ea,
						Widget		dialog,
						Widget		relative,
						Widget		option,
						QuestionContext * qc,
						const char *	okText,
						const char *	cancelText )
    {
    appQuestionChangeOkText( dialog,
				    okText?okText:qc->qcOkText );
    appQuestionChangeCancelText( dialog,
				    cancelText?cancelText:qc->qcCancelText );

    XtAddCallback( qc->qcOkButton, XmNactivateCallback,
				    appQuestionButtonPushed, (void *)qc );
    XtAddCallback( qc->qcCancelButton, XmNactivateCallback,
				    appQuestionButtonPushed, (void *)qc );

    appQuestionRunDialog( ea, qc, relative, option, dialog );

    XtRemoveCallback( qc->qcOkButton, XmNactivateCallback,
				    appQuestionButtonPushed, (void *)qc );
    XtRemoveCallback( qc->qcCancelButton, XmNactivateCallback,
				    appQuestionButtonPushed, (void *)qc );

    return qc->qcResponse;
    }

static int appQuestionRunAnyYesNoDialog(	EditApplication *	ea,
						Widget		dialog,
						Widget		relative,
						Widget		option,
						QuestionContext * qc,
						const char *	yesText,
						const char *	noText )
    {
    appQuestionChangeYesText( dialog, yesText?yesText:qc->qcYesText );
    appQuestionChangeNoText( dialog, noText?noText:qc->qcNoText );

    XtAddCallback( qc->qcYesButton, XmNactivateCallback,
				    appQuestionButtonPushed, (void *)qc );
    XtAddCallback( qc->qcNoButton, XmNactivateCallback,
				    appQuestionButtonPushed, (void *)qc );

    appQuestionRunDialog( ea, qc, relative, option, dialog );

    XtRemoveCallback( qc->qcYesButton, XmNactivateCallback,
				    appQuestionButtonPushed, (void *)qc );
    XtRemoveCallback( qc->qcNoButton, XmNactivateCallback,
				    appQuestionButtonPushed, (void *)qc );

    return qc->qcResponse;
    }

int appQuestionRunSubjectYesNoCancelDialog(	EditApplication *	ea,
						Widget		relative,
						Widget		option,
						const char *	subject,
						const char *	question,
						const char *	yesText,
						const char *	noText,
						const char *	cancelText )
    {
    int			rval;
    QuestionContext *	qc= (QuestionContext *)0;

    static Widget	subjectYesNoCancelDialog;

    if  ( ! subjectYesNoCancelDialog )
	{
	Widget				paned;
	Widget				textPart;
	Widget				bottomForm;

	subjectYesNoCancelDialog= appMakeQuestionDialog( ea, relative,
							&paned, &textPart );

	appQuestionMakeSubjectQuestion( subjectYesNoCancelDialog, textPart,
							subject, question );

	XtVaGetValues( subjectYesNoCancelDialog,
			    XmNuserData,	&qc,
			    NULL );

	bottomForm= appQuestionMakeYesNoCancelRow( paned,
					subjectYesNoCancelDialog,
					qc->qcYesText,
					qc->qcNoText,
					qc->qcCancelText );

	XtManageChild( textPart );
	XtManageChild( paned );

#	ifdef USE_POPUP
	XtManageChild( subjectYesNoCancelDialog );
#	endif
	}
    else{
	XtVaGetValues( subjectYesNoCancelDialog,
			XmNuserData,	&qc,
			NULL );

	appQuestionChangeSubject( subjectYesNoCancelDialog, subject );
	appQuestionChangeQuestion( subjectYesNoCancelDialog, question );
	}

    rval= appQuestionRunAnyYesNoCancelDialog( ea, subjectYesNoCancelDialog,
					relative, option, qc,
					yesText, noText, cancelText );

    XtDestroyWidget( XtParent( subjectYesNoCancelDialog ) );
    subjectYesNoCancelDialog= (Widget )0;

    return rval;
    }

int appQuestionRunYesNoCancelDialog(	EditApplication *	ea,
					Widget		relative,
					Widget		option,
					const char *	question,
					const char *	yesText,
					const char *	noText,
					const char *	cancelText )
    {
    int			rval;
    QuestionContext *	qc= (QuestionContext *)0;

    static Widget	yesNoCancelDialog;

    if  ( ! yesNoCancelDialog )
	{
	Widget				paned;
	Widget				textPart;
	Widget				bottomForm;

	yesNoCancelDialog= appMakeQuestionDialog( ea, relative,
							    &paned, &textPart );

	appQuestionMakeQuestion( yesNoCancelDialog, textPart, question );

	XtVaGetValues( yesNoCancelDialog,
			    XmNuserData,	&qc,
			    NULL );

	bottomForm= appQuestionMakeYesNoCancelRow( paned,
					yesNoCancelDialog,
					qc->qcYesText,
					qc->qcNoText,
					qc->qcCancelText );

	XtManageChild( textPart );
	XtManageChild( paned );

#	ifdef USE_POPUP
	XtManageChild( yesNoCancelDialog );
#	endif
	}
    else{
	XtVaGetValues( yesNoCancelDialog,
			XmNuserData,	&qc,
			NULL );

	appQuestionChangeQuestion( yesNoCancelDialog, question );
	}

    rval= appQuestionRunAnyYesNoCancelDialog( ea,
					yesNoCancelDialog, relative, option,
					qc, yesText, noText, cancelText );

    XtDestroyWidget( XtParent( yesNoCancelDialog ) );
    yesNoCancelDialog= (Widget )0;

    return rval;
    }

void appQuestionRunSubjectErrorDialog(	EditApplication *	ea,
					Widget			relative,
					Widget			option,
					const char *		subject,
					const char *		message )
    {
    QuestionContext *	qc= (QuestionContext *)0;

    static Widget	subjectErrorDialog;

    if  ( ! subjectErrorDialog )
	{
	Widget				paned;
	Widget				textPart;
	Widget				bottomForm;

	subjectErrorDialog= appMakeQuestionDialog( ea, relative,
							&paned, &textPart );

	appQuestionMakeSubjectQuestion( subjectErrorDialog, textPart,
							subject, message );

	XtVaGetValues( subjectErrorDialog,
			XmNuserData,	&qc,
			NULL );

	bottomForm= appQuestionMakeOkRow( paned,
					    subjectErrorDialog, qc->qcOkText );

	XtManageChild( textPart );
	XtManageChild( paned );

#	ifdef USE_POPUP
	XtManageChild( subjectErrorDialog );
#	endif
	}
    else{
	XtVaGetValues( subjectErrorDialog,
			XmNuserData,	&qc,
			NULL );

	appQuestionChangeSubject( subjectErrorDialog, subject );
	appQuestionChangeQuestion( subjectErrorDialog, message );
	appQuestionChangeOkText( subjectErrorDialog, qc->qcOkText );
	}

    appQuestionRunOkDialog( ea, relative, option, subjectErrorDialog );

    XtDestroyWidget( XtParent( subjectErrorDialog ) );
    subjectErrorDialog= (Widget )0;
    }

int appQuestionRunOkCancelDialog(	EditApplication *	ea,
					Widget			relative,
					Widget			option,
					const char *		question,
					const char *		okText,
					const char *		cancelText )
    {
    int			rval;
    QuestionContext *	qc= (QuestionContext *)0;

    static Widget	okCancelDialog;

    if  ( ! okCancelDialog )
	{
	Widget				paned;
	Widget				textPart;
	Widget				bottomForm;

	okCancelDialog= appMakeQuestionDialog( ea, relative,
							    &paned, &textPart );

	appQuestionMakeQuestion( okCancelDialog, textPart, question );

	XtVaGetValues( okCancelDialog,
			    XmNuserData,	&qc,
			    NULL );

	bottomForm= appQuestionMakeOkCancelRow( paned,
					okCancelDialog,
					qc->qcOkText,
					qc->qcCancelText );

	XtManageChild( textPart );
	XtManageChild( paned );

#	ifdef USE_POPUP
	XtManageChild( okCancelDialog );
#	endif
	}
    else{
	XtVaGetValues( okCancelDialog,
			XmNuserData,	&qc,
			NULL );

	appQuestionChangeQuestion( okCancelDialog, question );
	}

    rval= appQuestionRunAnyOkCancelDialog( ea,
					okCancelDialog, relative, option,
					qc, okText, cancelText );

    XtDestroyWidget( XtParent( okCancelDialog ) );
    okCancelDialog= (Widget )0;

    return rval;
    }

int appQuestionRunSubjectOkCancelDialog( EditApplication *	ea,
					Widget			relative,
					Widget			option,
					const char *		subject,
					const char *		message,
					const char *		okText,
					const char *		cancelText )
    {
    int			rval;
    QuestionContext *	qc= (QuestionContext *)0;

    static Widget	subjectOkCancelDialog;

    if  ( ! subjectOkCancelDialog )
	{
	Widget				paned;
	Widget				textPart;
	Widget				bottomForm;

	subjectOkCancelDialog= appMakeQuestionDialog( ea, relative,
							    &paned, &textPart );

	appQuestionMakeSubjectQuestion( subjectOkCancelDialog, textPart,
							subject, message );

	XtVaGetValues( subjectOkCancelDialog,
			    XmNuserData,	&qc,
			    NULL );

	bottomForm= appQuestionMakeOkCancelRow( paned,
					subjectOkCancelDialog,
					qc->qcOkText,
					qc->qcCancelText );

	XtManageChild( textPart );
	XtManageChild( paned );

#	ifdef USE_POPUP
	XtManageChild( subjectOkCancelDialog );
#	endif
	}
    else{
	XtVaGetValues( subjectOkCancelDialog,
			XmNuserData,	&qc,
			NULL );

	appQuestionChangeQuestion( subjectOkCancelDialog, message );
	appQuestionChangeSubject( subjectOkCancelDialog, subject );
	}

    rval= appQuestionRunAnyOkCancelDialog( ea,
				    subjectOkCancelDialog, relative, option,
				    qc, okText, cancelText );

    XtDestroyWidget( XtParent( subjectOkCancelDialog ) );
    subjectOkCancelDialog= (Widget )0;

    return rval;
    }

int appQuestionRunSubjectYesNoDialog(	EditApplication *	ea,
					Widget			relative,
					Widget			option,
					const char *		subject,
					const char *		message,
					const char *		yesText,
					const char *		noText )
    {
    int			rval;
    QuestionContext *	qc= (QuestionContext *)0;

    static Widget	subjectYesNoDialog;

    if  ( ! subjectYesNoDialog )
	{
	Widget				paned;
	Widget				textPart;
	Widget				bottomForm;

	subjectYesNoDialog= appMakeQuestionDialog( ea, relative,
							    &paned, &textPart );

	appQuestionMakeSubjectQuestion( subjectYesNoDialog, textPart,
							subject, message );

	XtVaGetValues( subjectYesNoDialog,
			    XmNuserData,	&qc,
			    NULL );

	bottomForm= appQuestionMakeYesNoRow( paned,
					subjectYesNoDialog,
					qc->qcYesText,
					qc->qcNoText );

	XtManageChild( textPart );

	XtManageChild( paned );

#	ifdef USE_POPUP
	XtManageChild( subjectYesNoDialog );
#	endif
	}
    else{
	XtVaGetValues( subjectYesNoDialog,
			XmNuserData,	&qc,
			NULL );

	appQuestionChangeQuestion( subjectYesNoDialog, message );
	appQuestionChangeSubject( subjectYesNoDialog, subject );
	}

    rval= appQuestionRunAnyYesNoDialog( ea,
				    subjectYesNoDialog, relative, option,
				    qc, yesText, noText );

    XtDestroyWidget( XtParent( subjectYesNoDialog ) );
    subjectYesNoDialog= (Widget )0;

    return rval;
    }

void appQuestionRunErrorDialog(	EditApplication *	ea,
				Widget			relative,
				Widget			option,
				const char *		message )
    {
    QuestionContext *	qc= (QuestionContext *)0;

    static Widget	errorDialog;

    if  ( ! errorDialog )
	{
	Widget				paned;
	Widget				textPart;
	Widget				bottomForm;

	errorDialog= appMakeQuestionDialog( ea, relative,
							&paned, &textPart );

	appQuestionMakeQuestion( errorDialog, textPart, message );

	XtVaGetValues( errorDialog,
			XmNuserData,	&qc,
			NULL );

	bottomForm= appQuestionMakeOkRow( paned,
					    errorDialog, qc->qcOkText );

	XtManageChild( textPart );

	XtManageChild( paned );

#	ifdef USE_POPUP
	XtManageChild( errorDialog );
#	endif
	}
    else{
	XtVaGetValues( errorDialog,
			XmNuserData,	&qc,
			NULL );

	appQuestionChangeQuestion( errorDialog, message );
	appQuestionChangeOkText( errorDialog, qc->qcOkText );
	}

    appQuestionRunOkDialog( ea, relative, option, errorDialog );

    XtDestroyWidget( XtParent( errorDialog ) );
    errorDialog= (Widget )0;

    return;
    }
