/****************************************************************************
|                         Digital Audio Processor
|                         =======================
|
| Filename    : DPTich_linux_audiofile.cc
|
| Revision    : 1.0
| Date        : 26/03/97
|
| Object      : None
|
| Description : Audiofile replacement code for linux
|
| (c) Richard Kent 1997
|
| $Id$
|
****************************************************************************/

static char DPTich_linux_audiofile_cc [] = "$Id$";

#ifdef LINUX

#include "DPTich_linux_audiofile.h"
#include "DPTich_linux_macros.h"

// Limitations :-
//   Only one track (AF_DEFAULT_TRACK) permitted
//   Only one instrument (AF_DEFAULT_INST) permitted
//   Only two's complement samples
//   No built in compression codecs (must be AF_COMPRESSION_NONE)
//   Maximum 65535 markers
//   Maximum 2 loops
//   Only one ANNO chunk
//   Only one MIDI chunk
//   Only one APPL chunk

AFerrfunc audioFileErrorFunction = 0;

/*---------------------------------------------------------------------------
| FUNCTION AFseterrorhandler
---------------------------------------------------------------------------*/
AFerrfunc AFseterrorhandler (AFerrfunc efunc)
{
  AFerrfunc temp;
  
  temp = audioFileErrorFunction;
  audioFileErrorFunction = efunc;
  return temp;
}

/*---------------------------------------------------------------------------
| FUNCTION AFnewfilesetup
---------------------------------------------------------------------------*/
AFfilesetup AFnewfilesetup (void)
{
  AFfilesetup setup;
  
  setup = (aiffType *) calloc (1,sizeof (aiffType));
  initialiseSetup (setup);
  return setup;
}

/*---------------------------------------------------------------------------
| FUNCTION AFfreefilesetup
---------------------------------------------------------------------------*/
void AFfreefilesetup (AFfilesetup setup)
{
  if (!setup) return;
  freeSetup (setup);
  free (setup);
}

/*---------------------------------------------------------------------------
| FUNCTION AFinitfilefmt
---------------------------------------------------------------------------*/
void AFinitfilefmt (AFfilesetup setup,long fmt)
{
  if (!setup) return;
  if (fmt == AF_FILE_AIFFC || fmt == AF_FILE_AIFF)
    setup->filefmt = fmt;
}

/*---------------------------------------------------------------------------
| FUNCTION AFinitrate
---------------------------------------------------------------------------*/
void AFinitrate (AFfilesetup setup,long trackid,double rate)
{
  if (!setup) return;
  if (trackid == AF_DEFAULT_TRACK)
    if (rate > 0.0) setup->rate = rate;
}

/*---------------------------------------------------------------------------
| FUNCTION AFinitsampfmt
---------------------------------------------------------------------------*/
void AFinitsampfmt (AFfilesetup setup,long trackid,long fmt,long width)
{
  if (!setup) return;
  if (trackid == AF_DEFAULT_TRACK)
  {
    if (fmt == AF_SAMPFMT_TWOSCOMP) setup->sampfmt = fmt;
    if (width > 1 && width < 32) setup->width = width;
  }
}

/*---------------------------------------------------------------------------
| FUNCTION AFinitchannels
---------------------------------------------------------------------------*/
void AFinitchannels (AFfilesetup setup,long trackid,long channels)
{
  if (!setup) return;
  if (trackid == AF_DEFAULT_TRACK)
    if (channels == 1 || channels == 2 || channels == 4)
      setup->channels = channels;
}

/*---------------------------------------------------------------------------
| FUNCTION AFinitaeschanneldata
---------------------------------------------------------------------------*/
void AFinitaeschanneldata (AFfilesetup setup,long trackid)
{
  if (!setup) return;
    if (trackid == AF_DEFAULT_TRACK)
      setup->aesdatapresent = 1;
}

/*---------------------------------------------------------------------------
| FUNCTION AFsetaeschanneldata
---------------------------------------------------------------------------*/
void AFsetaeschanneldata (
  AFfilehandle file,
  long trackid,
  unsigned char buf [24])
{
  int i;
  
  if (!file) return;
  if (trackid == AF_DEFAULT_TRACK)
    for (i=0; i<24; i++)
      file->aiff.aesdata [i] = buf [i];
}

/*---------------------------------------------------------------------------
| FUNCTION AFinitcompression
---------------------------------------------------------------------------*/
void AFinitcompression (AFfilesetup setup,long trackid,long compression)
{
  if (!setup) return;
    if (trackid == AF_DEFAULT_TRACK)
      if (compression == AF_COMPRESSION_NONE)
        setup->compression = compression;
}

/*---------------------------------------------------------------------------
| FUNCTION AFinitmarkids
---------------------------------------------------------------------------*/
void AFinitmarkids (
  AFfilesetup setup,
  long trackid,
  long *markids,
  long nmarks)
{
  int i;

  if (!setup) return;
  if (trackid == AF_DEFAULT_TRACK)
  {
    if (nmarks >= 0 && nmarks <= 65535)
    {
      if (setup->mark)
      {
        for (i=0; i<setup->marks; i++)
          if (setup->mark [i].name)
            free (setup->mark [i].name);
        free (setup->mark);
      }

      if (!nmarks || !markids)
      {
        setup->marks = 0;
        setup->mark  = 0;
      }
      else
      {
        setup->marks = nmarks;
        setup->mark  = (markType *) calloc (nmarks,sizeof (markType));
        for (i=0; i<nmarks; i++)
        {
          setup->mark [i].id       = markids [i];
          setup->mark [i].name     = 0;
          setup->mark [i].position = 0;
        }
      }
    }
  }
}

/*---------------------------------------------------------------------------
| FUNCTION AFinitmarkname
---------------------------------------------------------------------------*/
void AFinitmarkname (
  AFfilesetup setup,
  long trackid,
  long markid,
  const char *name)
{
  int i;

  if (!setup) return;
  if (trackid == AF_DEFAULT_TRACK)
  {
    for (i=0; i<setup->marks; i++)
    {
      if (setup->mark [i].id == markid)
      {
        if (setup->mark [i].name)
          free (setup->mark [i].name);
        if (name)
        {
          setup->mark [i].name =
            (char *) calloc (strlen (name) + 1,sizeof (char));
          strcpy (setup->mark [i].name,name);
        }
        else
        {
          setup->mark [i].name = 0;
        }
      }
    }
  }
}

/*---------------------------------------------------------------------------
| FUNCTION AFinitinstids
---------------------------------------------------------------------------*/
void AFinitinstids (AFfilesetup setup,long *instids,long ninsts)
{
  int i;

  if (!setup) return;
  if (ninsts >= 0 && ninsts <= 1)
  {
    if (setup->inst)
      free (setup->inst);

    if (!ninsts || !instids || instids [0] != AF_DEFAULT_INST)
    {
      setup->insts = 0;
      setup->inst  = 0;
    }
    else
    {
      setup->insts = ninsts;
      setup->inst  = (instType *) calloc (ninsts,sizeof (instType));
      for (i=0; i<ninsts; i++)
      {
        setup->inst [i].id             = instids [i];
        setup->inst [i].midibasenote   = 60;
        setup->inst [i].midihinote     = 127;
        setup->inst [i].midihivelocity = 127;
        setup->inst [i].midilonote     = 0;
        setup->inst [i].midilovelocity = 1;
        setup->inst [i].numcentsdetune = 0;
        setup->inst [i].numdbsgain     = 0;
        setup->inst [i].susloopid      = 1;
        setup->inst [i].relloopid      = 2;
      }
    }
  }
}

/*---------------------------------------------------------------------------
| FUNCTION AFinitloopids
---------------------------------------------------------------------------*/
void AFinitloopids (AFfilesetup setup,long instid,long *loopids,long nloops)
{
  int i;

  if (!setup) return;
  if (instid == AF_DEFAULT_INST)
  {
    if (nloops >= 0 && nloops <= 2)
    {
      if (setup->loop)
        free (setup->loop);
  
      if (!nloops || !loopids)
      {
        setup->loops = 0;
        setup->loop  = 0;
      }
      else
      {
        setup->loops = nloops;
        setup->loop  = (loopType *) calloc (nloops,sizeof (loopType));
        for (i=0; i<nloops; i++)
        {
          setup->loop [i].id             = loopids [i];
          setup->loop [i].mode           = AF_LOOP_MODE_NOLOOP;
          setup->loop [i].start          = (2 * i) + 1;
          setup->loop [i].end            = (2 * i) + 2;
        }
      }
    }
  }
}

/*---------------------------------------------------------------------------
| FUNCTION AFinitmiscids
---------------------------------------------------------------------------*/
void AFinitmiscids (AFfilesetup setup,long *miscids,long nmiscs)
{
  int i;

  if (!setup) return;
  if (nmiscs >= 0)
  {
    if (setup->misc)
    {
      for (i=0; i<setup->miscs; i++)
        if (setup->misc [i].data)
          free (setup->misc [i].data);
      free (setup->misc);
    }
  
    if (!nmiscs || !miscids)
    {
      setup->miscs = 0;
      setup->misc  = 0;
    }
    else
    {
      setup->miscs = nmiscs;
      setup->misc  = (miscType *) calloc (nmiscs,sizeof (miscType));
      for (i=0; i<nmiscs; i++)
      {
        setup->misc [i].id       = miscids [i];
        setup->misc [i].type     = AF_MISC_AIFF_ANNO;
        setup->misc [i].size     = 0;
        setup->misc [i].current  = 0;
        setup->misc [i].data     = 0;
      }
    }
  }
}
  
/*---------------------------------------------------------------------------
| FUNCTION AFinitmisctype
---------------------------------------------------------------------------*/
void AFinitmisctype (AFfilesetup setup,long miscid,long type)
{
  int i;

  if (!setup) return;
  for (i=0; i<setup->miscs; i++)
  {
    if (setup->misc [i].id == miscid)
      if (type == AF_MISC_AIFF_ANNO || type == AF_MISC_AIFF_APPL ||
      type == AF_MISC_AIFF_AUTH || type == AF_MISC_AIFF_COPY ||
      type == AF_MISC_AIFF_MIDI || type == AF_MISC_AIFF_NAME)
        setup->misc [i].type = type;
  }
}

/*---------------------------------------------------------------------------
| FUNCTION AFinitmiscsize
---------------------------------------------------------------------------*/
void AFinitmiscsize (AFfilesetup setup,long miscid,long size)
{
  int i;

  if (!setup) return;
  for (i=0; i<setup->miscs; i++)
  {
    if (setup->misc [i].id == miscid)
    {
      if (size >= 0)
      {
        if (setup->misc [i].data)
          free (setup->misc [i].data);
      
        setup->misc [i].size = size;
        if (size)
          setup->misc [i].data = (char *) calloc (size,sizeof (char));
        else
          setup->misc [i].data = 0;
      }
    }
  }
}

/*---------------------------------------------------------------------------
| FUNCTION AUchecklicense
---------------------------------------------------------------------------*/
int AUchecklicense (int product,int *errorval,char **message)
{
  if (product == AU_LICENSE_AWARE_MPEG_ENCODER ||
    product == AU_LICENSE_AWARE_MPEG_DECODER ||
    product == AU_LICENSE_AWARE_MULTIRATE_ENCODER ||
    product == AU_LICENSE_AWARE_MULTIRATE_DECODER)
  {
    *errorval = 0;
    *message  = "License not available";
    return AU_LICENSE_ERR;
  }
  else
  {
    return AU_BAD_PRODUCT;
  }
  return AU_LICENSE_OK;
}

/*---------------------------------------------------------------------------
| FUNCTION AFopenfile
---------------------------------------------------------------------------*/
AFfilehandle AFopenfile (
  const char *name,
  const char *mode,
  AFfilesetup setup)
{
  int file;
  AFfilehandle tempHandle;

  if (!mode) return AF_NULL_FILEHANDLE;

  if (!strcmp (mode,"r"))
  {
    file = open (name,O_RDONLY);
    if (file == -1) return AF_NULL_FILEHANDLE;
    if ((tempHandle = AFopenfd (file,mode,setup)) == AF_NULL_FILEHANDLE)
      close (file);
  }
  else if (!strcmp (mode,"w"))
  {
    // NB decimal 438 = octal 666
    file = open (name,O_RDWR | O_CREAT | O_TRUNC,438);
    if (file == -1) return AF_NULL_FILEHANDLE;
    if ((tempHandle = AFopenfd (file,mode,setup)) == AF_NULL_FILEHANDLE)
      close (file);
  }
  else
  {
    return AF_NULL_FILEHANDLE;
  }
  return tempHandle;
}

/*---------------------------------------------------------------------------
| FUNCTION AFopenfd
---------------------------------------------------------------------------*/
AFfilehandle AFopenfd (int file,const char *mode,AFfilesetup setup)
{
  int i;
  int j;
  int found;
  aiffHandle handleAct;
  AFfilehandle handle = &handleAct;
  char id [4];
  long chunksize;
  long seeksize;
  long ltemp;
  unsigned long ultemp;
  short stemp;
  unsigned short ustemp;
  signed char ctemp;
  unsigned char uctemp;
  unsigned char etemp [10];
  double dtemp;
  char idtemp [4];
  char *tempstring;
  unsigned long offset;
  unsigned long blocksize;
  long bytes;

  int done  = FALSE;
  int error = FALSE;
  int fver  = FALSE;
  int comm  = FALSE;
  int ssnd  = FALSE;
  int mark  = FALSE;
  int comt  = FALSE;
  int inst  = FALSE;
  int aesd  = FALSE;
  int name  = FALSE;
  int auth  = FALSE;
  int copy  = FALSE;
  int anno  = FALSE;
  int midi  = FALSE;
  int appl  = FALSE;

  if (!mode) return AF_NULL_FILEHANDLE;
  if (!strcmp (mode,"r"))
  {
    handle->mode = READONLY;
    handle->file = file;
    handle->actualBytes  = 0;
    handle->actualFrames = 0;
    handle->ssndStart    = 0;
    
    // Setup defaults in handle->aiff
    blankSetup (&(handle->aiff));

    // Read info from file into handle->aiff
    if (!file) return AF_NULL_FILEHANDLE;
    
    // Check FORM chunk
    if (!idRead (file,id) || strncmp (id,"FORM",4))
      fail ("AIFF header does not begin with magic word 'FORM'\n")
    if (!lRead (file,&(handle->actualBytes)))
      fail ("No overall chunk size\n")
    if (!idRead (file,id))
      fail ("AIFF header does not contain a form type\n")

    if (!strncmp (id,"AIFF",4))
      handle->aiff.filefmt = AF_FILE_AIFF;
    else if (!strncmp (id,"AIFC",4))
      handle->aiff.filefmt = AF_FILE_AIFFC;
    else
      fail ("AIFF header does not contain a valid form type\n")

    while (!done && !error) // && !feof (file))
    {
      if (!idRead (file,id))
      {
        // message ("done\n");
        done = TRUE;
        break;
      }
      else if (!strncmp (id,"FVER",4))
      {
        // fver
        // message ("fver\n");
        if (fver) aiffError
        lread (chunksize)
        if (chunksize != 4) aiffError;
        ulread (handle->aiff.version)
        fver = TRUE;
      }
      else if (!strncmp (id,"COMM",4))
      {
        // comm
        // message ("comm\n");
        if (comm) aiffError
        lread (chunksize)
        if (handle->aiff.filefmt == AF_FILE_AIFF && chunksize != 18)
          aiffError
        sread (handle->aiff.channels)
        ulread (handle->aiff.framecount)
        sread (handle->aiff.width)
        eread (handle->aiff.rate)

        if (handle->aiff.filefmt == AF_FILE_AIFFC)
        {
          idread (idtemp)

          // No compression supported so if idtemp not "NONE" set
          // handle->aiff.compression to AF_COMPRESSION_UNKNOWN
          // Ignore compression string as it is not needed
          
          if (!strncmp (idtemp,"NONE",4))
            handle->aiff.compression = AF_COMPRESSION_NONE;
          else
            handle->aiff.compression = AF_COMPRESSION_UNKNOWN;

          pstringread (tempstring)

          int checksize = 22 + 1;
          if (tempstring) checksize += strlen (tempstring);
          if (checksize % 2) checksize++;
          if (chunksize != checksize)
          {
            if (tempstring) free (tempstring);
            aiffError
          }
          if (tempstring) free (tempstring);
        }
        comm = TRUE;
      }
      else if (!strncmp (id,"SSND",4))
      {
        // ssnd
        // message ("ssnd\n");
        if (ssnd) aiffError
        lread (chunksize)
        
        // offset and blocksize not used (should be zero anyway)
        ulread (offset)
        ulread (blocksize)
        
        // Skip past data for now
        handle->ssndStart = lseek (file,0L,SEEK_CUR);
        if (handle->ssndStart == -1) aiffError
        handle->ssndStart += offset;
        if (chunksize > 8)
        {
          seeksize = chunksize - 8;
          if (seeksize % 2) seeksize++;
          if (lseek (file,seeksize,SEEK_CUR) == -1) aiffError
        }
        else
        {
          aiffError
        }
        ssnd = TRUE;
      }
      else if (!strncmp (id,"MARK",4))
      {
        // mark
        // message ("mark\n");
        if (mark) aiffError
        lread (chunksize)
        usread (handle->aiff.marks)
        
        if (handle->aiff.marks)
        {
          handle->aiff.mark =
            (markType *) calloc (handle->aiff.marks,sizeof (markType));
          for (i=0; i<handle->aiff.marks; i++)
          {
            sread (handle->aiff.mark [i].id)
            ulread (handle->aiff.mark [i].position)
            pstringread (handle->aiff.mark [i].name)
          }
        }
        else
        {
          handle->aiff.mark = 0;
        }
        mark = TRUE;
      }
      else if (!strncmp (id,"COMT",4))
      {
        // comt
        // message ("comt\n");
        long comments;
        commentType *comment;

        if (comt) aiffError
        lread (chunksize)
        
        usread (comments)
        
        if (comments)
        {
          comment = (commentType *) calloc (comments,sizeof (commentType));
          for (i=0; i<comments; i++)
          {
            ulread (comment [i].timeStamp)
            sread (comment [i].marker)
            usread (comment [i].count)
            if (comment [i].count)
            {
              comment [i].text =
                (char *) calloc (comment [i].count + 1,sizeof (char));
              if (read (file,comment [i].text,comment [i].count)
                != comment [i].count)
              {
                if (comment)
                {
                  for (i=0; i<comments; i++)
                    if (comment [i].text)
                      free (comment [i].text);
                  free (comment);
                }
                aiffError;
              }
              else
              {
                comment [i].text [comment [i].count] = 0;
                if (comment [i].count % 2) ucread (uctemp);
              }
            }
            else
            {
              comment [i].text = 0;
            }
          }
        }
        else
        {
          comment = 0;
        }
        
        // Don't need the comments for now so just free
        if (comment)
        {
          for (i=0; i<comments; i++)
            if (comment [i].text)
              free (comment [i].text);
          free (comment);
        }
        comt = TRUE;
      }
      else if (!strncmp (id,"INST",4))
      {
        // inst
        // message ("inst\n");
        if (inst) aiffError
        lread (chunksize)
        if (chunksize != 20) aiffError

        handle->aiff.insts = 1;
        handle->aiff.inst =
          (instType *) calloc (handle->aiff.insts,sizeof (instType));
        for (i=0; i<handle->aiff.insts; i++)
        {
          handle->aiff.inst [i].id = AF_DEFAULT_INST;
          cread (handle->aiff.inst [i].midibasenote)
          cread (handle->aiff.inst [i].numcentsdetune)
          cread (handle->aiff.inst [i].midilonote)
          cread (handle->aiff.inst [i].midihinote)
          cread (handle->aiff.inst [i].midilovelocity)
          cread (handle->aiff.inst [i].midihivelocity)
          sread (handle->aiff.inst [i].numdbsgain)
          handle->aiff.inst [i].susloopid  = 1;
          handle->aiff.inst [i].relloopid  = 2;

          handle->aiff.loops = 2;
          handle->aiff.loop =
            (loopType *) calloc (handle->aiff.loops,sizeof (loopType));
          for (j=0; j<handle->aiff.loops; j++)
          {
            handle->aiff.loop [j].id = j + 1;
            sread (handle->aiff.loop [j].mode)
            sread (handle->aiff.loop [j].start)
            sread (handle->aiff.loop [j].end)
          }
        }
        inst = TRUE;
      }
      else if (!strncmp (id,"AESD",4))
      {
        // aesd
        // message ("aesd\n");
        if (aesd) aiffError
        lread (chunksize)
        if (chunksize != 24) aiffError
        
        handle->aiff.aesdatapresent = TRUE;
        if (read (file,handle->aiff.aesdata,24) != 24) aiffError
        aesd = TRUE;
      }
      else if (!strncmp (id,"NAME",4))
      {
        // name
        // message ("name\n");
        if (name) aiffError
        if (!name && !auth && !copy && !anno && !midi && !anno)
          initialiseMiscs (handle);
        lread (chunksize)
        if (chunksize)
        {
          handle->aiff.misc [0].size    = chunksize;
          handle->aiff.misc [0].current = 0;
          handle->aiff.misc [0].data    =
            (char *) calloc (chunksize,sizeof (char));
          if (read (file,handle->aiff.misc [0].data,chunksize)
            != chunksize)
            aiffError
          if (chunksize % 2) ucread (uctemp)
        }
        else
        {
          handle->aiff.misc [0].size = 0;
          handle->aiff.misc [0].data = 0;
        }
        name = TRUE;
      }
      else if (!strncmp (id,"AUTH",4))
      {
        // auth
        // message ("auth\n");
        if (auth) aiffError
        if (!name && !auth && !copy && !anno && !midi && !anno)
          initialiseMiscs (handle);
        lread (chunksize)
        if (chunksize)
        {
          handle->aiff.misc [1].size    = chunksize;
          handle->aiff.misc [1].current = 0;
          handle->aiff.misc [1].data    =
            (char *) calloc (chunksize,sizeof (char));
          if (read (file,handle->aiff.misc [1].data,chunksize)
            != chunksize)
            aiffError
          if (chunksize % 2) ucread (uctemp)
        }
        else
        {
          handle->aiff.misc [1].size = 0;
          handle->aiff.misc [1].data = 0;
        }
        auth = TRUE;
      }
      else if (!strncmp (id,"(c) ",4))
      {
        // copy
        // message ("copy\n");
        if (copy) aiffError
        if (!name && !auth && !copy && !anno && !midi && !anno)
          initialiseMiscs (handle);
        lread (chunksize)
        if (chunksize)
        {
          handle->aiff.misc [2].size    = chunksize;
          handle->aiff.misc [2].current = 0;
          handle->aiff.misc [2].data    =
            (char *) calloc (chunksize,sizeof (char));
          if (read (file,handle->aiff.misc [2].data,chunksize)
            != chunksize)
            aiffError
          if (chunksize % 2) ucread (uctemp)
        }
        else
        {
          handle->aiff.misc [2].size = 0;
          handle->aiff.misc [2].data = 0;
        }
        copy = TRUE;
      }
      else if (!strncmp (id,"ANNO",4))
      {
        // anno
        // message ("anno\n");
        if (anno)
        {
          // Simply replace current anno (ie only supports one)
          if (handle->aiff.misc [3].data)
            free (handle->aiff.misc [3].data);
        }
        if (!name && !auth && !copy && !anno && !midi && !anno)
          initialiseMiscs (handle);
        lread (chunksize)
        if (chunksize)
        {
          handle->aiff.misc [3].size    = chunksize;
          handle->aiff.misc [3].current = 0;
          handle->aiff.misc [3].data    =
            (char *) calloc (chunksize,sizeof (char));
          if (read (file,handle->aiff.misc [3].data,chunksize)
            != chunksize)
            aiffError
          if (chunksize % 2) ucread (uctemp)
        }
        else
        {
          handle->aiff.misc [3].size = 0;
          handle->aiff.misc [3].data = 0;
        }
        anno = TRUE;
      }
      else if (!strncmp (id,"MIDI",4))
      {
        // midi
        // message ("midi\n");
        if (midi)
        {
          // Simply replace current midi (ie only supports one)
          if (handle->aiff.misc [4].data)
            free (handle->aiff.misc [4].data);
        }
        if (!name && !auth && !copy && !anno && !midi && !anno)
          initialiseMiscs (handle);
        lread (chunksize)
        if (chunksize)
        {
          handle->aiff.misc [4].size    = chunksize;
          handle->aiff.misc [4].current = 0;
          handle->aiff.misc [4].data    =
            (char *) calloc (chunksize,sizeof (char));
          if (read (file,handle->aiff.misc [4].data,chunksize)
            != chunksize)
            aiffError
          if (chunksize % 2) ucread (uctemp)
        }
        else
        {
          handle->aiff.misc [4].size = 0;
          handle->aiff.misc [4].data = 0;
        }
        midi = TRUE;
      }
      else if (!strncmp (id,"APPL",4))
      {
        // appl
        // message ("appl\n");
        if (appl)
        {
          // Simply replace current appl (ie only supports one)
          if (handle->aiff.misc [5].data)
            free (handle->aiff.misc [5].data);
        }
        if (!name && !auth && !copy && !anno && !midi && !anno)
          initialiseMiscs (handle);
        lread (chunksize)
        if (chunksize)
        {
          handle->aiff.misc [5].size    = chunksize;
          handle->aiff.misc [5].current = 0;
          handle->aiff.misc [5].data    =
            (char *) calloc (chunksize,sizeof (char));
          if (read (file,handle->aiff.misc [5].data,chunksize)
            != chunksize)
            aiffError
          if (chunksize % 2) ucread (uctemp)
        }
        else
        {
          handle->aiff.misc [5].size = 0;
          handle->aiff.misc [5].data = 0;
        }
        appl = TRUE;
      }
      else
      {
        // other
        // message ("other\n");
        lread (chunksize)
        
        // Skip past data
        seeksize = chunksize;
        if (seeksize % 2) seeksize++;
        if (lseek (file,seeksize,SEEK_CUR) == -1) aiffError
      }
    }

    if (error)
      fail ("error while reading file\n")
    
    // Check for necessary chunks
    if (!comm)
      fail ("no COMM chunk\n")
    if (handle->aiff.framecount > 0 && !ssnd)
      fail ("framecount > 0 but no SSND chunk\n")
    if (handle->aiff.filefmt == AF_FILE_AIFFC)
    {
      if (!fver)
        fail ("no FVER chunk\n")
      if (handle->aiff.version != AIFCVersion1)
        alert ("possibly incompatible AIFFC version\n")
    }

    // Read to start of sound data
    if (lseek (file,handle->ssndStart,SEEK_SET) == -1)
      fail ("error seeking to start of sound data\n")

    handle = new aiffHandle;
    *handle = handleAct;
    return handle;
  }
  else if (!strcmp (mode,"w"))
  {
    if (!setup) return AF_NULL_FILEHANDLE;
    handle->mode = WRITEONLY;
    handle->file = file;
    handle->actualBytes  = 0;
    handle->actualFrames = 0;
    handle->ssndStart    = 0;

    // Copy info from setup into handle->aiff
    copySetup (setup,&(handle->aiff));

    // Write info from handle->aiff to file (without closing)
    if (!file) return AF_NULL_FILEHANDLE;

    idwrite ("FORM")
    
    // Don't know length yet (will update later)
    lwrite (0);
    
    // Reset counter as count doesn't include FORM or count itself
    handle->actualBytes = 0;

    if (handle->aiff.filefmt == AF_FILE_AIFF)
      idwrite ("AIFF")
    else if (handle->aiff.filefmt == AF_FILE_AIFFC)
      idwrite ("AIFC")
    else
      return AF_NULL_FILEHANDLE;
    
    if (handle->aiff.filefmt == AF_FILE_AIFFC)
    {
      idwrite ("FVER")
      chunksize = 4;
      lwrite (chunksize)
      ulwrite (handle->aiff.version)
    }

    idwrite ("COMM")
    if (handle->aiff.filefmt == AF_FILE_AIFF)
      chunksize = 18;
    else
      chunksize = 24; // Assume no compression name
    lwrite (chunksize)
    
    swrite (handle->aiff.channels)
    ulwrite (handle->aiff.framecount)
    swrite (handle->aiff.width)
    ewrite (handle->aiff.rate)
    if (handle->aiff.filefmt == AF_FILE_AIFFC)
    {
      if (handle->aiff.compression == AF_COMPRESSION_NONE)
        idwrite ("NONE")
      else if (handle->aiff.compression == AF_COMPRESSION_APPLE_ACE2)
        idwrite ("ACE2")
      else if (handle->aiff.compression == AF_COMPRESSION_APPLE_ACE8)
        idwrite ("ACE8")
      else if (handle->aiff.compression == AF_COMPRESSION_APPLE_MAC3)
        idwrite ("MAC3")
      else if (handle->aiff.compression == AF_COMPRESSION_APPLE_MAC6)
        idwrite ("MAC6")
      else
        idwrite ("UNKN")
      pstringwrite ("")
    }
    
    if (handle->aiff.marks)
    {
      idwrite ("MARK")
      chunksize = 2;
      for (i=0; i<handle->aiff.marks; i++)
      {
        chunksize += 7;
        if (handle->aiff.mark [i].name)
        {
          chunksize += strlen (handle->aiff.mark [i].name);
          if (!strlen (handle->aiff.mark [i].name) % 2)
            chunksize += 1;
        }
        else chunksize += 1;
      }
      lwrite (chunksize)
      uswrite (handle->aiff.marks)
      for (i=0; i<handle->aiff.marks; i++)
      {
        swrite (handle->aiff.mark [i].id)
        ulwrite (handle->aiff.mark [i].position)
        pstringwrite (handle->aiff.mark [i].name)
      }
    }

    if (handle->aiff.insts)
    {
      idwrite ("INST")
      chunksize = 20;
      lwrite (chunksize)
      for (i=0; i<handle->aiff.insts; i++)
      {
        cwrite (handle->aiff.inst [i].midibasenote)
        cwrite (handle->aiff.inst [i].numcentsdetune)
        cwrite (handle->aiff.inst [i].midilonote)
        cwrite (handle->aiff.inst [i].midihinote)
        cwrite (handle->aiff.inst [i].midilovelocity)
        cwrite (handle->aiff.inst [i].midihivelocity)
        swrite (handle->aiff.inst [i].numdbsgain)
        
        // Find sustain loop details
        found = FALSE;
        j = 0;
        while (!found && j<handle->aiff.loops)
        {
          if (handle->aiff.loop [j].id == handle->aiff.inst [i].susloopid)
            found = TRUE;
          else
            j++;
        }
        if (found)
        {
          swrite (handle->aiff.loop [j].mode)
          swrite (handle->aiff.loop [j].start)
          swrite (handle->aiff.loop [j].end)
        }
        else
        {
          swrite (AF_LOOP_MODE_NOLOOP)
          swrite (1)
          swrite (2)
        }

        // Find release loop details
        found = FALSE;
        j = 0;
        while (!found && j<handle->aiff.loops)
        {
          if (handle->aiff.loop [j].id == handle->aiff.inst [i].relloopid)
            found = TRUE;
          else
            j++;
        }
        if (found)
        {
          swrite (handle->aiff.loop [j].mode)
          swrite (handle->aiff.loop [j].start)
          swrite (handle->aiff.loop [j].end)
        }
        else
        {
          swrite (AF_LOOP_MODE_NOLOOP)
          swrite (3)
          swrite (4)
        }
      }
    }
    
    if (handle->aiff.aesdatapresent)
    {
      idwrite ("AESD")
      chunksize = 24;
      lwrite (chunksize)
      if (write (file,handle->aiff.aesdata,24) != 24)
        return AF_NULL_FILEHANDLE;
      handle->actualBytes += 24;
    }
    
    if (handle->aiff.miscs)
    {
      for (i=0; i<handle->aiff.miscs; i++)
      {
        if (handle->aiff.misc [i].type == AF_MISC_AIFF_NAME)
          idwrite ("NAME")
        else if (handle->aiff.misc [i].type == AF_MISC_AIFF_AUTH)
          idwrite ("AUTH")
        else if (handle->aiff.misc [i].type == AF_MISC_AIFF_COPY)
          idwrite ("(c) ")
        else if (handle->aiff.misc [i].type == AF_MISC_AIFF_ANNO)
          idwrite ("ANNO")
        else if (handle->aiff.misc [i].type == AF_MISC_AIFF_MIDI)
          idwrite ("MIDI")
        else if (handle->aiff.misc [i].type == AF_MISC_AIFF_APPL)
          idwrite ("APPL")
        else
          idwrite ("APPL")

        if (handle->aiff.misc [i].data)
        {
          chunksize = handle->aiff.misc [i].size;
          lwrite (chunksize)
          if (chunksize)
          {
            if (write (file,handle->aiff.misc [i].data,chunksize)
              != chunksize)
              return AF_NULL_FILEHANDLE;
            handle->actualBytes += chunksize;
          }
        }
        else
        {
          chunksize = 0;
          lwrite (chunksize)
        }
      }
    }
    
    // Write sample data chunk header and reset frame counter
    idwrite ("SSND")
    
    // Don't know length yet
    lwrite (0)
    
    // Offset and blocksize zero
    ulwrite (0)
    ulwrite (0)
    // fflush (file); no flushing needed
    
    handle = new aiffHandle;
    *handle = handleAct;
    return handle;
  }
  else
  {
    return AF_NULL_FILEHANDLE;
  }
}

/*---------------------------------------------------------------------------
| FUNCTION AFgetfd
---------------------------------------------------------------------------*/
int AFgetfd (AFfilehandle handle)
{
  if (!handle) return 0;
  return handle->file;
}

/*---------------------------------------------------------------------------
| FUNCTION AFclosefile
---------------------------------------------------------------------------*/
long AFclosefile (AFfilehandle handle)
{
  int i;
  int j;
  int found;
  long chunksize;
  long seeksize;
  long ltemp;
  unsigned long ultemp;
  short stemp;
  unsigned short ustemp;
  char ctemp;
  double dtemp;
  char *tempString;
  char idtemp [4];
  int file;

  // If handle->mode is WRITEONLY update header info (as in handle->aiff)
  // If error then return negative number

  if (!handle) return -1;
  file = handle->file;
  
  if (handle->mode == READONLY)
  {
    if (close (handle->file) == -1) return -1;
    delete handle;
    return 0;
  }

  if (handle->mode == WRITEONLY)
  {
    // Update frame count
    handle->aiff.framecount = handle->actualFrames;
    
    // Reset file descriptor
    if (lseek (file,0L,SEEK_SET) == -1)
    {
      fprintf (stderr,"unable to seek on file\n");
      return -1;
    }
    
    // Write info from handle->aiff to file (without closing)
    if (!file)
      fail ("file won't open for update\n")

    idupdate ("FORM")
    
    // Update length
    lupdate (handle->actualBytes);
    
    if (handle->aiff.filefmt == AF_FILE_AIFF)
      idupdate ("AIFF")
    else if (handle->aiff.filefmt == AF_FILE_AIFFC)
      idupdate ("AIFC")
    else
      return -1;
    
    if (handle->aiff.filefmt == AF_FILE_AIFFC)
    {
      idupdate ("FVER")
      chunksize = 4;
      lupdate (chunksize)
      ulupdate (handle->aiff.version)
    }

    idupdate ("COMM")
    if (handle->aiff.filefmt == AF_FILE_AIFF)
      chunksize = 18;
    else
      chunksize = 24; // Assume no compression name
    lupdate (chunksize)
    
    supdate (handle->aiff.channels)
    ulupdate (handle->aiff.framecount)
    supdate (handle->aiff.width)
    eupdate (handle->aiff.rate)
    if (handle->aiff.filefmt == AF_FILE_AIFFC)
    {
      if (handle->aiff.compression == AF_COMPRESSION_NONE)
        idupdate ("NONE")
      else if (handle->aiff.compression == AF_COMPRESSION_APPLE_ACE2)
        idupdate ("ACE2")
      else if (handle->aiff.compression == AF_COMPRESSION_APPLE_ACE8)
        idupdate ("ACE8")
      else if (handle->aiff.compression == AF_COMPRESSION_APPLE_MAC3)
        idupdate ("MAC3")
      else if (handle->aiff.compression == AF_COMPRESSION_APPLE_MAC6)
        idupdate ("MAC6")
      else
        idupdate ("UNKN")
      pstringupdate ("")
    }
    
    if (handle->aiff.marks)
    {
      idupdate ("MARK")
      chunksize = 2;
      for (i=0; i<handle->aiff.marks; i++)
      {
        chunksize += 7;
        if (handle->aiff.mark [i].name)
        {
          chunksize += strlen (handle->aiff.mark [i].name);
          if (!strlen (handle->aiff.mark [i].name) % 2)
            chunksize += 1;
        }
        else chunksize += 1;
      }
      lupdate (chunksize)
      usupdate (handle->aiff.marks)
      for (i=0; i<handle->aiff.marks; i++)
      {
        supdate (handle->aiff.mark [i].id)
        ulupdate (handle->aiff.mark [i].position)
        pstringupdate (handle->aiff.mark [i].name)
      }
    }

    if (handle->aiff.insts)
    {
      idupdate ("INST")
      chunksize = 20;
      lupdate (chunksize)
      for (i=0; i<handle->aiff.insts; i++)
      {
        cupdate (handle->aiff.inst [i].midibasenote)
        cupdate (handle->aiff.inst [i].numcentsdetune)
        cupdate (handle->aiff.inst [i].midilonote)
        cupdate (handle->aiff.inst [i].midihinote)
        cupdate (handle->aiff.inst [i].midilovelocity)
        cupdate (handle->aiff.inst [i].midihivelocity)
        supdate (handle->aiff.inst [i].numdbsgain)
        
        // Find sustain loop details
        found = FALSE;
        j = 0;
        while (!found && j<handle->aiff.loops)
        {
          if (handle->aiff.loop [j].id == handle->aiff.inst [i].susloopid)
            found = TRUE;
          else
            j++;
        }
        if (found)
        {
          supdate (handle->aiff.loop [j].mode)
          supdate (handle->aiff.loop [j].start)
          supdate (handle->aiff.loop [j].end)
        }
        else
        {
          supdate (AF_LOOP_MODE_NOLOOP)
          supdate (1)
          supdate (2)
        }

        // Find release loop details
        found = FALSE;
        j = 0;
        while (!found && j<handle->aiff.loops)
        {
          if (handle->aiff.loop [j].id == handle->aiff.inst [i].relloopid)
            found = TRUE;
          else
            j++;
        }
        if (found)
        {
          supdate (handle->aiff.loop [j].mode)
          supdate (handle->aiff.loop [j].start)
          supdate (handle->aiff.loop [j].end)
        }
        else
        {
          supdate (AF_LOOP_MODE_NOLOOP)
          supdate (3)
          supdate (4)
        }
      }
    }
    
    if (handle->aiff.aesdatapresent)
    {
      idupdate ("AESD")
      chunksize = 24;
      lupdate (chunksize)
      if (write (file,handle->aiff.aesdata,24) != 24)
        return AF_NULL_FILEHANDLE;
    }
    
    if (handle->aiff.miscs)
    {
      for (i=0; i<handle->aiff.miscs; i++)
      {
        if (handle->aiff.misc [i].type == AF_MISC_AIFF_NAME)
          idupdate ("NAME")
        else if (handle->aiff.misc [i].type == AF_MISC_AIFF_AUTH)
          idupdate ("AUTH")
        else if (handle->aiff.misc [i].type == AF_MISC_AIFF_COPY)
          idupdate ("(c) ")
        else if (handle->aiff.misc [i].type == AF_MISC_AIFF_ANNO)
          idupdate ("ANNO")
        else if (handle->aiff.misc [i].type == AF_MISC_AIFF_MIDI)
          idupdate ("MIDI")
        else if (handle->aiff.misc [i].type == AF_MISC_AIFF_APPL)
          idupdate ("APPL")
        else
          idupdate ("APPL")

        if (handle->aiff.misc [i].data)
        {
          chunksize = handle->aiff.misc [i].size;
          lupdate (chunksize)
          if (chunksize)
          {
            if (write (file,handle->aiff.misc [i].data,chunksize)
              != chunksize)
              return AF_NULL_FILEHANDLE;
          }
        }
        else
        {
          chunksize = 0;
          lupdate (chunksize)
        }
      }
    }
    
    // Write sample data chunk header and reset frame counter
    idupdate ("SSND")
    
    // Update chunk size
    chunksize = handle->aiff.framecount * handle->aiff.channels;
    chunksize *= ((handle->aiff.width + 7) / 8);
    chunksize += 8; // For offset and blocksize
    lupdate (chunksize)
    
    // Offset and blocksize zero
    ulupdate (0)
    ulupdate (0)

    if (close (handle->file) == -1) return -1;
    delete handle;
    return 0;
  }
  return -1;
}

/*---------------------------------------------------------------------------
| FUNCTION AFgetfilefmt
---------------------------------------------------------------------------*/
long AFgetfilefmt (AFfilehandle handle,long *version)
{
  if (!handle) return 0;
  if (handle->aiff.filefmt == AF_FILE_AIFF)
  {
    *version = 0;
    return AF_FILE_AIFF;
  }
  if (handle->aiff.filefmt == AF_FILE_AIFFC)
  {
    *version = handle->aiff.version;
    return AF_FILE_AIFFC;
  }
  return AF_FILE_UNKNOWN;
}

/*---------------------------------------------------------------------------
| FUNCTION AFidentifyfd
---------------------------------------------------------------------------*/
long AFidentifyfd (int file)
{
  char id [4];
  long bytes;

  // Read info from file
  if (!file) return AF_FILE_UNKNOWN;
    
  // Check FORM chunk
  if (read (file,id,4) != 4 || strncmp (id,"FORM",4))
    return AF_FILE_UNKNOWN;

  if (!lRead (file,&bytes))
    return AF_FILE_UNKNOWN;

  if (read (file,id,4) != 4)
    return AF_FILE_UNKNOWN;

  if (!strncmp (id,"AIFF",4))
    return AF_FILE_AIFF;
  else if (!strncmp (id,"AIFC",4))
    return AF_FILE_AIFFC;
  else
    return AF_FILE_UNSUPPORTED;
}

/*---------------------------------------------------------------------------
| FUNCTION AFgetrate
---------------------------------------------------------------------------*/
double AFgetrate (AFfilehandle handle,long trackid)
{
  if (!handle) return 0.0;
  return handle->aiff.rate;
}

/*---------------------------------------------------------------------------
| FUNCTION AFgetsampfmt
---------------------------------------------------------------------------*/
void AFgetsampfmt (
  AFfilehandle handle,
  long trackid,
  long *sampfmt,
  long *width)
{
  if (!handle || trackid != AF_DEFAULT_TRACK)
  {
    *sampfmt = 0;
    *width   = 0;
    return;
  }
  *sampfmt = handle->aiff.sampfmt;
  *width   = handle->aiff.width;
}

/*---------------------------------------------------------------------------
| FUNCTION AFgetchannels
---------------------------------------------------------------------------*/
long AFgetchannels (AFfilehandle handle,long trackid)
{
  if (!handle || trackid != AF_DEFAULT_TRACK) return 0;
  return handle->aiff.channels;
}

/*---------------------------------------------------------------------------
| FUNCTION AFgetaeschanneldata
---------------------------------------------------------------------------*/
long AFgetaeschanneldata (
  AFfilehandle handle,
  long trackid,
  unsigned char buf [24])
{
  int i;
  
  if (!handle || trackid != AF_DEFAULT_TRACK) return 0;
  if (!handle->aiff.aesdatapresent) return 0;
  
  for (i=0; i<24; i++)
    buf [i] = handle->aiff.aesdata [i];
    
  return 1;
}

/*---------------------------------------------------------------------------
| FUNCTION AFgetcompression
---------------------------------------------------------------------------*/
long AFgetcompression (AFfilehandle handle,long trackid)
{
  if (!handle || trackid != AF_DEFAULT_TRACK) return AF_COMPRESSION_NONE;
  return handle->aiff.compression;
}

/*---------------------------------------------------------------------------
| FUNCTION AFgetcompressionname
---------------------------------------------------------------------------*/
char *AFgetcompressionname (AFfilehandle handle,long trackid)
{
  if (!handle || trackid != AF_DEFAULT_TRACK) return NULL;
  if (handle->aiff.compression == AF_COMPRESSION_NONE)
    return "No compression";
  
  // Should really handle all compression cases
  return "Compression";
}

/*---------------------------------------------------------------------------
| FUNCTION AFgetframecnt
---------------------------------------------------------------------------*/
long AFgetframecnt (AFfilehandle handle,long trackid)
{
  if (!handle || trackid != AF_DEFAULT_TRACK) return 0;
  return handle->aiff.framecount;
}

/*---------------------------------------------------------------------------
| FUNCTION AFgetmarkids
---------------------------------------------------------------------------*/
long AFgetmarkids (AFfilehandle handle,long trackid,long *markids)
{
  int i;
  
  if (!handle || trackid != AF_DEFAULT_TRACK) return 0;
  if (markids)
    for (i=0; i<handle->aiff.marks; i++)
      markids [i] = handle->aiff.mark [i].id;
  return handle->aiff.marks;
}

/*---------------------------------------------------------------------------
| FUNCTION AFgetmarkname
---------------------------------------------------------------------------*/
char *AFgetmarkname (AFfilehandle handle,long trackid,long markid)
{
  int i;
  int found;
  
  if (!handle || trackid != AF_DEFAULT_TRACK) return 0;
  
  i = 0;
  found = FALSE;
  while (i<handle->aiff.marks && !found)
  {
    if (handle->aiff.mark [i].id == markid)
      found = TRUE;
    else
      i++;
  }
  
  if (!found) return 0;
  return handle->aiff.mark [i].name;
}

/*---------------------------------------------------------------------------
| FUNCTION AFgetmarkpos
---------------------------------------------------------------------------*/
long AFgetmarkpos (AFfilehandle handle,long trackid,long markid)
{
  int i;
  int found;
  
  if (!handle || trackid != AF_DEFAULT_TRACK) return 0;
  
  i = 0;
  found = FALSE;
  while (i<handle->aiff.marks && !found)
  {
    if (handle->aiff.mark [i].id == markid)
      found = TRUE;
    else
      i++;
  }
  
  if (!found) return 0;
  return handle->aiff.mark [i].position;
}

/*---------------------------------------------------------------------------
| FUNCTION AFsetmarkpos
---------------------------------------------------------------------------*/
void AFsetmarkpos (
  AFfilehandle handle,
  long trackid,
  long markid,
  long markpos)
{
  int i;
  int found;
  
  if (!handle || trackid != AF_DEFAULT_TRACK) return;
  
  i = 0;
  found = FALSE;
  while (i<handle->aiff.marks && !found)
  {
    if (handle->aiff.mark [i].id == markid)
      found = TRUE;
    else
      i++;
  }
  
  if (!found) return;
  handle->aiff.mark [i].position = markpos;
}

/*---------------------------------------------------------------------------
| FUNCTION AFgetinstids
---------------------------------------------------------------------------*/
long AFgetinstids (AFfilehandle handle,long *instids)
{
  int i;
  
  if (!handle) return 0;

  if (instids)
    for (i=0; i<handle->aiff.insts; i++)
      instids [i] = handle->aiff.inst [i].id;
  return handle->aiff.insts;
}

/*---------------------------------------------------------------------------
| FUNCTION AFgetinstparamlong
---------------------------------------------------------------------------*/
long AFgetinstparamlong (AFfilehandle handle,long instid,long param)
{
  int i;
  int found;
  
  if (!handle || instid != AF_DEFAULT_INST) return 0;
  
  i = 0;
  found = FALSE;
  while (i<handle->aiff.insts && !found)
  {
    if (handle->aiff.inst [i].id == instid)
      found = TRUE;
    else
      i++;
  }
  
  if (!found) return 0;
  switch (param)
  {
    case AF_INST_MIDI_BASENOTE :
      return handle->aiff.inst [i].midibasenote;
    case AF_INST_NUMCENTS_DETUNE :
      return handle->aiff.inst [i].numcentsdetune;
    case AF_INST_MIDI_LONOTE :
      return handle->aiff.inst [i].midilonote;
    case AF_INST_MIDI_HINOTE :
      return handle->aiff.inst [i].midihinote;
    case AF_INST_MIDI_LOVELOCITY :
      return handle->aiff.inst [i].midilovelocity;
    case AF_INST_MIDI_HIVELOCITY :
      return handle->aiff.inst [i].midihivelocity;
    case AF_INST_NUMDBS_GAIN :
      return handle->aiff.inst [i].numdbsgain;
    case AF_INST_SUSLOOPID :
      return handle->aiff.inst [i].susloopid;
    case AF_INST_RELLOOPID :
      return handle->aiff.inst [i].relloopid;
    default :
      return 0;
  }
}

/*---------------------------------------------------------------------------
| FUNCTION AFsetinstparamlong
---------------------------------------------------------------------------*/
void AFsetinstparamlong (
  AFfilehandle handle,
  long instid,
  long param,
  long value)
{
  int i;
  int found;
  
  if (!handle || instid != AF_DEFAULT_INST) return;
  
  i = 0;
  found = FALSE;
  while (i<handle->aiff.insts && !found)
  {
    if (handle->aiff.inst [i].id == instid)
      found = TRUE;
    else
      i++;
  }
  
  if (!found) return;
  switch (param)
  {
    case AF_INST_MIDI_BASENOTE :
      handle->aiff.inst [i].midibasenote = value;
    case AF_INST_NUMCENTS_DETUNE :
      handle->aiff.inst [i].numcentsdetune = value;
    case AF_INST_MIDI_LONOTE :
      handle->aiff.inst [i].midilonote = value;
    case AF_INST_MIDI_HINOTE :
      handle->aiff.inst [i].midihinote = value;
    case AF_INST_MIDI_LOVELOCITY :
      handle->aiff.inst [i].midilovelocity = value;
    case AF_INST_MIDI_HIVELOCITY :
      handle->aiff.inst [i].midihivelocity = value;
    case AF_INST_NUMDBS_GAIN :
      handle->aiff.inst [i].numdbsgain = value;
    case AF_INST_SUSLOOPID :
      handle->aiff.inst [i].susloopid = value;
    case AF_INST_RELLOOPID :
      handle->aiff.inst [i].relloopid = value;
    default :
      return;
  }
}

/*---------------------------------------------------------------------------
| FUNCTION AFgetloopids
---------------------------------------------------------------------------*/
long AFgetloopids (AFfilehandle handle,long *loopids)
{
  int i;
  
  if (!handle) return 0;
  if (loopids)
    for (i=0; i<handle->aiff.loops; i++)
      loopids [i] = handle->aiff.loop [i].id;
  return handle->aiff.loops;
}

/*---------------------------------------------------------------------------
| FUNCTION AFgetloopmode
---------------------------------------------------------------------------*/
long AFgetloopmode (AFfilehandle handle,long instid,long loopid)
{
  int i;
  int found;
  
  if (!handle || instid != AF_DEFAULT_INST) return AF_LOOP_MODE_NOLOOP;
  
  i = 0;
  found = FALSE;
  while (i<handle->aiff.loops && !found)
  {
    if (handle->aiff.loop [i].id == loopid)
      found = TRUE;
    else
      i++;
  }
  
  if (!found) return AF_LOOP_MODE_NOLOOP;
  return handle->aiff.loop [i].mode;
}

/*---------------------------------------------------------------------------
| FUNCTION AFsetloopmode
---------------------------------------------------------------------------*/
void AFsetloopmode (AFfilehandle handle,long instid,long loopid,long mode)
{
  int i;
  int found;
  
  if (!handle || instid != AF_DEFAULT_INST) return;
  
  i = 0;
  found = FALSE;
  while (i<handle->aiff.loops && !found)
  {
    if (handle->aiff.loop [i].id == loopid)
      found = TRUE;
    else
      i++;
  }
  
  if (!found) return;
  handle->aiff.loop [i].mode = mode;
}

/*---------------------------------------------------------------------------
| FUNCTION AFgetloopstart
---------------------------------------------------------------------------*/
long AFgetloopstart (AFfilehandle handle,long instid,long loopid)
{
  int i;
  int found;
  
  if (!handle || instid != AF_DEFAULT_INST) return 0;
  
  i = 0;
  found = FALSE;
  while (i<handle->aiff.loops && !found)
  {
    if (handle->aiff.loop [i].id == loopid)
      found = TRUE;
    else
      i++;
  }
  
  if (!found) return 0;
  return handle->aiff.loop [i].start;
}

/*---------------------------------------------------------------------------
| FUNCTION AFsetloopstart
---------------------------------------------------------------------------*/
void AFsetloopstart (
  AFfilehandle handle,
  long instid,
  long loopid,
  long markid)
{
  int i;
  int found;
  
  if (!handle || instid != AF_DEFAULT_INST) return;
  
  i = 0;
  found = FALSE;
  while (i<handle->aiff.loops && !found)
  {
    if (handle->aiff.loop [i].id == loopid)
      found = TRUE;
    else
      i++;
  }
  
  if (!found) return;
  handle->aiff.loop [i].start = markid;
}

/*---------------------------------------------------------------------------
| FUNCTION AFgetloopend
---------------------------------------------------------------------------*/
long AFgetloopend (AFfilehandle handle,long instid,long loopid)
{
  int i;
  int found;
  
  if (!handle || instid != AF_DEFAULT_INST) return 0;
  
  i = 0;
  found = FALSE;
  while (i<handle->aiff.loops && !found)
  {
    if (handle->aiff.loop [i].id == loopid)
      found = TRUE;
    else
      i++;
  }
  
  if (!found) return 0;
  return handle->aiff.loop [i].end;
}

/*---------------------------------------------------------------------------
| FUNCTION AFsetloopend
---------------------------------------------------------------------------*/
void AFsetloopend (AFfilehandle handle,long instid,long loopid,long markid)
{
  int i;
  int found;
  
  if (!handle || instid != AF_DEFAULT_INST) return;
  
  i = 0;
  found = FALSE;
  while (i<handle->aiff.loops && !found)
  {
    if (handle->aiff.loop [i].id == loopid)
      found = TRUE;
    else
      i++;
  }
  
  if (!found) return;
  handle->aiff.loop [i].end = markid;
}

/*---------------------------------------------------------------------------
| FUNCTION AFgetmiscids
---------------------------------------------------------------------------*/
long AFgetmiscids (AFfilehandle handle,long *miscids)
{
  int i;
  
  if (!handle) return 0;
  if (miscids)
    for (i=0; i<handle->aiff.miscs; i++)
      miscids [i] = handle->aiff.misc [i].id;
  return handle->aiff.miscs;
}

/*---------------------------------------------------------------------------
| FUNCTION AFgetmisctype
---------------------------------------------------------------------------*/
long AFgetmisctype (AFfilehandle handle,long miscid)
{
  int i;
  int found;
  
  if (!handle) return AF_MISC_AIFF_UNRECOGNIZED;
  
  i = 0;
  found = FALSE;
  while (i<handle->aiff.miscs && !found)
  {
    if (handle->aiff.misc [i].id == miscid)
      found = TRUE;
    else
      i++;
  }
  
  if (!found) return AF_MISC_AIFF_UNRECOGNIZED;
  return handle->aiff.misc [i].type;
}

/*---------------------------------------------------------------------------
| FUNCTION AFgetmiscsize
---------------------------------------------------------------------------*/
long AFgetmiscsize (AFfilehandle handle,long miscid)
{
  int i;
  int found;
  
  if (!handle) return 0;
  
  i = 0;
  found = FALSE;
  while (i<handle->aiff.miscs && !found)
  {
    if (handle->aiff.misc [i].id == miscid)
      found = TRUE;
    else
      i++;
  }
  
  if (!found) return 0;
  return handle->aiff.misc [i].size;
}

/*---------------------------------------------------------------------------
| FUNCTION AFreadmisc
---------------------------------------------------------------------------*/
long AFreadmisc (AFfilehandle handle,long miscid,void *buf,long nbytes)
{
  int i;
  int j;
  int count;
  int found;
  
  if (!handle) return 0;
  
  i = 0;
  found = FALSE;
  while (i<handle->aiff.miscs && !found)
  {
    if (handle->aiff.misc [i].id == miscid)
      found = TRUE;
    else
      i++;
  }
  
  if (!found) return 0;

  count = 0;
  j = handle->aiff.misc [i].current;
  
  while (count < nbytes && j < handle->aiff.misc [i].size)
  {
    ((char *) buf) [j] = handle->aiff.misc [i].data [j];
    j++;
    count++;
    handle->aiff.misc [i].current++;
  }
  return count;
}

/*---------------------------------------------------------------------------
| FUNCTION AFwritemisc
---------------------------------------------------------------------------*/
long AFwritemisc (AFfilehandle handle,long miscid,void *buf,long nbytes)
{
  int i;
  int j;
  int count;
  int found;
  
  if (!handle) return 0;
  
  i = 0;
  found = FALSE;
  while (i<handle->aiff.miscs && !found)
  {
    if (handle->aiff.misc [i].id == miscid)
      found = TRUE;
    else
      i++;
  }
  
  if (!found) return 0;

  count = 0;
  j = handle->aiff.misc [i].current;
  
  while (count < nbytes && j < handle->aiff.misc [i].size)
  {
    handle->aiff.misc [i].data [j] = ((char *) buf) [j];
    j++;
    count++;
    handle->aiff.misc [i].current++;
  }
  return count;
}

/*---------------------------------------------------------------------------
| FUNCTION AFseekmisc
---------------------------------------------------------------------------*/
void AFseekmisc (AFfilehandle handle,long miscid,long offset)
{
  int i;
  int found;
  
  if (!handle) return;
  
  i = 0;
  found = FALSE;
  while (i<handle->aiff.miscs && !found)
  {
    if (handle->aiff.misc [i].id == miscid)
      found = TRUE;
    else
      i++;
  }
  
  if (!found) return;

  handle->aiff.misc [i].current = offset;
  if (handle->aiff.misc [i].current < 0)
    handle->aiff.misc [i].current = 0;
  if (handle->aiff.misc [i].current > handle->aiff.misc [i].size)
    handle->aiff.misc [i].current = handle->aiff.misc [i].size;
}

/*---------------------------------------------------------------------------
| FUNCTION AFreadframes
---------------------------------------------------------------------------*/
long AFreadframes (
  AFfilehandle handle,
  long trackid,
  void *frames,
  long count)
{
  long i;
  long j;
  long frameCount;
  int file;
  signed char ctemp;
  short stemp;
  long  ltemp;

  if (!handle || trackid != AF_DEFAULT_TRACK) return 0;
  if (handle->mode != READONLY) return 0;

  file = handle->file;
  if (!file) return 0;

  i = 0;
  frameCount = 0;
  while (handle->actualFrames < handle->aiff.framecount &&
    frameCount < count) // && !(feof (file)))
  {
    for (j=0; j<handle->aiff.channels; j++)
    {
      if (handle->aiff.width <= 8)
      {
        if (!cRead (file,&ctemp)) return frameCount;
        ((signed char *) frames) [i++] = ctemp;
      }
      else if (handle->aiff.width <= 16)
      {
        if (!sRead (file,&stemp)) return frameCount;
        ((short *) frames) [i++] = stemp;
      }
      else if (handle->aiff.width <= 32)
      {
        if (!lRead (file,&ltemp)) return frameCount;
        ((long *) frames) [i++] = ltemp;
      }
      else
      {
        return frameCount;
      }
    }
    frameCount++;
    handle->actualFrames++;
  }
  return frameCount;
}

/*---------------------------------------------------------------------------
| FUNCTION AFwriteframes
---------------------------------------------------------------------------*/
long AFwriteframes (
  AFfilehandle handle,
  long trackid,
  void *frames,
  long count)
{
  long i;
  long j;
  long frameCount;
  int  file;
  signed char  ctemp;
  short stemp;
  long  ltemp;

  if (!handle || trackid != AF_DEFAULT_TRACK) return 0;
  if (handle->mode != WRITEONLY) return 0;

  file = handle->file;
  if (!file) return 0;

  i = 0;
  frameCount = 0;
  while (frameCount < count) // && !(feof (file)))
  {
    for (j=0; j<handle->aiff.channels; j++)
    {
      if (handle->aiff.width <= 8)
      {
        ctemp = ((signed char *) frames) [i++];
        if (!cWrite (file,ctemp)) return frameCount;
        handle->actualBytes += 1;
      }
      else if (handle->aiff.width <= 16)
      {
        stemp = ((short *) frames) [i++];
        if (!sWrite (file,stemp)) return frameCount;
        handle->actualBytes += 2;
      }
      else if (handle->aiff.width <= 32)
      {
        ltemp = ((long *) frames) [i++];
        if (!lWrite (file,ltemp)) return frameCount;
        handle->actualBytes += 4;
      }
      else
      {
        return frameCount;
      }
    }
    frameCount++;
    handle->actualFrames++;
  }
  return frameCount;
}

/*---------------------------------------------------------------------------
| FUNCTION AFseekframe
---------------------------------------------------------------------------*/
long AFseekframe (AFfilehandle handle,long trackid,long offset)
{
  long seeksize;
  int file;

  if (!handle || trackid != AF_DEFAULT_TRACK) return 0;
  if (handle->mode != READONLY) return 0;

  file = handle->file;
  if (!file) return 0;

  if (offset < 0)
    offset = 0;
  if (offset >= handle->aiff.framecount)
    offset = handle->aiff.framecount;

  seeksize = offset - handle->actualFrames;

  if (handle->aiff.width <= 8)
  {
    seeksize = seeksize;
  }
  else if (handle->aiff.width <= 16)
  {
    seeksize = 2 * seeksize;
  }
  else if (handle->aiff.width <= 32)
  {
    seeksize = 4 * seeksize;
  }
  else
  {
    seeksize = 0;
  }
  
  handle->actualFrames = offset;
  if (lseek (file,seeksize,SEEK_CUR) == -1)
    return -1;
  else
    return offset;
}

/*---------------------------------------------------------------------------
| FUNCTION initialiseMiscs
---------------------------------------------------------------------------*/
void initialiseMiscs (AFfilehandle handle)
{
  // Initialises the miscs in the order shown below

  int i;
  
  if (!handle) return;
  handle->aiff.miscs = 6;
  handle->aiff.misc  =
    (miscType *) calloc (handle->aiff.miscs,sizeof (miscType));
  for (i=0; i<handle->aiff.miscs; i++)
  {
    handle->aiff.misc [i].id = i + 1;
    if (i == 0) handle->aiff.misc [i].type = AF_MISC_AIFF_NAME;
    if (i == 1) handle->aiff.misc [i].type = AF_MISC_AIFF_AUTH;
    if (i == 2) handle->aiff.misc [i].type = AF_MISC_AIFF_COPY;
    if (i == 3) handle->aiff.misc [i].type = AF_MISC_AIFF_ANNO;
    if (i == 4) handle->aiff.misc [i].type = AF_MISC_AIFF_MIDI;
    if (i == 5) handle->aiff.misc [i].type = AF_MISC_AIFF_APPL;
    handle->aiff.misc [i].size    = 0;
    handle->aiff.misc [i].current = 0;
    handle->aiff.misc [i].data    = 0;
  }
}

/*---------------------------------------------------------------------------
| FUNCTION initialiseSetup
---------------------------------------------------------------------------*/
void initialiseSetup (AFfilesetup setup)
{
  // Initialises a default setup

  int i;

  if (!setup) return;
  
  setup->filefmt        = AF_FILE_AIFFC;
  setup->version        = AIFCVersion1;
  setup->audiotrack     = AF_DEFAULT_TRACK;
  setup->rate           = 8000.0;
  setup->sampfmt        = AF_SAMPFMT_TWOSCOMP;
  setup->width          = 16;
  setup->channels       = 2;
  setup->compression    = AF_COMPRESSION_NONE;
  setup->aesdatapresent = 0;
  for (i=0; i<24; i++) setup->aesdata [i] = 0;
  setup->framecount     = 0;
  setup->marks          = 4;
  setup->mark           = (markType *) calloc (4,sizeof (markType));

  for (i=0; i<4; i++)
  {
    setup->mark [i].id       = i + 1;
    setup->mark [i].name     = 0;
    setup->mark [i].position = 0;
  }

  setup->insts          = 1;
  setup->inst           = (instType *) calloc (1,sizeof (instType));

  for (i=0; i<1; i++)
  {
    setup->inst [i].id             = AF_DEFAULT_INST;
    setup->inst [i].midibasenote   = 60;
    setup->inst [i].midihinote     = 127;
    setup->inst [i].midihivelocity = 127;
    setup->inst [i].midilonote     = 0;
    setup->inst [i].midilovelocity = 1;
    setup->inst [i].numcentsdetune = 0;
    setup->inst [i].numdbsgain     = 0;
    setup->inst [i].susloopid      = 1;
    setup->inst [i].relloopid      = 2;
  }

  setup->loops = 2;
  setup->loop  = (loopType *) calloc (2,sizeof (loopType));

  for (i=0; i<2; i++)
  {
    setup->loop [i].id             = i + 1;
    setup->loop [i].mode           = AF_LOOP_MODE_NOLOOP;
    setup->loop [i].start          = (2 * i) + 1;
    setup->loop [i].end            = (2 * i) + 2;
  }
  
  setup->miscs = 0;
  setup->misc  = 0;
}

/*---------------------------------------------------------------------------
| FUNCTION blankSetup
---------------------------------------------------------------------------*/
void blankSetup (AFfilesetup setup)
{
  // Initialises a completely blank setup

  int i;
  
  if (!setup) return;

  setup->filefmt        = AF_FILE_AIFF;
  setup->version        = AIFCVersion1;
  setup->audiotrack     = AF_DEFAULT_TRACK;
  setup->rate           = 8000.0;
  setup->sampfmt        = AF_SAMPFMT_TWOSCOMP;
  setup->width          = 16;
  setup->channels       = 2;
  setup->compression    = AF_COMPRESSION_NONE;
  setup->aesdatapresent = 0;
  for (i=0; i<24; i++) setup->aesdata [i] = 0;
  setup->framecount     = 0;
  setup->marks          = 0;
  setup->mark           = 0;
  setup->insts          = 0;
  setup->inst           = 0;
  setup->loops          = 0;
  setup->loop           = 0;
  setup->miscs          = 0;
  setup->misc           = 0;
}

/*---------------------------------------------------------------------------
| FUNCTION copySetup
---------------------------------------------------------------------------*/
void copySetup (AFfilesetup from,AFfilesetup to)
{
  // Copies a setup

  int i;
  
  if (!from || !to) return;

  to->filefmt        = from->filefmt;
  to->version        = from->version;
  to->audiotrack     = from->audiotrack;
  to->rate           = from->rate;
  to->sampfmt        = from->sampfmt;
  to->width          = from->width;
  to->channels       = from->channels;
  to->compression    = from->compression;
  to->aesdatapresent = from->aesdatapresent;
  for (i=0; i<24; i++) to->aesdata [i] = from->aesdata [i];
  to->framecount     = from->framecount;
  to->marks          = from->marks;
  to->mark           = (markType *) calloc (to->marks,sizeof (markType));
  for (i=0; i<to->marks; i++)
  {
    to->mark [i].id = from->mark [i].id;
    if (from->mark [i].name)
    {
      to->mark [i].name =
        (char *) calloc (strlen (from->mark [i].name) + 1,sizeof (char));
      strcpy (to->mark [i].name,from->mark [i].name);
    }
    else
    {
      to->mark [i].name = 0;
    }
    to->mark [i].position = from->mark [i].position;
  }

  to->insts = from->insts;
  to->inst  = (instType *) calloc (to->insts,sizeof (instType));
  for (i=0; i<to->insts; i++)
    to->inst [i] = from->inst [i];
  
  to->loops = from->loops;
  to->loop  = (loopType *) calloc (to->loops,sizeof (loopType));
  for (i=0; i<to->loops; i++)
    to->loop [i] = from->loop [i];

  to->miscs = from->miscs;
  to->misc  = (miscType *) calloc (to->miscs,sizeof (miscType));
  for (i=0; i<to->miscs; i++)
  {
    to->misc [i].id      = from->misc [i].id;
    to->misc [i].type    = from->misc [i].type;
    to->misc [i].size    = from->misc [i].size;
    to->misc [i].current = from->misc [i].current;
    if (from->misc [i].data)
    {
      to->misc [i].data = (char *) calloc (to->misc [i].size,sizeof (char));
      memcpy (to->misc [i].data,from->misc [i].data,to->misc [i].size);
    }
    else
    {
      to->misc [i].data = 0;
    }
  }
}

/*---------------------------------------------------------------------------
| FUNCTION freeSetup
---------------------------------------------------------------------------*/
void freeSetup (AFfilesetup setup)
{
  // Frees all memory from a setup

  int i;
  
  if (!setup) return;

  if (setup->mark)
  {
    for (i=0; i<setup->marks; i++)
    {
      if (setup->mark [i].name)
      {
        free (setup->mark [i].name);
        setup->mark [i].name = 0;
      }
    }
    free (setup->mark);
    setup->mark  = 0;
    setup->marks = 0;
  }
  
  if (setup->inst)
  {
    free (setup->inst);
    setup->inst  = 0;
    setup->insts = 0;
  }
  
  if (setup->loop)
  {
    free (setup->loop);
    setup->loop  = 0;
    setup->loops = 0;
  }
  
  if (setup->misc)
  {
    for (i=0; i<setup->miscs; i++)
    {
      if (setup->misc [i].data)
      {
        free (setup->misc [i].data);
        setup->misc [i].data = 0;
      }
    }
    free (setup->misc);
    setup->misc  = 0;
    setup->miscs = 0;
  }
}

#endif // linux

/***************************************************************************/
