/*
   Written by Pieter J. Schoenmakers <tiggr@ics.ele.tue.nl>

   Copyright (C) 1996-1998 Pieter J. Schoenmakers.

   This file is part of TOM.  TOM is distributed under the terms of the
   TOM License, a copy of which can be found in the TOM distribution; see
   the file LICENSE.

   $Id: tr.m,v 1.72 1999/03/03 14:40:53 tiggr Exp $  */

#import "tr.h"
#import "tom/trt.h"

LTTInstance *tom_string_instance;

/* The TOP_UNIT is the one for which we work; the APP_UNIT is either the
   TOP_UNIT, or the unit into which the TOP_UNIT is going to be loaded.  */
LTTUnit *app_unit, *top_unit;

/* This is the TOP_UNIT iff FLAG_1, otherwise, it is NIL.  */
LTTUnit *single_unit;

/* Any units named with -x.  */
TLVector *extra_units;

/* Iff !0, we're doing single-unit resolution.  */
int flag_1 = 0;

/* Iff !0, we're doing FLAG_1 on the tom unit.  */
int single_tom_unit;

TLSet *app_sel_arg_types;
TLSet *app_selectors;
TLSet *app_units;

/* The units which are part of this module.  */
TLSet *included_units;

TLSet *app_strings;

TLCons *super_refs;
int num_super_refs;

/* Flags.  */
int flag_no_def, flag_verbose_deferred;
int flag_instances_deferred, flag_instances_non_deferred;

TLCons *instances_deferred, *instances_non_deferred;

void
usage (int rc, char *s)
{
  if (s)
    formac (V_stderr_, @"%@: bad option/argument: %s\n", prog, s);

  formac (rc || s ? V_stderr_ : V_stdout_,
@"tr (%@), version %s\n\
usage: %@ [options] [unit]\n\
options:\n\
 --version	print version on stdout and exit\n\
 -I dir		add <dir> to the directories in which to find depended units\n\
 -u unit	name the unit to be resolved\n\
\n\
 -1		create single-unit definition file\n\
 -x unit	in addition to what the unit file says, require the <unit>\n\
 -n		do not create definition file\n\
 -y unit	create def for dynamic loading into <unit>\n\
 -static	create def for static resolution\n\
 -Y		create def for dynamic resolution (default)\n\
\n\
 -v				increase verbosity\n\
 -v-deferred			report deferred selectors of each object\n\
 -v-deferred-instances		list deferred instances\n\
 -v-non-deferred-instances	list non-deferred instances\n\
", prog, long_version, prog);

  exit (rc);
}

void
tr_init (int argc, char **argv)
{
  lti_init (argc, argv);
}

void
tr_finish_init (void)
{
  lti_finish_init ();

  tom_string_instance = [ltt_unit_tom instanceNamed: TOM_CLASS_NAME_STRING];
  if (!tom_string_instance)
    error (@"Can't find instance %@.%@",
	   [ltt_unit_tom lttName], TOM_CLASS_NAME_STRING);
}

void
list_metas (TLCons *l)
{
  LTIMeta *m;

  while (l)
    {
      DECONS (l, m, l);
      formac (V_stderr_, @" %@.%@", [[[[m structure] unit] lttName] internal],
	      [[[m structure] lttName] internal]);
    }
}

int
main (int argc, char **argv)
{
  TLString *unit_name = nil, *app_unit_name = nil;
  int num_dyn_sel, num_posers = 0;
  id <TLEnumerator> e;
  int i, flag_Y = 1;
  LTTUnit *u;
  char *opt;

  tr_init (argc, argv);

  for (i = 1; i < argc; i++)
    if (argv[i][0] == '-')
      {
	BOOL ok = YES;

	switch (argv[i][1])
	  {
	  case '-':
	    if (!strcmp (argv[i], "--version"))
	      {
		if (flag_verbose)
		  printf ("tr, version %s\n", long_version);
		else
		  printf ("%s\n", short_version);
		exit (0);
	      }
	    break;

	  case '1':
	    flag_1 = 1;
	    break;

	  case 'h':
	  case '?':
	    /* Allow any option starting with a `h' or `?' to be an
	       indication of the need for help.  */
	    usage (0, NULL);
	    break;

	  case 'I':
	    opt = get_arg (&i, 2, argv, argc);
	    if (opt)
	      add_to_load_path ([[TLString stringWithCString: opt]
				 componentsSeparatedByString: @":"]);
	    break;

	  case 'n':
	    if (!strcmp (argv[i], "-n"))
	      flag_no_def = 1;
	    else
	      ok = NO;
	    break;

	  case 'u':
	    opt = get_arg (&i, 2, argv, argc);
	    if (opt)
	      unit_name = [TLString stringWithCString: opt];
	    break;

	  case 'x':
	    opt = get_arg (&i, 2, argv, argc);
	    if (opt)
	      {
		if (!extra_units)
		  extra_units = [CO_TLVector new];
		[extra_units addElement: [TLString stringWithCString: opt]];
	      }
	    break;

	  case 'v':
	    if (!strcmp (argv[i], "-v"))
	      flag_verbose++;
	    else if (!strcmp (argv[i], "-v-deferred"))
	      flag_verbose_deferred++;
	    else if (!strcmp (argv[i], "-v-deferred-instances"))
	      flag_instances_deferred++;
	    else if (!strcmp (argv[i], "-v-non-deferred-instances"))
	      flag_instances_non_deferred++;
	    else
	      ok = NO;
	    break;

	  case 'y':
	    if (!app_unit_name)
	      {
		opt = get_arg (&i, 2, argv, argc);
		if (opt)
		  app_unit_name = [TLString stringWithCString: opt];
	      }
	    else
	      ok = NO;
	    break;

	  case 's':
	    if (!strcmp (argv[i], "-static"))
	      flag_Y = NO;
	    else
	      ok = NO;
	    break;

	  case 'Y':
	    /* Re-enforce the default.  */
	    if (!strcmp (argv[i], "-Y"))
	      flag_Y = YES;
	    else
	      ok = NO;
	    break;

	  default:
	    ok = NO;
	    break;
	  }

	if (!ok)
	  usage (1, argv[i]);
      }
    else if (unit_name)
      usage (1, argv[i]);
    else
      unit_name = [TLString stringWithCString: argv[i]];

  if (!NUM_ERRORS && !unit_name)
    error (@"missing unit name");
  if (NUM_ERRORS)
    return 1;

  add_to_load_path ([TLVector vectorWithElements: DEFAULT_LOAD_PATH]);

  tr_finish_init ();

  /* Make some sense.  */
  if (flag_1)
    {
      app_unit_name = nil;
      flag_Y = YES;
    }

  /* Iff needed, read the app unit.  */
  if (app_unit_name)
    {
      app_unit = ltt_find_unit (app_unit_name);
      if (!app_unit || NUM_ERRORS)
	return 1;

      flag_Y = YES;

      e = [ltt_units valueEnumerator];
      while ((u = [e nextObject]))
	if (u != ltt_builtin_unit && u != ltt_missing_unit)
	  lti_load_unit_info (u);

      app_sel_arg_types = [TLSet setWithEnumerator:
				   [LTTSelArgTypes argumentTypes]];
      app_selectors = [TLSet setWithEnumerator:
			       [(TLVector *) [CO_LTISelector selectors]
					     enumerator]];
      app_units = [TLSet setWithEnumerator: [ltt_units valueEnumerator]];

      // XXX These can be removed...
      // Wed Jun 26 15:31:15 1996, tiggr@cobra.es.ele.tue.nl
      app_strings = [TLSet setWithEnumerator:
			     [(TLVector *) [CO_LTIStringCST allStrings]
					   enumerator]];

      /* Forget anything defined for the app unit which is not needed (or
	 desired) for the top unit.  */
      [CO_LTIStringCST sufferAmnesia];
      load_imps = nil;
    }

  /* Read the top unit.  */
  top_unit = ltt_find_unit (unit_name);
  if (!top_unit || NUM_ERRORS)
    return 1;
  if (app_unit && [app_unit dependsOn: top_unit])
    error (@"app unit `%@' depends on unit `%@'", app_unit_name, unit_name);
  if (flag_1)
    {
      ltt_current_unit = single_unit = top_unit;
      if (single_unit == ltt_unit_tom)
	single_tom_unit = 1;
    }

  if (flag_Y && !app_unit)
    app_unit = top_unit;

  included_units = [TLSet set];
  if (flag_1)
    {
      lti_load_unit_info (top_unit);
      [included_units addElement: top_unit];
    }
  else
    {
      e = [ltt_units valueEnumerator];
      while ((u = [e nextObject]))
	if (u != ltt_builtin_unit && u != ltt_missing_unit
	    && (!app_units || ![app_units memq: u]))
	  {
	    lti_load_unit_info (u);
	    [included_units addElement: u];
	  }
    }

  if (NUM_ERRORS)
    return 1;

  {
    int num_included_selectors = 0, num_load_imps = 0;
    /* If we have a main selector, it will reside in the top unit.  */
    LTTSelector *main_sel = [LTTSelector searchSelectorNamed: TO_SEL_MAIN];
    LTTMeta *main_class = 0;
    TLString *decl_name, *def_name;
    TLFILEStream *decl, *def;
    TLCons *emit_load_imps = load_imps;
    time_t t = time (0);
    LTTSelArgTypes *args;
    LTIStringCST *cst;
    TLString *string;
    LTISelector *sel;
    TLVector *vec;
    LTTMeta *str;
    int i = 0;
    int j, n;
    char *s;

    def_name = (id) formac (nil, @"%@%@.c", unit_name,
			    TOM_RESOLVER_UNIT_SUFFIX);
    def = flag_no_def ? nil : open_output_file (def_name);

    decl_name = (id) formac (nil, @"%@%@.h", unit_name,
			     TOM_RESOLVER_UNIT_SUFFIX);
    decl = open_output_file (decl_name);

    if (!decl || (!def && !flag_no_def))
      return 1;

    s = ctime (&t);
    s[24] = 0;
    string = formac (nil, @"/* Generated by tr (%@), on %s\n"
		     @"   version %s.  */\n", prog, s, long_version);
    if (def)
      {
	formac (def, @"%@#include \"%@%@.h\"\n",
		string, unit_name, TOM_RESOLVER_UNIT_SUFFIX);
	if (!app_unit)
	  formac (def, @"#define _ trt_forward\n");
      }

    formac (decl, @"%@#ifndef __include_%@_resolved\n",
	    string, quote (unit_name));
    formac (decl, @"#define __include_%@_resolved\n", quote (unit_name));
    if (flag_1)
      {
	formac (decl,
		@"#ifndef CURRENT_UNIT\n#define CURRENT_UNIT %@\n#endif\n",
		quote (unit_name));
      }
    formac (decl, @"#include <tom/trt.h>\n", string);

    /* If FLAG_1, include the directly depending units' resolved
       declaration files.  */
    if (flag_1)
      {
	e = [[top_unit unitsUsed] enumerator];
	while ((u = [e nextObject]))
	  formac (decl, @"#include <%@/%@-r.h>\n",
		  [[u lttName] internal], [[u lttName] internal]);
      }

    /* Output the selector argument types (both DECL and DEF).  */
    e = [LTTSelArgTypes argumentTypes];
    while ((args = [e nextObject]))
      {
	[args compileDeclaration: decl];
	if (def && (!app_sel_arg_types || ![app_sel_arg_types memq: args]))
	  [args compileDefinition: def];
      }

    /* Output the selectors (both DECL and DEF).  */
    vec = [CO_LTISelector selectors];
    for (i = 0, n = [vec length]; i < n; i++)
      {
	sel = [vec _elementAtIndex: i];
	[sel compileDeclaration: decl];
	if (!app_selectors || ![app_selectors memq: sel])
	  {
	    num_included_selectors++;
	    if (def)
	      [sel compileDefinition: def];
	  }
      }

    if (def)
      {
	/* Output TRTD_SELECTORS.  */
	formac (def, @"static struct trtd_selectors selectors = {{%d, 0}, {\n",
		num_included_selectors);
	vec = [CO_LTISelector selectors];
	for (j = 0, n = [vec length]; j < n; j++)
	  {
	    sel = [vec _elementAtIndex: j];
	    if (!app_selectors || ![app_selectors memq: sel])
	      formac (def, @"&%@,\n", [[sel structure] outputDefinitionName]);
	  }
	formac (def, @"}};\n");

	/* Output the string definitions.  */
	e = [CO_LTIStringCST strings];
	while ((cst = [e nextObject]))
	  [cst compileDefinition: def];

	/* Output the string references.  */
	vec = [CO_LTIStringCST allStrings];
	for (j = 0, n = [vec length]; j < n; j++)
	  {
	    cst = [vec _elementAtIndex: j];
	    [cst compileReference: def];
	  }

	/* Output information on the dynamic selectors.  */
	num_dyn_sel = [dynamic_selectors length];
	if (num_dyn_sel)
	  {
	    LTISelector *alias;

	    formac (def, @"\nstatic struct trtd_selector_map "
		    @"dynamic_selectors[] = {");
	    e = [dynamic_selectors keyEnumerator];
	    while ((alias = [e nextObject]))
	      {
		LTISelector *orig = [dynamic_selectors objectForKey: alias];

		formac (def, @"\n{&%@, &%@},",
			[[alias structure] outputDefinitionName],
			[[orig structure] outputDefinitionName]);
	      }
	    formac (def, @"\n};\n");

	    if (!app_unit)
	      {
		formac (def, @"static struct trt_dynamic_selectors "
			@"dynamic_selectors_def = {{%d, 0}, {", num_dyn_sel);

		for (i = 0; i < num_dyn_sel; i++)
		  formac (def, @"\n&dynamic_selectors[%d],", i);

		formac (def, @"}};\nstruct trt_dynamic_selectors "
			@"*trt_dynamic_selectors = &dynamic_selectors_def;\n");
	      }
	  }

	if (!app_unit)
	  {
	    /* Output the thread-local stuff.  */
	    formac (def, @"\nint trt_thread_local_size = %d;",
		    next_thread_local_offset);
	    formac (def, @"\nstatic char trt_thread_data[%d];",
		    next_thread_local_offset);
	    formac (def, @"\nchar *trt_start_thread_data = "
		    @"&trt_thread_data[0];");

	    /* Output next extension id.  */
	    formac (def, @"\nint trt_num_eids = %d;",
		    [LTIExtension numExtensions]);
	  }
      }

    /* Compile the reference type declarations.  */
    e = [LTTMeta metas];
    while ((str = [e nextObject]))
      {
	LTIMeta *m = [str semantics];
	LTTUnit *u = [str unit];

	if (!flag_1 || u == top_unit
	    || (single_tom_unit && u == ltt_builtin_unit))
	  [m compileReferenceDeclaration: decl];
	else
	  [m dontCompileReferenceDeclaration];
      }

    /* Compile the class struct declarations.  */
    e = [LTTMeta metas];
    while ((str = [e nextObject]))
      {
	LTTUnit *u = [str unit];

	/* Handle the builtin metas for the unit tom.  */
	if (!flag_1 || u == top_unit
	    || (single_tom_unit && u == ltt_builtin_unit))
	  [[str semantics] compileDeclaration: decl];
      }

    if (def)
      {
	/* Generate the ordered class sequences.  */
	[CO_LTIMeta generateClassSequences];

	/* For static resolution, create TRT_SELECTORS
	   and output the buckets.  */
	if (!app_unit)
	  {
	    formac (def,
		    @"struct trtd_selectors *trt_selectors = &selectors;\n");

	    formac (def, @"extern void _ ();\n");
	    e = [LTTMeta metas];
	    while ((str = [e nextObject]))
	      [[str semantics] compileBuckets: def];
	  }

	e = [LTTMeta metas];
	while ((str = [e nextObject]))
	  [[str semantics] compileTables: def];

	e = [LTTMeta metas];
	while ((str = [e nextObject]))
	  {
	    LTTUnit *u = [str unit];

	    if (!flag_1 || u == top_unit
		|| (single_tom_unit && u == ltt_builtin_unit))
	      [[str semantics] compileDefinition: def];
	  }

	/* Report deferred instances.  */
	if (instances_deferred)
	  {
	    formac (V_stderr_, @"deferred instances:");
	    list_metas (instances_deferred);
	    formac (V_stderr_, @"\n");
	  }

	/* Report non-deferred instances.  */
	if (instances_non_deferred)
	  {
	    formac (V_stderr_, @"non-deferred instances:");
	    list_metas (instances_non_deferred);
	    formac (V_stderr_, @"\n");
	  }

	/* For static resolution, output runtime structures.  */
	if (!app_unit)
	  {
	    /* Output bucket count information.  */
	    int ns = [CO_LTISelector numSelectors];

	    formac (def, @"\nint num_buckets = %d;",
		    (ns + TRT_BUCKET_SIZE - 1) / TRT_BUCKET_SIZE);
	    formac (def, @"\nint trt_num_buckets = %d;", [mdt_buckets length]);

	    /* Output the runtime TRT_METAS.  */
	    formac (def, @"\nstruct trt_metas trt_metas_def = {{%d, 0}, {",
		    [LTTMeta numMetas] / 2);
	    e = [LTTMeta metas];
	    while ((str = [e nextObject]))
	      if ([str classp])
		formac (def, @"\n(void *) &%@,", [str definitionName]);
	    formac (def, @"\n}};\n");

	    formac (def, @"struct trt_metas *trt_metas = &trt_metas_def;\n");
	  }

	if (flag_verbose)
	  {
	    int ns = [CO_LTISelector numSelectors];
	    int nb = [mdt_buckets length];
	    int rns = ((ns + TRT_BUCKET_SIZE - 1) / TRT_BUCKET_SIZE
		       * TRT_BUCKET_SIZE);

	    formac (V_stdout_, @"%d classes, %d selectors, %d buckets, "
		    @"saving %2.1f%%/%2.1f%%\n", i, ns, nb,
		    100 - 100.0 * ((double) (nb * TRT_BUCKET_SIZE)
				   / (2 * i * ns)),
		    100 - 100.0 * ((double) (nb * TRT_BUCKET_SIZE
					     + 2 * i * rns / TRT_BUCKET_SIZE)
				   / (2 * i * ns)));
	  }

#if !GLOBAL_RESOLUTION
	/* Declare the load imps so their addresses can be referenced.  */
	{
	  TLCons *c = emit_load_imps;
	  LTIMethod *s;

	  while (c)
	    {
	      DECONS (c, s, c);
	      formac (def, @"int %@ ();\n", [s methodName]);
	    }
	}
#endif

	/* Output information on the `load' implementations.  */
	while (emit_load_imps)
	  {
	    LTIMethod *s;

	    DECONS (emit_load_imps, s, emit_load_imps);

	    if (!num_load_imps)
	      formac (def, @"\nstatic struct trtd_load_imp load_imps[] = {");
	    formac (def, @"%@{%@, (void *) &%@}",
		    num_load_imps ? @",\n" : @"\n",
		    [s methodName], [[[s meta] structure] referenceName]);
	    num_load_imps++;		
	  }
	if (num_load_imps)
	  formac (def, @"\n};\n");

	/* For single-unit resolution, retrieve correct information on the
           super references, emitting them.  */
	if (single_unit)
	  {
	    TLDictionary *dict = [single_unit superReferences];
	    id <TLEnumerator> e;
	    LTTMeta *from, *to;

	    super_refs = nil;

	    e = [dict keyEnumerator];
	    while ((from = [e nextObject]))
	      {
		TLSet *s = [dict objectForKey: from];
		id <TLEnumerator> f = [s enumerator];

		while ((to = [f nextObject]))
		  {
		    formac (def, 
		      @"struct _es_c_tom_State *%@ = &%@.c_tom_State;\n",
			    [from outputSuperReference: to],
			    [[to posedSelf] metaDefinitionName]);
		    super_refs = CONS (CONS ([from semantics], [to semantics]),
				       super_refs);
		  }
	      }
	  }

	/* Output information on the super references.  */
	while (super_refs)
	  {
	    LTIMeta *subm, *superm;
	    TLCons *c;

	    DECONS (super_refs, c, super_refs);
	    DECONS (c, subm, superm);

	    if (!num_super_refs)
	      formac (def, @"\nstatic struct trtd_super super_refs[] = {");
	    formac (def, @"%@\n{(void *) &%@, (void *) &%@, (void *) &%@}",
		    num_super_refs ? @"," : @"",
		    [[subm structure] outputSuperReference: [superm structure]],
		    [[subm structure] metaReferenceName],
		    [[superm structure] metaReferenceName]);
	    num_super_refs++;
	  };
	if (num_super_refs)
	  formac (def, @"\n};\n");

	/* Deduce the main information, if we're supposed to have a `main'.
	   With FLAG_1 we may or may not have a main selector, and we may or
	   may not have a main method.  */
	if ((!flag_1 && (!app_unit || app_unit == top_unit))
	    || (flag_1 && main_sel))
	  if (!main_sel)
	    error (@"definition of main not found");
	  else
	    {
	      TLVector *methods = [[main_sel semantics] methods];

	      if (![methods length])
		if (flag_1)
		  main_sel = nil;
		else
		  error (@"definition of main not found");
	      else if ([methods length] > 1)
		{
		  /* This is an internal error as it does not check posing
		     and such.  */
		  internal (@"ambiguous definition of main");
		}
	      else
		{
		  LTIMethod *m = [methods _elementAtIndex: 0];
		  
		  main_class = [[m meta] structure];
		}
	    }

	/* Output information on the units contained in this module.  */
	e = [included_units enumerator];
	while ((u = [e nextObject]))
	  {
	    int num_extensions = 0, num_metas = 0;
	    TLCons *included_extensions;
	    id <TLEnumerator> ie;

	    formac (def, @"\nstatic struct trtd_extension *"
		    @"trtd_extensions_%@[] = {\n", [[u lttName] external]);

	    ++search_mark;
	    included_extensions = [[ltt_class_all semantics]
				    compile: def unit: u extensionList:
				      [[ltt_instance_all semantics]
					compile: def unit: u extensionList: nil
					mark: search_mark]
				    mark: search_mark];
	    formac (def, @"};\n");

	    num_extensions = [included_extensions length];

	    ie = [u instances];
	    while ((str = [ie nextObject]))
	      {
		if (!num_metas)
		  formac (def, @"\nstatic struct trt_class *"
			  @"trtd_metas_%@[] = {", [[u lttName] external]);
		formac (def, @"%@(void *) &%@", num_metas ? @",\n" : @"\n",
			[str metaDefinitionName]);
		num_metas++;
	      }
	    if (num_metas)
	      formac (def, @"\n};\n");

	    formac (def, @"\nstatic struct trtd_unit %@ = {\n"
		    @"{%#, %d},\n%@, %d,\n%@, %d\n};\n",
		    [u outputDefinitionName], [[u lttName] internal],
		    [[[u lttName] internal] length],
		    (num_extensions
		     ? formac (nil, @"trtd_extensions_%@",
			       [[u lttName] external])
		     : (id) @"0"), num_extensions,
		    (num_metas
		     ? formac (nil, @"trtd_metas_%@", [[u lttName] external])
		     : (id) @"0"), num_metas);
	  }

	i = 0;
	e = [[top_unit orderedUnits: nil] enumerator];
	formac (def, @"\nstatic struct trtd_unit *trtd_units[] = {");
	while ((u = [e nextObject]))
	  if (!included_units || [included_units memq: u])
	    {
	      formac (def, @"%@&%@", i ? @",\n" : @"\n",
		      [u outputDefinitionName]);
	      i++;
	    }
	formac (def, @"\n};\n");

	e = [included_units enumerator];
	while ((u = [e nextObject]))
	  {
	    TLCons *l = [u posers];

	    while (l)
	      {
		LTTMeta *poser, *posed;
		TLCons *p;

		DECONS (l, p, l);
		DECONS (p, poser, posed);

		formac (def, @"%@{(void *) &%@, (void *) &%@}", !num_posers
			? @"\nstatic struct trtd_poser posers[] =\n{" : @"\n,",
			[poser metaDefinitionName], [posed metaDefinitionName]);
		num_posers++;
	      }
	  }
	if (num_posers)
	  formac (def, @"};");

	/* Output module description.  */
	if (single_unit)
	  {
	    formac (def, @"\nstruct trtd_module %s_%@ = {\ntrtd_units",
		    TRT_MODULE_INFO, [single_unit outputName]);
	  }
	else
	  formac (def, @"\nstruct trtd_module %s = {\ntrtd_units",
		  (!app_unit || app_unit == top_unit ? "trt_main_module_info"
		   : TRT_MODULE_INFO));

	/* num_units */
	formac (def, @",\n%d", [included_units length]);

	/* selectors */
	formac (def, @",\n&selectors");

	/* dynamic_selectors, num_dynamic_selectors */
	if (num_dyn_sel)
	  formac (def, @",\ndynamic_selectors, %d", num_dyn_sel);
	else
	  formac (def, @",\n0, 0");

	/* super_refs, num_super_refs */
	if (num_super_refs)
	  formac (def, @",\nsuper_refs, %d", num_super_refs);
	else
	  formac (def, @",\n0, 0");

	/* load_imps, num_load_imps */
	if (num_load_imps)
	  formac (def, @",\nload_imps, %d", num_load_imps);
	else
	  formac (def, @",\n0, 0");

	/* main_selector, main_class */
	if (main_sel)
	  formac (def, @",\n&%@, (void *) &%@",
		  [main_sel outputDefinitionName], [main_class referenceName]);
	else
	  formac (def, @",\n0, 0");

	if (num_posers)
	  formac (def, @",\nposers, %d", num_posers);
	else
	  formac (def, @",\n0, 0");

	formac (def, @"\n};\n");

	if (single_unit)
	  {
	    /* This being a constructor won't harm?  */
	    formac (def, @"\nvoid trt_register_unit_%@ (void)"
		    @" __attribute__ ((constructor));\n",
		    [single_unit outputName]);

	    formac (def, @"\nvoid\ntrt_register_unit_%@ (void)\n\
{\n\
  static char initialized = 0;\n\
  if (initialized)\n\
    return;\n\
  initialized = 1;\n", [single_unit outputName]);

	    e = [[top_unit unitsUsed] enumerator];
	    while ((u = [e nextObject]))
	      formac (def, @"  trt_register_unit_%@ ();\n", [u outputName]);

	    if (extra_units)
	      { 
		e = [extra_units enumerator];
		while ((u = [e nextObject]))
		  formac (def, @"  trt_register_unit_%@ ();\n", u);
	      }

	    formac (def, @"  trt_resolve_module (&%s_%@);\n", TRT_MODULE_INFO,
		    [single_unit outputName]);

	    formac (def, @"}\n");

	    if (main_sel)
	      formac (def, @"void (*trt_main_registration) (void)"
		      @" = trt_register_unit_%@;\n", [single_unit outputName]);
	  }

	[def close];
      }

    formac (decl, @"#endif\n");
    [decl close];

    if (NUM_ERRORS)
      {
	if (!flag_no_def && unlink ([def_name cString]))
	  error (@"unlink %@: %@", def_name, ERRMSG);
	if (unlink ([decl_name cString]))
	  error (@"unlink %@: %@", decl_name, ERRMSG);
      }
  }

  if (flag_verbose)
    formac (V_stderr_,
	    @"total %d line%s, %d error%s, %d warning%s\n",
	    total_num_lines, total_num_lines == 1 ? "" : "s",
	    total_num_errors, total_num_errors == 1 ? "" : "s",
	    total_num_warnings, total_num_warnings == 1 ? "" : "s");

  return !!NUM_ERRORS;
}
