/* Implementation of TLString class.
   This file is part of TL, Tiggr's Library.
   Written by Tiggr <tiggr@es.ele.tue.nl>
   Copyright (C) 1995, 1996 Pieter J. Schoenmakers
   TL is distributed WITHOUT ANY WARRANTY.
   See the file LICENSE in the TL distribution for details.

   $Id: TLString.m,v 1.3 1998/01/10 11:07:36 tiggr Exp $  */

#import "tl/support.h"
#import "tl/TLString.h"
#import "tl/TLStream.h"
#import "tl/TLStringStream.h"
#import "tl/TLPatchedRoots.h"
#import "tl/TLSymbol.h"
#import <string.h>
#import <ctype.h>
#import <sys/param.h>

/* Hash the LEN bytes at S.  XXX Maybe LEN should be clipped.  */
unsigned int
tol_hash_string (const char *s, int len)
{
  unsigned int h, i;

  for (h = i = 0; i < len; i++)
    h = (h << 3) ^ s[i];
  return (h);
} /* tol_hash_string */

@implementation NXConstantString (TLNXConstantString)

#include "StringMethods.m"

#if NEXT_RUNTIME
/* This method is needed to complement NeXT's NXConstantString.  */
-(const char *) cString
{
  return (c_string);
} /* cString */
#endif

-(TLString *) string
{
  return [CO_TLString stringWithCString: c_string length: len];
} /* -string */

/******************** doing difficult avoidance ********************/

/* Make sure NXConstantString is never retained or released.  */

-autorelease
{
  return (self);
} /* -autorelease */

-free
{
  return (nil);
} /* -free */

-(void) gcMark
{
} /* -gcMark */

-(void) release
{
} /* -release */

-retain
{
  return (self);
} /* -retain */

-(unsigned int) retainCount
{
  return (42);
} /* retainCount */

@end

@implementation TLString

#include "StringMethods.m"

/******************** private methods ********************/

+initialize
{
  if (!tll_full_range)
    {
      tll_full_range = [TLRange rangeWithStart: 0 length: -1];
      [(id) tll_full_range gcLock];
    }
  return (self);
} /* +initialize */

-initWithCString: (const char *) an_s length: (int) l
{
  len = l;
  c_string = xmalloc (len + 1);
  memcpy (c_string, an_s, len);
  c_string[len] = 0;
  return (self);
} /* -initWithCString: */

-initWithCStringNoCopy: (char *) an_s length: (int) l
{
  len = l;
  c_string = an_s;
  return (self);
} /* -initWithCStringNoCopy: */

-initWithStream: (id <TLInputStream>) stream length: (int) length
{
  int i;

  c_string = xmalloc (length + 1);

  /* XXX Wouldn't persistent read be nice?  */
  for (len = 0; len < length; len += i)
    {
      i = [stream readBytes: length - len intoBuffer: c_string + len];
      if (i <= 0)
	{
	  [self warning: "error or EOF"];
	  INIT_RETURN_NIL ();
	}
    }

  c_string[len] = 0;
  return (self);
} /* -initWithStream:length: */

-initWithStream: (id <TLInputStream>) stream
{
  int i, cap = 512;

  for (;;)
    {
      cap *= 2;
      c_string = xrealloc (c_string, cap);
      i = [stream readBytes: cap - len intoBuffer: c_string + len];
      if (i < 0)
	INIT_RETURN_NIL ();
      else if (!i)
	break;
      len += i;
    }

  c_string = xrealloc (c_string, len + 1);
  c_string[len] = 0;
  return (self);
} /* -initWithStream:length: */

-initWithString: (id <TLString>) s length: (int) l byPadding: (int) pad
	   with: (char) c
{
  int i, sl = [s length];
  int ls, rs;

  if (l < sl)
    l = sl;
  len = l;
  c_string = xmalloc (1 + len);

  if (pad < 0)
    ls = 0, rs = l - sl;
  else if (pad > 0)
    rs = 0, ls = l - sl;
  else
    {
      rs = ls = (l - sl) / 2;
      if (rs + ls != l - sl)
	rs++;
    }

  if (ls)
    for (i = 0; i < ls; i++)
      c_string[i] = c;
  if (rs)
    for (i = len - rs; i < len; i++)
      c_string[i] = c;
  memcpy (c_string + ls, [s cString], sl);

  c_string[len] = 0;
  return (self);
} /* +initWithString:length:byPadding:with: */

/******************** public methods ********************/

+(TLString *) stringWithCString: (const char *) an_s
{
  return ([(TLString *) [self gcAlloc]
	   initWithCString: an_s length: strlen (an_s)]);
} /* +stringWithCString: */

+(TLString *) stringWithCStringNoCopy: (char *) an_s
{
  return ([(TLString *) [self gcAlloc]
	   initWithCStringNoCopy: an_s length: strlen (an_s)]);
} /* +stringWithCStringNoCopy: */

+(TLString *) stringWithCStringNoCopy: (char *) an_s length: (int) l
{
  return ([(TLString *) [self gcAlloc]
	   initWithCStringNoCopy: an_s length: l]);
} /* +stringWithCStringNoCopy:length: */

+(TLString *) stringWithCString: (const char *) an_s length: (int) l
{
  return ([(TLString *) [self gcAlloc] initWithCString: an_s length: l]);
} /* +stringWithCString:length: */

+(id <TLString>) stringWithHTMLQuotedCString: (const char *) s length: (int) l
{
  /* Slow...  */
  id r = [CO_TLMutableString mutableString];
  int i, j;

  for (i = 0; i < l; i = j)
    {
      for (j = i; j < l; j++)
	if (s[j] == '%')
	  break;
      if (j != i)
	[r writeBytes: j - i fromBuffer: s + i];
      if (j != l)
	{
	  int k, v;

	  for (k = j + 1, v = 0; k - j < 3 && k < l; k++)
	    if (!isxdigit (s[k]))
	      break;
	    else
	      {
		int digit = s[k] - '0';
		if (digit > 9)
		  digit -= 'A' - 1 - '9';
		if (digit > 15)
		  digit -= 'a' - 'A';
		v = 16 * v + digit;
	      }

	  [r writeByte: v];
	  j = k;
	}
    }
  return (r);
} /* stringWithHTMLQuotedCString:length: */

+(TLString *) stringWithFormat: (id <TLString>) fmt, ...
{
  va_list ap;
  id retval;

  va_start (ap, fmt);
  /* XXX Return a real string, not the mutable one returned by vformac.  */
  retval = [self stringWithString: vformac (nil, fmt, ap)];
  va_end (ap);
  return (retval);
} /* +stringWithFormat: */

+(TLString *) stringWithFormat: (id) fmt: (va_list) ap
{
  /* Return a real string, not the mutable one returned by vformac.  */
  return ([self stringWithString: vformac (nil, fmt, ap)]);
} /* +stringWithFormat:: */

+(TLString *) stringWithFormatVector: (TLVector *) vector
{
  return ([self stringWithString:
	   [CO_TLMutableString mutableStringWithFormatVector: vector]]);
} /* +stringWithFormatVector: */

+(TLString *) stringWithStream: (id <TLInputStream>) stream
{
  return ([[self gcAlloc] initWithStream: stream]);
} /* +stringWithStream: */

+(TLString *) stringWithStream: (id <TLInputStream>) stream
 length: (int) length
{
  return ([[self gcAlloc] initWithStream: stream length: length]);
} /* +stringWithStream:length: */

+(TLString *) stringWithString: (id <TLString>) s
{
  return ([(TLString *) [self gcAlloc]
	   initWithCString: [s cString] length: [s length]]);
} /* +stringWithString */

+(TLString *) stringWithString: (id <TLString>) s withLength: (int) l
 byPadding: (int) pad
{
  return ([[self gcAlloc] initWithString: s length: l
	   byPadding: pad with: ' ']);
} /* +stringWithString:withLength:byPadding: */

-(id <TLString>) basename
{
  TLRange *r = [self rangeOfString: @"/" options: TLSEARCH_BACKWARD];

  return (r ? [self stringWithRange: [TLRange rangeWithStart: 1 + [r _start]
				      length: -1]] : self);
} /* -basename */

-(int) characterAtIndex: (int) idx
{
  if (idx < -len || idx >= len)
    return (-1);
  return (c_string[idx < 0 ? len + idx : idx]);
} /* -characterAtIndex: */

-(id <TLVector>) componentsSeparatedByString: (id <TLString>) s
{
  id <TLMutableVector> v;
  id <TLRange> r;

  /* Splitting into a <TLVector> of characters is not very useful, as a
     TLString itself is such a beast.  */
  if (![s length])
    return (self);

  v = [CO_TLVector vector];
  for (r = tll_full_range;;)
    {
      int i = [r _start], j;

      r = [self rangeOfString: s range: r];
      if (r)
	{
	  j = [r _start];
	  [v addElement: [self stringWithStart: i length: j - i]];
	  r = [TLRange rangeWithStart: j + [r length] length: -1];
	}
      else
	{
	  [v addElement: [self stringWithStart: i length: -1]];
	  break;
	}
    }
  return (v);
} /* -componentsSeparatedByString: */

-(const char *) cString
{
  return (c_string);
} /* cString */

-elementAtIndex: io
{
  /* Do not call `_elementAtIndex:' for the sake of speed.  */
  int i = [io integerPIntValue];
  return (i < len && i >= 0 ? [CO_TLNumber numberWithChar: c_string[i]] : nil);
} /* -elementAtIndex: */

-_elementAtIndex: (int) i
{
  return (i < len && i >= 0 ? [CO_TLNumber numberWithChar: c_string[i]] : nil);
} /* -_elementAtIndex: */

-(id <TLString>) expandedFilename
{
  return ([self expandedFilenameRelativeTo: nil]);
} /* -expandedFilename */

-(id <TLString>) expandedFilenameRelativeTo: (id <TLString>) dir
{
  if (len < 1 || *c_string == '/')
    return (self);

  if (*c_string == '~')
    {
      if (len == 1 || c_string[1] == '/')
	return ([isa stringWithFormat: @"%s%s", getenv ("HOME"), c_string + 1]);
    }
  else
    {
      if (!dir)
	{
	  char buf[MAXPATHLEN + 1];

	  /* XXX Check.  */
	  getcwd (buf, sizeof (buf));
	  dir = [isa stringWithFormat: @"%s", buf];
	}
      return ([isa stringWithFormat: @"%@/%s", dir, c_string]);
    }

  return (self);
} /* -expandedFilenameRelativeTo: */

-(BOOL) _fileExistsP
{
  return (!access (c_string, F_OK));
} /* _fileExistsP */

-fileExistsP
{
  return ([self _fileExistsP] ? Qt : nil);
} /* -fileExistsP */

-(TLMutableString *) htmlQuoted
{
  id r = [CO_TLMutableString mutableString];
  int i, j;

  /* XXX This is, semantically, not good enough!  */
  for (i = 0; i < len; i = j)
    {
      for (j = i; j < len; j++)
	if (!isalnum (c_string[j]) && c_string[j] != '/' && c_string[j] != ':')
	  break;
      if (j != i)
	[r writeBytes: j - i fromBuffer: c_string + i];
      if (j != len)
	{
	  int v = (unsigned char) c_string[j];
	  /* XXX Should have zero left-padding.  */
	  formac (r, @"%%%s%X", v < 16 ? "0" : "", v);
	  j++;
	}
    }
  return (r);
} /* -htmlQuoted */

-(TLMutableString *) htmlUnquoted
{
  id r = [CO_TLMutableString mutableString];
  int i, j;

  /* XXX This is, semantically, not good enough!  */
  for (i = 0; i < len; i = j)
    {
      for (j = i; j < len; j++)
	if (c_string[j] == '%')
	  break;
      if (j != i)
	[r writeBytes: j - i fromBuffer: c_string + i];
      if (j != len)
	{
	  int k, v;

	  for (k = j + 1, v = 0; k - j < 3 && k < len; k++)
	    if (!isxdigit (c_string[k]))
	      break;
	    else
	      {
		int digit = c_string[k] - '0';
		if (digit > 9)
		  digit -= 'A' - 1 - '9';
		if (digit > 15)
		  digit -= 'a' - 'A';
		v = 16 * v + digit;
	      }

	  [r writeByte: v];
	  j = k;
	}
    }
  return (r);
} /* -htmlUnquoted */

-(int) indexOfCharacterMatchFrom: (id <TLString>) s
{
  const char *cs = [s cString];
  int i, l = [s length];
  
  for (i = 0; i < len; i++)
    if (memchr (cs, c_string[i], l))
      return (i);
  return (-1);
} /* -indexOfCharacterMatchFrom: */

-(int) length
{
  return (len);
} /* -length */

-(id <TLRange>) rangeOfString: (id <TLString>) s
{
  return ([self rangeOfString: s range: tll_full_range options: 0]);
} /* -rangeOfString: */

-(id <TLRange>) rangeOfString: (id <TLString>) s options: (unsigned int) options
{
  return ([self rangeOfString: s range: tll_full_range options: options]);
} /* -rangeOfString:options: */

-(id <TLRange>) rangeOfString: (id <TLString>) s range: (id <TLRange>) r
{
  return ([self rangeOfString: s range: r options: 0]);
} /* -rangeOfString:range: */

-(id <TLRange>) rangeOfString: (id <TLString>) s
			range: (id <TLRange>) r
		      options: (unsigned int) options
{
  int i, start, length, end, sl, fold_case;
  const char *ss;
  TLRange *ret = nil;

  start = [r _start];
  if (start > len)
    return (nil);

  length = [r length];
  if (length < 0)
    length = len - start;

  if (start < 0)
    length += start, start = 0;
  if (start + length > len)
    length = len - start;

  ss = [s cString];
  sl = [s length];

  fold_case = options & TLSEARCH_FOLD_CASE;

  /* If this must be a full match, the lengths must be equal.  */
  if ((options & TLSEARCH_FULL_MATCH)
      && sl != length)
    return (nil);

  if (!sl)
    {
      /* The empty string always matched at the start of the search range.  */
      ret = [TLRange rangeWithStart: ((options & TLSEARCH_BACKWARD)
				      ? start + length : start) length: 0];
    }
  else if (options & TLSEARCH_BACKWARD)
    {
      for (i = start + length - sl; i >= start; i--)
	if (fold_case ? !memcasecmp (c_string + i, ss, sl)
	    : (c_string[i] == *ss
	       && (!memcmp (c_string + i, ss, sl))))
	  {
	    ret = [TLRange rangeWithStart: i length: sl];
	    break;
	  }
    }
  else
    {
      for (i = start, end = start + length - sl; i <= end; i++)
	if (fold_case ? !memcasecmp (c_string + i, ss, sl)
	    : (c_string[i] == *ss
	       && (!memcmp (c_string + i, ss, sl))))
	  {
	    ret = [TLRange rangeWithStart: i length: sl];
	    break;
	  }
    }
    
  return (ret);
} /* -rangeOfString:range: */

-(id <TLRange>) strcmp: (id <TLString>) s
{
  return ([self rangeOfString: s options: TLSEARCH_FULL_MATCH]);
} /* -strcmp: */

-(id <TLRange>) strcasecmp: (id <TLString>) s
{
  return ([self rangeOfString: s
	   options: TLSEARCH_FULL_MATCH | TLSEARCH_FOLD_CASE]);
} /* -strcasecmp: */

-(id <TLRange>) strstr: (id <TLString>) s
{
  return ([self rangeOfString: s]);
} /* -strstr: */

-(id <TLRange>) strcasestr: (id <TLString>) s
{
  return ([self rangeOfString: s options: TLSEARCH_FOLD_CASE]);
} /* -strcasestr: */

-(id <TLRange>) strrstr: (id <TLString>) s
{
  return ([self rangeOfString: s options: TLSEARCH_BACKWARD]);
} /* -strrstr: */

-(id <TLRange>) strcaserstr: (id <TLString>) s
{
  return ([self rangeOfString: s
	   options: TLSEARCH_FOLD_CASE | TLSEARCH_BACKWARD]);
} /* -strcaserstr: */

-(id <TLInputStream>) stream
{
  return ([TLStringStream streamWithString: self]);
} /* -stream */

-(TLString *) string
{
  return self;
} /* -string */

-(TLString *) stringWithRange: (id <TLRange>) r
{
  return ([self stringWithStart: [r _start] length: [r length]]);
} /* -stringWithRange: */

-(TLString *) stringWithStart: (int) start length: (int) length
{
  if (start > len)
    return (nil);
  if (length < 0)
    length = len - start;
  if (start < 0)
    length += start, start = 0;
  if (start + length > len)
    length = len - start;
  return ([isa stringWithCString: c_string + start length: length]);
} /* -stringWithStart:length: */

-(id <TLString>) stringWithUppercase
{
  char *nb = xmalloc (1 + len);
  int i;

  memcpy (nb, c_string, len + 1);

  for (i = 0; i < len; i++)
    if (nb[i] >= 'a' && nb[i] <= 'z')
      nb[i] = nb[i] - 'a' + 'A';
  return ([(TLString *) [isa gcAlloc] initWithCStringNoCopy: nb length: len]);
} /* -stringWithUppercase */

-(id <TLString>) stringWithLowercase
{
  char *nb = xmalloc (1 + len);
  int i;

  memcpy (nb, c_string, len + 1);

  for (i = 0; i < len; i++)
    if (nb[i] >= 'A' && nb[i] <= 'Z')
      nb[i] = nb[i] - 'A' + 'a';
  return ([(TLString *) [isa gcAlloc] initWithCStringNoCopy: nb length: len]);
} /* -stringWithLowercase */

-(id <TLString>) stringWithCapitals
{
  return (UNIMPLEMENTED_POINTER_METHOD);
} /* -stringWithCapitals */

-(id <TLNumber>) writeRange: (id <TLRange>) r
 toStream: (id <TLOutputStream>) stream
{
  int start = [r _start];
  int length = [r length];
  int n;

  if (start > len)
    return (tll_small_int[0]);
  if (length < 0)
    length = len - start;
  if (start < 0)
    length += start, start = 0;
  if (start + length > len)
    length = len - start;
  n = [stream writeBytes: length fromBuffer: c_string + start];
  if (n < 0)
    {
      [self warning: "%#: failed to write (%d, %d) to %#: %s",
       self, start, length, stream, ERRMSG];
      n = 0;
    }
  return ((n || !length) ? [CO_TLNumber numberWithInt: n] : nil);
} /* -writeRange:toStream: */

/******************** garbage collection ********************/

-(void) dealloc
{
  xfree (c_string);
} /* -dealloc */

@end
