/* Implementation of TLLSubroutine: TLL C messaging.
   This file is part of TL, Tiggr's Library.
   Written by Tiggr <tiggr@es.ele.tue.nl>
   Copyright (C) 1995, 1996 Pieter J. Schoenmakers
   TL is distributed WITHOUT ANY WARRANTY.
   See the file LICENSE in the TL distribution for details.

   $Id: TLLSubroutine.m,v 1.2 1998/02/23 14:17:33 tiggr Exp $  */

#import "tl/support.h"
#import "tl/TLLSubroutine.h"
#import "tl/TLCons.h"
#import "tl/subr.h"
#import "tl/TLGC.h"

#if SUPPORT_DEBUG
tll_invocation_info *tll_invocation_stack;
int tll_invocation_num, tll_invocation_cap, max_lisp_eval_depth = 200;
#endif

/* Invoke a method of the RECEIVER.  SELECTOR is the selector symbol to
   invoke and ARGV are the args to that selector.  NUM_ARGS is always the
   number of elements in ARGV.  Iff EVAL, the elements of ARGV have not yet
   been evaluated.  In that case, ARGV is modified!  */
id
tll_invoke_method (id receiver, TLSymbol *selector,
		   id *argv, int num_args, int eval)
{
  id retval, rcv = nil;
  int i, expected_args;
  SEL sel;
  GCDECL3;

#if SUPPORT_DEBUG
  int invocation_level = tll_invocation_num - 1;
  tll_invocation_info *ii;
#endif

  if (!selector)
    [TLLSubroutine error: "NIL selector to method invocation"];

  expected_args = [selector selNumArguments];
  if (num_args != expected_args)
    [TLLSubroutine error: "wrong #arguments to selector %#: %d != %d",
     selector, num_args, expected_args];

  if (num_args)
    {
      GCPRO3 (receiver, rcv, *argv);
      _gcpro3.n = num_args;
    }
  else
    GCPRO2 (receiver, rcv);

#if SUPPORT_DEBUG
  ii = &tll_invocation_stack[invocation_level];
  if (!tll_invocation_num || ii->argc != -2)
    {
      invocation_level = tll_invocation_new ();
      ii = &tll_invocation_stack[invocation_level];
      ii->name = selector;
    }
  ii->receiver = nil;
  ii->argc = -1;
#endif

  if (!eval)
    {
      rcv = receiver;
#if SUPPORT_DEBUG
      ii->receiver = rcv;
#endif
    }
  else
    {
      /* Evaluate the arguments.  */
      rcv = EVAL (receiver);
#if SUPPORT_DEBUG
      ii->receiver = rcv;
#endif
      for (i = 0; i < num_args; i++)
	{
	  argv[i] = EVAL (argv[i]);
	  ASGN_SPROT (argv[i]);
	}
    }

#if SUPPORT_DEBUG
  /* Now the arguments have been evaluated, fill them in.  */
  ii->argl = nil;
  ii->argv = argv;
  ii->argc = num_args;
#endif

  sel = [selector selSelector];
  if (!rcv)
    {
      /* Do not actually send something to NIL, but do allow a breakpoint to
         be set by inserting some nonsense code which will not be optimized
         away.  */
      atoi ("0");

      retval = nil;
    }
  else
    {
      switch (num_args)
	{
	case 0:
	  retval = [rcv perform: sel];
	  break;

	case 1:
	  retval = [rcv perform: sel with: argv[0]];
	  break;

	case 2:
	  retval = [rcv perform: sel with: argv[0] with: argv[1]];
	  break;

	default:
	  retval = nil;		/* Prevent compiler from complaining.  */
	  [TLLSubroutine
	   error: "too many args for method invocation: %d (fix me)", num_args];
	  break;
	}

      /* XXX Patch the return type to some constant in case the return
         type was VOID.  */
    }

#if SUPPORT_DEBUG
  tll_invocation_pop (invocation_level);
#endif
  GCUNPRO;
  return (retval);
} /* tll_invoke_method */

@implementation TLLSubroutine

+(void) registerSubroutines: (struct tl_subdef *) defs
{
  id su = [TLLSubroutine class];
  id sy = [CO_TLSymbol class];
  int i;

#if SUPPORT_DEBUG
  /* XXX Naughty!  */
  static int initialized;

  if (!initialized)
    {
      initialized = 1;
      [[sy symbolWithName: @"max-lisp-eval-depth"]
	setCValue: &max_lisp_eval_depth
	encoding: @encode (__typeof__ (max_lisp_eval_depth))];
    }
#endif

  for (i = 0; defs[i].name; i++)
    {
      TLSymbol *s = [sy symbolWithName: defs[i].name
		     funValue: [su subroutineWithFunction: defs[i].fun
				args: defs[i].min : defs[i].max]];
      if (defs[i].sym)
	*defs[i].sym = s;
    }
} /* +registerSubroutines: */

+(TLLSubroutine *) subroutineWithFunction: (tll_subroutine) f
 args: (int) min : (int) max;
{
  return ([[self gcAlloc] initWithFunction: f args: min : max]);
} /* +subroutineWithFunction:args:: */

-(id <TLString>) documentation
{
  return (documentation);
} /* -documentation */

-evalWithArguments: (TLCons *) in_args
{
  id retval = nil, args = in_args;
  int num_args;
  GCDECL2;

#if SUPPORT_DEBUG
  int invocation_level = tll_invocation_num - 1;
  tll_invocation_info *ii;

  ii = &tll_invocation_stack[invocation_level];
  if (!tll_invocation_num || ii->argc != -2)
    {
      invocation_level = tll_invocation_new ();
      ii = &tll_invocation_stack[invocation_level];
      ii->name = nil;
    }
  ii->receiver = nil;
#endif

  /* Do not compute the number of arguments if the argspec if (0, uneval).  */
  if (!min_args && max_args == ARGS_UNEVAL)
    {
#if SUPPORT_DEBUG
      ii->argc = 0;
      ii->argl = args;
#endif
      retval = fun (args, -1);
#if SUPPORT_DEBUG
      tll_invocation_pop (invocation_level);
#endif
      return (retval);
    }

#if SUPPORT_DEBUG
  ii->argc = -1;
#endif

  num_args = in_args ? [in_args length] : 0;
  if (num_args < min_args || (max_args >= 0 && num_args > max_args))
    [self error: "bad number of arguments: (%d, %d, %d)",
     min_args, num_args, max_args];

  if (max_args == ARGS_UNEVAL)
    {
#if SUPPORT_DEBUG
      ii->argc = 0;
      ii->argl = args;
#endif
      retval = fun (args, num_args);
#if SUPPORT_DEBUG
      tll_invocation_pop (invocation_level);
#endif
      return (retval);
    }

  if (max_args == ARGS_NUM_ANY)
    {
      /* XXX YYY Should be a simple `id *argv'.  */
      TLVector *argv = [CO_TLVector vectorWithCapacity: num_args];

      GCPRO2 (in_args, argv);

      for (; args; args = [args cdr])
	[argv addElement: EVAL ([args car])];

#if SUPPORT_DEBUG
      ii->argc = 0;
      ii->argl = (id) argv;
#endif
      retval = fun (argv, num_args);

      GCUNPRO;
    }
  else
    {
      id *argv = alloca (max_args * sizeof (*argv));

      GCPRO2 (in_args, in_args);

      for (_gcpro2.n = 0, _gcpro2.v = argv; _gcpro2.n < max_args;
	   _gcpro2.n++, args = args ? [args cdr] : args)
	{
	  argv[_gcpro2.n] = (args ? EVAL ([args car]) : args);
	  ASGN_SPROT (argv[_gcpro2.n]);
	}
#if SUPPORT_DEBUG
      ii->argc = _gcpro2.n;
      ii->argv = argv;
#endif

      switch (max_args)
	{
	case 0:
	  retval = fun ();
	  break;
	case 1:
	  retval = fun (argv[0]);
	  break;
	case 2:
	  retval = fun (argv[0], argv[1]);
	  break;
	case 3:
	  retval = fun (argv[0], argv[1], argv[2]);
	  break;
	case 4:
	  retval = fun (argv[0], argv[1], argv[2], argv[3]);
	  break;
	default:
	  [self error: "too many arguments for subr %d (fix me)", max_args];
	  break;
	}

      GCUNPRO;
    }

#if SUPPORT_DEBUG
  tll_invocation_pop (invocation_level);
#endif
  return (retval);
} /* -evalWithArguments: */

-initWithFunction: (tll_subroutine) f args: (int) min : (int) max;
{
  fun = f;
  min_args = min;
  max_args = max;
  return (self);
} /* -initWithFunction:args:: */

-setDocumentation: (id <TLString>) doc
{
  ASGN_IVAR (documentation, doc);
  return (self);
} /* -setDocumentation: */

-subroutinep
{
  return (Qt);
} /* -subroutinep */

/******************** garbage collection ********************/

-(void) gcReference
{
  MARK (documentation);
} /* -gcReference */

@end
