      SUBROUTINE PBCSYMM( ICONTXT, MATBLK, SIDE, UPLO, M, N, NB, ALPHA,
     $                    A, LDA, B, LDB, BETA, C, LDC, IAROW, IACOL,
     $                    IBPOS, ICPOS, ACOMM, ABWORK, CWORK, MULLEN,
     $                    WORK )
*
*  -- PB-BLAS routine (version 2.1) --
*     University of Tennessee, Knoxville, Oak Ridge National Laboratory.
*     April 28, 1996
*
*     Jaeyoung Choi, Oak Ridge National Laboratory
*     Jack Dongarra, University of Tennessee and Oak Ridge National Lab.
*     David Walker,  Oak Ridge National Laboratory
*
*     .. Scalar Arguments ..
      CHARACTER*1        ABWORK, ACOMM, CWORK, MATBLK, SIDE, UPLO
      INTEGER            IACOL, IAROW, IBPOS, ICONTXT, ICPOS, LDA, LDB,
     $                   LDC, M, MULLEN, N, NB
      COMPLEX            ALPHA, BETA
*     ..
*     .. Array Arguments ..
      COMPLEX            A( LDA, * ), B( LDB, * ), C( LDC, * ),
     $                   WORK( * )
*
*  Purpose
*  =======
*
*  PBCSYMM is a parallel blocked version of CSYMM.
*  PBCSYMM  performs one of the matrix-matrix operations
*
*     C := alpha*A*B + beta*C,
*
*  or
*
*     C := alpha*B*A + beta*C,
*
*  where alpha and beta are scalars, A is a symmetric matrix and B and
*  C are  m-by-n matrices.
*
*  The first elements  of the matrices A, B, and C  should be located at
*  the beginnings of their first blocks. (not the middle of the blocks.)
*  B can be broadcast if necessary, and C is collected.
*
*  Parameters
*  ==========
*
*  ICONTXT (input) INTEGER
*          ICONTXT is the BLACS mechanism for partitioning communication
*          space.  A defining property of a context is that a message in
*          a context cannot be sent or received in another context.  The
*          BLACS context includes the definition of a grid, and each
*          process' coordinates in it.
*
*  MATBLK  (input) CHARACTER*1
*          MATBLK specifies whether A is a (full) block matrix or
*          a single block as follows:
*
*             MATBLK = 'M',  A is a (full) block matrix
*             MATBLK = 'B',  A is a single block
*
*  SIDE    (input) CHARACTER*1
*          SIDE specifies whether the symmetric matrix A appears on the
*          left or right  in the  operation as follows:
*
*             SIDE = 'L',  C := alpha*A*B + beta*C,
*             SIDE = 'R',  C := alpha*B*A + beta*C,
*
*  UPLO    (input) CHARACTER*1
*          UPLO specifies whether the upper or lower triangular part of
*          the symmetric matrix A is to be referenced as follows:
*
*             UPLO = 'U',  Only the upper triangular part of the
*                          symmetric matrix is to be referenced.
*             UPLO = 'L',  Only the lower triangular part of the
*                          symmetric matrix is to be referenced.
*
*  M       (input) INTEGER
*          M specifies the (global) number of  rows of the block matrix
*          B and C.  If SIDE = 'L', it also specifies the (global)
*          number of rows and columns of the matrix A.  M >= 0.
*          If SIDE = 'R', M <= NB.
*
*  N       (input) INTEGER
*          N specifies the (global) number of columns of the block
*          matrix B and C.  If SIDE = 'R', it also specifies the
*          (global) number of rows and columns of the matrix A. M >= 0.
*          If SIDE = 'L', N >= NB.
*
*  NB      (input) INTEGER
*          NB specifies the row and column block size of matrix A.
*          It also specifies the row block size of the matrices B and C
*          if MATBLK = 'M' and SIDE = 'L', or MATBLK = 'B' and SIDE =
*          'R'; and the column block size of the matrices B and C if
*          MATBLK = 'M' and SIDE = 'R', or MATBLK = 'B' and SIDE = 'L'.
*          NB >= 1.
*
*  ALPHA   (input) COMPLEX
*          ALPHA specifies the scalar alpha.
*
*  A       (input) COMPLEX array of DIMENSION ( LDA, ka ), where ka is
*          Mq when SIDE = 'L' and is Nq otherwise.
*          Before entry with SIDE = 'L', the M-by-M part of the (global)
*          array A must contain the symmetric matrix, such that when
*          UPLO = 'U', the leading M-by-M upper triangular part of the
*          array A  must contain  the upper triangular part of the
*          symmetric matrix and the  strictly lower triangular part of
*          A is not referenced,  and  when  UPLO = 'L', the leading
*          M-by-M lower triangular part  of the (global) array A must
*          contain the lower triangular part of the symmetric matrix and
*          the strictly upper triangular part of A is not referenced.
*          Before entry with SIDE = 'R', the N-by-N part of the (global)
*          array A must contain the symmetric matrix, such that when
*          UPLO = 'U', the leading n by n upper triangular part of the
*          (global) array A must contain the upper triangular part of
*          the symmetric matrix and the strictly lower triangular part
*          of A is not referenced, and when UPLO = 'L', the leading
*          N-by-N lower triangular part of the array A must contain the
*          lower triangular part of the symmetric matrix and the
*          strictly upper triangular part of A is not referenced.
*
*  LDA     (input) INTEGER
*          On entry, LDA specifies the first dimension of (local) A  as
*          declared in the calling (sub) program.  When  SIDE = 'L',
*          LDA >= MAX(1,Mp),  otherwise LDA >= MAX(1,Np).
*
*  B       (input) COMPLEX array of DIMENSION ( LDB, n ).
*          The leading  Mp-by-Nq part of the (local) array
*          B  must contain the matrix B.
*
*  LDB     (input) INTEGER
*          On entry,  LDB specifies the first dimension of (local) B as
*          declared in the  calling  (sub) program.   LDB >= MAX(1,Mp).
*
*  BETA    (input) COMPLEX
*          BETA specifies the scalar beta. When BETA  is supplied as
*          zero then C need not be set on input.
*
*  C       (input/output) COMPLEX array of DIMENSION ( LDC, n ).
*          On entry, the leading Mp-by-Nq part of the array  C must
*          contain the (local) matrix C,  except when  beta is zero, in
*          which case C need not be set on entry.
*          On exit, the array C  is overwritten by the Mp-by-Nq updated
*          matrix. Input values of C would be changed after the
*          computation in the processes which don't have the resultant
*          column block or row block of C.
*
*  LDC     (input) INTEGER
*          LDC specifies the first dimension of C as declared
*          in  the  calling  (sub)  program.   LDC >= MAX(1,Mp).
*
*  IAROW   (input) INTEGER
*          It specifies a row of process template which has the
*          first block of A.  When MATBLK = 'B', and all rows of
*          processes have their own copies of A, set IAROW =  -1.
*
*  IACOL   (input) INTEGER
*          It specifies a column of process template which has the
*          first block of A.  When MATBLK = 'B', and all columns of
*          processes have their own copies of A, set IACOL = -1.
*
*  IBPOS   (input) INTEGER
*          When MATBLK = 'M', if SIDE = 'L', IBPOS specifies a column of
*          the process template, which holds the column of blocks of B
*          (-1 <= IBPOS < NPCOL).  And if SIDE = 'R', it specifies a row
*          of the template, which holds the row of blocks of B (-1 <=
*          IBPOS < NPROW).  If all columns or rows of the template have
*          their own copies of B, set IBPOS = -1.
*          When MATBLK = 'B', if SIDE = 'L', it specifies a column of
*          the template which has the first block of B (0 <= IBPOS
*          < NPCOL), and if SIDE = 'R', it specifies a row of the
*          template, which has the first block of B (0 <=IBPOS <NPROW).
*          IBPOS should be the same as ICPOS if MATBLK = 'B'.
*
*  ICPOS   (input) INTEGER
*          When MATBLK = 'M', if SIDE = 'L', ICPOS specifies a column of
*          the process template, which holds the column of blocks of C
*          (0 <= ICPOS < NPCOL).  And if SIDE = 'R', it specifies a row
*          of the template, which holds the row of blocks of C (0 <=
*          ICPOS < NPROW).
*          When MATBLK = 'B', if SIDE = 'L', it specifies a column of
*          the template which has the first block of C (0 <= ICPOS
*          < NPCOL), and if SIDE = 'R', it specifies a row of the
*          template, which has the first block of C (0 <=ICPOS <NPROW).
*          ICPOS should be the same as IBPOS if MATBLK = 'B'.
*
*  ACOMM   (input) CHARACTER*1
*          When MATBLK = 'B', ACOMM specifies the communication scheme
*          of a block of A.  And it is ignored when MATBLK = 'M'.
*          It follows topology definition of BLACS.
*
*  ABWORK  (input) CHARACTER*1
*          When MATBLK = 'M', ABWORK determines whether B is a
*          workspace or not.
*
*             ABWORK = 'Y':  B is workspace in other processes.
*                            B is sent to B position in other processes.
*                            It is assumed that processes have
*                            sufficient space to store (local) B.
*             ABWORK = 'N':  Data in B will be untouched (unchanged).
*
*          And MATBLK = 'B', ABWORK determines whether A is a
*          workspace or not.
*
*             ABWORK = 'Y':  A is workspace in other processes.
*                            A is sent to A position in other processes.
*                            It is assumed that processes have
*                            sufficient space to store a single block A.
*             ABWORK = 'N':  A is data space, not to be touched.
*
*  CWORK   (input) CHARACTER*1
*          When MATBLK = 'M', CWORK determines whether C is a
*          workspace or not.
*
*             CWORK = 'Y':   C is workspace in other processes.
*                            It is assumed that processes have
*                            sufficient space to store temporary
*                            (local) C.
*             CWORK = 'N':   Data in C will be untouched (unchanged)
*                            in other processes.
*
*          And MATBLK = 'B', it is ignored.
*
*  MULLEN  (input) INTEGER
*          It specifies  multiplication  length  of the  optimum column
*          number of A  for multiplying A with B.  The value depends on
*          machine characteristics.
*
*  WORK    (workspace) COMPLEX array of dimension Size(WORK).
*          It will store copies of B and/or C (see Requirements).
*
*  Parameters Details
*  ==================
*
*  Lx      It is  a local portion  of L  owned  by  a process,  (L is
*          replaced by M, or N,  and x  is replaced  by  either  p
*          (=NPROW) or q (=NPCOL)).  The value is determined by  L, LB,
*          x, and MI,  where  LB is  a block size  and MI is a  row  or
*          column position in a process template.  Lx is equal to  or
*          less than  Lx0 = CEIL( L, LB*x ) * LB.
*
*  Communication Scheme
*  ====================
*
*  When MATBLK = 'M', the communication schemes of the routine are
*  fixed as fan-out and fan-in schemes (COMM = '1-tree').
*
*  Memory Requirement of WORK
*  ==========================
*
*  Mqb    = CEIL( M, NB*NPCOL )
*  Npb    = CEIL( N, NB*NPROW )
*  Mq0    = NUMROC( M, NB, 0, 0, NPCOL ) ~= Mqb * NB
*  Np0    = NUMROC( N, NB, 0, 0, NPROW ) ~= Npb * NB
*  LCMQ   = LCM / NPCOL
*  LCMP   = LCM / NPROW
*  ISZCMP = CEIL(MULLEN, LCMQ*NB)
*  SZCMP  = ISZCMP * ISZCMP * LCMQ*NB * LCMP*NB
*
*  (1) MATBLK = 'M'
*    (a) SIDE = 'Left'
*    Size(WORK) = 2 * N * Mq0
*               + N * Mp0                       ( if CWORK <> 'Y' )
*               + N * Mp0      ( if IBPOS <> -1 and ABWORK <> 'Y' )
*               + MAX[ SZCMP,
*                      N*CEIL(Mqb,LCMQ)*NB*MIN(LCMQ,CEIL(M,NB)) ]
*    (b) SIDE = 'Right'
*    Size(WORK) = 2 * M * Np0
*               + M * Nq0                       ( if CWORK <> 'Y' )
*               + M * Nq0      ( if IBPOS <> -1 and ABWORK <> 'Y' )
*               + MAX[ SZCMP,
*                      M*CEIL(Npb,LCMP)*NB*MIN(LCMP,CEIL(N,NB)) ]
*
*  (2) MATBLK = 'B'
*    (a) SIDE = 'Left'
*     Size(WORK) = M * M     ( if IACOL <> -1 and ABWORK <> 'Y' )
*    (b) SIDE = 'Right'
*     Size(WORK) = N * N     ( if IAROW <> -1 and ABWORK <> 'Y' )
*
*  Notes
*  -----
*  More precise space can be computed as
*
*  CEIL(Mqb,LCMQ)*NB => NUMROC( NUMROC(M,NB,0,0,NPCOL), NB, 0, 0, LCMQ )
*                    = NUMROC( Mq0, NB, 0, 0, LCMQ )
*  CEIL(Npb,LCMP)*NB => NUMROC( NUMROC(N,NB,0,0,NPROW), NB, 0, 0, LCMP )
*                    = NUMROC( Np0, NB, 0, 0, LCMP )
*
*  =====================================================================
*
*     .. Parameters ..
      COMPLEX            ONE, ZERO
      PARAMETER          ( ONE  = ( 1.0E+0, 0.0E+0 ),
     $                   ZERO = ( 0.0E+0, 0.0E+0 ) )
*     ..
*     .. Local Scalars ..
      CHARACTER*1        FORM, COMMA
      LOGICAL            ADATA, AMAT, ASPACE, BDATA, BSPACE, CDATA,
     $                   CSPACE, LSIDE, RSIDE, UPPER
      INTEGER            INFO, IPB, IPBZ, IPC, IPD, IPT, IPW, IQBZ,
     $                   ISZCMP, ITER, JJ, JNPBZ, JNQBZ, JPBZ, JQBZ, KI,
     $                   KIZ, KJ, KJZ, LCM, LCMP, LCMQ, LMW, LNW, LPBZ,
     $                   LQBZ, MRCOL, MRROW, MYCOL, MYROW, MZCOL, MZROW,
     $                   NDIM, NP, NP1, NPCOL, NPROW, NQ
      COMPLEX            DUMMY
*     ..
*     .. External Functions ..
      LOGICAL            LSAME
      INTEGER            ICEIL, ILCM, NUMROC
      EXTERNAL           ICEIL, ILCM, LSAME, NUMROC
*     ..
*     .. External Subroutines ..
      EXTERNAL           BLACS_GRIDINFO, CGEBR2D, CGEBS2D, CGEMM,
     $                   CGSUM2D, CSYMM, CTRBR2D, CTRBS2D, PBCDZERO,
     $                   PBCLACPZ, PBCMATADD, PBCTRAN, PXERBLA
*     ..
*     .. Intrinsic Functions ..
      INTRINSIC          MAX, MIN
*     ..
*     .. Executable Statements ..
*
*     Quick return if possible.
*
      IF( M.EQ.0 .OR. N.EQ.0 .OR. ( ALPHA.EQ.ZERO .AND. BETA.EQ.ONE ) )
     $   RETURN
*
      CALL BLACS_GRIDINFO( ICONTXT, NPROW, NPCOL, MYROW, MYCOL )
*
      AMAT  = LSAME( MATBLK,  'M' )
      UPPER = LSAME( UPLO, 'U' )
      LSIDE = LSAME( SIDE, 'L' )
      RSIDE = LSAME( SIDE, 'R' )
*
*     Test the input parameters.
*
      INFO = 0
      IF(      ( .NOT.AMAT                 ).AND.
     $         ( .NOT.LSAME( MATBLK, 'B' ) )      ) THEN
        INFO = 2
      ELSE IF( .NOT.LSIDE .AND. .NOT.RSIDE        ) THEN
        INFO = 3
      ELSE IF( ( .NOT.UPPER              ).AND.
     $         ( .NOT.LSAME( UPLO, 'L' ) )        ) THEN
        INFO = 4
      ELSE IF( M  .LT.0                           ) THEN
        INFO = 5
      ELSE IF( N  .LT.0                           ) THEN
        INFO = 6
      ELSE IF( NB .LT.1                           ) THEN
        INFO = 7
      END IF
*
   10 CONTINUE
      IF( INFO.NE.0 ) THEN
        CALL PXERBLA( ICONTXT, 'PBCSYMM ', INFO )
        RETURN
      END IF
*
* === If A is a general matrix ( MATBLK = 'M' ) ===
*
      IF( LSAME( MATBLK, 'M' ) ) THEN
        IF( LSIDE ) THEN
          NDIM = M
        ELSE
          NDIM = N
        END IF
        NP = NUMROC( NDIM, NB, MYROW, IAROW, NPROW )
        NQ = NUMROC( NDIM, NB, MYCOL, IACOL, NPCOL )
*
        NP1 = MAX( 1, NP )
        IF( LDA.LT.NP1                          ) THEN
          INFO = 10
        ELSE IF( IAROW.LT.0 .OR. IAROW.GE.NPROW ) THEN
          INFO = 16
        ELSE IF( IACOL.LT.0 .OR. IACOL.GE.NPCOL ) THEN
          INFO = 17
        END IF
*
*       Quick return if alpha = zero
*
        IF( ALPHA.EQ.ZERO ) THEN
          IF( LSIDE .AND. MYCOL.EQ.ICPOS ) THEN
            CALL PBCMATADD( ICONTXT, 'V', NP, N, ZERO, DUMMY, 1, BETA,
     $                      C, LDC )
          ELSE IF( .NOT.LSIDE .AND. MYROW.EQ.ICPOS ) THEN
            CALL PBCMATADD( ICONTXT, 'G', M, NQ, ZERO, DUMMY, 1, BETA,
     $                      C, LDC )
          END IF
          RETURN
        END IF
*
*       LCM : the least common multiple of NPROW and NPCOL
*
        LCM  = ILCM( NPROW, NPCOL )
        LCMP = LCM  / NPROW
        LCMQ = LCM  / NPCOL
        LPBZ = LCMP * NB
        LQBZ = LCMQ * NB
*
        MRROW = MOD( NPROW+MYROW-IAROW, NPROW )
        MRCOL = MOD( NPCOL+MYCOL-IACOL, NPCOL )
*
        BDATA  = .FALSE.
        IF( IBPOS.EQ.-1 ) BDATA = .TRUE.
        CDATA  = .FALSE.
        BSPACE = LSAME( ABWORK, 'Y' )
        CSPACE = LSAME( CWORK,  'Y' )
*
*       PART 1: Distribute a column (or row) block B and its transpose
*       ==============================================================
*
        IF( LSIDE ) THEN
*
*         Form  C := alpha*A*B + beta*C, if SIDE = 'Left'.
*          _             _____________     _            _
*         | |           |\_           |   | |          | |
*         | |           |  \_         |   | |          | |
*         | |           |    \_       |   | |          | |
*         |C| = alpha * |      A_     | * |B| + beta * |C|
*         | |           |        \_   |   | |          | |
*         | |           |          \_ |   | |          | |
*         |_|           |____________\|   |_|          |_|
*
          IF( LDB.LT.NP1 .AND. ( BSPACE .OR.
     $             IBPOS.EQ.MYCOL .OR. IBPOS.EQ.-1 ) ) THEN
            INFO = 12
          ELSE IF( LDC.LT.NP1 .AND. ( CSPACE .OR.
     $             ICPOS.EQ.MYCOL .OR. ICPOS.EQ.-1 ) ) THEN
            INFO = 15
          ELSE IF( IBPOS.LT.-1 .OR. IBPOS.GE.NPCOL   ) THEN
            INFO = 18
          ELSE IF( ICPOS.LT.0  .OR. ICPOS.GE.NPCOL   ) THEN
            INFO = 19
          END IF
          IF( INFO.NE.0 ) GO TO 10
*
*         Initialize parameters
*
          IF( CSPACE ) THEN
            IPD = 1
            CDATA = .TRUE.
            IF( MYCOL.EQ.ICPOS ) THEN
              CALL PBCMATADD( ICONTXT, 'G', NP, N, ZERO, DUMMY, 1, BETA,
     $                        C, LDC )
            ELSE
              CALL PBCMATADD( ICONTXT, 'G', NP, N, ZERO, DUMMY, 1, ZERO,
     $                        C, LDC )
            END IF
          ELSE
            IPC = 1
            IPD = N * NP + IPC
            CALL PBCMATADD( ICONTXT, 'G', NP, N, ZERO, DUMMY, 1, ZERO,
     $                      WORK(IPC), NP )
          END IF
*
          CALL PBCMATADD( ICONTXT, 'G', N, NQ, ZERO, DUMMY, 1, ZERO,
     $                    WORK(IPD), N )
*
          IPT = N * NQ + IPD
          IPB = N * NQ + IPT
          IPW = N * NP + IPB
*
*         Broadcast B if necessary
*
          IF( .NOT.BDATA ) THEN
            IF( BSPACE ) THEN
              IF( MYCOL.EQ.IBPOS ) THEN
                CALL CGEBS2D( ICONTXT, 'Row', '1-tree', NP, N, B, LDB )
              ELSE
                CALL CGEBR2D( ICONTXT, 'Row', '1-tree', NP, N, B, LDB,
     $                        MYROW, IBPOS )
              END IF
              BDATA = .TRUE.
              IPW = IPB
            ELSE
              IF( MYCOL.EQ.IBPOS ) THEN
                CALL PBCMATADD( ICONTXT, 'V', NP, N, ONE, B, LDB, ZERO,
     $                          WORK(IPB), NP )
                CALL CGEBS2D( ICONTXT, 'Row', '1-tree', NP, N,
     $                        WORK(IPB), NP )
              ELSE
                CALL CGEBR2D( ICONTXT, 'Row', '1-tree', NP, N,
     $                        WORK(IPB), NP, MYROW, IBPOS )
              END IF
            END IF
          END IF
*
*         Transpose col block of B to WORK(IPT), where B is distributed
*
          IF( BDATA ) THEN
            CALL PBCTRAN( ICONTXT, 'Col', 'T', M, N, NB, B, LDB, ZERO,
     $                    WORK(IPT), N, IAROW, -1, -1, IACOL,
     $                    WORK(IPW) )
          ELSE
            CALL PBCTRAN( ICONTXT, 'Col', 'T', M, N, NB, WORK(IPB), NP,
     $                    ZERO, WORK(IPT),N, IAROW, -1, -1, IACOL,
     $                    WORK(IPW) )
          END IF
*
        ELSE
*
*         Form  C := alpha*B*A + beta*C, if SIDE = 'Right'.
*                                        _____________
*                                       |\_           |
*                                       |  \_         |
*      ___________       _____________  |    \_       |    ___________
*     |____C______| = a*|______B______|*|      A_     |+b*|____C______|
*                                       |        \_   |
*                                       |          \_ |
*                                       |____________\|
*
          IF( LDB.LT.MAX(1,M) .AND. ( BSPACE .OR.
     $             IBPOS.EQ.MYROW .OR. IBPOS.EQ.-1 ) ) THEN
            INFO = 12
          ELSE IF( LDC.LT.MAX(1,M) .AND. ( CSPACE .OR.
     $             ICPOS.EQ.MYROW .OR. ICPOS.EQ.-1 ) ) THEN
            INFO = 15
          ELSE IF( IBPOS.LT.-1 .OR. IBPOS.GE.NPROW   ) THEN
            INFO = 18
          ELSE IF( ICPOS.LT.0  .OR. ICPOS.GE.NPROW   ) THEN
            INFO = 19
          END IF
          IF( INFO.NE.0 ) GO TO 10
*
*         Initialize parameters
*
          IF( CSPACE ) THEN
            IPD = 1
            CDATA = .TRUE.
            IF( MYROW.EQ.ICPOS ) THEN
              CALL PBCMATADD( ICONTXT, 'G', M, NQ, ZERO, DUMMY, 1, BETA,
     $                        C, LDC )
            ELSE
              CALL PBCMATADD( ICONTXT, 'G', M, NQ, ZERO, DUMMY, 1, ZERO,
     $                        C, LDC )
            END IF
          ELSE
            IPC = 1
            IPD = M * NQ + IPC
            CALL PBCMATADD( ICONTXT, 'G', M, NQ, ZERO, DUMMY, 1, ZERO,
     $                      WORK(IPC), M )
          END IF
*
          CALL PBCMATADD( ICONTXT, 'G', NP, M, ZERO, DUMMY, 1, ZERO,
     $                    WORK(IPD), NP )
*
          IPT = M * NP + IPD
          IPB = M * NP + IPT
          IPW = M * NQ + IPB
*
*         Broadcast B if necessary
*
          IF( .NOT.BDATA ) THEN
            IF( BSPACE ) THEN
              IF( MYROW.EQ.IBPOS ) THEN
                CALL CGEBS2D( ICONTXT, 'Col', '1-tree', M, NQ, B, LDB )
              ELSE
                CALL CGEBR2D( ICONTXT, 'Col', '1-tree', M, NQ, B, LDB,
     $                        IBPOS, MYCOL )
              END IF
              BDATA = .TRUE.
              IPW = IPB
            ELSE
              IF( MYROW.EQ.IBPOS ) THEN
                CALL PBCMATADD( ICONTXT, 'V', M, NQ, ONE, B, LDB, ZERO,
     $                          WORK(IPB), M )
                CALL CGEBS2D( ICONTXT, 'Col', '1-tree', M, NQ,
     $                        WORK(IPB), M )
              ELSE
                CALL CGEBR2D( ICONTXT, 'Col', '1-tree', M, NQ,
     $                        WORK(IPB), M, IBPOS, MYCOL )
              END IF
            END IF
          END IF
*
*         Transpose row block of B to WORK(IPT), where B is distributed
*
          IF( BDATA ) THEN
            CALL PBCTRAN( ICONTXT, 'Row', 'T', M, N, NB, B, LDB, ZERO,
     $                    WORK(IPT), NP, -1, IACOL, IAROW, -1,
     $                    WORK(IPW) )
          ELSE
            CALL PBCTRAN( ICONTXT, 'Row', 'T', M, N, NB, WORK(IPB), M,
     $                    ZERO, WORK(IPT), NP, -1, IACOL, IAROW, -1,
     $                    WORK(IPW) )
          END IF
        END IF
*
*       PART 2: Compute C (= WORK(IPC)) and WORK(IPD)
*       =============================================
*
        IF( NP.EQ.0 .OR. NQ.EQ.0 ) GO TO 160
*
*       If A is a symmetric upper triangular matrix,
*
        IF( UPPER ) THEN
          ISZCMP = ICEIL( MULLEN, LQBZ )
          IF( ISZCMP.LE.0 ) ISZCMP = 1
          IPBZ = ISZCMP * LPBZ
          IQBZ = ISZCMP * LQBZ
          ITER = ICEIL( NQ, IQBZ )
          JPBZ = 0
          JQBZ = 0
*
          DO 80 JJ = 0, ITER-1
            LMW = MIN( IPBZ, NP-JPBZ )
            LNW = MIN( IQBZ, NQ-JQBZ )
            JNPBZ = JPBZ + LMW
            JNQBZ = JQBZ + LNW
*
*           Copy the upper triangular matrix A to WORK(IPW)
*
            MZROW = MRROW
            MZCOL = MRCOL
            KI = 0
*
            DO 30 KJ = 0, LCMQ-1
   20         CONTINUE
              IF( MZROW.LT.MZCOL ) THEN
                MZROW = MZROW + NPROW
                KI = KI + 1
                GO TO 20
              END IF
              KIZ = KI * NB
              KJZ = KJ * NB
              IF( KJZ.GE.LNW ) GO TO 40
              FORM = 'G'
              IF( MZROW.EQ.MZCOL ) FORM = 'T'
              MZCOL = MZCOL + NPCOL
*
              CALL PBCLACPZ( ICONTXT, 'Upper', FORM, 'No', KIZ, NB,
     $                       A(JPBZ+1,JQBZ+KJZ+1), LDA,
     $                       WORK(KJZ*LMW+IPW), LMW,
     $                       LPBZ, LQBZ, LMW, LNW-KJZ )
   30       CONTINUE
   40       CONTINUE
*
*           Compute C if SIDE = 'Left'
*
            IF( LSIDE ) THEN
              IF( CDATA ) THEN
                CALL CGEMM( 'No', 'Trans', LMW, N, LNW, ALPHA,
     $                      WORK(IPW), MAX(1,LMW), WORK(JQBZ*N+IPT), N,
     $                      ONE, C(JPBZ+1,1), LDC )
                CALL CGEMM( 'No', 'Trans', JPBZ, N, LNW, ALPHA,
     $                      A(1,JQBZ+1), LDA, WORK(JQBZ*N+IPT), N,
     $                      ONE, C, LDC )
              ELSE
                CALL CGEMM( 'No', 'Trans', LMW, N, LNW, ALPHA,
     $                      WORK(IPW), MAX(1,LMW), WORK(JQBZ*N+IPT), N,
     $                      ZERO, WORK(JPBZ+IPC), NP1 )
                CALL CGEMM( 'No', 'Trans', JPBZ, N, LNW, ALPHA,
     $                      A(1,JQBZ+1), LDA, WORK(JQBZ*N+IPT), N,
     $                      ONE, WORK(IPC), NP1 )
              END IF
*
*           Compute C if SIDE = 'Right'
*
            ELSE
              IF( BDATA ) THEN
                CALL CGEMM( 'No', 'Trans', LMW, M, LNW, ALPHA,
     $                      WORK(IPW), MAX(1,LMW), B(1,JQBZ+1), LDB,
     $                      ONE, WORK(JPBZ+IPD), NP1 )
                CALL CGEMM( 'No', 'Trans', JPBZ, M, LNW, ALPHA,
     $                      A(1,JQBZ+1), LDA, B(1,JQBZ+1), LDB, ONE,
     $                      WORK(IPD), NP1 )
              ELSE
                CALL CGEMM( 'No', 'Trans', LMW, M, LNW, ALPHA,
     $                      WORK(IPW), MAX(1,LMW), WORK(JQBZ*M+IPB),
     $                      M, ZERO, WORK(JPBZ+IPD), NP1 )
                CALL CGEMM( 'No', 'Trans', JPBZ, M, LNW, ALPHA,
     $                      A(1,JQBZ+1), LDA, WORK(JQBZ*M+IPB), M,
     $                      ONE, WORK(IPD), NP1 )
              END IF
            END IF
*
*           Delete the diagonal elements of upper tri. matrix WORK(IPW)
*
            MZROW = MRROW
            MZCOL = MRCOL
            KI = 0
*
            DO 60 KJ = 0, LCMQ-1
   50         CONTINUE
              IF( MZROW.LT.MZCOL ) THEN
                MZROW = MZROW + NPROW
                KI = KI + 1
                GO TO 50
              END IF
              KIZ = KI * NB
              KJZ = KJ * NB
              IF( KJZ.GE.LNW ) GO TO 70
              IF( MZROW.EQ.MZCOL )
     $          CALL PBCDZERO( KIZ, NB, WORK(KJZ*LMW+IPW), LMW,
     $                         LPBZ, LQBZ, LNW-KJZ )
              MZCOL = MZCOL + NPCOL
   60       CONTINUE
   70       CONTINUE
*
*           Compute C if SIDE = 'Left'
*
            IF( LSIDE ) THEN
              IF( BDATA ) THEN
                CALL CGEMM( 'Trans', 'No', N, LNW, LMW, ALPHA,
     $                      B(JPBZ+1,1), LDB, WORK(IPW), MAX(1,LMW),
     $                      ZERO, WORK(N*JQBZ+IPD), N )
                CALL CGEMM( 'Trans', 'No', N, LNW, JPBZ, ALPHA, B, LDB,
     $                      A(1,JQBZ+1), LDA, ONE, WORK(N*JQBZ+IPD),N )
              ELSE
                CALL CGEMM( 'Trans', 'No', N, LNW, LMW, ALPHA,
     $                      WORK(JPBZ+IPB), NP1, WORK(IPW), MAX(1,LMW),
     $                      ZERO, WORK(N*JQBZ+IPD), N )
                CALL CGEMM( 'Trans', 'No', N, LNW, JPBZ, ALPHA,
     $                      WORK(IPB), NP1, A(1,JQBZ+1), LDA, ONE,
     $                      WORK(N*JQBZ+IPD), N )
              END IF
*
*           Compute C if SIDE = 'Right'
*
            ELSE
              IF( CDATA ) THEN
                CALL CGEMM( 'Trans', 'No', M, LNW, LMW, ALPHA,
     $                      WORK(JPBZ+IPT), NP1, WORK(IPW), MAX(1,LMW),
     $                      ONE, C(1,JQBZ+1), LDC )
                CALL CGEMM( 'Trans', 'No', M, LNW, JPBZ, ALPHA,
     $                      WORK(IPT), NP1, A(1,JQBZ+1), LDA, ONE,
     $                      C(1,JQBZ+1), LDC )
              ELSE
                CALL CGEMM( 'Trans', 'No', M, LNW, LMW, ALPHA,
     $                      WORK(JPBZ+IPT), NP1, WORK(IPW), MAX(1,LMW),
     $                      ZERO, WORK(M*JQBZ+IPC), M )
                CALL CGEMM( 'Trans', 'No', M, LNW, JPBZ, ALPHA,
     $                      WORK(IPT), NP1, A(1,JQBZ+1), LDA, ONE,
     $                      WORK(M*JQBZ+IPC), M )
              END IF
            END IF
*
            JPBZ = JNPBZ
            JQBZ = JNQBZ
   80     CONTINUE
*
*       If A is a symmetric lower triangular matrix,
*
        ELSE
          ISZCMP = ICEIL( MULLEN, LQBZ )
          IF( ISZCMP.LE.0 ) ISZCMP = 1
          IPBZ = ISZCMP * LPBZ
          IQBZ = ISZCMP * LQBZ
          ITER = ICEIL( NQ, IQBZ )
          JPBZ = 0
          JQBZ = 0
*
          DO 150 JJ = 0, ITER-1
            LMW = MIN( IPBZ, NP-JPBZ )
            LNW = MIN( IQBZ, NQ-JQBZ )
            JNPBZ = JPBZ + LMW
            JNQBZ = JQBZ + LNW
*
*           Copy the lower triangular matrix A to WORK(IPW)
*
            MZROW = MRROW
            MZCOL = MRCOL
            KI = 0
*
            DO 100 KJ = 0, LCMQ-1
   90         CONTINUE
              IF( MZROW.LT.MZCOL ) THEN
                MZROW = MZROW + NPROW
                KI = KI + 1
                GO TO 90
              END IF
              KIZ = KI * NB
              KJZ = KJ * NB
              IF( KJZ.GE.LNW ) GO TO 110
              FORM = 'G'
              IF( MZROW.EQ.MZCOL ) FORM = 'T'
              MZCOL = MZCOL + NPCOL
*
              CALL PBCLACPZ( ICONTXT, 'Lower', FORM, 'No', KIZ, NB,
     $                       A(JPBZ+1,JQBZ+KJZ+1), LDA,
     $                       WORK(KJZ*LMW+IPW), LMW, LPBZ, LQBZ,
     $                       LMW, LNW-KJZ )
  100       CONTINUE
  110       CONTINUE
*
*           Compute C if SIDE = 'Left'
*
            IF( LSIDE ) THEN
              IF( CDATA ) THEN
                CALL CGEMM( 'No', 'Trans', LMW, N, LNW, ALPHA,
     $                      WORK(IPW), MAX(1,LMW), WORK(JQBZ*N+IPT), N,
     $                      ONE, C(JPBZ+1,1), LDC )
                CALL CGEMM( 'No', 'Trans', NP-JNPBZ, N, LNW, ALPHA,
     $                      A(JNPBZ+1,JQBZ+1), LDA, WORK(JQBZ*N+IPT),
     $                      N, ONE, C(JNPBZ+1,1), LDC )
              ELSE
                CALL CGEMM( 'No', 'Trans', LMW, N, LNW, ALPHA,
     $                      WORK(IPW), MAX(1,LMW), WORK(JQBZ*N+IPT), N,
     $                      ONE, WORK(JPBZ+IPC), NP1 )
                CALL CGEMM( 'No', 'Trans', NP-JNPBZ, N, LNW, ALPHA,
     $                      A(JNPBZ+1,JQBZ+1), LDA, WORK(JQBZ*N+IPT),
     $                      N, ONE, WORK(JNPBZ+IPC), NP1 )
              END IF
*
*           Compute C if SIDE = 'Right'
*
            ELSE
              IF( BDATA ) THEN
                CALL CGEMM( 'No', 'Trans', LMW, M, LNW, ALPHA,
     $                      WORK(IPW), MAX(1,LMW), B(1,JQBZ+1), LDB,
     $                      ONE, WORK(JPBZ+IPD), NP1 )
                CALL CGEMM( 'No', 'Trans', NP-JNPBZ, M, LNW, ALPHA,
     $                      A(JNPBZ+1,JQBZ+1), LDA, B(1,JQBZ+1), LDB,
     $                      ONE, WORK(JNPBZ+IPD), NP1 )
              ELSE
                CALL CGEMM( 'No', 'Trans', LMW, M, LNW, ALPHA,
     $                      WORK(IPW), MAX(1,LMW), WORK(JQBZ*M+IPB), M,
     $                      ONE, WORK(JPBZ+IPD), NP1 )
                CALL CGEMM( 'No', 'Trans', NP-JNPBZ, M, LNW, ALPHA,
     $                      A(JNPBZ+1,JQBZ+1), LDA, WORK(JQBZ*M+IPB),
     $                      M, ONE, WORK(JNPBZ+IPD), NP1 )
              END IF
            END IF
*
*           Delete the diagonal elements of lower tri. matrix WORK(IPW)
*
            MZROW = MRROW
            MZCOL = MRCOL
            KI = 0
*
            DO 130 KJ = 0, LCMQ-1
  120         CONTINUE
              IF( MZROW.LT.MZCOL ) THEN
                MZROW = MZROW + NPROW
                KI = KI + 1
                GO TO 120
              END IF
              KIZ = KI * NB
              KJZ = KJ * NB
              IF( KJZ.GE.LNW ) GO TO 140
              IF( MZROW.EQ.MZCOL )
     $          CALL PBCDZERO( KIZ, NB, WORK(KJZ*LMW+IPW), LMW,
     $                         LPBZ, LQBZ, LNW-KJZ )
              MZCOL = MZCOL + NPCOL
  130       CONTINUE
  140       CONTINUE
*
*           Compute C if SIDE = 'Left'
*
            IF( LSIDE ) THEN
              IF( BDATA ) THEN
                CALL CGEMM( 'Trans', 'No', N, LNW, LMW, ALPHA,
     $                      B(JPBZ+1,1), LDB, WORK(IPW), MAX(1,LMW),
     $                      ZERO, WORK(N*JQBZ+IPD), N )
                CALL CGEMM( 'Trans', 'No', N, LNW, NP-JNPBZ, ALPHA,
     $                      B(JNPBZ+1,1), LDB, A(JNPBZ+1,JQBZ+1), LDA,
     $                      ONE, WORK(N*JQBZ+IPD), N )
              ELSE
                CALL CGEMM( 'Trans', 'No', N, LNW, LMW, ALPHA,
     $                      WORK(JPBZ+IPB), NP, WORK(IPW), MAX(1,LMW),
     $                      ZERO, WORK(N*JQBZ+IPD), N )
                CALL CGEMM( 'Trans', 'No', N, LNW, NP-JNPBZ, ALPHA,
     $                      WORK(JNPBZ+IPB), NP, A(JNPBZ+1,JQBZ+1),
     $                      LDA, ONE, WORK(N*JQBZ+IPD), N )
              END IF
*
*           Compute C if SIDE = 'Right'
*
            ELSE
              IF( CDATA ) THEN
                CALL CGEMM( 'Trans', 'No', M, LNW, LMW, ALPHA,
     $                      WORK(JPBZ+IPT), NP, WORK(IPW), MAX(1,LMW),
     $                      ONE, C(1,JQBZ+1), LDC )
                CALL CGEMM( 'Trans', 'No', M, LNW, NP-JNPBZ, ALPHA,
     $                      WORK(JNPBZ+IPT), NP, A(JNPBZ+1,JQBZ+1),
     $                      LDA, ONE, C(1,JQBZ+1), LDC )
              ELSE
                CALL CGEMM( 'Trans', 'No', M, LNW, LMW, ALPHA,
     $                      WORK(JPBZ+IPT), NP1, WORK(IPW), MAX(1,LMW),
     $                      ZERO, WORK(M*JQBZ+IPC), M )
                CALL CGEMM( 'Trans', 'No', M, LNW, NP-JNPBZ, ALPHA,
     $                      WORK(JNPBZ+IPT), NP1, A(JNPBZ+1,JQBZ+1),
     $                      LDA, ONE, WORK(M*JQBZ+IPC), M )
              END IF
            END IF
*
            JPBZ = JNPBZ
            JQBZ = JNQBZ
  150     CONTINUE
        END IF
*
  160   CONTINUE
*
*       PART 3: Collect and Add C, C := C + op(WORK(IPC))+op(WORK(IPD))
*       ===============================================================
*
*       C is a column block if SIDE = 'Left'
*
        IF( LSIDE ) THEN
          IF( CDATA ) THEN
            CALL CGSUM2D( ICONTXT, 'Row', '1-tree', NP, N, C, LDC,
     $                    MYROW, ICPOS )
          ELSE
            IF( MYCOL.EQ.ICPOS ) THEN
              CALL PBCMATADD( ICONTXT, 'V', NP, N, ONE, WORK(IPC), NP,
     $                        BETA, C, LDC )
              CALL CGSUM2D( ICONTXT, 'Row', '1-tree', NP, N, C, LDC,
     $                      MYROW, ICPOS )
            ELSE
              CALL CGSUM2D( ICONTXT, 'Row', '1-tree', NP, N, WORK(IPC),
     $                      NP, MYROW, ICPOS )
            END IF
          END IF
*
          CALL CGSUM2D( ICONTXT, 'Col', '1-tree', N, NQ, WORK(IPD), N,
     $                  IAROW, MYCOL )
          CALL PBCTRAN( ICONTXT, 'Row', 'T', N, M, NB, WORK(IPD), N,
     $                  ONE, C, LDC, IAROW, IACOL, IAROW, ICPOS,
     $                  WORK(IPT) )
*
*       C is a row block if SIDE = 'Right'
*
        ELSE
          IF( CDATA ) THEN
            CALL CGSUM2D( ICONTXT, 'Col', '1-tree', M, NQ, C, LDC,
     $                    ICPOS, MYCOL )
          ELSE
            IF( MYROW.EQ.ICPOS ) THEN
              CALL PBCMATADD( ICONTXT, 'G', M, NQ, ONE, WORK(IPC), M,
     $                        BETA,  C, LDC )
              CALL CGSUM2D( ICONTXT, 'Col', '1-tree', M, NQ, C, LDC,
     $                      ICPOS, MYCOL )
            ELSE
              CALL CGSUM2D( ICONTXT, 'Col', '1-tree', M, NQ,
     $                      WORK(IPC), M, ICPOS, MYCOL )
            END IF
          END IF
*
          CALL CGSUM2D( ICONTXT, 'Row', '1-tree', NP, M, WORK(IPD), NP,
     $                  MYROW, IACOL )
          CALL PBCTRAN( ICONTXT, 'Col', 'T', N, M, NB, WORK(IPD), NP,
     $                  ONE, C, LDC, IAROW, IACOL, ICPOS, IACOL,
     $                  WORK(IPT) )
        END IF
*
* === If A is just a block ( MATBLK = 'B' ) ===
*
      ELSE
        ADATA = .FALSE.
        ASPACE = LSAME( ABWORK, 'Y' )
        COMMA = ACOMM
        IF( LSAME( COMMA, ' ' ) ) COMMA = '1'
*
        IF( LSIDE .AND. MYROW.EQ.IAROW ) THEN
*
*         Form  C := alpha*A*B + beta*C
*      _____________       _   _____________       _____________
*     |______C______| = a*|_|*|______B______| + b*|______C______|
*                          A
*
          IF( IACOL.EQ.-1 )  ADATA = .TRUE.
          NQ = NUMROC( N, NB, MYCOL, IBPOS, NPCOL )
*
          IF( LDA.LT.MAX(1,M) .AND. ( ASPACE .OR.
     $          IACOL.EQ.MYCOL .OR. IACOL.EQ.-1 ) ) THEN
            INFO = 10
          ELSE IF( LDB.LT.MAX(1,M)                ) THEN
            INFO = 12
          ELSE IF( LDC.LT.MAX(1,M)                ) THEN
            INFO = 15
          ELSE IF( IAROW.LT.0 .OR. IAROW.GE.NPROW ) THEN
            INFO = 16
          ELSE IF( IACOL.LT.-1.OR. IACOL.GE.NPCOL ) THEN
            INFO = 17
          ELSE IF( IBPOS.LT.0 .OR. IBPOS.GE.NPCOL ) THEN
            INFO = 18
          ELSE IF( ICPOS.NE.IBPOS                 ) THEN
            INFO = 19
          END IF
          IF( INFO.NE.0 ) GO TO 10
*
*         Broadcast A if necessary
*
          IF( .NOT.ADATA ) THEN
            IF( ASPACE ) THEN
              IF( MYCOL.EQ.IACOL ) THEN
                CALL CTRBS2D( ICONTXT, 'Row', COMMA, UPLO, 'No', M, M,
     $                        A, LDA )
              ELSE
                CALL CTRBR2D( ICONTXT, 'Row', COMMA, UPLO, 'No', M, M,
     $                        A, LDA, MYROW, IACOL )
              END IF
              ADATA = .TRUE.
            ELSE
              IF( MYCOL.EQ.IACOL ) THEN
                CALL CTRBS2D( ICONTXT, 'Row', COMMA, UPLO, 'No', M, M,
     $                        A, LDA )
                CALL PBCMATADD( ICONTXT, UPLO, M, M, ONE, A, LDA, ZERO,
     $                          WORK, M )
              ELSE
                CALL CTRBR2D( ICONTXT, 'Row', COMMA, UPLO, 'No', M, M,
     $                        WORK, M, MYROW, IACOL )
              END IF
            END IF
          END IF
*
*         Compute CSYMM
*
          IF( ADATA ) THEN
            CALL CSYMM( 'Left', UPLO, M, NQ, ALPHA, A, LDA, B, LDB,
     $                  BETA, C, LDC )
          ELSE
            CALL CSYMM( 'Left', UPLO, M, NQ, ALPHA, WORK, M, B, LDB,
     $                  BETA, C, LDC )
          END IF
*
        ELSE IF( LSAME( SIDE, 'R' ) .AND. MYCOL.EQ.IACOL ) THEN
*
*         Form  B := alpha*B*A + beta*C.
*            _             _                  _
*           | |           | |                | |
*           | |           | |                | |
*           | |           | |    _           | |
*           |C| = alpha * |B| * |_| + beta * |C|
*           | |           | |    A           | |
*           | |           | |                | |
*           |_|           |_|                |_|
*
          IF( IAROW.EQ.-1 )  ADATA = .TRUE.
          NP = NUMROC( M, NB, MYROW, IBPOS, NPROW )
*
          IF( LDA.LT.MAX(1,N) .AND. ( ASPACE .OR.
     $          IAROW.EQ.MYROW .OR. IAROW.EQ.-1 ) ) THEN
            INFO = 10
          ELSE IF( LDB.LT.MAX(1,NP)               ) THEN
            INFO = 12
          ELSE IF( LDC.LT.MAX(1,NP)               ) THEN
            INFO = 15
          ELSE IF( IAROW.LT.-1.OR. IAROW.GE.NPROW ) THEN
            INFO = 16
          ELSE IF( IACOL.LT.0 .OR. IACOL.GE.NPCOL ) THEN
            INFO = 17
          ELSE IF( IBPOS.LT.0 .OR. IBPOS.GE.NPROW ) THEN
            INFO = 18
          ELSE IF( ICPOS.NE.IBPOS                 ) THEN
            INFO = 19
          END IF
          IF( INFO.NE.0 ) GO TO 10
*
*         Broadcast B if necessary
*
          IF( .NOT.ADATA ) THEN
            IF( ASPACE ) THEN
              IF( MYROW.EQ.IAROW ) THEN
                CALL CTRBS2D( ICONTXT, 'Col', COMMA, UPLO, 'No', N, N,
     $                        A, LDA )
              ELSE
                CALL CTRBR2D( ICONTXT, 'Col', COMMA, UPLO, 'No', N, N,
     $                        A, LDA, IAROW, MYCOL )
              END IF
              ADATA = .TRUE.
            ELSE
              IF( MYROW.EQ.IAROW ) THEN
                CALL CTRBS2D( ICONTXT, 'Col', COMMA, UPLO, 'No', N, N,
     $                        A, LDA )
                CALL PBCMATADD( ICONTXT, UPLO, N, N, ONE, A, LDA, ZERO,
     $                          WORK, N )
              ELSE
                CALL CTRBR2D( ICONTXT, 'Col', COMMA, UPLO, 'No', N, N,
     $                        WORK, N, IAROW, MYCOL )
              END IF
            END IF
          END IF
*
*         Compute CSYMM
*
          IF( ADATA ) THEN
            CALL CSYMM( 'Right', UPLO, NP, N, ALPHA, A, LDA, B, LDB,
     $                  BETA, C, LDC )
          ELSE
            CALL CSYMM( 'Right', UPLO, NP, N, ALPHA, WORK, N, B, LDB,
     $                  BETA, C, LDC )
          END IF
        END IF
      END IF
*
      RETURN
*
*     End of PBCSYMM
*
      END
