/*
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * License for the specific language governing rights and limitations
 * under the License.
 *
 * The Initial Developer of this code is Rodd Zurcher.
 * Portions created by Rodd Zurcher are Copyright (C) 1998 Rodd Zurcher.
 * All Rights Reserved.
 */

/*
 * NOTE - this is only a partial implementation of the PSerial abstract
 * class.  It defaults to 2400 bps, 8 data bits, 1 stop, odd parity.  It
 * does not support changing the baud rate or other attributes.  It does
 * now support manual control of RTS and DTR lines.
 */
 
#include <sys/time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <unistd.h>
#include <fcntl.h>
#include <string.h>
#include <termios.h>

#include "PSerial.h"

#ifndef DEFAULT_SERIAL_NAME
#define DEFAULT_SERIAL_NAME "/dev/ttyS0"
#endif


class PSerial_unix : public PSerial
{
public:
  PSerial_unix(void);
	virtual	bool	Open(const char *name);
	virtual void	Close();
	virtual long	Write(const void *ptr, long count);
	virtual void	FlushWrite();
	
	virtual long	Read(void *ptr, long count);
	virtual bool	SetTimeout(long timeout_ms);
	virtual bool	SetSpeed(int speed, int opts = 0);
	virtual bool	SetDTR(bool state);
	virtual bool	SetRTS(bool state);

private:
  
  bool SetRaw(void);
  inline int tv_ge(const struct timeval *a,
		   const struct timeval *b) {
    if (a->tv_sec < b->tv_sec)
      return 0;			// asec < bsec
    
    if (a->tv_sec == b->tv_sec) {
      if (a->tv_usec < b->tv_usec) {
	return 0;		// ausec < busec
      }
      return 1;			// ausec >= busec
    }
    return 1;			// asec > bsec
  };

  inline void tv_sub(const struct timeval *a,
		     struct timeval *b) {
    b->tv_sec = a->tv_sec - b->tv_sec;
    b->tv_usec = a->tv_usec - b->tv_usec;
    if (b->tv_usec < 0) {
      b->tv_usec += 1000000;
      b->tv_sec--;
    }
  };
	  
  int fTerm;
  long fTimeout;
};

PSerial* PSerial::NewSerial()
{
  return new PSerial_unix();
}

const char *PSerial::GetDefaultName()
{
  return DEFAULT_SERIAL_NAME;
}

PSerial_unix::PSerial_unix(void) :
  fTerm(-1),
  fTimeout(kPStream_NeverTimeout)
{}

bool PSerial_unix::Open(const char *name)
{
  if ((fTerm = open(name, O_RDWR)) < 0) { 
    return false;
  }

  if (!SetRaw()) {
    return false;
  }

  return true;
}

// Put the tty into raw mode
// Adapted from tty_raw, "Advanced Programing in the UNIX
// Environment", W. Richard Stevens, pp354-355
bool
PSerial_unix::SetRaw(void) {
  termios tios;

  if (tcgetattr(fTerm, &tios) < 0) {
    return false;
  }

  // echo off, canonical mode off, extended input processing off
  // signal chars off
  tios.c_lflag &= ~(ECHO | ICANON | IEXTEN | ISIG);

  // no SIGINT on BREAK, CR-to-NL off, input parity check off,
  // don't strip 8th bit on input, output flow control off
  tios.c_iflag &= ~(BRKINT | ICRNL | INPCK | ISTRIP | IXON);

  // clear size bits, 1 stop bit
  tios.c_cflag &= ~(CSIZE | CSTOPB);
 
  // set 8 bits/char, Parity Enabled, OddParity
  tios.c_cflag |= (CS8 | PARENB | PARODD);

  // output processing off
  tios.c_oflag &= ~(OPOST);

  // 1 byte at a time, no timer
  tios.c_cc[VMIN] = 1;
  tios.c_cc[VTIME] = 0;

  // Baudrate 2400
  if (cfsetispeed(&tios, B2400) < 0) {
    return false;
  }
  if (cfsetospeed(&tios, B2400) < 0) {
    return false;
  }

  if (tcsetattr(fTerm, TCSAFLUSH, &tios) < 0) {
    return false;
  }

  return true;
}

void PSerial_unix::Close()
{
  if (fTerm != -1) {
    close(fTerm);
  }
  fTerm = -1;
}

bool PSerial_unix::SetSpeed(int /* speed */, int /* opts*/)
{
  return true;
}


long PSerial_unix::Write(const void *ptr, long count)
{
  return write(fTerm, ptr, count);
}


void PSerial_unix::FlushWrite()
{
  tcdrain(fTerm);
}


bool PSerial_unix::SetTimeout(long timeout_ms)
{
  fTimeout = timeout_ms;
  return true;
}

long PSerial_unix::Read(void *ptr, long count)
{
  // Blocking read
  if (fTimeout == kPStream_NeverTimeout) {
    long rval = read(fTerm, ptr, count);
    return rval;
  }
	
  // time limited read
  char *cur = (char *) ptr;
  struct timeval expire;
  struct timezone tz;

  if (gettimeofday(&expire, &tz) < 0) {
    return -1;
  }
  
  expire.tv_sec += fTimeout / 1000;
  expire.tv_usec += (fTimeout % 1000) * 1000;

  while (count) {
    fd_set rfds;
    FD_ZERO(&rfds);
    FD_SET(fTerm, &rfds);
    struct timeval delay;
    long nread;

    if (gettimeofday(&delay, &tz) < 0) {
      return -1;
    }
    if (tv_ge(&delay, &expire)) { // delay (current time) >= expire
      break;
    }
    tv_sub(&expire, &delay);	// delay = expire - delay

    if (select(fTerm + 1, &rfds, NULL, NULL, &delay)) {
      if (ioctl(fTerm, FIONREAD, &nread) < 0) {
	return -1;
      }
      nread = (count < nread) ? count : nread;
      if ((nread = read(fTerm, cur, nread)) < 0) {
	return -1;
      }
      count -= nread;
      cur += nread;
    }
    else {
      // timeout
      break;
    }
  } // while

  return (cur - (char *)ptr);
}

//
// This part implemented by Peter Ljungstrand (SetDTR and SetRTS)
//

bool PSerial_unix::SetDTR(bool state)
{
  int lineData;

  if (ioctl(fTerm, TIOCMGET, &lineData)) {
    return false;
  } else {
    if (state) {
      lineData |= TIOCM_DTR;
    } else {
      lineData &= ~TIOCM_DTR;
    }
    return !ioctl(fTerm, TIOCMSET, &lineData);
  }
}

bool PSerial_unix::SetRTS(bool state)
{
  int lineData;

  if (ioctl(fTerm, TIOCMGET, &lineData)) {
    return false;
  } else {
    if (state) {
      lineData |= TIOCM_RTS;
    } else {
      lineData &= ~TIOCM_RTS;
    }
    return !ioctl(fTerm, TIOCMSET, &lineData);
  }
}














