/*
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * License for the specific language governing rights and limitations
 * under the License.
 *
 * The Initial Developer of this code is David Baum.
 * Portions created by David Baum are Copyright (C) 1998 David Baum.
 * All Rights Reserved.
 */

#include "WhileStmt.h"
#include "Bytecode.h"
#include "CheckState.h"

WhileStmt::WhileStmt(Condition *c, Stmt *s)
{
	fCondition = c;
	fBody = s;
}


WhileStmt::~WhileStmt()
{
	delete fCondition;
	delete fBody;
}


bool WhileStmt::Check(CheckState &state)
{
	bool ok = true;
	bool value;
	
	state.fLoopDepth++;
	
	if (!fBody->Check(state)) ok = false;
	
	if (fCondition->Evaluate(value) && (value==false))
		fNullable = true;
	else
		fNullable = false;

	state.fLoopDepth--;
	
	return ok;
}


void WhileStmt::Emit(Bytecode &b)
{
	int cPos;
	int startLabel;
	bool value;
	
	b.PushLoopContext();
	cPos = b.GetLength();	// general case
		
	if (fCondition->Evaluate(value))
	{
		if (value)
		{
			/*
				cPos:
					body
					jump -> 1
			*/
			fBody->Emit(b);
			b.AddJump(b.GetContinueLabel());
		}
	}
	else
	{
		if (fBody->IsNullable())
		{
			/*
				cPos:	test !C -> cPos
			*/
			fCondition->Emit(b, b.GetContinueLabel());
		}
		else
		{
			/* 		jump -> cPos
				startLabel:
					body
				cPos:
					test C -> startLabel
			*/
			b.AddJump(b.GetContinueLabel());
			startLabel = b.NewLabel();
			b.SetLabel(startLabel);
			fBody->Emit(b);
			cPos = b.GetLength();	// we need to move cPos
			fCondition->Emit(b, startLabel);
		}
	}
	
	b.PopLoopContext(cPos, b.GetLength());
}


WhileStmt* WhileStmt::Clone(Mapping *b) const
{
	return new WhileStmt(fCondition->Clone(b), fBody->Clone(b));
}
