/*
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See the
 * License for the specific language governing rights and limitations
 * under the License.
 *
 * The Initial Developer of this code is David Baum.
 * Portions created by David Baum are Copyright (C) 1998 David Baum.
 * All Rights Reserved.
 */

#include "IfStmt.h"
#include "Bytecode.h"

IfStmt::IfStmt(Condition *c, Stmt *s1, Stmt *s2)
{
	fCondition = c;
	fThen = s1;
	fElse = s2;
}


IfStmt::~IfStmt()
{
	delete fCondition;
	delete fThen;
	delete fElse;
}


bool IfStmt::Check(CheckState &state)
{
	bool ok = true;
	bool v;
	
	if (!fThen->Check(state)) ok = false;
	if (fElse && !fElse->Check(state)) ok = false;
	
	fNullable = false;

	if (fCondition->Evaluate(v))
	{
		if ((v == true) && fThen->IsNullable())
			fNullable = true;
		else if ((v==false) && (!fElse || fElse->IsNullable()))
			fNullable = true;
	}

	return ok;
}


void IfStmt::Emit(Bytecode &b)
{
	bool value;
	
	if (fCondition->Evaluate(value))
	{
		if (value)
			fThen->Emit(b);
		else if (fElse)
			fElse->Emit(b);
	}
	else
	{
		if (fElse)
			EmitIfElse(b);
		else
			EmitIf(b);
	}
}


void IfStmt::EmitIf(Bytecode &b)
{
	int label;
	
	label = b.NewLabel();	
	fCondition->EmitInv(b, label);
	fThen->Emit(b);	
	b.SetLabel(label);
}


void IfStmt::EmitIfElse(Bytecode &b)
{
	int testLabel = b.NewLabel();
	int outLabel= b.NewLabel();
		
	fCondition->EmitInv(b, testLabel);
	
	// generate A and jump
	fThen->Emit(b);
	b.AddJump(outLabel);
	b.SetLabel(testLabel);
	
	// generate B
	fElse->Emit(b);
	b.SetLabel(outLabel);
}


IfStmt* IfStmt::Clone(Mapping *b) const
{
	return new IfStmt(
		fCondition->Clone(b),
		fThen->Clone(b),
		fElse ? fElse->Clone(b) : 0);
}
