/* GtkHint - prototype hint for the source editor widget.
 * Copyright (C) 1998 Kristian Hgsberg.
 *
 * Hints was written (or mostly stolen from gtktooltips) by Kristian
 * Hgsberg <hogsberg@daimi.au.dk>.  Bug reports should be send there.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */
#include <stdlib.h>
#include <string.h>
#include <stdio.h>

#include <gtk/gtkmain.h>
#include <gtk/gtkwidget.h>
#include <gtk/gtkwindow.h>
#include <gtk/gtksignal.h>
#include <gtk/gtkstyle.h>
#include <gtk/gtktext.h>

#include "gtkeditor.h"
#include "regex.h"
#include "gtkhint.h"


#define DEFAULT_DELAY 500           /* Default delay in ms */

static void gtk_hint_class_init        (GtkHintClass   *klass);
static void gtk_hint_init              (GtkHint        *hint);
static void gtk_hint_destroy           (GtkObject      *object);

static gint gtk_hint_event_handler     (GtkWidget      *widget,
					GdkEvent       *event);
static void gtk_hint_widget_unmap      (GtkWidget      *widget,
					gpointer        data);
static void gtk_hint_widget_remove     (GtkWidget      *widget,
					gpointer        data);
static void gtk_hint_set_active_widget (GtkHint        *hint,
					GtkWidget      *widget);
static gint gtk_hint_timeout           (gpointer        data);
static gint gtk_hint_expose            (GtkHint        *hint, 
					GdkEventExpose *event);

static void gtk_hint_draw_hint         (GtkHint        *hint,
					int cursor_x, int cursor_y);

static GtkDataClass *parent_class;
static const gchar  *hint_data_key = "_GtkHintData";

GtkType
gtk_hint_get_type (void)
{
  static GtkType hint_type = 0;

  if (!hint_type)
    {
      GtkTypeInfo hint_info =
      {
	"GtkHint",
	sizeof (GtkHint),
	sizeof (GtkHintClass),
	(GtkClassInitFunc) gtk_hint_class_init,
	(GtkObjectInitFunc) gtk_hint_init,
	/* reserved_1 */ NULL,
	/* reserved_2 */ NULL,
        (GtkClassInitFunc) NULL,
      };

      hint_type = gtk_type_unique (GTK_TYPE_DATA, &hint_info);
    }

  return hint_type;
}

static void
gtk_hint_class_init (GtkHintClass *class)
{
  GtkObjectClass *object_class;

  object_class = (GtkObjectClass*) class;
  parent_class = gtk_type_class (GTK_TYPE_DATA);

  object_class->destroy = gtk_hint_destroy;
}

static void
gtk_hint_init (GtkHint *hint)
{
  hint->hint_window = NULL;
  hint->active_hint_data = NULL;
  hint->hint_data_list = NULL;
  hint->gc = NULL;
  hint->foreground = NULL;
  hint->background = NULL;
  
  hint->delay = DEFAULT_DELAY;
  hint->enabled = TRUE;
  hint->timer_tag = 0;
}

GtkHint *
gtk_hint_new (void)
{
  return gtk_type_new (GTK_TYPE_HINT);
}

static void
gtk_hint_free_string (gpointer data, gpointer user_data)
{
  if (data)
    g_free (data);
}

static void
gtk_hint_destroy_data (GtkHintData *hintdata)
{
  g_free (hintdata->text);
  g_list_foreach (hintdata->row, gtk_hint_free_string, 0);
  if (hintdata->row)
    g_list_free (hintdata->row);
  gtk_signal_disconnect_by_data (GTK_OBJECT (hintdata->widget),
 				 (gpointer) hintdata);
  gtk_object_remove_data (GTK_OBJECT (hintdata->widget), hint_data_key);
  gtk_widget_unref (hintdata->widget);
  g_free (hintdata);
}

static void
gtk_hint_destroy (GtkObject *object)
{
  GtkHint *hint = GTK_HINT (object);
  GList *current;
  GtkHintData *hintdata;

  g_return_if_fail (hint != NULL);

  if (hint->timer_tag)
    {
      gtk_timeout_remove (hint->timer_tag);
      hint->timer_tag = 0;
    }

  if (hint->hint_data_list != NULL)
    {
      current = g_list_first (hint->hint_data_list);
      while (current != NULL)
	{
	  hintdata = (GtkHintData*) current->data;
	  current = current->next;
	  gtk_hint_widget_remove (hintdata->widget, hintdata);
	}
    }

  if (hint->hint_window)
    gtk_widget_destroy (hint->hint_window);

  if (hint->gc != NULL)
    {
      gdk_gc_destroy (hint->gc);
      hint->gc = NULL;
    }
}

void
gtk_hint_force_window (GtkHint *hint)
{
  g_return_if_fail (hint != NULL);
  g_return_if_fail (GTK_IS_HINT (hint));

  if (!hint->hint_window)
    {
      hint->hint_window = gtk_window_new (GTK_WINDOW_POPUP);
      gtk_widget_set_app_paintable (hint->hint_window, TRUE);
      gtk_widget_set_name (hint->hint_window, "GtkHint");
      gtk_widget_ref (hint->hint_window);
      gtk_window_set_policy (GTK_WINDOW (hint->hint_window), FALSE, FALSE, TRUE);

      gtk_signal_connect_object (GTK_OBJECT (hint->hint_window), 
				 "expose_event",
				 GTK_SIGNAL_FUNC (gtk_hint_expose), 
				 GTK_OBJECT (hint));

      gtk_signal_connect (GTK_OBJECT (hint->hint_window),
			  "destroy",
			  gtk_widget_destroyed,
			  &hint->hint_window);
    }
}

static void
gtk_hint_layout_text (GtkHint *hint, GtkHintData *data)
{
  gchar *row_end, *text, *row_text, *break_pos;
  gint i, row_width, window_width = 0;
  size_t len;

  if (!hint->hint_window)
    gtk_hint_force_window (hint);

  g_list_foreach (data->row, gtk_hint_free_string, 0);
  if (data->row)
    g_list_free (data->row);
  data->row = 0;
  data->font = hint->hint_window->style->font;
  data->width = 0;

  text = data->text;
  if (!text)
    return;

  while (*text)
    {
      row_end = strchr (text, '\n');
      if (!row_end)
       row_end = strchr (text, '\0');

      len = row_end - text + 1;
      row_text = g_new(gchar, len);
      memcpy (row_text, text, len - 1);
      row_text[len - 1] = '\0';

      /* now either adjust the window's width or shorten the row until
        it fits in the window */

      while (1)
       {
         row_width = gdk_string_width (data->font, row_text);
         if (!window_width)
           {
             /* make an initial guess at window's width: */

             if (row_width > gdk_screen_width () / 4)
               window_width = gdk_screen_width () / 4;
             else
               window_width = row_width;
           }
         if (row_width <= window_width)
           break;

         if (strchr (row_text, ' '))
           {
             /* the row is currently too wide, but we have blanks in
                the row so we can break it into smaller pieces */

             gint avg_width = row_width / strlen (row_text);

             i = window_width;
             if (avg_width)
               i /= avg_width;
             if ((size_t) i >= len)
               i = len - 1;

             break_pos = strchr (row_text + i, ' ');
             if (!break_pos)
               {
                 break_pos = row_text + i;
                 while (*--break_pos != ' ');
               }
             *break_pos = '\0';
           }
         else
           {
             /* we can't break this row into any smaller pieces, so
                we have no choice but to widen the window: */

             window_width = row_width;
             break;
           }
       }
      if (row_width > data->width)
       data->width = row_width;
      data->row = g_list_append (data->row, row_text);
      text += strlen (row_text);
      if (!*text)
       break;

      if (text[0] == '\n' && text[1])
       /* end of paragraph and there is more text to come */
       data->row = g_list_append (data->row, 0);
      ++text;  /* skip blank or newline */
    }
  data->width += 8;    /* leave some border */
}

void
gtk_hint_enable (GtkHint *hint)
{
  g_return_if_fail (hint != NULL);

  hint->enabled = TRUE;
}

void
gtk_hint_disable (GtkHint *hint)
{
  g_return_if_fail (hint != NULL);

  gtk_hint_set_active_widget (hint, NULL);

  hint->enabled = FALSE;
}

void
gtk_hint_set_delay (GtkHint *hint,
                        guint         delay)
{
  g_return_if_fail (hint != NULL);

  hint->delay = delay;
}

GtkHintData*
gtk_hint_data_get (GtkWidget       *widget)
{
  g_return_val_if_fail (widget != NULL, NULL);

  return gtk_object_get_data ((GtkObject*) widget, hint_data_key);
}

void
gtk_hint_set_hint (GtkHint         *hint,
		   GtkWidget       *widget,
		   const char      *regex,
		   GtkHintCallback  callback,
		   gpointer         callback_data)

{
  GtkHintData *hintdata;

  g_return_if_fail (hint != NULL);
  g_return_if_fail (GTK_IS_HINT (hint));
  g_return_if_fail (widget != NULL);

  hintdata = gtk_hint_data_get (widget);
  if (hintdata)
    gtk_hint_widget_remove (hintdata->widget, hintdata);

  if (!callback)
    return;

  hintdata = g_new0 (GtkHintData, 1);

  if (hintdata != NULL)
    {
      hintdata->hint = hint;
      hintdata->widget = widget;
      gtk_widget_ref (widget);

      hintdata->regex = g_new0 (struct re_pattern_buffer, 1);
      re_compile_pattern (regex, strlen (regex), hintdata->regex);

      hintdata->callback = callback;
      hintdata->callback_data = callback_data;

      hint->hint_data_list = g_list_append (hint->hint_data_list,
                                             hintdata);
      gtk_signal_connect_after(GTK_OBJECT (widget), "event",
                               (GtkSignalFunc) gtk_hint_event_handler,
 	                       (gpointer) hintdata);

      gtk_object_set_data (GTK_OBJECT (widget), hint_data_key,
                           (gpointer) hintdata);

      gtk_signal_connect (GTK_OBJECT (widget), "unmap",
                          (GtkSignalFunc) gtk_hint_widget_unmap,
                          (gpointer) hintdata);

      gtk_signal_connect (GTK_OBJECT (widget), "unrealize",
                          (GtkSignalFunc) gtk_hint_widget_unmap,
                          (gpointer) hintdata);

      gtk_signal_connect (GTK_OBJECT (widget), "destroy",
			  (GtkSignalFunc) gtk_hint_widget_remove,
			  (gpointer) hintdata);
    }
}

void
gtk_hint_set_colors (GtkHint *hint,
			 GdkColor    *background,
			 GdkColor    *foreground)
{
  g_return_if_fail (hint != NULL);

  if (background != NULL)
    hint->foreground = foreground;
  if (foreground != NULL)
    hint->background = background;
}

static gint
gtk_hint_expose (GtkHint *hint, GdkEventExpose *event)
{
  GtkStyle *style;
  gint y, baseline_skip, gap;
  GtkHintData *data;
  GList *el;

  style = hint->hint_window->style;

  gap = (style->font->ascent + style->font->descent) / 4;
  if (gap < 2)
    gap = 2;
  baseline_skip = style->font->ascent + style->font->descent + gap;

  data = hint->active_hint_data;
  if (!data)
    return FALSE;

  gtk_paint_flat_box(style, hint->hint_window->window,
		     GTK_STATE_NORMAL, GTK_SHADOW_OUT, 
		     NULL, GTK_WIDGET(hint->hint_window), "tooltip",
		     0, 0, -1, -1);

  y = style->font->ascent + 4;
  
  for (el = data->row; el; el = el->next)
    {
      if (el->data)
	{
	  gtk_paint_string (style, hint->hint_window->window, 
			    GTK_STATE_NORMAL, 
			    NULL, GTK_WIDGET(hint->hint_window), "tooltip",
			    4, y, el->data);
	  y += baseline_skip;
	}
      else
	y += baseline_skip / 2;
    }

  return FALSE;
}

static void
gtk_hint_draw_hint (GtkHint * hint, int cursor_x, int cursor_y)
{
  GtkWidget *widget;
  GtkStyle *style;
  gint gap, x, y, w, h, scr_w, scr_h, baseline_skip;
  GtkHintData *data;
  GList *el;

  if (!hint->hint_window)
    gtk_hint_force_window (hint);
  else if (GTK_WIDGET_VISIBLE (hint->hint_window))
    gtk_widget_hide (hint->hint_window);

  style = hint->hint_window->style;
  
  widget = hint->active_hint_data->widget;

  scr_w = gdk_screen_width ();
  scr_h = gdk_screen_height ();

  data = hint->active_hint_data;
  if (data->font != style->font)
    gtk_hint_layout_text (hint, data);

  gap = (style->font->ascent + style->font->descent) / 4;
  if (gap < 2)
    gap = 2;
  baseline_skip = style->font->ascent + style->font->descent + gap;

  w = data->width;
  h = 8 - gap;
  for (el = data->row; el; el = el->next)
    if (el->data)
      h += baseline_skip;
    else
      h += baseline_skip / 2;
  if (h < 8)
    h = 8;

#if 0
  gdk_window_get_pointer (NULL, &x, NULL, NULL);
  gdk_window_get_origin (widget->window, NULL, &y);
#else
  x = cursor_x;
  y = cursor_y;
#endif

#if 0
  x -= ((w >> 1) + 4);
#else
  x -= 4;
#endif

  if ((x + w) > scr_w)
    x -= (x + w) - scr_w;
  else if (x < 0)
    x = 0;

#if 0
  if ((y + h + widget->allocation.height + 4) > scr_h)
    y = y - h - 4;
  else
    y = y + widget->allocation.height + 4;
#else
  if ((y + h + 4) > scr_h)
    y = y - h - 8;
  else
    y = y + 8;
#endif

  gtk_widget_set_usize (hint->hint_window, w, h);
  gtk_widget_popup (hint->hint_window, x, y);
}

static gint
gtk_hint_timeout (gpointer data)
{
  GtkHint *hint = (GtkHint *) data;
  GtkHintData *hint_data = hint->active_hint_data; 
  GtkSCText *text;
  GtkWidget *widget;
  char *match;
  int i, x, y, index;
  struct re_registers *regs;

  text = GTK_SCTEXT (hint_data->widget);
  widget = GTK_WIDGET (text);
  index = GTK_EDITABLE (text)->current_pos;
  regs = g_new0 (struct re_registers, 1);

  if (re_search_2 (hint_data->regex, text->text.ch, text->gap_position,
		   text->text.ch + text->gap_position + text->gap_size,
		   text->text_end - text->gap_position - text->gap_size,
		   index, -index, regs, index) >= 0) {
    match = g_new (char, regs->end[1] - regs->start[1] + 1);
    for (i = regs->start[1]; i < regs->end[1]; i++) 
      match[i - regs->start[1]] = GTK_SCTEXT_INDEX (text, i);
    match[i - regs->start[1]] = 0;

    if (hint_data->callback && GTK_WIDGET_DRAWABLE (widget)) {
      hint_data->text = hint_data->callback(match, 
						 hint_data->callback_data);
      if (hint_data->text != NULL) {
	gtk_hint_layout_text (hint, hint_data);
	gdk_window_get_origin (widget->window, &x, &y);
	x += text->cursor_pos_x;
	y += text->cursor_pos_y;
	gtk_hint_draw_hint (hint, x, y);
      }
    }
    g_free (match);
  }
  g_free (regs);
  return FALSE;
}

static void
gtk_hint_set_active_widget (GtkHint *hint,
                                GtkWidget   *widget)
{
  if (hint->hint_window)
    gtk_widget_hide (hint->hint_window);
  if (hint->timer_tag)
    {
      gtk_timeout_remove (hint->timer_tag);
      hint->timer_tag = 0;
    }
  
  hint->active_hint_data = NULL;
  
  if (widget)
    {
      GList *list;
      
      for (list = hint->hint_data_list; list; list = list->next)
	{
	  GtkHintData *hintdata;
	  
	  hintdata = list->data;
	  
	  if (hintdata->widget == widget &&
	      GTK_WIDGET_DRAWABLE (widget))
	    {
	      hint->active_hint_data = hintdata;
	      break;
	    }
	}
    }
}

static gint
gtk_hint_event_handler (GtkWidget *widget,
			GdkEvent  *event)
{
  GtkHint *hint;
  GtkHintData *old_hint_data, *hint_data;
  GtkWidget *event_widget;

#if 0
  if ((event->type == GDK_LEAVE_NOTIFY || event->type == GDK_ENTER_NOTIFY) &&
      event->crossing.detail == GDK_NOTIFY_INFERIOR)
    return FALSE;
#endif

  event_widget = gtk_get_event_widget (event);
#if 0
  if (event_widget != widget)
    return FALSE;
#endif
  
  hint_data = gtk_hint_data_get (widget);
  hint = hint_data->hint;

  switch (event->type)
    {
    case GDK_EXPOSE:
      /* do nothing */
      break;
      
    case GDK_KEY_PRESS:
      old_hint_data = hint->active_hint_data;
      if (hint->enabled)
	{
	  gtk_hint_set_active_widget (hint, widget);
	  
	  hint->timer_tag = gtk_timeout_add (hint->delay,
					     gtk_hint_timeout,
					     (gpointer) hint);
	}
      break;

    default:
      gtk_hint_set_active_widget (hint, NULL);
      break;
    }

  return FALSE;
}

static void
gtk_hint_widget_unmap (GtkWidget *widget,
		       gpointer   data)
{
  GtkHintData *hintdata = (GtkHintData *)data;
  GtkHint *hint = hintdata->hint;
  
  if (hint->active_hint_data &&
      (hint->active_hint_data->widget == widget))
    gtk_hint_set_active_widget (hint, NULL);
}

static void
gtk_hint_widget_remove (GtkWidget *widget,
			    gpointer   data)
{
  GtkHintData *hintdata = (GtkHintData*) data;
  GtkHint *hint = hintdata->hint;

  gtk_hint_widget_unmap (widget, data);
  hint->hint_data_list = g_list_remove (hint->hint_data_list,
					    hintdata);
  gtk_hint_destroy_data (hintdata);
}
