{
Heap management routines

Copyright (C) 1991-99 Free Software Foundation, Inc.

Authors: Jukka Virtanen <jtv@hut.fi>
         Frank Heckenbach <frank@pascal.gnu.de>

This file is part of the GNU Pascal Library. The GNU Pascal
Library is free software; you can redistribute it and/or modify
it under the terms of the GNU Library General Public License as
published by the Free Software Foundation; either version 2 of
the License, or (at your option) any later version.

The GNU Pascal Library is distributed in the hope that it will
be useful, but WITHOUT ANY WARRANTY; without even the implied
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
See the GNU Library General Public License for more details.

You should have received a copy of the GNU Library General Public
License along with this library; see the file COPYING.LIB.
If not, write to the Free Software Foundation, Inc.,
59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
}

unit Heap;

interface

uses Internal, Error;

{ C heap management routines. NOTE: when using Release, CFreeMem and
  CReAlloc must only be used for pointers allocated by CGetMem. }
function  CGetMem     (Size : SizeType) : Pointer;                        asmname 'malloc';
procedure CFreeMem    (aPointer : Pointer);                               asmname 'free';
function  CReAlloc    (aPointer : Pointer; NewSize : SizeType) : Pointer; asmname 'realloc';

type
  GetMemType  = ^function (Size : SizeType) : Pointer;
  FreeMemType = ^procedure (aPointer : Pointer);
  ReAllocType = ^function (aPointer : Pointer; NewSize : SizeType) : Pointer;

{ These variables can be set to user-defined routines for memory
  allocation/deallocation. GetMemPtr may return nil when insufficient
  memory is available. GetMem/New will produce a runtime error then. }
var
  GetMemPtr  : asmname '_p_getmem_ptr'  GetMemType;
  GetMemPtr  : GetMemType = @CGetMem;
  FreeMemPtr : asmname '_p_freemem_ptr' FreeMemType;
  FreeMemPtr : FreeMemType = @CFreeMem;
  ReAllocPtr : asmname '_p_realloc_ptr' ReAllocType;
  ReAllocPtr : ReAllocType = @CReAlloc;

const
  UndocumentedReturnNil = Pointer (- 1);

{ GPC_GetMem, GPC_FreeMem and GPC_ReAlloc call the actual routines
  through GetMemPtr, FreeMemPtr and ReAllocPtr. }
function  GPC_GetMem   (Size : SizeType) : Pointer;    asmname '_p_malloc';
procedure GPC_FreeMem  (aPointer : Pointer);           asmname '_p_free';
procedure GPC_ReAlloc  (var aPointer : Pointer; NewSize : SizeType); asmname '_p_gpc_realloc';

procedure GPC_Mark     (var aPointer : Pointer);       asmname '_p_mark';
procedure GPC_Release  (aPointer : Pointer);           asmname '_p_release';

{ Returns the number of pointers that would be released. aPointer must
  have been marked with Mark. For an example of its usage, see the
  HeapMon unit. }
function  ReleaseCount (aPointer : Pointer) : Integer; asmname '_p_releasecount';

{ GPC_New, GPC_Dispose and ReAlloc call GPC_GetMem, GPC_FreeMem and
  GPC_ReAlloc, but also do the stuff necessary for Mark and Release.
  Therefore, GPC_GetMem, GPC_FreeMem and GPC_ReAlloc should not be
  called directly when using Mark and Release. GetMem and FreeMem in
  a Pascal program will call GPC_New and GPC_Dispose internally, not
  GPC_GetMem and GPC_FreeMem. }
function  GPC_New      (Size : SizeType) : Pointer;    asmname '_p_new';
procedure GPC_Dispose  (aPointer : Pointer);           asmname '_p_dispose';
procedure ReAlloc (var aPointer : Pointer; NewSize : SizeType); asmname '_p_realloc';

procedure HeapInit;                                    asmname '_p_heap_init';

implementation

type
  PMarkList = ^TMarkList;
  TMarkList = record
    Next         : PMarkList;
    PointersUsed : Integer;
    Pointers     : array [0 .. 255] of Pointer
  end;

var
  CurrentMarkList : PMarkList = nil;
  HeapBegin : Pointer = nil;
  HeapEnd   : Pointer = nil; { HeapEnd points to the highest byte of heap used }

function GPC_GetMem (Size : SizeType) = p : Pointer;
begin
  p := GetMemPtr^(Size);
  if (p = nil) or (p = UndocumentedReturnNil) then RuntimeErrorInteger (853, Size) { out of heap when allocating %d bytes }
end;

procedure GPC_FreeMem (aPointer : Pointer);
begin
  FreeMemPtr^(aPointer)
end;

procedure GPC_ReAlloc (var aPointer : Pointer; NewSize : SizeType);
begin
  aPointer := ReallocPtr^(aPointer, NewSize);
  if (aPointer = nil) or (aPointer = UndocumentedReturnNil) then RuntimeErrorInteger (854, NewSize) { out of heap when reallocating %ld bytes }
end;

procedure GPC_Mark (var aPointer : Pointer);
var Temp : PMarkList;
begin
  Temp := GPC_GetMem (SizeOf (Temp^)); { don't use `New' here! }
  Temp^.Next := CurrentMarkList;
  Temp^.PointersUsed := 0;
  CurrentMarkList := Temp;
  if @aPointer <> nil then aPointer := Temp { GPC_New calls GPC_Mark (null) }
end;

procedure GPC_Release (aPointer : Pointer);
var
  Temp : PMarkList;
  i : Integer;
begin
  Temp := CurrentMarkList;
  while (Temp <> nil) and (Temp <> aPointer) do Temp := Temp^.Next;
  if Temp = nil then
    RuntimeErrorInteger (852, PtrCard (aPointer)); { cannot release object at address $%lx }
  repeat
    for i := CurrentMarkList^.PointersUsed - 1 downto 0 do
      if CurrentMarkList^.Pointers [i] <> nil then
        GPC_FreeMem (CurrentMarkList^.Pointers [i]);
    Temp := CurrentMarkList;
    CurrentMarkList := CurrentMarkList^.Next;
    GPC_FreeMem (Temp)
  until Temp = aPointer
end;

function ReleaseCount (aPointer : Pointer) = Count : Integer;
var
  Temp, Last : PMarkList;
  i : Integer;
begin
  Count := 0;
  Temp := CurrentMarkList;
  Last := nil;
  while (Temp <> nil) and (Last <> aPointer) do
    begin
      for i := Temp^.PointersUsed - 1 downto 0 do
        if Temp^.Pointers [i] <> nil then Inc (Count);
      Last := Temp;
      Temp := Temp^.Next
    end
end;

procedure AddToMarkList (p : Pointer; Size : SizeType);
type
  PBytes = ^TBytes;
  TBytes = array [0 .. MaxInt] of Byte;
var pEnd : Pointer;
begin
  pEnd := @PBytes (p)^[Size - 1];
  if PtrCard (pEnd) > PtrCard (HeapEnd) then HeapEnd := pEnd;
  if CurrentMarkList <> nil then
    begin
      if CurrentMarkList^.PointersUsed > High (CurrentMarkList^.Pointers) then
        GPC_Mark (null); { this creates a new TMarkList item }
      CurrentMarkList^.Pointers [CurrentMarkList^.PointersUsed] := p;
      Inc (CurrentMarkList^.PointersUsed)
    end
end;

procedure RemoveFromMarkList (aPointer : Pointer);
var
  p : PMarkList;
  Found : Boolean;
  i : Integer;
begin
  if aPointer = nil then Exit;
  Found := False;
  p := CurrentMarkList;
  while (p <> nil) and not Found do
    begin
      for i := p^.PointersUsed - 1 downto 0 do
        if p^.Pointers [i] = aPointer then
          begin
            p^.Pointers [i] := nil;
            Found := True;
            Break
          end;
      p := p^.Next
    end
end;

function GPC_New (Size : SizeType) = p : Pointer;
begin
  p := GetMemPtr^(Size);
  if p = nil then RuntimeErrorInteger (853, Size); { out of heap when allocating %d bytes }
  if p = UndocumentedReturnNil then return nil;
  AddToMarkList (p, Size)
end;

procedure GPC_Dispose (aPointer : Pointer);
begin
  RemoveFromMarkList (aPointer);
  if aPointer <> nil then GPC_FreeMem (aPointer)
end;

procedure ReAlloc (var aPointer : Pointer; NewSize : SizeType);
begin
  RemoveFromMarkList (aPointer);
  GPC_ReAlloc (aPointer, NewSize);
  AddToMarkList (aPointer, NewSize)
end;

procedure HeapInit;
var p : Pointer;
begin
  p := GPC_GetMem (1);
  HeapBegin := p;
  HeapEnd := p
end;

end.
