(****************************************************************************)
(*                 The Calculus of Inductive Constructions                  *)
(*                                                                          *)
(*                                Projet Coq                                *)
(*                                                                          *)
(*                     INRIA        LRI-CNRS        ENS-CNRS                *)
(*              Rocquencourt         Orsay          Lyon                    *)
(*                                                                          *)
(*                                 Coq V6.3                                 *)
(*                               July 1st 1999                              *)
(*                                                                          *)
(****************************************************************************)
(*                               equality.ml                                *)
(****************************************************************************)

open Impuniv
open Std
open Initial
open Names
open Vectops
open Generic
open Term
open Reduction
open Typing
open Termenv
open Mach
open More_util
open Pp

open Tacmach
open Proof_trees

open Wcclausenv
open Pattern
open Tacticals
open Tactics
open Vernacinterp;;

(* Rewriting tactics *)

(* Warning : rewriting from left to right only works
   if there exists in the context a theorem named <eqname>_<suffsort>_r
   with type (A:<sort>)(x:A)(P:A->Prop)(P x)->(y:A)(eqname A y x)->(P y).
   If another equality myeq is introduced, then corresponding theorems
   myeq_ind_r, myeq_rec_r and myeq_rect_r have to be proven. See below.
   -- Eduardo (19/8/97
*)

let general_rewrite_bindings lft2rgt (c,l) gl =
  let ctype  = pf_type_of gl c in 
  let sigma = project gl in 
  let sign,t = splay_prod sigma ctype in
    match match_with_equation t with
      	None -> error "The term provided does not end with an equation" 
      | Some (hdcncl,_) -> 
          let hdcncls = string_head hdcncl in 
          let elim =
	    if lft2rgt then
              pf_global gl (id_of_string 
			      (hdcncls^(suff gl (pf_concl gl))^"_r"))
            else
	      pf_global gl (id_of_string (hdcncls^(suff gl (pf_concl gl))))
          in tclNOTSAMEGOAL 
(* was tclWEAK_PROGRESS which only fails for tactics generating one subgoal 
   and did not fail for useless conditional rewritings generating an
   extra condition *)
            (general_elim (c,l) (elim,[])) gl
;;

let general_rewrite lft2rgt c = general_rewrite_bindings lft2rgt (c,[]);;




(* The following version of the rewriting tactic has the advantage of
   being independent form the context. However, its application raises
   errors during the compilation of some theories. I do not understand
   exactly why, but I guess there is a bad interaction with the number
   of subgoals remaining after an application of the tactic. It seems
   quite hard to modify the theories, specially ZARITH. The error
   seems to arise when a rewriting appears into a tactical.  
   -- Eduardo (19/8/97)


let general_rewrite lft2rgt c gl =
   let rc      = project gl           in
   let ctype   = (pf_type_of gl c)    in 
   let sign,t  = prod_decompose ctype 
   in match match_with_equation rc t with
        None -> error "The term provided does not end with an equation" 
      | Some (hdcncl,args) -> 
         if lft2rgt
         then let [typ;c1;c2] = args in
              let symeq = prod_it
                           (List.fold_left apply hdcncl [typ;c2;c1]) 
                           sign
              in tclWEAK_PROGRESS 
                     (tclTHENS (cut symeq)
                       [(tclTHEN intro
                        (tclTHEN (tclLAST_HYP simplest_case) (onLastHyp clear_clause)));
                        (tclTHEN intros
                        (tclTHEN symmetry
                              (apply c)))]) gl
         else tclWEAK_PROGRESS (simplest_case c) gl
;;
*)

let rewriteLR_bindings = general_rewrite_bindings true;;
let rewriteRL_bindings = general_rewrite_bindings false;;


let rewriteLR = general_rewrite true;;
let rewriteRL = general_rewrite false;;


let dyn_rewriteLR = function
    [(COMMAND com);(BINDINGS binds)] 
    -> tactic_com_bind_list rewriteLR_bindings (com,binds)
  | [(CONSTR c);(CBINDINGS binds)] -> rewriteLR_bindings (c,binds)
  | _ -> assert false
;;
let dyn_rewriteRL = function
    [(COMMAND com);(BINDINGS binds)] 
    -> tactic_com_bind_list rewriteRL_bindings (com,binds)
  | [(CONSTR c);(CBINDINGS binds)] -> rewriteRL_bindings (c,binds)
  | _ -> assert false
;;

(* Replacing tactics *)

(* eq,symeq : equality on Set and its symmetry theorem
   eqt,sym_eqt : equality on Type and its symmetry theorem
   c2 c1 : c1 is to be replaced by c2
   unsafe : If true, do not check that c1 and c2 are convertible
   gl : goal
*)

let abstract_replace (eq,sym_eq) (eqt,sym_eqt) c2 c1 unsafe gl =
  let t1 = pf_type_of gl c1 
  and t2 = pf_type_of gl c2 in
    if unsafe or (pf_conv_x gl t1 t2) then
      let (e,sym) = 
        match hnf_type_of gl t1 with 
            DOP0(Sort(Prop(Pos))) -> (eq,sym_eq)
          | DOP0(Sort(Type(_)))   -> (eqt,sym_eqt)
          | _                     -> error "replace"
      in (tclTHENL (elim_type (applist (e, [t1;c1;c2])))
            (tclORELSE assumption 
               (tclTRY (tclTHEN (apply sym) assumption)))) gl
    else
      error "terms does not have convertible types"
;;
  

(* Only for internal use *)
let unsafe_replace c2 c1 gl = 
  let eq        = (pf_parse_const gl "eq")     in
  let eqt       = (pf_parse_const gl "eqT")    in 
  let sym_eq    = (pf_parse_const gl "sym_eq") in
  let sym_eqt   = (pf_parse_const gl "sym_eqT") in
    abstract_replace (eq,sym_eq) (eqt,sym_eqt) c2 c1 true gl;;

let replace c2 c1 gl = 
  let eq        = (pf_parse_const gl "eq")     in
  let eqt       = (pf_parse_const gl "eqT")    in 
  let sym_eq    = (pf_parse_const gl "sym_eq") in
  let sym_eqt   = (pf_parse_const gl "sym_eqT") in
    abstract_replace (eq,sym_eq) (eqt,sym_eqt) c2 c1 false gl;;

let dyn_replace args gl = 
  match args with 
      [(COMMAND c1);(COMMAND c2)] -> 
       	replace (pf_constr_of_com gl c1) (pf_constr_of_com gl c2) gl
    | [(CONSTR c1);(CONSTR c2)] -> 
       	replace c1 c2 gl
    | _ -> assert false
;;
                 
let v_rewriteLR = 
    hide_tactic "RewriteLR" dyn_rewriteLR
let h_rewriteLR_bindings (c,bl) = v_rewriteLR [(CONSTR c);(CBINDINGS bl)] ;;
let h_rewriteLR c = h_rewriteLR_bindings (c,[]);;

let v_rewriteRL = 
    hide_tactic 
         "RewriteRL"     
          dyn_rewriteRL
let h_rewriteRL_bindings (c,bl) = v_rewriteRL [(CONSTR c);(CBINDINGS bl)] ;;
let h_rewriteRL c = h_rewriteRL_bindings (c,[]);;


let v_replace = 
    hide_tactic 
         "Replace"       
          dyn_replace;;
let h_replace c1 c2 = v_replace [(CONSTR c1);(CONSTR c2)];;

(* Conditional rewriting, the success of a rewriting is related 
   to the resolution of the conditions by a given tactic 
*)

let conditional_rewrite lft2rgt tac (c,bl) = 
  tclTHEN_i (general_rewrite_bindings lft2rgt (c,bl))
    (fun i -> if i=1 then tclIDTAC else tclCOMPLETE tac) 1;;

let dyn_conditional_rewrite lft2rgt = function
    [(TACEXP tac); (COMMAND com);(BINDINGS binds)] 
    -> tactic_com_bind_list 
	(conditional_rewrite lft2rgt (Tacinterp.interp tac)) 
	(com,binds)
  | [(TACEXP tac); (CONSTR c);(CBINDINGS binds)] 
    -> conditional_rewrite lft2rgt (Tacinterp.interp tac) (c,binds)
  | _ -> assert false
;;

let v_conditional_rewriteLR = hide_tactic "CondRewriteLR" (dyn_conditional_rewrite true);;

let v_conditional_rewriteRL = hide_tactic "CondRewriteRL" (dyn_conditional_rewrite false);;


(* End of Eduardo's code. The rest of this file could be improved
   using the functions match_with_equation, etc that I defined
   in Pattern.ml.
   -- Eduardo (19/8/97)
*)


(* Tactics for equality reasoning with the "eq"  or "eqT"
   relation This code  will work with any equivalence relation which 
   is substitutive *)

let find_constructor sigma c =
  match whd_betadeltaiota_stack sigma c [] with
      DOPN(MutConstruct _,_) as hd,stack -> (hd,stack)
    | _ -> error "find_constructor"
;;

type leibniz_eq = {eq   : marked_term;
                   ind  : marked_term;
                   rrec : marked_term option;
                   rect : marked_term option;
                   congr: marked_term;
                   sym  : marked_term
                   };;

let mmk = make_module_marker 
	    [ "#Prelude.obj"; "#Logic_Type.obj"; "#Specif.obj";
	      "#Logic.obj"; "#Core.obj"]

let eq_pattern = put_pat mmk "(eq ? ? ?)"
let not_pattern = put_pat mmk "(not ?)"
let imp_False_pattern = put_pat mmk "? -> False"

let pat_True = put_pat mmk "True"
let pat_False = put_pat mmk "False"
let pat_I = put_pat mmk "I"

let eq= {eq  = put_pat mmk "eq";
         ind = put_pat mmk "eq_ind" ;
         rrec = Some (put_pat mmk "eq_rec");
         rect = Some (put_pat mmk "eq_rect");
         congr = put_pat mmk "f_equal"  ;
         sym  = put_pat mmk "sym_eq"};;

let eqT_pattern = put_pat mmk "(eqT ? ? ?)";;


let eqT= {eq  = put_pat mmk "eqT";
          ind = put_pat mmk "eqT_ind" ;
          rrec = None;
          rect = None;
          congr = put_pat mmk "congr_eqT"  ;
          sym  = put_pat mmk "sym_eqT"};;


let idT_pattern = put_pat mmk "(identityT ? ? ?)";;

let idT=  {eq  = put_pat mmk "identityT";
          ind = put_pat mmk "identityT_ind" ;
          rrec = Some (put_pat mmk "identityT_rec")  ;
          rect = Some (put_pat mmk "identityT_rect");
          congr = put_pat mmk "congr_idT"  ;
          sym  = put_pat mmk "sym_idT"};;

let pat_EmptyT = put_pat mmk "EmptyT";;
let pat_UnitT = put_pat mmk "UnitT";;
let pat_IT = put_pat mmk "IT";;
let notT_pattern = put_pat mmk "(notT ?)";;

let rec hd_of_prod prod =
  match strip_outer_cast prod with
      (DOP2(Prod,c,DLAM(n,t'))) -> hd_of_prod t'
    |  t -> t
;;

type elimination_types =
    Set_Type
  | Type_Type
  | Set_SetorProp
  | Type_SetorProp ;;

let necessary_elimination sort_arity  sort =
  if (is_Type sort) then
    if is_Set sort_arity then
      Set_Type
    else 
      if is_Type sort_arity then
	Type_Type
      else  errorlabstrm "necessary_elimination" 
        [< 'sTR "no primitive equality on proofs" >]  
  else
    if is_Set sort_arity then
      Set_SetorProp
    else
      if is_Type sort_arity then
	Type_SetorProp
      else  errorlabstrm "necessary_elimination" 
        [< 'sTR "no primitive equality on proofs" >]
;;


(* [sort] is the sort of the goal. If sort<>Type then 
   depending on the sort of the arity we choose eq or eqT. 
   If sort=Type then we choose eq or identityT.
*)
(****
let find_eq_pattern arity sort = 
let ty=hd_of_prod arity in
 if  (is_Type sort) 
      then    if is_Set ty  then  eq.eq
                else  if is_Type ty then idT.eq
                       else  errorlabstrm "make_inv_predicate" 
                             [< 'sTR "no primitive equality on proofs" >]  
     else    if is_Set ty  then  eq.eq
                else  if is_Type ty then eqT.eq
                       else  errorlabstrm "make_inv_predicate" 
                                [< 'sTR "no primitive equality on proofs" >]
;;
****)


let find_eq_pattern arity sort = 
  let mt =
    match necessary_elimination (hd_of_prod arity) sort with
        Set_Type       ->  eq.eq
      | Type_Type      ->  idT.eq
      | Set_SetorProp  ->  eq.eq
      | Type_SetorProp ->  eqT.eq
(*    |       _         -> errorlabstrm "find_eq_pattern" 
                         [< 'sTR "no primitive equality on proofs" >]*)
  in get_pat mt
;;

(* [find_positions t1 t2]

   will find the positions in the two terms which are suitable for
   discrimination, or for injection.  Obviously, if there is a
   position which is suitable for discrimination, then we want to
   exploit it, and not bother with injection.  So when we find a
   position which is suitable for discrimination, we will just raise
   an exception with that position.

   So the algorithm goes like this:

   if [t1] and [t2] start with the same constructor, then we can
   continue to try to find positions in the arguments of [t1] and
   [t2].

   if [t1] and [t2] do not start with the same constructor, then we
   have found a discrimination position

   if one [t1] or [t2] do not start with a constructor and the two
   terms are not already convertible, then we have found an injection
   position.

   A discriminating position consists of a constructor-path and a pair
   of operators.  The constructor-path tells us how to get down to the
   place where the two operators, which must differ, can be found.

   An injecting position has two terms instead of the two operators,
   since these terms are different, but not manifestly so.

   A constructor-path is a list of pairs of (operator * int), where
   the int (based at 0) tells us which argument of the operator we
   descended into.

 *)

exception DiscrFound of (sorts oper * int) list * sorts oper * sorts oper;;

let find_positions sigma sign t1 t2 =
  
  let rec findrec posn t1 t2 =
    match (whd_betadeltaiota_stack sigma t1 [],
           whd_betadeltaiota_stack sigma t2 []) with
  	
    	((DOPN(MutConstruct _ as oper1,_) as hd1,args1),
	 (DOPN(MutConstruct _ as oper2,_) as hd2,args2)) ->
        (* both sides are constructors, so either we descend, or we can
           discriminate here.
         *)
	  if oper1 = oper2 then
            List.flatten (map2_i (fun i arg1 arg2 ->
				    findrec ((oper1,i)::posn) arg1 arg2)
			    0 args1 args2)
	  else
	    raise (DiscrFound(List.rev posn,oper1,oper2))

      | (t1_0,t2_0) ->
	  let t1_0 = applist t1_0
          and t2_0 = applist t2_0 in
            if conv_x sigma t1_0 t2_0 then []
            else (match whd_castapp ((fexecute sigma sign t1_0)._KIND) with
		      DOP0(Sort(Prop Pos)) ->
			[(List.rev posn,t1_0,t2_0)] (* Set *)
		    | DOP0(Sort(Type(_))) ->
			[(List.rev posn,t1_0,t2_0)] (* Type *)
		    | _ -> [])
	  in (try Inr(findrec [] t1 t2)
	      with DiscrFound (x_0,x_1,x_2) -> Inl (x_0,x_1,x_2))
;;


let discriminable sigma sign t1 t2 =
  match find_positions sigma sign t1 t2 with
      Inl _ -> true
    | _ -> false
;;

(* Once we have found a position, we need to project down to it.  If
   we are discriminating, then we need to produce False on one of the
   branches of the discriminator, and True on the other one.  So the
   result type of the case-expressions is always Prop.

   If we are injecting, then we need to discover the result-type.
   This can be difficult, since the type of the two terms at the
   injection-position can be different, and we need to find a
   dependent sigma-type which generalizes them both.

   We can get an approximation to the right type to choose by:

   (0) Before beginning, we reserve a metavariable for the default
   value of the match, to be used in all the bogus branches.

   (1) perform the case-splits, down to the site of the injection.  At
   each step, we have a term which is the "head" of the next
   case-split.  At the point when we actually reach the end of our
   path, the "head" is the term to return.  We compute its type, and
   then, backwards, make a sigma-type with every free debruijn
   reference in that type.  We can be finer, and first do a S(TRONG)NF
   on the type, so that we get the fewest number of references
   possible.

   (2) This gives us a closed type for the head, which we use for the
   types of all the case-splits.

   (3) Now, we can compute the type of one of T1, T2, and then unify
   it with the type of the last component of the result-type, and this
   will give us the bindings for the other arguments of the tuple.

 *)

(* The algorithm, then is to perform successive case-splits.  We have
   the result-type of the case-split, and also the type of that
   result-type.  We have a "direction" we want to follow, i.e. a
   constructor-number, and in all other "directions", we want to juse
   use the default-value.

   After doing the case-split, we call the afterfun, with the updated
   environment, to produce the term for the desired "direction".

   The assumption is made here that the result-type is not manifestly
   functional, so we can just use the length of the branch-type to
   know how many lambda's to stick in.

 *)

(* [descend_then sigma env head dirn]

   returns the number of products introduced, and the environment
   which is active, in the body of the case-branch given by [dirn],
   along with a continuation, which expects to be fed:

    (1) the value of the body of the branch given by [dirn]
    (2) the default-value

    (3) the type of the default-value, which must also be the type of
        the body of the [dirn] branch

   the continuation then constructs the case-split.
 *)

let descend_then sigma env head dirn =
  let headj = execute_rec sigma env head in
  let (construct,largs,nparams,arityind,mind,
       consnamev,case_fun,type_branch_fun)= 
    (match whd_betadeltaiota_stack sigma headj._TYPE [] with
      	(DOPN(MutInd (x_0,x_1),cl) as ity,largs) ->
	  let mispec = mind_specif_of_mind ity in 
    	  let nparams = mis_nparams mispec
	  and consnamev = mis_consnames mispec
	  and arity = mis_arity mispec in 
	    (DOPN(MutConstruct((x_0,x_1),dirn),cl),largs,nparams,
	     mis_arity mispec,ity,consnamev,mkMutCase,
	     type_case_branches env sigma)
	  | _ -> assert false)
  in
  let (globargs,largs) = chop_list nparams largs in
  let dirn_cty = 
    strong whd_castapp
      (type_of_rel sigma env (applist(construct,globargs))) in
  let dirn_nlams = nb_prod dirn_cty in
  let (_,dirn_env) = add_prods_rel sigma (dirn_cty,env) in

    (dirn_nlams,
     dirn_env,
     (fun dirnval (dfltval,resty) ->
	
	let nconstructors = Array.length consnamev in
	let arity =
	  hnf_prod_applist sigma "discriminate" arityind globargs in
	let p = lambda_ize (nb_prod arity) arity resty in
	let nb_prodP = nb_prod p in
	let (_,bty,_) =
	  type_branch_fun (DOP2(Cast,headj._TYPE,headj._KIND))
	    (type_of_rel sigma env p) p head in
	  
	let build_branch i =
	  let result = if i = dirn then dirnval else dfltval in
	  let nlams = nb_prod bty.(i-1) in
	  let typstack,_,_ =
	    push_and_liftl (nlams-nb_prodP) [] bty.(i-1) [] in
	  let _,branchval,_ =
	    lam_and_popl_named (nlams-nb_prodP) typstack result [] in
	    branchval in
	  
	  case_fun (ci_of_mind mind) p head 
	    (List.map build_branch (interval 1 nconstructors))))
;;


(* Now we need to construct the discriminator, given a discriminable
   position.  This boils down to:

   (1) If the position is directly beneath us, then we need to do a
   case-split, with result-type Prop, and stick True and False into
   the branches, as is convenient.

   (2) If the position is not directly beneath us, then we need to
   call descend_then, to descend one step, and then recursively
   construct the discriminator.

 *)

let necessary_elimination arity sort =
  let ty=hd_of_prod arity in
    if is_Type sort then
      if is_Set ty then
	Set_Type
      else
	if is_Type ty then
	  Type_Type
        else  errorlabstrm "necessary_elimination" 
          [< 'sTR "no primitive equality on proofs" >]  
    else
      if is_Set ty then
	Set_SetorProp
      else
	if is_Type ty then
	  Type_SetorProp
        else
	  errorlabstrm "necessary_elimination" 
            [< 'sTR "no primitive equality on proofs" >]
;;

(* [construct_discriminator env dirn headval]
   constructs a case-split on [headval], with the [dirn]-th branch
   giving [True], and all the rest giving False.
 *)

let construct_discriminator sigma env dirn c sort=
  let t = type_of_rel sigma env c in
  let (largs,nparams,arityind,mind,consnamev,case_fun,type_branch_fun) = 
    (match whd_betadeltaiota_stack sigma t [] with
      	(DOPN(MutInd (x_0,x_1),cl) as ity,largs) ->
    	  let mispec = mind_specif_of_mind ity in 
    	  let nparams = mis_nparams mispec
	  and consnamev = mis_consnames mispec
	  and arity = mis_arity mispec in 
	    (largs,nparams,mis_arity mispec,ity,consnamev,mkMutCase,
	     type_case_branches env sigma)
       | _ -> (* one can find Rel(k) in case of dependent constructors 
                 like T := c : (A:Set)A->T and a discrimination 
                 on (c bool true) = (c bool false)
                 CP : changed assert false in a more informative error
	       *)
              errorlabstrm "Equality.construct_discriminator"
		[< 'sTR "Cannot discriminate on inductive constructors with 
		     dependent types" >])
  in
  let nconstructors = Array.length consnamev in
  let (globargs,largs) = chop_list nparams largs in

  let arity =
    hnf_prod_applist sigma "construct_discriminator"  arityind globargs in
  let (true_0,false_0,sort_0) = 
    match necessary_elimination (hd_of_prod arity) sort with
        Type_Type ->
	  get_pat pat_UnitT, get_pat pat_EmptyT,
	  (DOP0(Sort (Type(dummy_univ))))
      | _ ->
	  get_pat pat_True, get_pat pat_False, (DOP0(Sort (Prop Null))) in
                    
  let eq = find_eq_pattern arity sort in
  let p = lambda_ize (nb_prod arity) arity sort_0 in

  let (_,bty,_) = type_branch_fun (DOP2(Cast,t,type_of_rel sigma env t)) 
                    (type_of_rel sigma env p) p c in

  let build_branch i =
    let nlams = nb_prod bty.(i-1) in
    let endpt = if i = dirn then true_0 else false_0 in
      lambda_ize nlams bty.(i-1) endpt in

  let build_match () =
    case_fun (ci_of_mind mind) p c 
      (List.map build_branch (interval 1 nconstructors)) in
    
    build_match()
;;



let rec build_discriminator sigma env dirn c sort = function
    [] -> construct_discriminator sigma env dirn c sort
  | (MutConstruct(sp,cnum),argnum)::l ->
      let cty = type_of_rel sigma env c in
      let (ity,_) = find_mrectype sigma cty in
      let arity=mind_arity ity in 
      let nparams = mind_nparams ity in
      let (cnum_nlams,cnum_env,kont) = descend_then sigma env c cnum in
      let newc = Rel(cnum_nlams-(argnum-nparams)) in
      let subval = build_discriminator sigma cnum_env dirn newc sort l  in
      	(match necessary_elimination (hd_of_prod arity) sort with
            Type_Type ->
	      kont subval (get_pat pat_EmptyT,DOP0(Sort(Type(dummy_univ))))
	  | _ -> kont subval (get_pat pat_False,DOP0(Sort(Prop Null))))
  | _ -> assert false
;;


let dest_somatch_eq eqn eq_pat =
  match dest_somatch eqn eq_pat with
      [t;x;y] -> (t,x,y)
    | _ -> anomaly "dest_somatch_eq: an eq pattern should match 3 terms";;

let find_eq_data_decompose eqn =
  if (somatches eqn eq_pattern) then
    (eq, dest_somatch_eq eqn eq_pattern)
  else if (somatches eqn eqT_pattern) then
    (eqT, dest_somatch_eq eqn eqT_pattern)
  else if (somatches eqn idT_pattern) then
    (idT, dest_somatch_eq eqn idT_pattern)
  else
    errorlabstrm  "find_eq_data_decompose" [< >]
;;

let gen_absurdity id gl =
  if   (matches gl (clause_type (Some id) gl) pat_False) 
    or (matches gl (clause_type  (Some id) gl) pat_EmptyT)
  then
    simplest_elim (VAR id) gl
  else
    errorlabstrm "Equality.gen_absurdity" 
      [< 'sTR "Not the negation of an equality" >]
;;

(*
let discr id gls =
let eqn = (pf_whd_betadeltaiota gls (clause_type (Some id) gls)) in
let sort = pf_type_of gls (pf_concl gls) in 
let (eq_data,(t,t1,t2))= try (find_eq_data_decompose   eqn) 
                        with UserError _ -> errorlabstrm  "Discriminate"
                [<'sTR (string_of_id id); 'sTR" Not a discriminable equation">] in

let tj = pf_fexecute gls t in
let sigma = project gls in
let sign = pf_untyped_hyps gls
in (match find_positions sigma sign t1 t2 with
    (Inr _) -> 
    errorlabstrm "Discr"
    [< 'sTR (string_of_id id); 'sTR" Not a discriminable equality" >]

  | (Inl(cpath,MutConstruct(_,dirn),_)) ->
let [e] = pf_get_new_ids [id_of_string "ee"] gls in
let e_env = 
  gLOB(add_sign (e,assumption_of_judgement sigma (gLOB sign) tj) sign) in
let discriminator = build_discriminator sigma e_env dirn (VAR e) sort cpath  in
let (indt,_) = find_mrectype sigma t in 
let arity=mind_arity indt in

let (pf, absurd_term) =  
    (match necessary_elimination (hd_of_prod arity) sort with
      Type_Type  -> let (Some rect) = idT.rect in
                    let  [eq_elim;eq_term;i; absurd_term] = 
                           List.map get_pat [rect; idT.eq; pat_IT; pat_EmptyT] in
                    let [h] = pf_get_new_ids   [id_of_string "HH"] gls in
                    let pred = lambda e t 
                                (lambda h (applist (eq_term, [t;t1;(Rel 1)])) 
                                      discriminator)
                    in (applist(eq_elim, [t;t1;pred;i;t2]), absurd_term)
 
    |   v  -> let [i;absurd_term]= List.map get_pat [pat_I; pat_False] and
                   eq_elim = (match v with
                              Set_SetorProp    -> get_pat eq.ind 
                             | Type_SetorProp  ->  get_pat eqT.ind )
              
               in  (applist(eq_elim, [t;t1;mkNamedLambda e t discriminator;i;t2]), absurd_term))
  

in tclCOMPLETE((tclTHENS (cut absurd_term)
            ([onLastHyp ((comp (gen_absurdity) (outSOME)));
             refine (apply pf (VAR id))]))) gls)
;;
*)


(* Precondition: eq is leibniz equality
 
  returns ((eq_elim t t1 P i t2), absurd_term)
  where  P=[e:t][h:(t1=e)]discrimator 
          absurd_term=EmptyT    if the necessary elimination is Type_Tyoe 

   and   P=[e:t][h[e:t]discriminator 
         absurd_term=Fale       if the necessary eliination is Type_ProporSet
                                   or Set_ProporSet
*)

let discrimination_pf e (t,t1,t2) discriminator lbeq gls =
  let (indt,_) = find_mrectype (project gls) t in 
  let arity = mind_arity indt in
  let sort = pf_type_of gls (pf_concl gls) in 
    match necessary_elimination (hd_of_prod arity) sort with
    	Type_Type  ->
 	  let rect = match lbeq.rect with Some x -> x | _ -> assert false in
	  let eq_elim     = get_pat rect in
	  let eq_term     = get_pat lbeq.eq in
	  let i           = get_pat pat_IT in
	  let absurd_term = get_pat pat_EmptyT in
          let h = pf_get_new_id (id_of_string "HH")gls in
          let pred= mkNamedLambda e t 
                      (mkNamedLambda h (applist (eq_term, [t;t1;(Rel 1)])) 
			 discriminator)
          in (applist(eq_elim, [t;t1;pred;i;t2]), absurd_term)
	       
      | _ ->
	  let i           = get_pat pat_I in
	  let absurd_term = get_pat pat_False in
	  let eq_elim     = get_pat lbeq.ind in
            (applist (eq_elim, [t;t1;mkNamedLambda e t discriminator;i;t2]),
 	     absurd_term)
;;


let discr id gls =
  let eqn = (pf_whd_betadeltaiota gls (clause_type (Some id) gls)) in
  let sort = pf_type_of gls (pf_concl gls) in 
  let (lbeq,(t,t1,t2)) =
    try (find_eq_data_decompose   eqn) 
    with UserError _ -> errorlabstrm  "Discriminate"
        [<'sTR (string_of_id id); 'sTR" Not a discriminable equality">] in

  let tj = pf_fexecute gls t in
  let sigma = project gls in
  let sign = pf_hyps gls in
    (match find_positions sigma sign t1 t2 with
	 (Inr _) -> 
	   errorlabstrm "Discr"
	     [< 'sTR (string_of_id id); 'sTR" Not a discriminable equality" >]
	     
       | (Inl(cpath,MutConstruct(_,dirn),_)) ->
	   let e = pf_get_new_id (id_of_string "ee") gls in
	   let e_env =
	     gLOB(add_sign 
		    (e,Machops.assumption_of_judgement sigma (gLOB sign) tj) 
		    sign) in
	   let discriminator =
	     build_discriminator sigma e_env dirn (VAR e) sort cpath in

	   let (indt,_) = find_mrectype sigma t in 
	   let arity=mind_arity indt in

	   let (pf, absurd_term) =
	     discrimination_pf e (t,t1,t2) discriminator lbeq gls in

	     tclCOMPLETE((tclTHENS (cut_intro absurd_term)
			    ([onLastHyp (comp gen_absurdity outSOME);
			      refine (mkAppL [| pf; VAR id |])]))) gls
       | _ -> assert false)
;;


let not_found_message  id =
  [<'sTR "the variable"; 'sPC ; 'sTR (string_of_id id) ; 'sPC;
    'sTR" was not found in the current environment" >]
;;

let insatisfied_prec_message cls =
  match cls with
      None -> [< 'sTR"goal does not satify the expected preconditions">] 
    |  Some id -> [< 'sTR(string_of_id id); 'sPC;
		     'sTR"does not satify the expected preconditions" >]
;;

let discrClause cls gls =
  match cls with
      None ->
    	if somatches (pf_concl gls) not_pattern then
          (tclTHEN (tclTHEN hnf_in_concl intro)
             (onLastHyp (comp discr outSOME))) gls
    	else if somatches (pf_concl gls) imp_False_pattern then
	  (tclTHEN intro (onLastHyp (comp discr outSOME))) gls
	else errorlabstrm "DiscrClause" (insatisfied_prec_message cls)
    | Some id ->
	try (discr id gls)
      	with Not_found -> errorlabstrm "DiscrClause" (not_found_message  id)
;;

let discrEverywhere = Tacticals.tryAllClauses discrClause;;
let discrConcl gls  = discrClause None gls;;
let discrHyp id gls = discrClause (Some id) gls;;

(**)
let h_discr      = hide_atomic_tactic "Discr"      discrEverywhere;;
let h_discrConcl = hide_atomic_tactic "DiscrConcl" discrConcl;;
let h_discrHyp   = hide_ident_tactic  "DiscrHyp"   discrHyp;;
(**)


(* [bind_ith na i T]
 *  will verify that T has no binders below [Rel i], and produce the
 *  term [na]T, binding [Rel i] in T.  The resulting term should be
 *  valid in the same environment as T, which means that we have to
 *  re-lift it.
 *)
let bind_ith na i t = lift i (DLAM(na,lift (-(i-1)) t));;

let existS_term = put_pat mmk "existS";;
let existS_pattern = put_pat mmk "(existS ? ? ? ?)";;
let sigS_term = put_pat mmk "sigS";;
let projS1_term = put_pat mmk "projS1";;
let projS2_term = put_pat mmk "projS2";;
let sigS_rec_term = put_pat mmk "sigS_rec";;


let existT_term = put_pat mmk "existT";;
let existT_pattern = put_pat mmk "(existT ? ? ? ?)";;
let sigT_term = put_pat mmk "sigT";;
let projT1_term = put_pat mmk "projT1";;
let projT2_term = put_pat mmk "projT2";;
let sigT_rect_term = put_pat mmk "sigT_rect";;

(* returns the sigma type (sigS, sigT) with the respective
    constructor depending on the sort
*)
let find_sigma_data s =
  match strip_outer_cast s with  
      DOP0(Sort(Prop Pos))     ->                                (* Set *) 
       	(projS1_term,projS2_term,sigS_rec_term,existS_term, sigS_term)     
    |  DOP0(Sort(Type(_))) ->                                (* Type *)
	 (projT1_term, projT2_term, sigT_rect_term, existT_term, sigT_term)  
    |   _     -> error "find_sigma_data"
;;


(* [make_tuple env na lind rterm rty]

   If [rty] depends on lind, then we will fabricate the term

          (existS A==[type_of(Rel lind)] P==(Lambda(type_of(Rel lind),
                                            [bind_ith na lind rty]))
                  [(Rel lind)] [rterm])

   [The term (Lambda(type_of(Rel lind),[bind_ith na lind rty])) is
    valid in [env] because [bind_ith] produces a term which does not
    "change" environments.]

   which should have type (sigS A P) - we can verify it by
   typechecking at the end.

 *)

let make_tuple sigma env na lind rterm rty =
  if dependent (Rel lind) rty then
    let (_,_,_,exist_term,sig_term) =
      find_sigma_data (type_of_rel sigma env rty) in
    let a = type_of_rel sigma env (Rel lind) in
    let p = DOP2(Lambda,a,
                 bind_ith (fst(lookup_rel lind env)) lind rty) in
      (applist(get_pat exist_term,[a;p;(Rel lind);rterm]),
       applist(get_pat sig_term,[a;p]))
  else
    (rterm,rty)
;;

(* check that the free-references of the type of [c] are contained in
   the free-references of the normal-form of that type.  If the normal
   form of the type contains fewer references, we want to return that
   instead.
 *)
let minimal_free_rels sigma (c,cty) =
  let cty_rels = free_rels cty in
  let nf_cty = nf_betadeltaiota sigma cty in
  let nf_rels = free_rels nf_cty in
    if Listset.subset cty_rels nf_rels then
      (cty,cty_rels)
    else
      (nf_cty,nf_rels)
;;


(* [sig_clausale_forme siglen ty]
    
   Will explode [siglen] [sigS,sigT ]'s on [ty] (depending on the 
   type of ty), and return:

   (1) a pattern, with meta-variables in it for various arguments,
       which, when the metavariables are replaced with appropriate
       terms, will have type [ty]

   (2) an integer, which is the last argument - the one which we just
       returned.

   (3) a pattern, for the type of that last meta

   (4) a typing for each metavariable

   WARNING: No checking is done to make sure that the 
            sigS(or sigT)'s are actually there.
          - Only homogenious pairs are built i.e. pairs where all the 
   dependencies are of the same sort
 *)


let sig_clausale_forme sort_of_ty siglen ty =
  let (_,_,_,exist_term,_)=find_sigma_data sort_of_ty in 
  let rec sigrec_clausale_forme siglen ty =
    if siglen = 0 then
      let mv = newMETA() in
      	(DOP0(Meta mv),(mv,ty),[(mv,ty)])
    else
      let (a,p) = match whd_stack (whd_beta ty) [] with
	  (_,[a;p]) -> (a,p)
 	| _ -> anomaly "sig_clausale_forme: should be a sigma type" in
      let mv = newMETA() in
      let rty = applist(p,[DOP0(Meta mv)]) in
      let (rpat,headinfo,mvenv) = sigrec_clausale_forme (siglen-1) rty in
        (applist(get_pat exist_term,[a;p;DOP0(Meta mv);rpat]),
         headinfo,
         (mv,a)::mvenv)
  in
    sigrec_clausale_forme siglen ty 
;;

(* [make_iterated_tuple sigma env DFLT c]

   Will find the free (DB) references of the S(TRONG)NF of [c]'s type,
   gather them together in left-to-right order (i.e. highest-numbered
   is farthest-left), and construct a big iterated pair out of it.
   This only works when the references are all themselves to members
   of [Set]s, because we use [sigS] to construct the tuple.

   Suppose now that our constructed tuple is of length [tuplen].

   Then, we need to construct the default value for the other
   branches.  The default value is constructed by taking the
   tuple-type, exploding the first [tuplen] [sigS]'s, and replacing at
   each step the binder in the right-hand-type by a fresh
   metavariable.

   In addition, on the way back out, we will construct the pattern for
   the tuple which uses these meta-vars.

   This gives us a pattern, which we use to match against the type of
   DFLT; if that fails, then against the S(TRONG)NF of that type.  If
   both fail, then we just cannot construct our tuple.  If one of
   those succeed, then we can construct our value easily - we just use
   the tuple-pattern.

 *)
(***** Ajout de la normalisation les types des termes a comparer pour 
       suporter de  definitions des  types

let make_iterated_tuple sigma env (dFLT,dFLTty) (c,cty) =
let (cty,rels) = minimal_free_rels sigma (c,cty) in
let sort_of_cty =type_of_rel sigma env cty in
let sorted_rels = sort (neg gt) rels in
let (tuple,tuplety) =
    List.fold_left (fun (rterm,rty) lind ->
                 let na = fst(lookup_rel lind env)
                 in make_tuple sigma env na lind rterm rty)
    (c,cty)
    sorted_rels in

if (not((closed tuplety))) then failwith "make_iterated_tuple";

let (tuplepat,(headmv,headpat),mvenv) = 
    sig_clausale_forme sort_of_cty (List.length sorted_rels) tuplety in

let headpat = nf_beta headpat in

let nf_ty = nf_betadeltaiota sigma dFLTty in

let dfltval =
    try_find (fun ty -> 
                  try let binding = if is_Type headpat & is_Type ty
                                     then []
                                     else  Somatch.somatch None headpat ty
                      in instance ((headmv,dFLT)::binding) tuplepat
                  with UserError _ -> failwith "caught")
    [dFLTty;nf_ty] in

    (tuple,tuplety,dfltval)
;;
************)

let make_iterated_tuple sigma env (dFLT,dFLTty) (c,cty) =
  let (cty,rels) = minimal_free_rels sigma (c,cty) in
  let sort_of_cty = type_of_rel sigma env cty in
  let sorted_rels = Sort.list (>=) (Listset.elements rels) in
  let (tuple,tuplety) =
    List.fold_left (fun (rterm,rty) lind ->
                      let na = fst(lookup_rel lind env)
                      in make_tuple sigma env na lind rterm rty)
      (c,cty)
      sorted_rels in

    if not(closed0 tuplety) then failwith "make_iterated_tuple";
    
    let (tuplepat,(headmv,headpat),mvenv) = 
      sig_clausale_forme sort_of_cty (List.length sorted_rels) tuplety in

    let headpat = nf_betadeltaiota sigma headpat in

    let nf_ty = nf_betadeltaiota sigma dFLTty in
      
    let dfltval =
      try_find (fun ty -> 
                  try let binding = 
		    if is_Type headpat & is_Type ty then
                      []
                    else
		      somatch None headpat ty in
                    instance ((headmv,dFLT)::binding) tuplepat
                  with UserError _ -> failwith "caught")
      	[dFLTty; nf_ty] in
      
      (tuple,tuplety,dfltval)
;;

let rec build_injrec sigma env (t1,t2) c = function
    [] ->
      make_iterated_tuple sigma env (t1,type_of_rel sigma env t1)
        (c,type_of_rel sigma env c)
  | (MutConstruct(sp,cnum),argnum)::l ->
      let cty = type_of_rel sigma env c in
      let (ity,_) = find_mrectype sigma cty in
      let nparams = mind_nparams ity in
      let (cnum_nlams,cnum_env,kont) = descend_then sigma env c cnum in
      let newc = Rel(cnum_nlams-(argnum-nparams)) in
      let (subval,tuplety,dfltval) =
      	build_injrec sigma cnum_env (t1,t2) newc l
      in
      	(kont subval (dfltval,tuplety),
	 tuplety,dfltval)
  | _ -> assert false
;;


let build_injector sigma env (t1,t2) c cpath =
  let (injcode,resty,_) = build_injrec sigma env (t1,t2) c cpath in
    (injcode,resty)
;;

let try_delta_expand sigma t =
  let whdt= whd_betadeltaiota sigma t  in 
  let rec hd_rec c  =
    match c with
       	DOPN(MutConstruct _,_) -> whdt
      | DOPN(AppL,cl)  -> hd_rec (hd_vect cl)
      | DOP2(Cast,c,_) -> hd_rec c
      | _  -> t
  in hd_rec whdt 
;;


(* Given t1=t2 Inj calculates the whd normal forms of t1 and t2 and it 
   expands then only when the whdnf has a constructor of an inductive type
   in hd position, otherwise delta expansion is not done
*)

let inj id gls =
  let eqn = (pf_whd_betadeltaiota gls (clause_type (Some id) gls)) in
  let (eq,(t,t1,t2))= 
    try (find_eq_data_decompose  eqn) 
    with UserError _ -> 
      errorlabstrm "Inj"  [<'sTR(string_of_id id); 
			    'sTR" Not a primitive  equality here " >] in
  let tj = pf_fexecute gls t in
  let sigma = project gls in
  let sign = pf_hyps gls in
    match find_positions sigma sign t1 t2 with
	(Inl _) ->
	  errorlabstrm "Inj" [<'sTR (string_of_id id);
			       'sTR" is not a projectable equality" >]
	    
      | (Inr posns) ->
	  let e = pf_get_new_id (id_of_string "e") gls in
	  let e_env =
	    gLOB(add_sign 
		   (e,Machops.assumption_of_judgement sigma (gLOB sign) tj)
		   sign)
	  in
	  let injectors =
	    map_succeed
	      (fun (cpath,t1_0,t2_0) ->
		 let (injbody,resty) =
		   build_injector sigma e_env (t1_0,t2_0) (VAR e) cpath in
		 let injfun = mkNamedLambda e t injbody in
		   try type_of sigma sign injfun ; (injfun,resty)
		   with UserError _ -> failwith "caught")
              posns in
	    
	    if injectors = [] then
	      errorlabstrm "Equality.inj" 
                  [<'sTR "Failed to decompose the equality">]
	        
	    else
	      tclMAP (fun (injfun,resty) ->
			let pf = applist(get_pat eq.congr,
					 [t;resty;injfun;
					  try_delta_expand sigma t1;
					  try_delta_expand sigma t2;
					  VAR id]) in
			let ty = pf_type_of gls pf in
			  ((tclTHENS  (cut  ty) ([tclIDTAC;refine pf]))))
		injectors
		gls
;;

let injClause cls gls =
  match cls with
      None ->
    	if somatches (pf_concl gls) not_pattern then
          (tclTHEN (tclTHEN hnf_in_concl intro)
             (onLastHyp (comp inj outSOME))) gls
    	else
	  errorlabstrm "InjClause" (insatisfied_prec_message  cls)
	    
    | Some id ->
	try 
	  inj id gls
        with
	    Not_found ->
	      errorlabstrm "InjClause" (not_found_message id)
          | UserError("refiner__FAIL",_) -> 
                errorlabstrm "InjClause" 
		  [< 'sTR (string_of_id id);
		     'sTR" Not a projectable equality" >]
;;

let injConcl gls  = injClause None gls;;
let injHyp id gls = injClause (Some id) gls;;

(**)
let h_injConcl = hide_atomic_tactic "Inj" injConcl;;
let h_injHyp   = hide_ident_tactic "InjHyp" injHyp;;
(**)


(****** fait trop d'unfold, pb. de arditi:
let DecompEqThen ntac id gls =
let eqn = (pf_whd_betadeltaiota gls (clause_type (SOME id) gls)) in
let (_,_,eq_congr_term,eq_ind_t,_,(T,t1,t2))= find_eq_data_decompose  eqn in
let Tj = pf_fexecute gls T in
let sigma = Project gls in
let sign = pf_untyped_hyps gls 
in (match find_positions sigma sign t1 t2 with
    (inl(cpath,MutConstruct(_,dirn),_)) ->
(let [e] = pf_get_new_ids [id_of_string "e"] gls in
let e_env = GLOB(add_sign (e,assumption_of_judgement sigma (gLOB sign) Tj) sign) in
let discriminator = build_discriminator sigma e_env dirn (VAR e) cpath in

let eq_ind = get_pat eq_ind_t and
    I = get_pat Pat_I in
let pf = applist(eq_ind,[T;t1;mkNamedLambda e T discriminator;I;t2])

in COMPLETE(Cut (get_pat Pat_False) THENS
            [OnLastHyp (Absurdity o outSOME);
             Refine (apply pf (VAR id))]) gls)

  | (inr posns) ->
(let [e] = pf_get_new_ids [id_of_string "e"] gls in
let e_env = GLOB(add_sign (e,assumption_of_judgement sigma (gLOB sign) Tj) sign) in
let injectors =
    map_succeed
    (fun (cpath,T1,T2) ->
         let (injbody,resty) = build_injector sigma e_env (T1,T2) (VAR e) cpath in
         let injfun = mkNamedLambda e T injbody
         in try type_of sigma sign injfun ; (injfun,resty)
            with UserError _ -> failwith "caught")
         posns in

if injectors = [] then FAIL gls else

(OnL (fun (injfun,resty) ->
         let pf = applist(get_pat eq_congr_term,
                          [T;resty;injfun;(whd_betadeltaiota sigma t1);(whd_betadeltaiota sigma t2);VAR id]) in
         let ty = pf_type_of gls pf in
             (tactics__cut ty THENS [IDTAC;Refine pf]))
    (rev injectors)
THEN (ntac (length injectors)))
    gls))
;;
*******)
(****
let decompEqThen ntac id gls =
let eqn = (pf_whd_betadeltaiota gls (clause_type (Some id) gls)) in
let (_,_,eq_congr_term,eq_ind_t,_,_,(t,t1,t2))= find_eq_data_decompose  eqn in
let tj = pf_fexecute gls t in
let sigma = project gls in
let sign = pf_untyped_hyps gls 
in (match find_positions sigma sign t1 t2 with
    (Inl(cpath,MutConstruct(_,dirn),_)) ->
(let [e] = pf_get_new_ids [id_of_string "e"] gls in
let e_env = gLOB(add_sign (e,assumption_of_judgement sigma (gLOB sign) tj) sign) in
let discriminator = build_discriminator sigma e_env dirn (VAR e) cpath in

let eq_ind = get_pat eq_ind_t and
    i = get_pat pat_I in
let pf = applist(eq_ind,[t;t1;mkNamedLambda e t discriminator;i;t2])

in tclCOMPLETE((tclTHENS (cut (get_pat pat_False))
            ([onLastHyp ((comp (absurdity) (outSOME)));
             refine (apply pf (VAR id))]))) gls)

  | (Inr posns) ->
(let [e] = pf_get_new_ids [id_of_string "e"] gls in
let e_env = gLOB(add_sign (e,assumption_of_judgement sigma (gLOB sign) tj) sign) in
let injectors =
    map_succeed
    (fun (cpath,t1_0,t2_0) ->
         let (injbody,resty) = build_injector sigma e_env (t1_0,t2_0) (VAR e) cpath in
         let injfun = mkNamedLambda e t injbody
         in try type_of sigma sign injfun ; (injfun,resty)
            with UserError _ -> failwith "caught")
         posns in

if injectors = [] then tclFAIL gls else

((tclTHEN
(tclMAP (fun (injfun,resty) ->
         let pf = applist(get_pat eq.congr,
                          [t;resty;injfun;t1;t2;VAR id]) in
         let ty = pf_type_of gls pf in
             ((tclTHENS (cut ty) ([tclIDTAC;refine pf]))))
    (List.rev injectors)) ((ntac (List.length injectors)))))
    gls))
;;
********)


let decompEqThen ntac id gls =
  let eqn = (pf_whd_betadeltaiota gls (clause_type (Some id) gls)) in
  let (lbeq,(t,t1,t2))= find_eq_data_decompose  eqn in
  let sort = pf_type_of gls (pf_concl gls) in 
  let tj = pf_fexecute gls t in
  let sigma = project gls in
  let sign = pf_hyps gls 
  in (match find_positions sigma sign t1 t2 with
	  (Inl(cpath,MutConstruct(_,dirn),_)) ->
	    
	    let e = pf_get_new_id (id_of_string "e") gls in
	    let e_env =
	      gLOB(add_sign 
		     (e,Machops.assumption_of_judgement sigma (gLOB sign) tj)
		     sign) in
	    let discriminator =
	      build_discriminator sigma e_env dirn (VAR e) sort cpath in
	    let (pf, absurd_term) =
	      discrimination_pf e (t,t1,t2) discriminator lbeq gls in
	      
	      tclCOMPLETE((tclTHENS (cut_intro absurd_term)
			     ([onLastHyp (comp gen_absurdity outSOME);
			       refine (mkAppL [| pf; VAR id |])]))) gls
		
  	| (Inr posns) ->
	    (let e = pf_get_new_id (id_of_string "e") gls in
	     let e_env =
	       gLOB(add_sign
		      (e,Machops.assumption_of_judgement sigma (gLOB sign) tj)
		      sign) in
	     let injectors =
	       map_succeed
		 (fun (cpath,t1_0,t2_0) ->
		    let (injbody,resty) =
		      build_injector sigma e_env (t1_0,t2_0) (VAR e) cpath in
		    let injfun = mkNamedLambda e t injbody in
		      try type_of sigma sign injfun ; (injfun,resty)
		      with UserError _ -> failwith "caught")
		 posns in
	       
	       if injectors = [] then
		 errorlabstrm "Equality.decompEqThen" 
                  [<'sTR "Discriminate failed to decompose the equality">]
	       else
		 ((tclTHEN
		     (tclMAP (fun (injfun,resty) ->
				let pf = applist(get_pat lbeq.congr,
						 [t;resty;injfun;t1;t2;
						  VAR id]) in
				let ty = pf_type_of gls pf in
				  ((tclTHENS (cut ty) 
				      ([tclIDTAC;refine pf]))))
			(List.rev injectors))
		     (ntac (List.length injectors))))
		 gls)
  	| _ -> assert false)
;;


let decompEq = decompEqThen (fun x -> tclIDTAC);;


let dEqThen ntac cls gls =
  match cls with
      None ->
    	if somatches (pf_concl gls) not_pattern then
	  (tclTHEN hnf_in_concl
	     (tclTHEN intro
         	(onLastHyp (comp (decompEqThen ntac) outSOME)))) gls
    	else
	  errorlabstrm "DEqThen" (insatisfied_prec_message  cls)
    | Some id ->
	try (decompEqThen ntac id gls)
      	with Not_found -> errorlabstrm "DEqThen" (not_found_message id)
          |  UserError _ ->
	       errorlabstrm "DEqThen" (insatisfied_prec_message cls)
;;

let dEq = dEqThen (fun x -> tclIDTAC);;

let dEqConcl gls = dEq None gls;;
let dEqHyp id gls = dEq (Some id) gls;;

(**)
let dEqConcl_tac = hide_atomic_tactic "DEqConcl" dEqConcl;;
let dEqHyp_tac = hide_ident_tactic "DEqHyp" dEqHyp;;
(**)


let rewrite_msg = function 
   None ->  
     [<'sTR "passed term is not a primitive equality">] 
| (Some id) ->[<'sTR (string_of_id id); 'sTR "does not satisfy preconditions ">]
;;


let swap_equands gls eqn =
  let (lbeq,(t,e1,e2)) =
    (try find_eq_data_decompose eqn
     with _ -> errorlabstrm "swap_equamds" (rewrite_msg None)) 
  in applist(get_pat lbeq.eq,[t;e2;e1])
;;

let swapEquandsInConcl gls =
  let (lbeq,(t,e1,e2)) =
    (try find_eq_data_decompose (pf_concl gls)
     with _-> errorlabstrm "SwapEquandsInConcl" (rewrite_msg None)) in
  let sym_equal = get_pat lbeq.sym in
    refine (applist(sym_equal,[t;e2;e1;DOP0(Meta(newMETA()))])) gls
;;

let swapEquandsInHyp id gls =
  ((tclTHENS (cut_replacing id (swap_equands gls (clause_type (Some id) gls)))
      ([tclIDTAC;
      	(tclTHEN (swapEquandsInConcl) (exact (VAR id)))]))) gls
;;

(*let find_elim p sigma sign eq_ind eq_rec eq_rect =
 match hd_of_prod (type_of_rel sigma (gLOB sign) p) with
    DOP0(Sort(Prop Null))  ->  get_pat eq_ind  (* Prop *)
 |  DOP0(Sort(Prop Pos))     -> 
      (match eq_rec with
         (Some eq_rec) -> get_pat eq_rec (* Set *) 
      | None -> errorlabstrm "find_elim"
           [< 'sTR "this type of elimination is not allowed">])
 |   _ (* Type *) -> 
         (match eq_rect with
            (Some eq_rect) -> get_pat eq_rect (* Type *)
          | None -> errorlabstrm "find_elim"
               [< 'sTR "this type of elimination is not allowed">])
;;
*)

(* find_elim determines which elimination principle is necessary to
   eliminate lbeq on sort_of_gl. It yields the boolean true wether
   it is a dependent elimination principle (as idT.rect) and false
   otherwise
*)
let find_elim  sort_of_gl  lbeq =
  match  sort_of_gl  with
      DOP0(Sort(Prop Null))  (* Prop *)  ->  (get_pat lbeq.ind, false)  
    |  DOP0(Sort(Prop Pos))   (* Set *)   ->  
	 (match lbeq.rrec with
              (Some eq_rec) -> (get_pat eq_rec, false) 
	    | None -> errorlabstrm "find_elim"
		  [< 'sTR "this type of elimination is not allowed">])
    |   _ (* Type *) -> 
          (match lbeq.rect with
               (Some eq_rect) -> (get_pat eq_rect, true) 
             | None -> errorlabstrm "find_elim"
		   [< 'sTR "this type of elimination is not allowed">])
;;


(* builds a predicate [e:t][H:(lbeq t e t1)](body e)
   to be used as an argument for equality dependent elimination principle:
   Preconditon: dependent body (Rel 1)
*)
let build_dependent_rewrite_predicate (t,t1,t2) body lbeq gls =
  let e = pf_get_new_id  (id_of_string "e") gls in 
  let h = pf_get_new_id  (id_of_string "HH") gls in 
  let eq_term = get_pat lbeq.eq in
    (mkNamedLambda e t 
       (mkNamedLambda h (applist (eq_term, [t;t1;(Rel 1)])) 
          (lift 1 body))) 
;;


(* builds a predicate [e:t](body e) ???
   to be used as an argument for equality non-dependent elimination principle:
   Preconditon: dependent body (Rel 1)
*)
let build_non_dependent_rewrite_predicate (t,t1,t2) body  gls =
  Environ.lambda_create (t,body)
;;


let bareRevSubstInConcl lbeq body (t,e1,e2) gls =
  let (eq_elim,dep) =
    try find_elim (pf_type_of gls (pf_concl gls)) lbeq  
    with UserError _ -> errorlabstrm "RevSubstIncConcl"
        [< 'sTR "this type of substitution is not allowed">]  in 
  let p =
    if dep then
      (build_dependent_rewrite_predicate (t,e1,e2)  body lbeq gls)
    else
      (build_non_dependent_rewrite_predicate (t,e1,e2)  body  gls)
  in
    refine (applist(eq_elim,[t;e1;p;DOP0(Meta(newMETA()));
                             e2;DOP0(Meta(newMETA()))])) gls
;;

(* [subst_tuple_term dep_pair B]

   Given that dep_pair looks like:

   (existS e1 (existS e2 ... (existS en en+1) ... ))

   and B might contain instances of the ei, we will return the term:

   ([x1:ty(e1)]...[xn:ty(en)]B
    (projS1 (Rel 1))
    (projS1 (projS2 (Rel 1)))
    ... etc ...)

   That is, we will abstract out the terms e1...en+1 as usual, but
   will then produce a term in which the abstraction is on a single
   term - the debruijn index [Rel 1], which will be of the same type
   as dep_pair.

   ALGORITHM for abstraction:

   We have a list of terms, [e1]...[en+1], which we want to abstract
   out of [B].  For each term [ei], going backwards from [n+1], we
   just do a [subst_term], and then do a lambda-abstraction to the
   type of the [ei].

 *)


let comp_carS_pattern = put_pat mmk "<<x>>(projS1 ? ? (?)@[x])";;
let comp_cdrS_pattern = put_pat mmk "<<x>>(projS2 ? ? (?)@[x])";;

let comp_carT_pattern = put_pat mmk "<<x>>(projT1 ? ? (?)@[x])";;
let comp_cdrT_pattern = put_pat mmk "<<x>>(projT2 ? ? (?)@[x])";;

let dest_somatch_sigma ex ex_pat =
  match dest_somatch ex ex_pat with
      [a;p;car;cdr] -> (a,p,car,cdr)
    | _ -> anomaly "dest_somatch_sigma: a sigma pattern should match 4 terms";;

let find_sigma_data_decompose ex =
  try (comp_carS_pattern, comp_cdrS_pattern,
       dest_somatch_sigma ex existS_pattern)
  with _ -> (try (comp_carT_pattern,comp_cdrT_pattern,
                  dest_somatch_sigma ex existT_pattern)   
             with _ -> errorlabstrm "find_sigma_data_decompose" [< >]);;


let decomp_tuple_term = 
  let rec decomprec to_here_fun ex =
    try
      let (comp_car_pattern,comp_cdr_pattern,(a,p,car,cdr)) =
	find_sigma_data_decompose  ex in
      let car_code = soinstance comp_car_pattern [a;p;to_here_fun]
      and cdr_code = soinstance comp_cdr_pattern [a;p;to_here_fun] in
      	(car,Environ.named_hd a Anonymous,car_code)::(decomprec cdr_code cdr)
    with UserError _ ->
      [(ex,Environ.named_hd ex Anonymous,to_here_fun)]
  in
    decomprec (DLAM(Anonymous,Rel 1))
;;

let subst_tuple_term sigma sign dep_pair b =
  let env=(gLOB sign) in 
  let sort_of_dep_pair =
    type_of_rel sigma env (type_of_rel sigma env dep_pair) in
  let (proj1_term,proj2_term,sig_elim_term,_,_) =
    find_sigma_data sort_of_dep_pair in 
  let e_list = decomp_tuple_term dep_pair in
  let abst_B =
    List.fold_right (fun (e,na,_) b ->
                       let body = subst_term e b in
                       let pB = DLAM(na,body) in
			 DOP2(Lambda,type_of sigma sign e,pB))
      e_list b in
    
  let app_B = 
    applist(abst_B,(List.map (fun (_,_,c) -> (sAPP c (Rel 1))) e_list)) in

  let (proj1_sp,_) = destConst (get_pat proj1_term)
  and (proj2_sp,_) = destConst (get_pat proj2_term)
  and (sig_elim_sp,_) = destConst (get_pat sig_elim_term) in
    
    strong (comp whd_betaiota 
	      (whd_const [proj1_sp;proj2_sp;sig_elim_sp] sigma)) app_B
;;

(* |- (P e2)
     BY RevSubstInConcl (eq T e1 e2)
     |- (P e1)
     |- (eq T e1 e2)
 *)
let revSubstInConcl eqn gls =
  let (lbeq,(t,e1,e2)) = find_eq_data_decompose eqn in
  let body = subst_tuple_term (project gls) (pf_hyps gls) e2 (pf_concl gls) in
    if (not (dependent (Rel 1) body)) then
      errorlabstrm  "RevSubstInConcl" [<>];
    bareRevSubstInConcl lbeq body (t,e1,e2) gls
;;



(* |- (P e1)
     BY SubstInConcl (eq T e1 e2)
     |- (P e2)
     |- (eq T e1 e2)
 *)
let substInConcl eqn gls =
  (tclTHENS (revSubstInConcl (swap_equands gls eqn))
     ([tclIDTAC;
       swapEquandsInConcl])) gls
;;


let substInHyp eqn id gls =
  let (lbeq,(t,e1,e2)) = (find_eq_data_decompose eqn) in 
  let body = subst_term e1 (clause_type (Some id) gls) in


    mSG [< 'sTR "body:"; 'fNL>];
    Term.constr_display body;
    flush stdout;

    if (not (dependent (Rel 1) body)) then
      errorlabstrm  "SubstInHyp" [<>];
    let pB = DLAM(Environ.named_hd t Anonymous,body) in
      (tclTHENS (cut_replacing id (sAPP pB e2))
	 ([tclIDTAC;
	   (tclTHENS (bareRevSubstInConcl lbeq body (t,e1,e2))
              ([exact (VAR id);tclIDTAC]))])) gls
;;

let revSubstInHyp eqn id gls =
  (tclTHENS (substInHyp (swap_equands gls eqn) id)
     ([tclIDTAC;
       swapEquandsInConcl])) gls
;;

(**** to detect no progress ingoal 
let try_rewrite tac gls =
 try tac gls
 with UserError ("find_eq_data_decompose",_) -> 
         errorlabstrm "find_eq_data_decompose " [< 'sTR "Not a primitive equality here">]
 | UserError("find_eq_elim",s) -> 
        errorlabstrm "find_eq_elim" [<'sTR "This type of elimination is not allowed ">]  
 | _ -> tclIDTAC gls
;;
*******)

let try_rewrite tac gls =
  try 
    tac gls
  with UserError ("find_eq_data_decompose",_) -> errorlabstrm 
      "try_rewrite" [< 'sTR "Not a primitive equality here">]
    | UserError ("swap_equamds",_) -> errorlabstrm 
          "try_rewrite" [< 'sTR "Not a primitive equality here">]
    | UserError("find_eq_elim",s) -> errorlabstrm "try_rew" 
          [<'sTR "This type of elimination is not allowed ">]  
    | UserError(_) -> errorlabstrm "try_rewrite"
          [< 'sTR "Cannot find a well type generalisation of the goal that";
             'sTR " makes progress the proof.">]
;;

(*let general_rewrite_in lft2rgt id (c,lb) gls = 
   (try lookup_sign id (pf_untyped_hyps gls) ; () 
   with Not_found -> 
     errorlabstrm "general_rewrite_in" [< 'sTR"No such hypothesis : " ;
				  print_id id >]) ;
  let (wc,_) = startWalk gls in
  let (_,_,t)    = reduce_to_ind (project gls) (pf_type_of gls c)  in
  let ctype = type_clenv_binding wc (c,t) lb  in
  let (_,ty) = decompose_prod ctype in
  let (e1,e2) = match match_with_equation ty with
      None -> error "The term provided does not end with an equation" 
    | Some (_,[_;e1;e2]) -> (e1,e2)
    | Some (_,[e1;e2]) -> (e1,e2)
    |_ -> assert false 
  in let (e1,e2) = if lft2rgt then (e1,e2) else (e2,e1) in
  let body = subst_term e1 (clause_type (Some id) gls) in
    if dependent (Rel 1) body then
      let pB = DLAM(Environ.named_hd t Anonymous,body) in
	  let pred = DOP2(Lambda, pf_type_of gls e2, pB) in
	    tclTHENS (cut_replacing id (sAPP pB e2))
	      [tclIDTAC;
	       (tclTHEN_i
		  (tclTHEN (change_in_concl (applist (pred,[e2])))
                     (general_rewrite_bindings (not lft2rgt) (c,lb)))
		  (fun i -> if i = 1 then exact (VAR id)
		   else tclIDTAC) 1)] gls
    else
      errorlabstrm  "general_rewrite_in" 
	[<'sTR"Nothing to rewrite in : "; print_id id>]
;;*)

(*list_int n 0 [] gives the list [1;2;...;n]*)
let rec list_int n cmr l=
  if cmr=n then
    l@[n]
  else
    list_int n (cmr+1) (l@[cmr]);;

(*The Rewrite in tactic*)
let general_rewrite_in lft2rgt id (c,lb) gls =
  let (_,typ_id)=
    (try
       (lookup_sign id (pf_untyped_hyps gls))
     with
        Not_found -> 
          errorlabstrm "general_rewrite_in" [< 'sTR"No such hypothesis : " ;
		  		  print_id id >])
  in
    let (wc,_)=startWalk gls
    and (_,_,t)=reduce_to_ind (project gls) (pf_type_of gls c)
    in
      let ctype=type_clenv_binding wc (c,t) lb
      in
        match (match_with_equation ctype) with
           None -> error "The term provided does not end with an equation" 
          |Some (hdcncl,l) ->
            let mbr_eq=
              if lft2rgt then
                List.hd (List.tl (List.rev l))
              else
                List.hd (List.rev l)
            in
              (match (sub_term_with_unif 
			(appl_elim (put_off_casts typ_id))
			(appl_elim mbr_eq)) 
	       with
                  None ->
                    errorlabstrm "general_rewrite_in" 
		      [<'sTR "Nothing to rewrite in: "; print_id id>]
                 |Some (l2,nb_occ) ->
                   (tclTHENSI 
		      (tclTHEN 
			 (tclTHEN (generalize [(pf_global gls id)]) 
			    (reduce (Pattern [(list_int nb_occ 1 [],l2,
					       pf_type_of gls l2)]) []))
			 (general_rewrite_bindings lft2rgt (c,lb))) 
		      [(tclTHEN (clear_one id) (introduction id))]) gls);;

let dyn_rewrite_in lft2rgt = function
    [IDENTIFIER id;(COMMAND com);(BINDINGS binds)] 
    -> tactic_com_bind_list (general_rewrite_in lft2rgt id) (com,binds)
  | [IDENTIFIER id;(CONSTR c);(CBINDINGS binds)] 
  -> general_rewrite_in lft2rgt id (c,binds)
  | _ -> assert false
;;

let rewriteLR_in_tac =  hide_tactic "RewriteLRin" (dyn_rewrite_in true);;
let rewriteRL_in_tac = hide_tactic "RewriteRLin" (dyn_rewrite_in false);;

let conditional_rewrite_in lft2rgt id tac (c,bl) = 
  tclTHEN_i (general_rewrite_in lft2rgt id (c,bl))
    (fun i -> if i=1 then tclIDTAC else tclCOMPLETE tac) 1;;

let dyn_conditional_rewrite_in lft2rgt = function
    [(TACEXP tac); IDENTIFIER id; (COMMAND com);(BINDINGS binds)] 
    -> tactic_com_bind_list 
	(conditional_rewrite_in lft2rgt id (Tacinterp.interp tac)) 
	(com,binds)
  | [(TACEXP tac); IDENTIFIER id; (CONSTR c);(CBINDINGS binds)] 
    -> conditional_rewrite_in lft2rgt id (Tacinterp.interp tac) (c,binds)
  | _ -> assert false
;;

let v_conditional_rewriteLR_in = 
  hide_tactic "CondRewriteLRin" (dyn_conditional_rewrite_in true);; 

let v_conditional_rewriteRL_in = 
  hide_tactic "CondRewriteRLin" (dyn_conditional_rewrite_in false);;


(* Rewrite c in id. Rewrite -> c in id. Rewrite <- c in id. 
   Does not work when c is a conditional equation 
 *)
let rewrite_in lR com id gls =
  (try lookup_sign id (pf_untyped_hyps gls) ; () 
   with Not_found -> 
    errorlabstrm "rewrite_in" [< 'sTR"No such hypothesis : " ;print_id id >]) ;
  let c = pf_constr_of_com gls com in
  let eqn = pf_type_of gls c in
    try
      (find_eq_data_decompose eqn ;
       try ((tclTHENS 
               ((if lR then substInHyp else revSubstInHyp) eqn id) 
               ([tclIDTAC ; exact c ]))) gls
       with UserError("SubstInHyp",_) -> tclIDTAC gls)
    with UserError ("find_eq_data_decompose",_)->  
      errorlabstrm "rewrite_in" [< 'sTR"No equality here" >] 
;;

(**)

(**)
let subst eqn cls gls =
  match cls with
      None ->    substInConcl eqn gls
    | Some id -> substInHyp eqn id gls
;;

(* |- (P a)
 * Subst_Concl a=b 
 *  |- (P b)
 *  |- a=b
 *)
(**)
let substConcl_LR eqn gls = try_rewrite (subst eqn None) gls;;
let substConcl_LR_tac = 
  let gentac = hide_tactic "SubstConcl_LR"
		 (function [COMMAND eqn] -> 
		    (fun gls ->  substConcl_LR (pf_constr_of_com gls eqn)  gls)
		    | _ -> assert false)
  in fun eqn  -> gentac [COMMAND eqn] 
;;
(**)

(* id:(P a) |- G
 * SubstHyp a=b id
 *  id:(P b) |- G
 *  id:(P a) |-a=b
*)
(*******
let SubstHyp_LR  eqn id gls  =  try_rewrite  (Subst eqn (SOME id)) gls ;;
let SubstHyp_LR_tac   = 
let gentac = hide_tactic "SubstHyp_LR"
 (function [COMMAND eqn; IDENTIFIER id] -> 
       (fun gls ->  SubstHyp_LR (pf_constr_of_com gls eqn) id gls))
 (fun sigma  goal (_,[COMMAND eqn; IDENTIFIER id]) ->
    [<'S"CutRewrite ->" ; 'SPC ; pr_com sigma goal eqn; 'SPC; 'S"in"; print_id id >])
in fun eqn id -> gentac [COMMAND eqn; IDENTIFIER id] 
;;
******)

let hypSubst id cls gls =
  match cls with
      None -> (tclTHENS (substInConcl (clause_type (Some id) gls))
		 ([tclIDTAC; exact (VAR id)])) gls
    | Some hypid -> (tclTHENS (substInHyp (clause_type (Some id) gls) hypid)
		       ([tclIDTAC;exact (VAR id)])) gls
;;

(* id:a=b |- (P a)
 * HypSubst id.
 *  id:a=b |- (P b)
 *)
let substHypInConcl_LR id gls = try_rewrite (hypSubst id None) gls;;
let substHypInConcl_LR_tac =
  let gentac= hide_tactic "SubstHypInConcl_LR" 
		(function [IDENTIFIER id] -> substHypInConcl_LR id
		   | _ -> assert false)
  in fun id -> gentac [IDENTIFIER id]
;;


(* id:a=b H:(P a) |- G
   SubstHypInHyp id H.
    id:a=b H:(P b) |- G
*)
(*******
let SubstHypInHyp_LR id H gls = try_rewrite (HypSubst id (SOME H)) gls;;
let SubstHypInHyp_LR_tac = 
let gentac = hide_tactic "SubstHypInHyp_LR"
    (fun [IDENTIFIER id; IDENTIFIER H] -> SubstHypInHyp_LR id H)
    (fun sigma goal  (_,[IDENTIFIER id; IDENTIFIER H]) ->
        [< 'S"GRewrite"; 'S "->"; 'SPC ; print_id id ; 'SPC ; 'S"in"; print_id H >])
in fun id H -> gentac [IDENTIFIER id; IDENTIFIER H]
;;
*******)


let revSubst eqn cls gls =
  match cls with
      None -> revSubstInConcl eqn gls
    | Some id -> revSubstInHyp eqn id gls
;;

(* |- (P b)
   SubstConcl_RL a=b
     |- (P a)
     |- a=b
*)
let substConcl_RL eqn gls = try_rewrite (revSubst eqn None) gls;;
let substConcl_RL_tac = 
  let gentac = hide_tactic "SubstConcl_RL"
		 (function [COMMAND eqn] -> 
		    (fun gls ->  substConcl_RL (pf_constr_of_com gls eqn)  gls)
		    | _ -> assert false)
  in fun eqn  -> gentac [COMMAND eqn] 
;;

 

(* id:(P b) |-G
   SubstHyp_RL a=b id 
      id:(P a) |- G
      |- a=b  
*)
let substHyp_RL  eqn id gls = try_rewrite (revSubst eqn (Some id)) gls;;
(******
let SubstHyp_RL_tac   = 
let gentac = hide_tactic "SubstHyp_RL"
 (fun [COMMAND eqn; IDENTIFIER id] -> 
       (fun gls ->  SubstHyp_RL (pf_constr_of_com gls eqn) id gls))
 (fun sigma  goal (_,[COMMAND eqn; IDENTIFIER id]) ->
    [<'S"CutRewrite <-" ; 'SPC ; pr_com sigma goal eqn; 'SPC; 'S"in"; print_id id >])
in fun eqn id -> gentac [COMMAND eqn; IDENTIFIER id] 
;;
*********)


let revHypSubst id cls gls =
  match cls with
      None -> (tclTHENS (revSubstInConcl (clause_type (Some id) gls))
		 ([tclIDTAC; exact (VAR id)])) gls
    | Some hypid -> (tclTHENS (revSubstInHyp (clause_type (Some id) gls) hypid)
		       ([tclIDTAC;exact (VAR id)])) gls
;;

(* id:a=b |- (P b)
 * HypSubst id.
 * id:a=b |- (P a)
 *)
let substHypInConcl_RL id gls = try_rewrite (revHypSubst id None) gls;;
let substHypInConcl_RL_tac =
  let gentac= hide_tactic "SubstHypInConcl_RL" 
		(function [IDENTIFIER id] -> substHypInConcl_RL id
                   | _ -> assert false)
  in fun id -> gentac [IDENTIFIER id]
;;


(* id:a=b H:(P b) |- G
   SubstHypInHyp id H.
    id:a=b H:(P a) |- G
*)
(*****
let SubstHypInHyp_RL id H gls = try_rewrite (RevHypSubst id (SOME H)) gls;;
let SubstHypInHyp_RL_tac = 
let gentac = hide_tactic "SubstHypInHyp_RL"
    (fun [IDENTIFIER id; IDENTIFIER H] -> SubstHypInHyp_RL id H)
    (fun sigma goal  (_,[IDENTIFIER id; IDENTIFIER H]) ->
        [< 'S"GRewrite"; 'S "<-"; 'SPC ; print_id id ; 'SPC ; 'S"in"; print_id H >])
in fun id H -> gentac [IDENTIFIER id; IDENTIFIER H]
;;
******)

(***************)
(* AutoRewrite *)
(***************)

(****Dealing with the rewriting rules****)

(*A rewriting is typically an equational constr with an orientation (true=LR
  and false=RL)*)
type rewriting_rule=constr*bool;;

(* The table of rewriting rules. The key is the name of the rule base.  
   the value is a list of [rewriting_rule] *)
let rew_tab=Hashtabl.create 53;;

(*Functions necessary to the summary*)
let init ()=Hashtabl.clear rew_tab;;
let freeze ()=Hashtabl.freeze rew_tab;;
let unfreeze ft=Hashtabl.unfreeze ft rew_tab;;

(*Declaration of the summary*)
Summary.declare_summary "autorewrite"
  {Summary.freeze_function=freeze;
   Summary.unfreeze_function=unfreeze;
   Summary.init_function=init};;

(*Adds a list of rules to the rule table*)
let add_list_rules rbase lrl=List.iter (Hashtabl.add rew_tab rbase) lrl;;

(*Gives the list of rules for the base named rbase*)
let rules_of_base rbase=List.rev (Hashtabl.find_all rew_tab rbase);;

(*Functions necessary to the library object declaration*)
let load_autorewrite_rule=fun _ -> ();;
let cache_autorewrite_rule (_,(rbase,lrl))=add_list_rules rbase lrl;;
let specification_autorewrite_rule=fun x -> x;;

(*Declaration of the AUTOREWRITE_RULE library object*)
let (in_autorewrite_rule,out_autorewrite_rule)=
  Libobject.declare_object("AUTOREWRITE_RULE",
    {Libobject.load_function=load_autorewrite_rule;
     Libobject.cache_function=cache_autorewrite_rule;
     Libobject.specification_function=specification_autorewrite_rule});;

(* Semantic of the HintRewrite vernacular command *)
vinterp_add("HintRewrite",
  let rec lrules_arg lrl=function
      [] -> lrl
    |(VARG_VARGLIST [VARG_COMMAND rule; VARG_STRING ort])::a 
	when ort="LR" or ort="RL" ->
          lrules_arg (lrl@[(Trad.constr_of_com empty_evd
			      (initial_sign()) rule,ort="LR")]) a
    |_ -> bad_vernac_args "HintRewrite"
  and lbases_arg lbs=function
      [] -> lbs
    | (VARG_VARGLIST ((VARG_IDENTIFIER rbase)::b))::a ->
      	lbases_arg (lbs@[(rbase,lrules_arg [] b)]) a
    |_ -> bad_vernac_args "HintRewrite"
  in
    fun largs ->
      fun () ->
      	List.iter (fun c -> Library.add_anonymous_object 
		       (in_autorewrite_rule c)) (lbases_arg [] largs))
;;

(****The tactic****)

(*To build the validation function. Length=number of unproven goals, Valid=a
  validation which solves*)
type valid_elem=
   Length of int
  |Valid of validation;;

(* Ce truc devrait aller dans Std -- papageno *)
(*Gives the sub_list characterized by the indexes i_s and i_e with respect to
  lref*)
let sub_list lref i_s i_e=
  let rec sub_list_rec l i=
    if i=i_e then
      l@[List.nth lref i]
    else if (i>=i_s) & (i<i_e) then
      sub_list_rec (l@[List.nth lref i]) (i+1)
    else
      anomalylabstrm "Equality.sub_list" [<'sTR "Out of range">]
  in
    sub_list_rec [] i_s;;

(*Cuts the list l2becut in lists which lengths are given by llth*)
let cut_list l2becut lval=
  let rec cut4_1goal cmr l1g=function
     [] -> (cmr,l1g)
    |a::b ->
      (match a with
          Length lth ->
            if lth=0 then
              cut4_1goal cmr l1g b
            else
              cut4_1goal (cmr+lth) (l1g@(sub_list l2becut cmr (cmr+lth-1))) b
         |Valid p ->
            cut4_1goal cmr (l1g@[p []]) b)	
  and cut_list_rec cmr l2b=function
     [] -> l2b
    |a::b ->
      let (cmr,l1g)=cut4_1goal cmr [] a
      in
        cut_list_rec cmr (l2b@[l1g]) b
  in
    cut_list_rec 0 [] lval;;

(*Builds the validation function with lvalid and with respect to l*)
let validation_gen lvalid l=
  let (lval,larg_velem)=List.split lvalid
  in
    let larg=cut_list l larg_velem
    in
      List.fold_left2 (fun a p l -> p ([a]@l)) (List.hd lval (List.hd larg))
        (List.tl lval) (List.tl larg);;

(*Adds the main argument for the last validation function*)
let mod_hdlist l=
  match (List.hd l) with
     (p,[Length 0]) -> l
    |(p,larg) -> (p,[Length 1]@larg)::(List.tl l);;

(*For the Step options*)
type option_step=
   Solve
  |Use
  |All;;

(* the user can give a base either by a name of by its full definition
  The definition is an Ast that will find its meaning only in the context
  of a given goal *)
type hint_base = 
    By_name of identifier
  | Explicit of (CoqAst.t*bool) list
;;

let explicit_hint_base gl = function 
  | By_name id -> 
      begin match rules_of_base id with
	| [] -> errorlabstrm "autorewrite" [<'sTR ("Base "^(string_of_id id)^
            " does not exist")>]
	| lbs -> lbs
      end 
  | Explicit lbs -> 
      List.map (fun (ast,b) ->
		  (pf_constr_of_com gl ast, b))
	lbs 
;;

(*AutoRewrite cannot be expressed with a combination of tacticals (due to the
  options). So, we make it in a primitive way*)
let autorewrite lbases ltacstp opt_step ltacrest opt_rest depth_step gls=
  let lst=List.flatten (List.map (explicit_hint_base gls) lbases)
  and unproven_goals=ref []
  and fails=ref 0
  and (sigr,g)=unpackage gls in

  let put_rewrite lrw=List.map (fun (x,y) -> general_rewrite y x) lrw
  and nbr_rules=List.length lst in

  let lst_rew=put_rewrite lst in
  
  let rec try2solve_main_goal mgl=function
      [] -> None
    |a::b ->
        try
          (let (gl_solve,p_solve)=apply_sig_tac sigr a mgl
           in
             if gl_solve=[] then
               Some (gl_solve,p_solve)
             else
               try2solve_main_goal mgl b)
        with
            UserError _ -> try2solve_main_goal mgl b

  and try_tacs4main_goal mgl=function
      [] -> None
    |a::b ->
        try
          (Some (apply_sig_tac sigr a mgl))
        with
            UserError _ -> try_tacs4main_goal mgl b

  and try2solve1gen_goal gl=function
      [] -> ([gl],Length 1)
    |a::b ->
        try
          (let (gl_solve,p_solve)=apply_sig_tac sigr a gl
           in
             if gl_solve=[] then
               ([],Valid p_solve)
             else
               try2solve1gen_goal gl b)
        with
            UserError _ -> try2solve1gen_goal gl b

  and try2solve_gen_goals (lgls,valg) ltac=function
      [] -> (lgls,valg)
    |a::b ->
        let (g,elem)=try2solve1gen_goal a ltac in
          try2solve_gen_goals (lgls@g,valg@[elem]) ltac b
  and iterative_rew cmr fails (cglob,cmod,warn) unp_goals lvalid=
    let cmd=ref cmod
    and wrn=ref warn in

    (if !cmd=depth_step then
       (wARNING [<'sTR ((string_of_int
			   cglob)^" rewriting(s) carried out")>];
        cmd:=0;
        wrn:=true));
    (if fails=nbr_rules then
       (unp_goals,lvalid,!wrn)
     else if cmr=nbr_rules then
       iterative_rew 0 0 (cglob,!cmd,!wrn) unp_goals lvalid
     else try


     let (gl,p)=apply_sig_tac sigr (List.nth lst_rew cmr)
		  (List.hd unp_goals) 
     in
     let (lgl_gen,lval_gen)=
       begin match ltacrest with
           None ->
             if (List.length gl)=1 then
               ([],[])
             else
               (List.tl gl,[Length ((List.length gl)-1)])
         |Some ltac ->
             try2solve_gen_goals ([],[]) ltac (List.tl gl)
       end in
       if opt_rest & (not(lgl_gen=[])) then
         iterative_rew (cmr+1) (fails+1) (cglob,!cmd,!wrn)
           unp_goals lvalid
       else
         (match ltacstp with
            | None ->
                iterative_rew (cmr+1) fails
                  (cglob+1,!cmd+1,!wrn) 
		  ((List.hd gl)::(lgl_gen@(List.tl unp_goals)))
                  ((p,lval_gen)::lvalid)
            | Some ltac ->
                (match opt_step with
                   | Solve ->
                       (match (try2solve_main_goal (List.hd gl)
                                 ltac) with
                          | None ->
                              iterative_rew (cmr+1) fails
                                (cglob+1,!cmd+1,!wrn) 
				((List.hd gl)::(lgl_gen@(List.tl unp_goals)))
                                ((p,lval_gen)::lvalid)
                          | Some (gl_solve,p_solve) ->
                              (lgl_gen@(List.tl unp_goals),
			       (p_solve,[Length 0])::(p,lval_gen)
			       ::lvalid,!wrn))
                   | Use ->	
                       (match (try_tacs4main_goal (List.hd gl)
                                 ltac) with
                          | None ->
                              iterative_rew (cmr+1) fails
                                (cglob+1,!cmd+1,!wrn) 
				((List.hd gl)::(lgl_gen@(List.tl unp_goals)))
                                ((p,lval_gen)::lvalid)
                          | Some(gl_trans,p_trans) ->
                              let lth=List.length gl_trans in
                                if lth=0 then
                                  (lgl_gen@(List.tl unp_goals),
				   (p_trans,[Length 0])::(p,lval_gen)::lvalid,!wrn)
                                else if lth=1 then
                                  iterative_rew (cmr+1) fails
                                    (cglob+1,!cmd+1,!wrn)
                                    (gl_trans@(lgl_gen@(List.tl
							  unp_goals)))
                                    ((p_trans,[])::(p,lval_gen)::
                                     lvalid)
                                else
                                  iterative_rew (cmr+1) fails
                                    (cglob+1,!cmd+1,!wrn)
                                    (gl_trans@(lgl_gen@(List.tl
							  unp_goals))) 
				    ((p_trans,
				      [Length ((List.length gl_trans)-1)])::
				     (p,lval_gen):: lvalid))
                   | All ->
                       (match (try2solve_main_goal (List.hd gl)
                                 ltac) with
                            None ->
                              (match (try_tacs4main_goal 
					(List.hd gl) ltac) with
                                 | None ->
                                     iterative_rew (cmr+1) fails
                                       (cglob+1,!cmd+1,!wrn)
                                       ((List.hd
                                           gl)::(lgl_gen@(List.tl
							    unp_goals)))
                                       ((p,lval_gen)::lvalid)
                                 | Some(gl_trans,p_trans) ->
                                     let lth=List.length gl_trans in
                                       if lth=0 then
                                         (lgl_gen@(List.tl unp_goals),
					  (p_trans,[Length 0])::
					  (p,lval_gen)::lvalid, !wrn)
                                       else if lth=1 then
                                         iterative_rew (cmr+1) fails
                                           (cglob+1,!cmd+1,!wrn)
                                           (gl_trans@
					    (lgl_gen@
					     (List.tl unp_goals)))
                                           ((p_trans,[])::
					    (p,lval_gen)::lvalid)
                                       else
                                         iterative_rew (cmr+1) fails
                                           (cglob+1,!cmd+1,!wrn)
                                           (gl_trans@
					    (lgl_gen@
					     (List.tl unp_goals)))
                                           ((p_trans,
					     [Length 
						((List.length gl_trans)-1)])::
					    (p, lval_gen)::lvalid))
                          | Some (gl_solve,p_solve) ->
                              (lgl_gen@(List.tl unp_goals),
			       (p_solve,[Length 0])::
			       (p,lval_gen)::lvalid,!wrn))))

     with UserError _ ->
       iterative_rew (cmr+1) (fails+1) (cglob,!cmd,!wrn) unp_goals lvalid)
    in
    let (gl,lvalid)=
      let (gl_res,lvalid_res,warn)=iterative_rew 0 0 (0,0,false) [g] [] in
        (if warn then
           mSGNL [<>]);
        (gl_res,lvalid_res)
    in
    let validation_fun=
      if lvalid=[] then
        (fun l -> List.hd l)
      else
        (let nlvalid=mod_hdlist lvalid
         in
           (fun l -> validation_gen nlvalid l))
    in
      (repackage sigr gl,validation_fun);;

(*Collects the arguments of AutoRewrite ast node*)
let dyn_autorewrite largs=
  let rec explicit_base largs =
    let tacargs = List.map cvt_arg largs in 
      List.map (function
		  | REDEXP ("LR", [CoqAst.Node(_,"COMMAND", [ast])]) -> ast, true
		  | REDEXP ("RL", [CoqAst.Node(_,"COMMAND", [ast])]) -> ast, false
		  | _ -> anomaly "Equality.explicit_base"
	       ) tacargs
  and list_bases largs =
    let tacargs = List.map cvt_arg largs in 
      List.map (function 
		  | REDEXP ("ByName", [CoqAst.Nvar(_,s)]) -> 
		      By_name (id_of_string s)
		  | REDEXP ("Explicit", l) ->
		      Explicit (explicit_base l)
		  | _ -> anomaly "Equality.list_bases"
	       ) tacargs
  and int_arg=function
     [(INTEGER n)] -> n
    |_ -> anomalylabstrm "dyn_autorewrite" [<'sTR
      "Bad call of int_arg (not an INTEGER)">]
  and list_args_rest (lstep,evstep) (ostep,evostep) (lrest,evrest)
    (orest,evorest) (depth,evdepth)=function
     [] -> (lstep,ostep,lrest,orest,depth)
    |(REDEXP (s,l))::tail ->
      if (s="Step") & (not(evstep)) then
        list_args_rest ((List.map Tacinterp.interp l),true) (ostep,evostep)
          (lrest,evrest) (orest,evorest) (depth,evdepth) tail
      else if (s="SolveStep") & (not(evostep)) then
        list_args_rest (lstep,evstep) (Solve,true) (lrest,evrest)
          (orest,evorest) (depth,evdepth) tail
      else if (s="Use") & (not(evostep)) then
        list_args_rest (lstep,evstep) (Use,true) (lrest,evrest) (orest,evorest)
          (depth,evdepth) tail
      else if (s="All") & (not(evostep)) then
        list_args_rest (lstep,evstep) (All,true) (lrest,evrest) (orest,evorest)
          (depth,evdepth) tail
      else if (s="Rest") & (not(evrest)) then
        list_args_rest (lstep,evstep) (ostep,evostep) ((List.map
          Tacinterp.interp l),true) (orest,evorest) (depth,evdepth) tail
      else if (s="SolveRest") & (not(evorest)) then
        list_args_rest (lstep,evstep) (ostep,evostep) (lrest,evrest)
          (false,true) (depth,evdepth) tail
      else if (s="Cond") & (not(evorest)) then
        list_args_rest (lstep,evstep) (ostep,evostep) (lrest,evrest)
          (true,true) (depth,evdepth) tail
      else if (s="Depth") & (not(evdepth)) then
        (let dth=int_arg (List.map cvt_arg l)
         in
           if dth>0 then
             list_args_rest (lstep,evstep) (ostep,evostep) (lrest,evrest)
               (orest,evorest) (dth,true) tail
           else
             errorlabstrm "dyn_autorewrite" [<'sTR
               "Depth value lower or equal to 0">])
      else
        anomalylabstrm "dyn_autorewrite" [<'sTR "Bad call of list_args_rest">]
  and list_args=function
     (REDEXP (s,lbases))::tail ->
       if s="BaseList" then
         (let (lstep,ostep,lrest,orest,depth)=list_args_rest ([],false)
            (Solve,false) ([],false) (false,false) (100,false) tail
          in
            autorewrite (list_bases lbases) (if lstep=[] then None else Some
              lstep) ostep (if lrest=[] then None else Some lrest) orest depth)
       else
         anomalylabstrm "dyn_autorewrite" [<'sTR
           "Bad call of list_args (not a BaseList tagged REDEXP)">]
    |_ ->
      anomalylabstrm "dyn_autorewrite" [<'sTR
        "Bad call of list_args (not a REDEXP)">]
  in
    list_args largs;;

(*Adds and hides the AutoRewrite tactic*)
let h_autorewrite=hide_tactic "AutoRewrite" dyn_autorewrite;;

(* $Id: equality.ml,v 1.49 1999/11/01 13:20:59 mohring Exp $ *)
