(****************************************************************************)
(*                 The Calculus of Inductive Constructions                  *)
(*                                                                          *)
(*                                Projet Coq                                *)
(*                                                                          *)
(*                     INRIA        LRI-CNRS        ENS-CNRS                *)
(*              Rocquencourt         Orsay          Lyon                    *)
(*                                                                          *)
(*                                 Coq V6.3                                 *)
(*                               July 1st 1999                              *)
(*                                                                          *)
(****************************************************************************)
(*                              proof_trees.ml                              *)
(****************************************************************************)


open Std;;
open Initial;;
open Names;;
open Evd;;
open More_util;;
open Term;;
open Pp;;
open Printer;;
open CoqAst;;
open Ast;;
open Termast;;
open Trad;;
open Closure;;
open Reduction;;

(* The module proof declares the structure of proof trees, global and
   readable constraints, and a few utilities on these types *)

(* We keep track of whether proofs are completed or not *)
type pf_status = COMPLETE_PROOF | INCOMPLETE_PROOF;;

type bindOcc = 
    Dep of identifier
  | NoDep of int
  | Com
;;

type 'a substitution = (bindOcc * 'a)   list
;;


(* prim_rules are the primitive rules of the base theory, here CIC *)
(* Ex: {name="INTRO"; hypspects=params=terms=[]; newids=["x"]} for Intro x.
       {name="EXACT"; hypspects=params=newids=[]; terms=[t]} for Exact t.
       {name="CONVERT_HYP"; hypspects=["h"]; params=newids=[]; terms=[T]} 
          for Change T in h.  *)

type prim_rule_name = 
   INTRO | INTRO_AFTER | INTRO_REPLACING | FIX | COFIX | REFINE 
 | CONVERT_CONCL | CONVERT_HYP | THIN | MOVE of bool;;

type prim_rule = {name : prim_rule_name;
                  hypspecs : identifier list;
                  newids : identifier list;
                  params : CoqAst.t list;
                  terms : constr list};;

(* A tactic expression (tacname,args) refers to tactic tacname (indexing into
   a table in tacmach) applied to parameters args *)

type tactic_arg = (* reflects the current version of vernac *)
    COMMAND       of CoqAst.t
  | CONSTR        of constr
  | IDENTIFIER    of identifier
  | INTEGER       of int
  | CLAUSE        of identifier list
  | BINDINGS      of CoqAst.t substitution
  | CBINDINGS     of constr   substitution 
  | QUOTED_STRING of string
  | TACEXP        of CoqAst.t
  | REDEXP        of string * CoqAst.t list
  | FIXEXP        of identifier * int * CoqAst.t
  | COFIXEXP      of identifier * CoqAst.t
  | LETPATTERNS   of (int list option * (identifier * int list) list)
  | INTROPATTERN  of intro_pattern

and intro_pattern =  IdPat   of identifier          |
                     DisjPat of intro_pattern list  |
                     ConjPat of intro_pattern list  | 
                     ListPat of intro_pattern list  

and tactic_expression = string * tactic_arg list
;;

let is_bind arg = 
 match arg with
   (BINDINGS _) -> true |
    _           -> false;;

(***************************)
(* The type of proof trees *)
(***************************)

type local_constraints = Spset.t (* a set of section paths *)

let lc_toList lc = Spset.elements lc;;


type goal =  ctxtty evar_info

and  proof_tree =
     {status : pf_status; (* the proof is complete or incomplete *)
      goal   : goal;     (* the goal *)
      ref    : (rule * proof_tree list) option; 
                      (* NONE       if the goal has still to be proved, 
                         SOME (r,l) if the rule r was applied to the goal
                                    and gave l as subproofs to be completed *)
     subproof : proof_tree option 
               (* (SOME p) if ref=(SOME(TACTIC t,l)) ;
                p is then the proof that the goal can be proven if the goals
                in l are solved *)
     }

and rule =
    PRIM        of prim_rule
  | TACTIC      of tactic_expression
  | CONTEXT     of ctxtty
  | LOCAL_CONSTRAINTS of  local_constraints
         (* for changing the lc of current goal *)

and ctxtty = {pgm    : constr option; (* for Program tactic *)
              mimick : proof_tree option;(* for a soon to exist mimick tactic *)
              lc     : local_constraints} 

and evar_declarations = ctxtty evar_map
;;

let empty_evd = Evd.mt_evd();;

(* Functions on tactic arguments *)

let cvt_bind = function
    Node(_,"BINDING", [Num(_,n); Node(_,"COMMAND",[c])])   -> (NoDep n,c)
  | Node(_,"BINDING", [Nvar(_,s); Node(_,"COMMAND",[c])]) -> 
      let id = id_of_string s in (Dep id,c)
  | Node(_,"BINDING", [Node(_,"COMMAND",[c])])             -> (Com,c)
  | x -> errorlabstrm "cvt_bind"
        [< 'sTR "Not the expected form in binding!"; print_ast x >]
;;


let rec cvt_intro_pattern = function
    Node(_,"IDENTIFIER", [Nvar(_,s)]) -> IdPat   (id_of_string s)
  | Node(_,"DISJPATTERN", l)  -> DisjPat (List.map cvt_intro_pattern l)
  | Node(_,"CONJPATTERN", l)  -> ConjPat (List.map cvt_intro_pattern l)
  | Node(_,"LISTPATTERN", l)  -> ListPat (List.map cvt_intro_pattern l)
  | x -> errorlabstrm "cvt_intro_pattern"
        [< 'sTR "Not the expected form for an introduction pattern!"; 
            print_ast x >]
;;

let cvt_letpattern (o,l) = function
    Node(_,"HYPPATTERN", Nvar(_,s)::nums) ->
      (o, (id_of_string s, List.map num_of_ast nums)::l)
  | Node(_,"CCLPATTERN", nums) ->
      if o<>None then error "\"Goal\" occurs twice"
      else (Some (List.map num_of_ast nums), l)
  | arg -> invalid_arg_loc (Ast.loc arg,"cvt_hyppattern");;

let cvt_letpatterns astl = List.fold_left cvt_letpattern (None,[]) astl;;

(* Translation of reduction expression: we need trad because of Fold
 * Moreover, reduction expressions are used both in tactics and in
 * vernac.
 *)
let glob_nvar com =
  let s = nvar_of_ast com in
    try Nametab.sp_of_id CCI (id_of_string s)
    with Not_found -> error ("unbound variable "^s);;

let cvt_pattern sigma sign = function
    Node(_,"PATTERN", Node(_,"COMMAND",[com])::nums) ->
      let occs = List.map num_of_ast nums in
      let j = fconstruct sigma sign com in
        (occs, j._VAL, j._TYPE)
  | arg -> invalid_arg_loc (Ast.loc arg,"cvt_pattern");;

let cvt_unfold = function
    Node(_,"UNFOLD", com::nums) -> (List.map num_of_ast nums, glob_nvar com)
  | arg -> invalid_arg_loc (Ast.loc arg,"cvt_unfold");;

let cvt_fold sigma sign = function
    Node(_,"COMMAND",[c]) -> constr_of_com sigma sign c
  | arg -> invalid_arg_loc (Ast.loc arg,"cvt_fold");;

let flag_of_ast lf =
  let beta = ref false in
  let delta = ref false in
  let iota = ref false in
  let idents = ref (None: (sorts oper -> bool) option) in
  let set_flag = function
      Node(_,"Beta",[]) -> beta := true
    | Node(_,"Delta",[]) -> delta := true
    | Node(_,"Iota",[]) -> iota := true
    | Node(loc,"Unf",l) ->
        if !delta then
          if !idents = None then
            let idl = List.map glob_nvar l in
              idents := Some
                (function
                     Const sp -> List.mem sp idl
                   | Abst sp -> List.mem sp idl
                   | _ -> false)
          else user_err_loc(loc,"flag_of_ast",
                  [< 'sTR"Cannot specify identifiers to unfold twice" >])
        else user_err_loc(loc,"flag_of_ast",
                  [< 'sTR"Delta must be specified before" >])
    | Node(loc,"UnfBut",l) ->
        if !delta then
          if !idents = None then
            let idl = List.map glob_nvar l in
              idents := Some
                (function
                     Const sp -> not (List.mem sp idl)
                   | Abst sp -> not (List.mem sp idl)
                   | _ -> false)
          else user_err_loc(loc,"flag_of_ast",
                  [< 'sTR"Cannot specify identifiers to unfold twice" >])
        else user_err_loc(loc,"flag_of_ast",
                  [< 'sTR"Delta must be specified before" >])
    | arg -> invalid_arg_loc (Ast.loc arg,"flag_of_ast")
  in
    List.iter set_flag lf;
    { r_beta = !beta;
      r_iota = !iota;
      r_delta = match (!delta,!idents) with
          (false,_) -> (fun _ -> false)
        | (true,None) -> (fun _ -> true)
        | (true,Some p) -> p };;


let redexp_of_ast sigma sign = function
  | ("Red", []) -> Red
  | ("Hnf", []) -> Hnf
  | ("Simpl", []) -> Simpl
  | ("Unfold", ul) -> Unfold (List.map cvt_unfold ul)
  | ("Fold", cl) -> Fold (List.map (cvt_fold sigma sign) cl)
  | ("Cbv",lf) -> Cbv(UNIFORM, flag_of_ast lf)
  | ("Lazy",lf) -> Lazy(UNIFORM, flag_of_ast lf)
  | ("Pattern",lp) -> Pattern (List.map (cvt_pattern sigma sign) lp)
  | (s,_) -> invalid_arg ("malformed reduction-expression: "^s);;


let cvt_arg = function
    Nvar(_,s) -> IDENTIFIER (id_of_string s)
  | Str(_,s) -> QUOTED_STRING s
  | Num(_,n) -> INTEGER n
  | Node(_,"COMMAND",[c]) -> COMMAND c

  | Node(_,"BINDINGS",astl) -> BINDINGS (List.map cvt_bind astl)

  | Node(_,"REDEXP",[Node(_,redname,args)]) -> REDEXP(redname,args)

  | Node(_,"CLAUSE",cl) -> CLAUSE (List.map (comp id_of_string nvar_of_ast) cl)

  | Node(_,"TACTIC",[ast]) -> TACEXP ast

  | Node(_,"FIXEXP", [Nvar(_,s); Num(_,n);Node(_,"COMMAND",[c])]) ->
      (FIXEXP (id_of_string s,n,c))

  | Node(_,"COFIXEXP", [Nvar(_,s); Node(_,"COMMAND",[c])]) ->
      (COFIXEXP (id_of_string s,c))

  | Node(_,"INTROPATTERN", [ast]) -> (INTROPATTERN (cvt_intro_pattern ast))

  | Node(_,"LETPATTERNS", astl) -> LETPATTERNS (cvt_letpatterns astl)

  | x -> anomaly_loc (Ast.loc x, "Tacinterp.cvt_bind",
                      [< 'sTR "Unrecognizable ast node : "; print_ast x >])
;;

let ast_of_cvt_bind f = function
    (NoDep n,c) -> ope ("BINDING", [(num n); ope ("COMMAND",[(f c)])])
  | (Dep  id,c) -> ope ("BINDING", [nvar (string_of_id id);
                                      ope  ("COMMAND",[(f c)])]) 
  | (Com,c)     -> ope ("BINDING", [ope ("COMMAND",[(f c)])])
;;

let rec ast_of_cvt_intro_pattern = function
    IdPat id  -> nvar (string_of_id id) 
   |DisjPat l -> ope ("DISJPATTERN",  (List.map ast_of_cvt_intro_pattern l))
   |ConjPat l -> ope ("CONJPATTERN",  (List.map ast_of_cvt_intro_pattern l))
   |ListPat l -> ope ("LISTPATTERN",  (List.map ast_of_cvt_intro_pattern l))
;;

let ast_of_cvt_arg = function
    (IDENTIFIER id)   -> nvar (string_of_id id) 
  | (QUOTED_STRING s) -> str s
  | (INTEGER n)       -> num n 
  | (COMMAND c)       -> ope ("COMMAND",[c])
  | (CONSTR  c)       -> ope ("COMMAND",[bdize false (assumptions_for_print []) c])
  | (CLAUSE idl)      -> ope ("CLAUSE", List.map (comp nvar string_of_id) idl)
  | (BINDINGS bl)     -> ope ("BINDINGS", 
            List.map (ast_of_cvt_bind (fun x -> x)) bl)
  | (CBINDINGS bl)    -> ope ("BINDINGS", 
            List.map  (ast_of_cvt_bind (bdize false (assumptions_for_print []))) bl)
  | (TACEXP ast)      -> ope ("TACTIC",[ast]) 
  | (REDEXP (s,args)) -> ope ("REDEXP", [ope(s,args)])
  | (FIXEXP (id,n,c)) -> ope ("FIXEXP",[(nvar (string_of_id id)); 
                                        (num n); 
                                        ope ("COMMAND",[c])]) 
  | (COFIXEXP (id,c)) -> ope ("COFIXEXP",[(nvar (string_of_id id)); 
                                          (ope ("COMMAND",[c]))])
  | (INTROPATTERN p) ->  ast_of_cvt_intro_pattern p
;;

 
(* Functions on goals *)

let mkGOAL ctxt sign cl = {hyps=sign;concl=cl;Evd.body=EVAR_EMPTY;info=Some ctxt};;

(* Functions on the information associated with existential variables  *)

let mt_ctxt lc          = {pgm=None; mimick = None;lc=lc};;

let get_ctxt ctxt = 
  match ctxt.info with
      Some x -> x
    | None -> assert false
;;

let get_pgm evd = 
  match evd.info with
      Some x -> x.pgm
    | None -> assert false
;;

let set_pgm pgm ctxt = {pgm = pgm; mimick = ctxt.mimick;lc=ctxt.lc};;

let get_mimick evd   = 
  match evd.info with
      Some x -> x.mimick
    | None -> assert false
;;
let set_mimick mimick ctxt = {mimick = mimick; pgm = ctxt.pgm;lc=ctxt.lc};;

let get_lc evd = 
  match evd.info with
      Some x -> x.lc
    | None -> assert false
;;

(* Functions on proof trees *)

let ref_of_proof pf =
    match pf.ref with
    None -> failwith"rule_of_proof"
  | Some r -> r
;;

let rule_of_proof pf =
let (r,_) = ref_of_proof pf in r
;;

let children_of_proof pf = 
let (_,cl) = ref_of_proof pf in cl;;

let goal_of_proof pf = pf.goal;;
let subproof_of_proof pf =
    match pf.subproof with
    None -> failwith "subproof_of_proof"
  | Some pf -> pf
;;

let status_of_proof pf = pf.status;;

let is_complete_proof pf = pf.status = COMPLETE_PROOF;;

let is_leaf_proof pf =
    match pf.ref with
    None -> true
  | Some _ -> false
;;

let is_tactic_proof pf =
    match pf.subproof with
    Some _ -> true
  | None -> false
;;

(*******************************************************************)
(*            Constraints for existential variables                *)
(*******************************************************************)

(* A local constraint is just a set of section_paths *)

(* recall : type local_constraints    = Spset.t;; *)


(* A global constraint is a mappings of existential variables
   with some extra information for the program and mimick
   tactics.
*)

type global_constraints   = evar_declarations timestamped;;


(* A readable constraint is a global constraint plus a focus set
   of existential variables and a signature.
*)

type evar_recordty = {focus : local_constraints;
                      sign  : type_judgement signature;
                      decls : evar_declarations}

and  readable_constraints  = evar_recordty timestamped
;;


(* Functions on readable constraints *)

let mt_evcty lc gc = ts_mk{focus = lc;
                           sign  = nil_sign;
                           decls = gc}
;;

let evc_of_evds evds gl = 
  ts_mk{focus = (get_lc gl); 
        sign  = gl.hyps ; 
        decls = evds};;

let rc_of_gc gc gl = evc_of_evds (ts_it gc) gl
;;

let rc_add evc (k,v) = 
   ts_mod
    (fun evc -> {focus =(Spset.add k evc.focus);
                 sign  = evc.sign;
                 decls = Evd.add_with_info evc.decls k v})
         evc;;

let get_hyps  evc = (ts_it evc).sign;;
let get_focus evc = (ts_it evc).focus;;
let get_decls evc = (ts_it evc).decls;;
let get_gc    evc = (ts_mk (ts_it evc).decls);;
let remap evc (k,v) = 
  ts_mod
    (fun evc -> {focus = evc.focus;
                 sign  = evc.sign;
                 decls = Evd.remap evc.decls k v})
         evc;;

let lc_exists f lc = Spset.fold (fun e b -> (f e) or b) lc false;;

(* MENTIONS SIGMA SP LOC is true exactly when LOC is defined, and SP is
 * on LOC's access list, or an evar on LOC's access list mentions SP.
 *)
let rec mentions sigma sp loc =
  let loc_evd = map (ts_it sigma).decls loc
  in match loc_evd.Evd.body with 
    EVAR_DEFINED _ -> (Spset.mem sp (get_lc loc_evd) 
                       or lc_exists (mentions sigma sp) (get_lc loc_evd))
  | _ -> false
;;

(* ACCESSIBLE SIGMA SP LOC is true exactly when SP is on LOC's access list,
 * or there exists a LOC' on LOC's access list such that
 * MENTIONS SIGMA SP LOC' is true.
 *)
let rec accessible sigma sp loc =
  let loc_evd = map (ts_it sigma).decls loc
  in lc_exists (fun loc' -> sp = loc' or mentions sigma sp loc') (get_lc loc_evd)
;;

(* [ctxt_access sigma sp] is true when SIGMA is accessing a current
 * accessibility list ACCL, and SP is either on ACCL, or is mentioned
 * in the body of one of the ACCL.
 *)
let ctxt_access sigma sp =
    lc_exists (fun sp' -> sp' = sp or mentions sigma sp sp') (ts_it sigma).focus
;;


(*
let mt_evc = mt_evcty Spset.empty (Evd.create sp_ord);;
let mt_lc = Spset.empty;;
let evd_hyps evd = evd.evgoal.hyps;;
*)


(*********************************************************************)
(*                  Pretty printing functions                        *)
(*********************************************************************)

(* Il faudrait parametrer toutes les pr_term, term0, etc. par la
strategie de renommage choisie pour Termast (en particulier, il
faudrait pouvoir etre sur que lookup_as_renamed qui est utilis par
Intros Until fonctionne exactement comme on affiche le but avec term0
*)

let pf_lookup_name_as_renamed hyps ccl s =
  Termast.lookup_name_as_renamed (gLOB hyps) ccl s

let pf_lookup_index_as_renamed ccl n =
  Termast.lookup_index_as_renamed ccl n

let pr_idl idl = prlist_with_sep pr_spc print_id idl;;

let pr_goal g =
  let penv = pr_env_opt (gLOB g.hyps) in
  let pc = term0_at_top (gLOB g.hyps) g.concl in
    [< 'sTR"  "; hV 0 [< penv; 'fNL;
		         'sTR (emacs_str (String.make 1 (Char.chr 253))) ;
                         'sTR "============================"; 'fNL ;
                         'sTR" " ; pc>]; 'fNL>]
;;


let pr_concl n g =
  let pc = term0_at_top (gLOB g.hyps) g.concl in
    [< 'sTR (emacs_str (String.make 1 (Char.chr 253))) ;
       'sTR "subgoal ";'iNT n;'sTR " is:";'cUT;'sTR" " ; pc >];;

(* print the subgoals but write Subtree proved! even in case some existential 
   variables remain unsolved, pr_subgoals_existential is a safer version 
   of pr_subgoals
*)
let pr_subgoals = function
    [] -> [< 'sTR"Subtree proved!" ; 'fNL >]
  | [g] ->
      let pg = pr_goal g in
        v 0 [< 'sTR ("1 "^"subgoal");'cUT; pg >]

  | g1::rest ->
      let rec pr_rec n = function
          [] -> [< >]
        | g::rest ->
            let pg = pr_concl n g in
            let prest = pr_rec (n+1) rest in
              [< 'cUT; pg; prest >] in
      let pg1 = pr_goal g1 in
      let pgr = pr_rec 2 rest in
        v 0 [< 'iNT(List.length rest+1) ; 'sTR" subgoals" ;'cUT; pg1; pgr >];;

let pr_subgoal n =
 let rec prrec p = function
     [] -> error "No such goal"
   | g::rest ->
       if p = 1
       then
         let pg = pr_goal g in
           v 0 [< 'sTR "subgoal ";'iNT n;'sTR " is:"; 'cUT; pg >]
       else prrec (p-1) rest
 in prrec n;;

let pr_pgm ctxt =
    match ctxt.pgm with
    None -> [< >]
  | Some pgm ->
      let ppgm = fprterm pgm in [< 'sTR"Realizer " ; ppgm >];;

let pr_ctxt ctxt =
  let pc = pr_pgm ctxt in
    [< 'sTR"[" ; pc; 'sTR"]" >];;

let pr_seq = function 
  | {hyps=(x,y) as hyps;concl=cl;info=Some info} ->
      let sign = List.rev(List.combine x y) in
      let pc = pr_ctxt info in
      let pdcl =
    	prlist_with_sep pr_spc
	  (fun (id,ty) -> hOV 0 [< print_id id; 'sTR" : ";prterm ty.body >])
	  sign in
      let pcl = term0_at_top (gLOB hyps) cl in
    	hOV 0 [< pc; pdcl ; 'sPC ; hOV 0 [< 'sTR"|- " ; pcl >] >]
  | _ -> anomaly "pr_seq : info = None"
;;

let prgl gl =
  let plc = pr_idl (List.map basename (lc_toList (get_lc gl))) in
  let pgl = pr_seq gl in
    [< 'sTR"[[" ; plc; 'sTR"]" ; pgl ; 'sTR"]" ; 'sPC >]
;;

let pr_evgl gl =
  let plc = pr_idl (List.map basename (lc_toList (get_lc gl))) in
  let phyps = pr_idl (ids_of_sign gl.hyps) in
  let pc = prterm gl.concl in
    hOV 0 [< 'sTR"[[" ; plc; 'sTR"] " ; phyps;
             'sPC; 'sTR"|- " ; pc; 'sTR"]" >];;

let pr_evgl_sign gl = 
  let plc = pr_idl (List.map basename (lc_toList (get_lc gl))) in
  let ps = pr_sign gl.hyps in
  let pc = prterm gl.concl in
    hOV 0 [< 'sTR"[[" ; plc ; 'sTR"] " ; ps;
             'sPC; 'sTR"|- " ; pc; 'sTR"]" >];;

(*  evd.evgoal.lc seems to be printed twice *)
let pr_decl evd =
  let pevgl = pr_evgl evd in
  let pb =
    match evd.Evd.body with
        EVAR_EMPTY -> [< 'fNL >]
      | EVAR_DEFINED c ->
          let pc = prterm c in [< 'sTR" => " ; pc;  'fNL  >]
  in
    h 0 [< pevgl; pb >];;

let pr_evd evd = 
  prlist_with_sep pr_fnl
    (fun (sp,evd) ->
       let pe = pr_decl evd in 
         h 0 [< print_id(basename sp) ; 'sTR"==" ; pe >])
    (Evd.toList evd);;

let pr_decls decls = pr_evd (ts_it decls);;

let pr_focus accl = pr_idl (List.map basename (lc_toList accl));;

let pr_evc evc =
  let stamp = ts_stamp evc in
  let evc   = ts_it evc in
  let pe = pr_evd evc.decls in
    [< 'sTR"#" ; 'iNT stamp ; 'sTR"[" ; pr_focus evc.focus ; 'sTR"]=" ; pe >]
;;


let pr_evars = 
  prlist_with_sep pr_fnl
    (fun (sp,evd) ->
       let pegl = pr_evgl_sign evd in 
         [< print_id(basename sp); 'sTR " : "; pegl >])
;;


(* Print an enumerated list of existential variables *)
let rec pr_evars_int i = function
    [] -> [< >]
 | (sp,evd)::rest ->
     let pegl = pr_evgl_sign evd in 
     let pei = pr_evars_int (i+1) rest in
       [< (hOV 0 [< 'sTR "Existential "; 'iNT i; 'sTR " ="; 'sPC;
                    print_id (basename sp) ; 'sTR " : "; pegl >]); 'fNL ;  
          pei >];;

let pr_subgoals_existential sigma = function 
    [] -> let exl = Evd.non_instantiated sigma in 
          if exl = [] then [< 'sTR"Subtree proved!" ; 'fNL >]
          else
            let pei = pr_evars_int 1 exl in
              [< 'sTR "No more subgoals but non-instantiated existential ";
                 'sTR "variables :" ;'fNL; (hOV 0 pei) >]
  | [g] ->
      let pg = pr_goal g in
        v 0 [< 'sTR ("1 "^"subgoal");'cUT; pg >]

  | g1::rest ->
      let rec pr_rec n = function
          [] -> [< >]
        | g::rest ->
            let pc = pr_concl n g in
            let prest = pr_rec (n+1) rest in
              [< 'cUT; pc; prest >] in
      let pg1 = pr_goal g1 in
      let prest = pr_rec 2 rest in
        v 0 [< 'iNT(List.length rest+1) ; 'sTR" subgoals" ;'cUT; pg1; prest;
               'fNL >];;

(* $Id: proof_trees.ml,v 1.27 1999/11/08 15:21:24 mohring Exp $ *)

