(* camlp4r *)
(***********************************************************************)
(*                                                                     *)
(*                             Camlp4                                  *)
(*                                                                     *)
(*        Daniel de Rauglaudre, projet Cristal, INRIA Rocquencourt     *)
(*                                                                     *)
(*  Copyright 1998 Institut National de Recherche en Informatique et   *)
(*  Automatique.  Distributed only by permission.                      *)
(*                                                                     *)
(***********************************************************************)

(* Id *)

(* Load files in core *)

let find_in_path path name =
  if not (Filename.is_implicit name) then
    if Sys.file_exists name then name else raise Not_found
  else
    let rec try_dir =
      function
        [] -> raise Not_found
      | dir :: rem ->
          let fullname = Filename.concat dir name in
          if Sys.file_exists fullname then fullname else try_dir rem
    in
    try_dir path
;;

exception Error of string * string;;

let objfiles = ref [];;
let nolib = ref false;;
let path = ref [];;

let loadfile file =
  let fname =
    try find_in_path !path file with
      Not_found -> raise (Error (file, "file not found in path"))
  in
  try Dynlink.loadfile fname with
    Dynlink.Error e -> raise (Error (fname, Dynlink.error_message e))
;;

let load objfiles =
  Dynlink.init ();
  Dynlink.allow_unsafe_modules true;
  List.iter loadfile objfiles
;;

let apply_load () =
  let i = ref 1 in
  let stop = ref false in
  while not !stop && !i < Array.length Sys.argv do
    let s = Sys.argv.(!i) in
    if s = "-I" && !i + 1 < Array.length Sys.argv then
      begin path := Sys.argv.(!i + 1) :: !path; i := !i + 2; () end
    else if s = "-nolib" then begin nolib := true; incr i; () end
    else if s = "-where" then
      begin
        print_string Odyl_config.standard_library;
        print_newline ();
        flush stdout;
        exit 0
      end
    else if s = "--" then begin incr i; stop := true; () end
    else if String.length s > 0 && s.[0] == '-' then stop := true
    else if
      Filename.check_suffix s ".cmo" || Filename.check_suffix s ".cma" then
      begin objfiles := s :: !objfiles; incr i; () end
    else stop := true
  done;
  if !nolib then () else path := !path @ [Odyl_config.standard_library];
  let objs = List.rev !objfiles in load objs
;;

let main () =
  begin try apply_load () with
    Error (fname, str) ->
      flush stdout;
      Printf.eprintf "Error while loading \"%s\": " fname;
      Printf.eprintf "%s.\n" str;
      flush stderr;
      exit 2
  end;
  !(Odyl_main.go) ();
  ()
;;

Printexc.catch main ();;
