/* ---------------------------------------------------------------------
 *
 * -- Automatically Tuned Linear Algebra Software (ATLAS)
 *    (C) Copyright 1999 All Rights Reserved
 *
 * -- ATLAS routine -- Version 2.0 -- December 25, 1999
 *
 * -- Suggestions,  comments,  bugs reports should be sent to the follo-
 *    wing e-mail address: atlas@cs.utk.edu
 *
 *  Author         : Antoine P. Petitet
 * University of Tennessee - Innovative Computing Laboratory
 * Knoxville TN, 37996-1301, USA.
 *
 * ---------------------------------------------------------------------
 *
 * -- Copyright notice and Licensing terms:
 *
 * Redistribution  and  use in  source and binary forms, with or without
 * modification, are  permitted provided  that the following  conditions
 * are met:
 *
 * 1) Redistributions  of  source  code  must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2) Redistributions in binary form must reproduce  the above copyright
 *    notice,  this list of  conditions and the  following disclaimer in
 *    the documentation and/or other materials provided with the distri-
 *    bution.
 * 3) All advertising materials mentioning features or use of this soft-
 *    ware must display the folowing acknowledgement:
 *    This product includes software developed by the ATLAS group of the
 *    University of Tennesee, Knoxville and its contributors.
 * 4) The names of the  University of Tennessee,  Knoxville,  the  ATLAS
 *    group, or the names of its contributors may not be used to endorse
 *    or  promote products derived  from  this software without specific
 *    prior written permission.
 *
 * -- Disclaimer:
 *
 * The University of Tennessee, Knoxville,  the ATLAS group,  or the au-
 * thors make  no representations about the suitability of this software
 * for any purpose.  This software is provided ``as is'' without express
 * or implied warranty.
 *
 * ---------------------------------------------------------------------
 */
/*
 * Include files
 */
#include "atlas_refmisc.h"
#include "atlas_reflevel1.h"

void ATL_srefrotg
(
   float                      * A,
   float                      * B,
   float                      * C,
   float                      * S
)
{
/*
 * Purpose
 * =======
 *
 * ATL_srefrotg  constructs a Givens plane rotation. Given the scalars a
 * and b, this routine computes the following quantities:
 *
 *    sigma = sgn(a) if |a| > |b|, sgn(b) otherwise;
 *    r     = sigma * sqrt( a^2 + b^2 );
 *    c     = a / r if r <> 0, 1 otherwise;
 *    s     = b / r if r <> 0, 0 otherwise.
 *
 * The numbers c, s and r then satisfy the matrix equation:
 *
 *    [ c  s ] [ a ]    [ r ]
 *    [ -s c ] [ b ] =  [ 0 ].
 *
 * The introduction of  sigma  is not essential  to the computation of a
 * Givens rotation matrix, but it permits later stable reconstruction of
 * c and s from just one stored number.  For  this purpose, this routine
 * also computes
 *
 *         s        if |a| > |b|,
 *    z =  1 / c    if |b| >= |a| and c <> 0,
 *         1        if c = 0.
 *
 * This subroutine returns r overwriting a, and z overwriting b, as well
 * as returning c and s. If one later wishes to reconstruct c and s from
 * z, it can be done as follows:
 *
 *    if  z  = 1,    set c = 0             and s = 1,
 *    if |z| < 1,    set c = sqrt(1 - z^2) and s = z,
 *    if |z| > 1,    set c = 1 / z         and s = sqrt(1 - c^2).
 *
 * See ``Basic Linear Algebra Subprograms for Fortran Usage'' by C. Law-
 * son, R. Hanson, D. Kincaid and F. Krogh, ACM Transactions on Mathema-
 * tical Software, 1979, 5(3) pp 308-323, for further information.
 *
 * Arguments
 * =========
 *
 * A       (input/output)                float *
 *         On entry, A specifies the scalar a. On exit, A is overwritten
 *         by the scalar r defined above.
 *
 * B       (input/output)                float *
 *         On entry, B specifies the scalar b. On exit, B is overwritten
 *         by the scalar z defined above.
 *
 * C       (output)                      float *
 *         On exit, C  specifies the scalar c defined above.
 *
 * S       (output)                      float *
 *         On exit, S  specifies the scalar s defined above.
 *
 * ---------------------------------------------------------------------
 */
/*
 * .. Local Variables ..
 */
   register float             absa, absb, roe, scale, r, tmpa, tmpb, z;
/* ..
 * .. Executable Statements ..
 *
 */
   absa  = Msabs( *A );
   absb  = Msabs( *B );
   roe   = ( absa > absb ? (*A) :  (*B) );
   scale = absa + absb;

   if( scale != ATL_sZERO )
   {
      tmpa = (*A) / scale;
      tmpb = (*B) / scale;
      if( roe < ATL_sZERO )
         r = - scale * sqrt( ( tmpa * tmpa ) + ( tmpb * tmpb ) );
      else
         r =   scale * sqrt( ( tmpa * tmpa ) + ( tmpb * tmpb ) );
      *C  = (*A) / r;
      *S  = (*B) / r;
      z   =  ATL_sONE;
      if(   absa >  absb )
         z = *S;
      if( ( absb >= absa ) && ( (*C) != ATL_sZERO ) )
         z = ATL_sONE / (*C);
   }
   else { *C = ATL_sONE; *S = r = z = ATL_sZERO; }

   *A = r;
   *B = z;
/*
 * End of ATL_srefrotg
 */
}
