/* ---------------------------------------------------------------------
 *
 * -- Automatically Tuned Linear Algebra Software (ATLAS)
 *    (C) Copyright 1999 All Rights Reserved
 *
 * -- ATLAS routine -- Version 2.0 -- December 25, 1999
 *
 * -- Suggestions,  comments,  bugs reports should be sent to the follo-
 *    wing e-mail address: atlas@cs.utk.edu
 *
 *  Author         : Antoine P. Petitet
 * University of Tennessee - Innovative Computing Laboratory
 * Knoxville TN, 37996-1301, USA.
 *
 * ---------------------------------------------------------------------
 *
 * -- Copyright notice and Licensing terms:
 *
 * Redistribution  and  use in  source and binary forms, with or without
 * modification, are  permitted provided  that the following  conditions
 * are met:
 *
 * 1) Redistributions  of  source  code  must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2) Redistributions in binary form must reproduce  the above copyright
 *    notice,  this list of  conditions and the  following disclaimer in
 *    the documentation and/or other materials provided with the distri-
 *    bution.
 * 3) All advertising materials mentioning features or use of this soft-
 *    ware must display the folowing acknowledgement:
 *    This product includes software developed by the ATLAS group of the
 *    University of Tennesee, Knoxville and its contributors.
 * 4) The names of the  University of Tennessee,  Knoxville,  the  ATLAS
 *    group, or the names of its contributors may not be used to endorse
 *    or  promote products derived  from  this software without specific
 *    prior written permission.
 *
 * -- Disclaimer:
 *
 * The University of Tennessee, Knoxville,  the ATLAS group,  or the au-
 * thors make  no representations about the suitability of this software
 * for any purpose.  This software is provided ``as is'' without express
 * or implied warranty.
 *
 * ---------------------------------------------------------------------
 */
/*
 * Include files
 */
#include "atlas_refmisc.h"
#include "atlas_reflevel1.h"

void ATL_srefrot
(
   const int                  N,
   float                      * X,
   const int                  INCX,
   float                      * Y,
   const int                  INCY,
   const float                C,
   const float                S
)
{
/*
 * Purpose
 * =======
 *
 * ATL_srefrot  applies a  plane  rotation to the two n-vectors x and y.
 * This routine computes:
 *
 *    [ x_i ]   [ c  s ] [ x_i ]
 *    [ y_i ] = [ -s c ] [ y_i ]    for all i = 1 .. n.
 *
 * If n <= 0 or if c = 1 and s = 0, this subroutine returns immediately.
 *
 * Arguments
 * =========
 *
 * N       (input)                       const int
 *         On entry, N specifies the length of the vector x. N  must  be
 *         at least zero. Unchanged on exit.
 *
 * X       (input/output)                float *
 *         On entry,  X  points to the  first entry to be accessed of an
 *         incremented array of size equal to or greater than
 *            ( 1 + ( n - 1 ) * abs( INCX ) ) * sizeof(   float   ),
 *         that contains the vector x.  On exit,  the entries of the in-
 *         cremented array  X are rotated with the entries of the incre-
 *         mented array  Y.
 *
 * INCX    (input)                       const int
 *         On entry, INCX specifies the increment for the elements of X.
 *         INCX must not be zero. Unchanged on exit.
 *
 * Y       (input/output)                float *
 *         On entry,  Y  points to the  first entry to be accessed of an
 *         incremented array of size equal to or greater than
 *            ( 1 + ( n - 1 ) * abs( INCY ) ) * sizeof(   float   ),
 *         that contains the vector y.  On exit,  the entries of the in-
 *         cremented array  Y are rotated with the entries of the incre-
 *         mented array  X.
 *
 * INCY    (input)                       const int
 *         On entry, INCY specifies the increment for the elements of Y.
 *         INCY must not be zero. Unchanged on exit.
 *
 * C       (input)                       const float
 *         On entry, C specifies the scalar c definiting the plane rota-
 *         tion. Unchanged on exit.
 *
 * S       (input)                       const float
 *         On entry, S specifies the scalar s definiting the plane rota-
 *         tion. Unchanged on exit.
 *
 * ---------------------------------------------------------------------
 */
/*
 * .. Local Variables ..
 */
   register int               i;
   int                        nu;
   const int                  incX2 = 2 * INCX, incY2 = 2 * INCY,
                              incX3 = 3 * INCX, incY3 = 3 * INCY,
                              incX4 = 4 * INCX, incY4 = 4 * INCY;
   float                      * StX;
   register float             x0, x1, x2, x3, y0, y1, y2, y3;
   register const float       co = C, si = S;
/* ..
 * .. Executable Statements ..
 *
 */
   if( ( N > 0 ) && !Msone( co, si ) )
   {
      if( ( nu = ( N >> 2 ) << 2 ) != 0 )
      {
         StX = (float *)X + nu * INCX;

         do
         {
            x0 = (*X);     y0 = (*Y);
            x1 = X[INCX ]; y1 = Y[INCY ];
            x2 = X[incX2]; y2 = Y[incY2];
            x3 = X[incX3]; y3 = Y[incY3];

            *X       = co * x0 + si * y0; *Y       = co * y0 - si * x0;
            X[INCX ] = co * x1 + si * y1; Y[INCY ] = co * y1 - si * x1;
            X[incX2] = co * x2 + si * y2; Y[incY2] = co * y2 - si * x2;
            X[incX3] = co * x3 + si * y3; Y[incY3] = co * y3 - si * x3;

            X  += incX4;
            Y  += incY4;

         } while( X != StX );
      }

      for( i = N - nu; i != 0; i-- )
      {
         x0  = (*X);
         y0  = (*Y);

         *X  = co * x0 + si * y0;
         *Y  = co * y0 - si * x0;

         X  += INCX;
         Y  += INCY;
      }
   }
/*
 * End of ATL_srefrot
 */
}
