/*
 *	VME Linux/m68k Loader
 *
 *	(c) Copyright 1997 by Nick Holgate
 *
 *	This file is subject to the terms and conditions of the GNU General Public
 *	License.  See the file COPYING for more details.
 */

/*--------------------------------------------------------------------------*/

#include "loader.h"
#include "bvmbug.h"

/*--------------------------------------------------------------------------*/
/* Perform architecture specific initialisations.
 *
 * Note: malloc() heap has not yet been initialised.
 */

void
loader_init
(	CALLREGS	*regs
)
{
	put_str("BVME4000/6000 Linux Loader V" VERNUMB "\n\n");
}

/*--------------------------------------------------------------------------*/
/* Print character.
 */

void
put_char
(	const int	c
)
{
	BVMBug_putchar(c);
}

/*--------------------------------------------------------------------------*/

unsigned long
get_time
(	void
)
{	unsigned long	date;

	BVMBug_gettime(NULL, &date);
	return date;
}

/*--------------------------------------------------------------------------*/
/* Wait for and return character from keyboard.
 */

int
get_char
(	unsigned long	timeout		/* maximum time to wait for character		*/
)
{	int				c;

	if (timeout)
	{
		/* get timeout second */
		timeout += get_time();

		while ((c = BVMBug_getchar_nowait()) == -1)
		{
			/* check for timeout */
			if (get_time() > timeout)
			{
				/* timeout */
				break;
			}
		}

		return c;
	}

	return BVMBug_getchar();
}

/*--------------------------------------------------------------------------*/

int
disk_read
(	void			*buf,
	unsigned long	sector,
	unsigned long	extent
)
{	int				status;

	if ((status = BVMBug_disk_read(buf, sector, extent)) != 0)
	{
		printf("\nREAD FAILED: status=%d\n", status);
		return -1;
	}

	return 0;
}

/*--------------------------------------------------------------------------*/

unsigned long
get_compat_booti_version
(	void
)
{
	return COMPAT_BVME6000_BOOTI_VERSION;
}

/*--------------------------------------------------------------------------*/

unsigned long
get_booti_version
(	void
)
{
	return BVME6000_BOOTI_VERSION;
}

/*--------------------------------------------------------------------------*/

unsigned long
get_compat_machtype
(	void
)
{
	return COMPAT_MACH_BVME6000;
}

/*--------------------------------------------------------------------------*/

unsigned long
get_machtype
(	void
)
{
	return MACH_BVME6000;
}

/*--------------------------------------------------------------------------*/

int
can_do_symbols
(	void
)
{
	return TRUE;
}

/*--------------------------------------------------------------------------*/

void
clear_symbols
(	void
)
{
	BVMBug_delete_symbol(NULL);
}

/*--------------------------------------------------------------------------*/

int
add_symbol
(	char			*data
)
{	unsigned long	value;
	char			*p;
	int				type;

	/* skip white space */
	while (*data && *data < '!')
		data++;

	/* find end of value */
	for (p = data; *p >= '!'; p++)
		;

	/* premature end of line */
	if (*p == '\0') return 0;

	/* null terminate value */
	*p++ = '\0';

	/* evaluate */
	if (BVMBug_atoi(data, 16, &value) != 0)
	{
		return 0;
	}

	/* skip white space */
	while (*p && *p < '!')
		p++;

	/* premature end of line */
	if (*p == '\0') return 0;

	/* get symbol type */
	type = *p++;	

	/* premature end of line */
	if (*p == '\0') return 0;

	/* skip white space */
	while (*p && *p < '!')
		p++;

	/* premature end of line */
	if (*p == '\0') return 0;

	/* remember start of symbol name */
	data = p;

	/* find end of symbol name */
	while (*p >= '!')
		p++;

	/* null terminate symbol name */
	*p = '\0';

	/* get symbol type */
	type = (type == 'T' || type == 't') ? 0 : 1;

	if (BVMBug_define_symbol(data, value, type) == OUT_OF_MEMORY)
	{
		printf("\nBVMBug symbol table full\n");
		return -1;
	}

	return 0;
}

/*--------------------------------------------------------------------------*/
/*
 *	This assembler code is copied into the base of the stack, and then executed.
 *	It copies the kernel and ramdisk images to their final resting places.
 *
 *	It is called with:
 *
 *      a0 = address of this code (very top of memory)
 *      a1 = kernel destination address (low memory 0x1000)
 *		a2 = kernel source address
 *		a3 = ramdisk destination address (just below this code)
 *		a4 = ramdisk source address
 *		d0 = kernel size + size of bootinfo data rounded up next multiple of 4
 *		d1 = ramdisk size rounded to next multiple of 1K
 *      d2 = non-zero if BVMBug should be called
 */

__asm(".text\n"
__ALIGN_STR "\n"
".globl " SYMBOL_NAME_STR(startcode_beg) ";\n"
".globl " SYMBOL_NAME_STR(startcode_end) ";\n"
SYMBOL_NAME_STR(startcode_beg) ":
								| /* copy the ramdisk to the top of memory */
								| /* (from back to front) */
		addl	%d1,%a4			| src   = (u_long *) (rd_src + rd_size);
		movel	%a3,%a5
		addl	%d1,%a5			| dest  = (u_long *) (rd_dest + rd_size);

		bras	2f				| do
1:		movel	-(%a4),-(%a5)	|   *--dest = *--src;
2:		cmpl	%a3,%a5			| while (dest > ramdisk_dest)
		jgt		1b				| 

								| /* copy kernel text and data, and bootinfo */
		movel	%a2,%a4			| limit = (u_long *) (kernel_src + kernel_size);
		addl	%d0,%a4
		movel	%a1,%a5			| dest  = (u_long *) kernel_dest

		bras	2f				| do
1:		movel	(%a2)+,(%a5)+	|  *dest++ = *src++;
2:		cmpl	%a4,%a2			| while (src < limit)
		jlt		1b				|

		tstl	%d2				| call BVMbug?
		jeq		1f				| branch if not

		movel	%a1,-(%a7)		| return to start of kernel (kernel_dest)
		movel	#0xf8000404,%a1	| get BVMBug entry point
		moveq	#22,%d0			| BVMBug enter

1:		jmp		(%a1)			| start kernel or enter BVMBug
"
SYMBOL_NAME_STR(startcode_end) ":
");

/*--------------------------------------------------------------------------*/

void
print_model
(	void
)
{
	printf("BVME%d00", v->cpu_type);
}

/*--------------------------------------------------------------------------*/

int
add_vme_bootinfo
(	void
)
{	unsigned long	vme_type;

	vme_type = (v->cpu_type == 40)
			 ? VME_TYPE_BVME4000
			 : VME_TYPE_BVME6000
			 ;

	if (!add_bi_record(BI_VME_TYPE, sizeof(vme_type), &vme_type))
		return FALSE;

	return TRUE;
}

/*--------------------------------------------------------------------------*/
/* Enter ROM debug monitor
 */

void
call_bug
(	void
)
{
	BVMBug_enter();
}

/*-----------------------------< end of file >------------------------------*/
