
/*
 * tcl_blob.cc
 * Copyright (C) 1999 by John Heidemann
 * $Id: tcl_blob.cc,v 1.34 1999/11/18 05:50:57 johnh Exp $
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License,
 * version 2, as published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place, Suite 330, Boston, MA 02111-1307, USA.
 *
 */

#ifdef USE_TCL_BLOB

#include <stdlib.h>  // atoi
#include <tcl.h>
#include <iostream.h>

#include <tcl.h>
#include <tk.h>

#include "main.hh"
#include "tcl_blob.hh"
#include "process_list.hh"

#include <algorithm>
#include <string>


#include "gen.hh"


// instantiate me
blob *
blob::create_real_blob(process_view *pv)
{
	ENTRY_TRACE(__FILE__,__LINE__);
	return (blob*) new tcl_blob(pv);
}

void
blob::splash(const char *msg)
{
	tcl_blob::tcl_splash(msg);
}





// static stuff
Tcl_Interp *tcl_blob::interp_ = NULL;
char *tcl_blob::canvas_ = NULL;
tcl_blob::tcl_blobs_t tcl_blob::tcl_blobs_;
char tcl_blob::result_buf_[TCL_BUF_LEN];

int tcl_blob::pixel_size_ = 1;

static process_list pl;


// static
int
tcl_blob::lava_tick_proc(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[])
{
	ENTRY_TRACE(__FILE__,__LINE__);
	if (argc != 1) {
		Tcl_SetResult(interp, "wrong # args: should be \"lava_tick_\"", NULL);
		return TCL_ERROR;
	};
	pl.scan();
	blob::flush_drawings();
	return TCL_OK;
}

// static
int
tcl_blob::lava_dump_proc(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[])
{
	ENTRY_TRACE(__FILE__,__LINE__);
	pl.dump();
	return TCL_OK;
}

// static
int
tcl_blob::lava_resize_proc(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[])
{
	ENTRY_TRACE(__FILE__,__LINE__);
	if (argc != 3) {
		Tcl_SetResult(interp, "wrong # args: should be \"lava_resize w h\"", NULL);
		return TCL_ERROR;
	};
	int w = atoi(argv[1]);
	int h = atoi(argv[2]);
	resize_window(w, h);
	double w_scale = x_scale_, h_scale = y_scale_;
	if (!vertical_)
		swap(w_scale, h_scale);
	snprintf(result_buf_, TCL_BUF_LEN, "%f %f",  w_scale, h_scale);
	Tcl_SetResult(interp, result_buf_, NULL);
	return TCL_OK;
}


// static
int
tcl_blob::lava_version_proc(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[])
{
	ENTRY_TRACE(__FILE__,__LINE__);
	strncpy(result_buf_, VERSION, TCL_BUF_LEN);
	Tcl_SetResult(interp, result_buf_, NULL);
	return TCL_OK;
}

// static
int
tcl_blob::lava_default_resources_proc(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[])
{
	ENTRY_TRACE(__FILE__,__LINE__);
	strncpy(result_buf_, default_resources.c_str(), TCL_BUF_LEN);
	Tcl_SetResult(interp, result_buf_, NULL);
	return TCL_OK;
}


// xxx: probably a better way to do this using bind2nd or something,
// but that requires too many brain cells.
struct blobs_find_id_p : public unary_function<tcl_blob *,bool> {
	blobs_find_id_p(int id) : id_(id) {};
	int id_;
	bool operator()(tcl_blob *p) { return p->id_ == id_ || p->did_ == id_; };
};

// static
int
tcl_blob::lava_id_to_info_proc(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[])
{
	ENTRY_TRACE(__FILE__,__LINE__);
	if (argc != 2) {
		Tcl_SetResult(interp, "wrong # args: should be \"lava_id_to_info id\"", NULL);
		return TCL_ERROR;
	};
	int id = atoi(argv[1]);
	// find it
	tcl_blobs_t::iterator result = find_if(tcl_blobs_.begin(), tcl_blobs_.end(), blobs_find_id_p(id));
	if (result == tcl_blobs_.end()) {
		Tcl_SetResult(interp, "cannot find id", NULL);
		return TCL_ERROR;
	};
	// copy result
	(*result)->info_to_buf(result_buf_, TCL_BUF_LEN);
	Tcl_SetResult(interp, result_buf_, NULL);
	return TCL_OK;
}

// static
int
tcl_blob::lava_menu_proc(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[])
{
	ENTRY_TRACE(__FILE__,__LINE__);
	if (argc != 3) {
		Tcl_SetResult(interp, "wrong # args: should be \"lava_menu KEY VALUE\"", NULL);
		return TCL_ERROR;
	};
	char *key = argv[1];
	char *value = argv[2];
	if (strcmp(key, "who") == 0) {
		filter_style = (strcmp(value, "me") == 0 ? FILTER_BY_UID :
				strcmp(value, "everyone") == 0 ? FILTER_NOTHING :
				FILTER_BY_PID );
		if (filter_style == FILTER_BY_PID)
			filter_good_pid = atoi(value);
	} else if (strcmp(key, "what") == 0) {
		vm_style_enum style =
			strcmp(value, "virtual") == 0 ? VIRTUAL_MEM :
			strcmp(value, "physical") == 0 ? PHYSICAL_MEM :
			BOTH_MEM;
		report_vm.tick_set((vm_style_enum)style);
	} else if (strcmp(key, "how") == 0) {
		if (strcmp(value, "1") == 0 ||
		    strcmp(value, "0") == 0) {
			allow_autosize = (strcmp(value, "1") == 0);
		} else if (strcmp(value, "shrink") == 0) {
			if (allow_autosize)
				process_view::mem_target_adjust(-1);
			else process_view::mem_adjust(1);
		} else if (strcmp(value, "grow") == 0) {
			if (allow_autosize)
				process_view::mem_target_adjust(1);
			else process_view::mem_adjust(-1);
		} else {
			Tcl_SetResult(interp, "unknown VALUE for how", NULL);
			return TCL_ERROR;
		};
	} else if (strcmp(key, "debug") == 0) {
		lava_debug = (strcmp(value, "1") == 0);
	} else {
		Tcl_SetResult(interp, "unknown KEY", NULL);
		return TCL_ERROR;
	};
	return TCL_OK;
}

// static
void
tcl_blob::init()
{
	ENTRY_TRACE(__FILE__,__LINE__);
	// const char* appname = "lavaps";
	interp_ = Tcl_CreateInterp();
	if (Tcl_Init(interp_) == TCL_ERROR)
		die(Tcl_GetStringResult(interp_));
	Tk_Window tk = NULL;
	if (Tk_Init(interp_) == TCL_OK)
		tk = Tk_MainWindow(interp_);
	if (tk == NULL)
		die("cannot start tk");

	/*
	 * add our new commands
	 */
	(void) Tcl_CreateCommand(interp_, "lava_tick", lava_tick_proc, NULL, NULL);
	(void) Tcl_CreateCommand(interp_, "lava_id_to_info", lava_id_to_info_proc, NULL, NULL);
	(void) Tcl_CreateCommand(interp_, "lava_menu", lava_menu_proc, NULL, NULL);
	(void) Tcl_CreateCommand(interp_, "lava_resize", lava_resize_proc, NULL, NULL);
	(void) Tcl_CreateCommand(interp_, "lava_dump", lava_dump_proc, NULL, NULL);
	(void) Tcl_CreateCommand(interp_, "lava_version", lava_version_proc, NULL, NULL);
	(void) Tcl_CreateCommand(interp_, "lava_default_resources", lava_default_resources_proc, NULL, NULL);
	// (void) Tcl_CreateCommand(interp_, "lava_move", lava_move_proc, NULL, NULL);
	// (void) Tcl_CreateCommand(interp_, "lava_grow", lava_grow_proc, NULL, NULL);
	// (void) Tcl_CreateCommand(interp_, "lava_print", lava_print_proc, NULL, NULL);
	// (void) Tcl_CreateCommand(interp_, "lava_reverse", lava_reverse_proc, NULL, NULL);

#if 0
	if (!b) {
		b = new tcl_blob(10, interp_, ".c");
		cout << "b created\n";
	};
#endif /* 0 */

	/*
	 * source some files
	 */
	if (TCL_OK != Tcl_Eval(interp_, (char *)lava_code)) {
		cerr << "error loading bootstrap: " << Tcl_GetStringResult(interp_) << endl;
		exit(1);
	};
	if (TCL_OK != Tcl_Eval(interp_, "main")) {
		cerr << "error evaling main: " << Tcl_GetStringResult(interp_) << endl;
		exit(1);
	};
	canvas_ = ".c";
}


// static
void
tcl_blob::mainloop()
{
	ENTRY_TRACE(__FILE__,__LINE__);
	Tk_MainLoop();
}


tcl_blob::tcl_blob(process_view *pv) :
	blob(pv), id_(0), did_(0)
{
	ENTRY_TRACE(__FILE__,__LINE__);
	// xxx: assume blob initializer called first!
	tcl_blobs_.push_front(this);
	me_in_blobs_ = tcl_blobs_.begin();
	assert(me_in_blobs_ != tcl_blobs_.end());
	bd_color_ = "white";
	light_color_ = "black";
	dark_color_ = "grey";
	schedule_redraw();
}

tcl_blob::~tcl_blob()
{
	ENTRY_TRACE(__FILE__,__LINE__);
	char *bufp = result_buf_;

	tcl_blobs_.erase(me_in_blobs_);

	if (id_) {
		sprintf(bufp, "%s delete %d\n", canvas_, id_);   bufp += strlen(bufp);
	};
	if (did_) {
		sprintf(bufp, "%s delete %d\n", canvas_, did_);   bufp += strlen(bufp);
	};
	if (TCL_OK != Tcl_Eval(interp_, result_buf_)) {
		cerr << result_buf_ << endl << "tcl_blob::~tcl_blob: eval error\n";;
		exit (1);
	};
	id_ = 0;
}


char *
tcl_blob::list_point(char *bufp, char *eobuf, int x, int y)
{
	int ox = int(x * x_scale_),
		oy = int(y * y_scale_);
	if (!vertical_)
		swap(ox, oy);
	snprintf(bufp, eobuf-bufp, "%d %d ", ox, oy);
	bufp += strlen(bufp);
	return bufp;
}

char *
tcl_blob::list_points(char *bufp, char *eobuf, int pct, int offset)
{
	int i, x, y;

	// scratch space
	static int *ranges = NULL, *midpoints = NULL;
	static int len = 0;
	if (len < num_) {
		delete[] ranges;
		delete[] midpoints;
		ranges = new int[num_];
		midpoints = new int[num_];
		len = num_;
	};

	for (i = 0, x = x_; i < num_; i++, x += x_step_) {
		if (pct != 100) {
			ranges[i] = y_highs_[i] - y_lows_[i];
			midpoints[i] = y_lows_[i] + ranges[i] / 2;
			y = midpoints[i] - ranges[i] * pct / 200;
			if (y <= y_lows_[i])
				y += pixel_size_;
		} else {
			y = y_lows_[i];
		};
		bufp = list_point(bufp, eobuf, x, y);
	};
	for (; --i >= 0; ) {
		if (pct != 100) {
			y = midpoints[i] + ranges[i] * pct / 200;
			if (y >= y_highs_[i])
				y -= pixel_size_;
		} else {
			y = y_highs_[i];
		};
		x -= x_step_;
		bufp = list_point(bufp, eobuf, x, y);
	};
	// must always have 3 points
	if (num_ == 1) {
		// make one up
		x = x_ + x_step_ / 2 * (right_tending_ ? 1 : -1);
		int y = (y_highs_[0] - y_lows_[0]) / 2 + y_lows_[0];
		bufp = list_point(bufp, eobuf, x, y);
	};
	return bufp;
}

void
tcl_blob::redraw()
{
	ENTRY_TRACE(__FILE__,__LINE__);
	char buf[LONG_TCL_BUF_LEN], *bufp = buf;
	char *eobuf = &buf[LONG_TCL_BUF_LEN];  // betcha no compiler optimizes this away :-(
	if (id_) {
		snprintf(bufp, eobuf-bufp, "%s delete %d\n", canvas_, id_);   bufp += strlen(bufp);
	};
	if (did_) {
		snprintf(bufp, eobuf-bufp, "%s delete %d\n", canvas_, did_);   bufp += strlen(bufp);
	};
	snprintf(bufp, eobuf-bufp, "set id [%s create polygon ", canvas_);   bufp += strlen(bufp);
	bufp = list_points(bufp, eobuf);
	snprintf(bufp, eobuf-bufp, "-fill %s -outline %s -smooth 1 -tag p]\n", light_color_.c_str(), bd_color_.c_str());   bufp += strlen(bufp);

	//
	// do shading
	//
	if (dark_percentage_) {
		snprintf(bufp, eobuf-bufp, "set did [%s create polygon ", canvas_);   bufp += strlen(bufp);
		bufp = list_points(bufp, eobuf, dark_percentage_, 50);
		snprintf(bufp, eobuf-bufp, "-fill %s -smooth 1 -tag p]\n", dark_color_.c_str());   bufp += strlen(bufp);
		// snprintf(bufp, eobuf-bufp, "-fill %s -outline %s -smooth 1 -tag p]\n", dark_color_.c_str(), bd_color_.c_str());   bufp += strlen(bufp);
	};

	blob *b = superior();
	int superior_id = b ? ((tcl_blob*)b)->id_ : 0;
	if (superior_id)
		snprintf(bufp, eobuf-bufp, "%s lower $id %d\n", canvas_, superior_id);   bufp += strlen(bufp);
	if (dark_percentage_) {
		if (superior_id)
			snprintf(bufp, eobuf-bufp, "%s lower $did %d\n", canvas_, superior_id);   bufp += strlen(bufp);
		snprintf(bufp, eobuf-bufp, "return \"{$id $did}\"\n");   bufp += strlen(bufp);
	} else {
		snprintf(bufp, eobuf-bufp, "return $id\n");   bufp += strlen(bufp);
	};

	if (eobuf - bufp < 5) {
		int cont = 0;
		cerr << "tcl_blob::redraw: buffer overrun.\n";
		if (!cont)
			exit (1);
	};
	int e;
	if ((e = Tcl_Eval(interp_, buf)) != TCL_RETURN) {
		cerr << Tcl_GetStringResult(interp_) << endl << "tcl_blob::redraw: eval error\nscript:\n" << buf << endl << endl;;
		exit (1);
	};
	if (dark_percentage_) {
		char *cp = Tcl_GetStringResult(interp_);
		assert(cp != NULL);
		assert(*cp != 0);
		id_ = atoi(cp+1);
		cp = strchr(cp, ' ');
		assert(cp != NULL);
		did_ = atoi(cp);
		
	} else {
		id_ = atoi(Tcl_GetStringResult(interp_));
		did_ = 0;
	};
}

void
tcl_blob::set_hsb(double h, double s, double b)
{
	ENTRY_TRACE(__FILE__,__LINE__);
	blob::set_hsb(h, s, b);

	const char *cs = hsb_to_rgbs(H_, S_, B_);
	light_color_ = cs;
	free((void*)cs);

	cs = hsb_to_rgbs(H_, S_, 1.0 - B_);
	bd_color_ = cs;
	free((void*)cs);

	/*
	 * xxx: this choice (of what the dar color should be)
	 * probably beongs in process_view.
	 * Constants:
	 * 0.0625 is too subtle.
	 * 0.1 is too much.
	 */
	double dark_h = H_ + 0.07;
	if (dark_h > 1.0)
		dark_h -= 1.0;
	double dark_s = S_ + 0.1;
	if (dark_s > 1.0)
		dark_s = 1.0;
	cs = hsb_to_rgbs(dark_h, dark_s, B_);
	dark_color_ = cs;
	free((void*)cs);

	schedule_redraw();
}

void
tcl_blob::tcl_splash(const char *msg)
{
	// cout << msg << endl;
	char buf[TCL_BUF_LEN];
	snprintf(buf, TCL_BUF_LEN, "splash_refresh {%s}", msg);
	if (TCL_OK != Tcl_Eval(interp_, buf)) {
		cerr << "tcl_blob::tcl_splash: eval error: " << Tcl_GetStringResult(interp_) << endl;
	};
}

#endif /* USE_TCL_BLOB */
