/*
   prompt.c : screen read and write

   Part of fbgetty 
   Copyright (C) 1999 Yann DRONEAUD (lch@multimania.com). 

   fbgetty is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   fbgetty is distributed in the hope that it will be useful, but 
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

 */

#include "global.h"
#include "errors.h"
#include "options.h"

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

#if TIME_WITH_SYS_TIME
# include <sys/time.h>
# include <time.h>
#else
# if HAVE_SYS_TIME_H
#  include <sys/time.h>
# else
#  include <time.h>
# endif
# include <time.h>
#endif

#include <ctype.h>
#include <errno.h>
#include <utmp.h>
#include <sys/param.h>
#include <sys/utsname.h>
#include <sys/mman.h>
#include <sys/ioctl.h>
#include <sys/wait.h>
#include <linux/vt.h>
#include <linux/fb.h>


typedef struct sysinfos
{
  /* some information about this host */
  struct utsname *uts;
  /* the hostname */ 
  char *hostname;
  /* current time */ 
  time_t *time;
} sysinfos_t;

struct sysinfos *sysinfos;

void do_prompt (void);
void print_issue(void);
void refresh_screen(void);
void output_escape (int c);
void output_special (int c);

/* Redraw the screen */
void refresh_screen(void)
{

#ifdef USE_FRAME_BUFFER
  struct fb_var_screeninfo *fb_var;
  struct fb_fix_screeninfo *fb_fix;
  unsigned char *framebuffer;
#endif

#ifdef USE_FRAME_BUFFER
  if (fgoptions->fb_device != NULL)
    {
      fb_fix = (struct fb_fix_screeninfo *) malloc (sizeof (struct fb_fix_screeninfo));
      if (fb_fix == NULL)
	fatal_error("Can't allocate memory for framebuffer infos");
      
      if (ioctl(fgoptions->fb_fd, FBIOGET_FSCREENINFO, fb_fix) == -1)
	error("ioctl FBIOGET_FSCREENINFO: %m");

      fb_var = (struct fb_var_screeninfo *) malloc (sizeof (struct fb_var_screeninfo));
      if (fb_var == NULL)
	fatal_error("Can't allocate memory for framebuffer infos");
      
      if (ioctl(fgoptions->fb_fd, FBIOGET_VSCREENINFO, fb_var) == -1)
	error("ioctl FBIOGET_VSCREENINFO: %m");
      
#ifdef FB_GETTY_DEBUG
      error("- fb info:");
      error("- X offset vis : %d", fb_var->xoffset);    
      error("- Y offset vis : %d", fb_var->yoffset);
      error("- X virtual : %d", fb_var->xres_virtual);
      error("- Y virtual : %d", fb_var->yres_virtual);
#endif
      
      /* map of framebuffer device */
      framebuffer = (unsigned char *) mmap(NULL, \
					   fb_fix->smem_len, \
					   PROT_READ | PROT_WRITE, \
					   MAP_SHARED, \
					   fgoptions->fb_fd, \
					   (off_t) NULL);
      if (framebuffer == MAP_FAILED)
	fatal_error("cannot mmap frame buffer");
      
#ifdef FB_GETTY_DEBUG
      error("- framebuffer mapped");
#endif

    }
#endif
  
  sysinfos = (struct sysinfos *) malloc(sizeof(struct sysinfos));
  if (sysinfos == NULL)
    fatal_error("Can't allocate memory for system informations");
  
  sysinfos->uts      = (struct utsname *) malloc(sizeof(struct utsname));
  sysinfos->hostname = (char *)           malloc ((MAXHOSTNAMELEN + 1) * sizeof (char));
  sysinfos->time     = (time_t *)         malloc (sizeof(time_t));
  
  if ((sysinfos->uts == NULL) || (sysinfos->hostname == NULL) || (sysinfos->time == NULL))
    fatal_error("Can't allocate memory for system informations");
  
  uname (sysinfos->uts);
  time (sysinfos->time);
  gethostname (sysinfos->hostname, MAXHOSTNAMELEN);
  
  /* clear screen (Linux specific) */
  if (fgoptions->screen_clear != FALSE) 
    printf("\033c");  
  
  print_issue();

  do_prompt();
 
  free(sysinfos->time);
  free(sysinfos->hostname);
  free(sysinfos->uts);
  free(sysinfos);


#ifdef USE_FRAME_BUFFER
    if (fgoptions->fb_device != NULL)
      {
	munmap(framebuffer,fb_fix->smem_len);
	free(fb_var); /* missing in 0.1.1 and past */
	free(fb_fix);

#ifdef FB_GETTY_DEBUG
	error("- framebuffer unmapped\n");
#endif  
      }
#endif /* USE_FRAME_BUFFER */


  return;

}




void print_issue()
{
  FILE *fd;
  int c;
  unsigned int i;
  pid_t pid;
  char var_name[255];
  char *var_value;

  if (fgoptions->issue_file != NULL)
    {
      if ((fd = fopen (fgoptions->issue_file, "r")) != NULL ) 
	{
	  while ((c = getc (fd)) != EOF) 
	    {
	      switch(c)
		{
		case '\\':
		  output_escape(getc(fd));
		  break;
		case '%':
		  output_special(getc(fd));
		  break;
		case '@':
		  /* Not supported yet : long special will be used to include picture */
		  putchar(c);
		  break;
		case '$':
		  i = 0;
		  do 
		    {
		      c = getc (fd);
		      if (c != EOF)
			{
			  if (isprint(c) && (( c == '_') || !ispunct(c)))
			      var_name[i++] = c;
			  else
			    {
			      ungetc(c,fd);
			      c = EOF;
			    }
			}
		    }
		  while ( c != EOF);
		  var_name[i]='\0';
		  if ((strlen(var_name) > 0) && (( var_value = getenv(var_name)) != NULL))
		    printf("%s",var_value);
		  break;
		  
		case '&': /* run a program &name */
		  i = 0;
		  do 
		    {
		      c = getc (fd);
		      if (c != EOF)
			{
			  if (isprint(c) && (( c == '_') || (c == '/') || !ispunct(c)))
			    {
			      var_name[i++] = c;
			    }
			  else
			    {
			      ungetc(c,fd);
			      c = EOF;
			    }
			}
		    }
		  while (c != EOF);
		  var_name[i]='\0';
		  if (strlen(var_name) > 0)
		    {
		      pid = fork();
		      switch(pid)
			{
			case 0: /* child code */
			  execl(var_name,var_name,NULL);
			  error("exec %s: %m",var_name);
			  exit(0);
			  break;
			case -1: /* error */
			  error("fork %s:%m",var_name);
			  break;
			default: /* wait child */
			  /* FIXME: add an alarm */
			  wait(NULL);
			  break;
			}
		    }
		  break;
		  
		default:
		  putchar (c);
		  break;
		  
		}
	    }
	  fclose (fd);
	}
      else
	{
	  error("Can't open issue file (%s): %m",fgoptions->issue_file);
	  fgoptions->issue_file = NULL;
	}
    }
  fflush (stdout);
}

void do_prompt ()
{
  unsigned int i;

  if (fgoptions->login_prompt != NULL)
    {
      for (i = 0; i < strlen(fgoptions->login_prompt) ; i++)
	{
	  switch(fgoptions->login_prompt[i])
	    {
	    case '\\':
	      output_escape(fgoptions->login_prompt[++i]);
	      break;
	    case '%':
	      output_special(fgoptions->login_prompt[++i]);
	      break;
	    default:
	      putchar (fgoptions->login_prompt[i]);
	      break;
	    }
	}
    }
  
  if ((fgoptions->login_name != NULL) && \
      (fgoptions->login_name_valid != FALSE) && \
      (fgoptions->login_hide == FALSE))
    printf("%s",fgoptions->login_name);
  fflush (stdout);
}


void output_special (int c)
{
  char *buff;
struct tm *tm;   

  switch (c) 
    {
    case 's':
      printf ("%s", sysinfos->uts->sysname);
      break;
    case 'n':
      printf ("%s", sysinfos->uts->nodename);
      break;
    case 'r':
      printf ("%s", sysinfos->uts->release);
      break;
    case 'v':
      printf ("%s", sysinfos->uts->version);
      break;
    case 'm':
      printf ("%s", sysinfos->uts->machine);
      break;
    case 'o':
#ifdef __USE_GNU
      printf ("%s", sysinfos->uts->domainname);
#else
      printf ("%s", sysinfos->uts->__domainname);
#endif
      break;
    case 'd':
    case 't':
      {
        
	buff = (char *) malloc (32);
        tm = localtime (sysinfos->time);
	strftime (buff, 32 , c == 'd' ? "%a %b %d %Y" : "%X", tm);
	printf ("%s",buff);
	free(buff);
	break;
      }
      
    case 'l':
      printf ("%s", fgoptions->tty_device);
      break;
      
#ifdef USE_FRAME_BUFFER
    case 'f':
      printf ("%s", fgoptions->fb_device);
      break;
#endif
      
    case 'u':
    case 'U':
      {
	int users = 0;
	struct utmp *ut;
	setutent ();
	while ((ut = getutent ()))
	  if (ut->ut_type == USER_PROCESS)
	    users++;
	endutent ();
	printf ("%d", users);
	if (c == 'U')
	  printf (" user%s", users == 1 ? "" : "s");
	break;
      }
    default:
      putchar (c);
      break;
    }
}

void output_escape (int c)
{
  switch (c)
    {
    case 'a':  /* WARNING: the meaning of \a is changed is -traditional is used */
      c = '\a'; 
      break;
    case 'v': 
      c = '\v'; 
      break;
    case 'b': 
      c = '\b'; 
      break;
    case 'e': /* ESC -- non-ANSI */
    case 'E':         
      c = '\033'; 
      break;
    case 'f': 
      c = '\f'; 
      break;
    case 'n': 
      c = '\n'; 
      break;
    case 'r': 
      c = '\r'; 
      break;
    case 't': 
      c = '\t'; 
      break;
    }
  putchar(c);
}
 

/* get_login_name */
void get_login_name (void)
{
  int c;
  int login_length;
  char *new_ptr; 
#ifdef FB_GETTY_DEBUG
  int index;
#endif

  ioctl (STDOUT_FILENO, TCFLSH, 0);   /* flush pending input */

  fgoptions->login_name_valid = FALSE;

  if (fgoptions->login_name != NULL)
    fgoptions->login_name = (char *) realloc (fgoptions->login_name,sizeof(char));
  else
    fgoptions->login_name = (char *) malloc (sizeof(char));

  login_length = 0;
  fgoptions->login_name[0] = '\0';

#ifdef FB_GETTY_DEBUG
  error("reading login");
#endif

  do
    {
      /* read a char */
      c = fgetc(stdin); 

#ifdef FB_GETTY_DEBUG
      error("* char read: %c (%x)",c,c);
#endif

      if (isprint(c) && !iscntrl (c))
 	{
	  /* get more memory for login name */
          new_ptr = (char *) realloc (fgoptions->login_name, (++login_length + 1) * sizeof(char));
	  if (new_ptr == NULL)
	    {
	      error("can't allocate more memory for login name");
	      c = EOF;
	    }
	  else
	    {
	      fgoptions->login_name = new_ptr;
	      fgoptions->login_name[login_length - 1] = c;
	      fgoptions->login_name[login_length] = '\0';
	    }
	}
	
    }
  while ((c != '\r') && (c != '\n') && (c != EOF));

  fgoptions->login_name_valid=TRUE;

#ifdef FB_GETTY_DEBUG
  error("* login read: %s",fgoptions->login_name);
  index = 0x00;
  while (fgoptions->login_name[index] != 0x00)
    {
      error("* '%c' 0x%02x",fgoptions->login_name[index],fgoptions->login_name[index]);
      index ++;
    }
  error("login read");
#endif

  return;
}






