/* winspector.c - window attribute inspector
 * 
 *  WindowMaker window manager
 * 
 *  Copyright (c) 1997 Alfredo K. Kojima
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "wconfig.h"

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include "generic/wwmlib.h"
#include "WindowMaker.h"
#include "screen.h"
#include "wcore.h"
#include "framewin.h"
#include "window.h"
#include "workspace.h"
#include "funcs.h"
#include "defaults.h"
#include "dialog.h"
#include "icon.h"
#include "stacking.h"
#include "application.h"
#include "appicon.h"
#include "actions.h"
#include "winspector.h"


#include <proplist.h>

extern WPreferences wPreferences;

static InspectorPanel *panelList=NULL;

static proplist_t ANoTitlebar = NULL;
static proplist_t ANoResizebar;
static proplist_t ANotMiniaturizable;
static proplist_t ANotClosable;
static proplist_t ANoHideOthers;
static proplist_t ANoMouseBindings;
static proplist_t ANoKeyBindings;
static proplist_t ANoAppIcon;
static proplist_t AKeepOnTop;
static proplist_t AOmnipresent;
static proplist_t ASkipWindowList;
static proplist_t AKeepInsideScreen;
static proplist_t AUnfocusable;
static proplist_t AAlwaysUserIcon;

static proplist_t AStartWorkspace;

static proplist_t AIcon;

static proplist_t AnyWindow;
static proplist_t EmptyString;
static proplist_t Yes, No;



#define PWIDTH	260
#define PHEIGHT	350

#define COUNTARG(table)		(sizeof(table)/sizeof(WMArgument))


static WMArgument panelArgs[] = {
    {WARG_WIDTH,		(WMValue)PWIDTH},
    {WARG_HEIGHT,		(WMValue)PHEIGHT}
};

static WMArgument buttonArgs[] = {
    {WARG_X,			(WMValue)0},
    {WARG_CAPTION,		(WMValue)NULL},
    {WARG_Y,			(WMValue)310},
    {WARG_WIDTH,		(WMValue)62}
};

static WMArgument pageBtnArgs[] = {
    {WARG_X,			(WMValue)0},
    {WARG_CAPTION,		(WMValue)NULL},
    {WARG_Y,			(WMValue)10},
    {WARG_WIDTH,		(WMValue)30},
    {WARG_HEIGHT,		(WMValue)30},
    {WARG_TAG,			(WMValue)1}
};

static WMArgument specFrmArgs[] = {
    {WARG_CAPTION,		(WMValue)NULL},
    {WARG_X,			(WMValue)40},
    {WARG_Y, 			(WMValue)65},
    {WARG_WIDTH,		(WMValue)180},
    {WARG_HEIGHT,		(WMValue)105}
};

static WMArgument specLblArgs[] = {
    {WARG_CAPTION,		(WMValue)NULL},
    {WARG_ALIGNMENT,		(WMValue)WACenter},
    {WARG_X,			(WMValue)20},
    {WARG_Y,			(WMValue)170},
    {WARG_WIDTH,		(WMValue)210},
    {WARG_HEIGHT,		(WMValue)100}
};

static WMArgument specRadArgs[] = {
    {WARG_Y,			(WMValue)0},
    {WARG_CAPTION, 		(WMValue)NULL},
    {WARG_SELECTED,		(WMValue)1},
    {WARG_X, 			(WMValue)10},
    {WARG_WIDTH,		(WMValue)160},
    {WARG_TAG,			(WMValue)1}
};


static WMArgument attrFrmArgs[] = {
    {WARG_CAPTION,		(WMValue)NULL},
    {WARG_X,			(WMValue)20},
    {WARG_Y, 			(WMValue)65},
    {WARG_WIDTH,		(WMValue)220},
    {WARG_HEIGHT,		(WMValue)210}
};

static WMArgument attrChkArgs[] = {
    {WARG_CAPTION,		(WMValue)NULL},
    {WARG_Y, 			(WMValue)20},
    {WARG_SELECTED,		(WMValue)0},
    {WARG_X,			(WMValue)5},
    {WARG_WIDTH,		(WMValue)205},
    {WARG_HEIGHT,		(WMValue)20}
};


static WMArgument iconFrmArgs[] = {
    {WARG_CAPTION,		(WMValue)NULL},
    {WARG_Y, 			(WMValue)50},
    {WARG_X,			(WMValue)15},
    {WARG_WIDTH,		(WMValue)230},
    {WARG_HEIGHT,		(WMValue)170}
};

static WMArgument wsFrmArgs[] = {
    {WARG_CAPTION,		(WMValue)NULL},
    {WARG_Y, 			(WMValue)225},
    {WARG_X,			(WMValue)15},
    {WARG_WIDTH,		(WMValue)230},
    {WARG_HEIGHT,		(WMValue)70}
};


static WMArgument iconLblArgs[] = {
    {WARG_X,			(WMValue)130},
    {WARG_Y, 			(WMValue)30},
    {WARG_WIDTH,		(WMValue)64},
    {WARG_HEIGHT,		(WMValue)64},
    {WARG_RELIEF,		(WMValue)WRRaised},
    {WARG_IMAGE_POSITION,	(WMValue)WIPImageOnly}
};


static WMArgument updateIconBtnArgs[] = {
    {WARG_CAPTION,		(WMValue)NULL},
    {WARG_Y, 			(WMValue)30},
    {WARG_X,			(WMValue)22},
    {WARG_WIDTH,		(WMValue)70}
};

static WMArgument fileLblArgs[] = {
    {WARG_CAPTION,		(WMValue)NULL},
    {WARG_X,			(WMValue)20},
    {WARG_Y, 			(WMValue)95},
    {WARG_WIDTH,		(WMValue)104}
};

static WMArgument fileArgs[] = {
    {WARG_TEXT,			(WMValue)NULL},
    {WARG_X,			(WMValue)20},
    {WARG_Y, 			(WMValue)115},
    {WARG_WIDTH,		(WMValue)186}
};

static WMArgument alwChkArgs[] = {
    {WARG_CAPTION,		(WMValue)NULL},
    {WARG_SELECTED,		(WMValue)0},
    {WARG_X,			(WMValue)20},
    {WARG_Y, 			(WMValue)140},
    {WARG_WIDTH,		(WMValue)186}
};



static WMArgument workspaceArgs[] = {
    {WARG_TEXT,			(WMValue)NULL},
    {WARG_X,			(WMValue)30},
    {WARG_Y, 			(WMValue)40},
    {WARG_WIDTH,		(WMValue)170}
};

static WMArgument wsRad1Args[] = {
    {WARG_CAPTION,		(WMValue)NULL},
    {WARG_SELECTED,		(WMValue)0},
    {WARG_Y, 			(WMValue)15},
    {WARG_X,			(WMValue)10},
    {WARG_WIDTH,		(WMValue)180},
    {WARG_TAG,			(WMValue)1}
};

static WMArgument wsRad2Args[] = {
    {WARG_CAPTION,		(WMValue)NULL},
    {WARG_SELECTED,		(WMValue)0},
    {WARG_Y, 			(WMValue)40},
    {WARG_X,			(WMValue)10},
    {WARG_WIDTH,		(WMValue)25},
    {WARG_TAG,			(WMValue)1}
};


static void
make_keys()
{
    if (ANoTitlebar!=NULL)
	return;
    
    AIcon = PLMakeString("Icon");
    ANoTitlebar = PLMakeString("NoTitlebar");
    ANoResizebar = PLMakeString("NoResizebar");
    ANotMiniaturizable = PLMakeString("NotMiniaturizable");
    ANotClosable = PLMakeString("NotClosable");
    ANoHideOthers = PLMakeString("NoHideOthers");
    ANoMouseBindings = PLMakeString("NoMouseBindings");
    ANoKeyBindings = PLMakeString("NoKeyBindings");
    ANoAppIcon = PLMakeString("NoAppIcon");
    AKeepOnTop = PLMakeString("KeepOnTop");
    AOmnipresent = PLMakeString("Omnipresent");
    ASkipWindowList = PLMakeString("SkipWindowList");
    AKeepInsideScreen = PLMakeString("KeepInsideScreen");
    AUnfocusable = PLMakeString("Unfocusable");
    AAlwaysUserIcon = PLMakeString("AlwaysUserIcon");
    
    AStartWorkspace = PLMakeString("StartWorkspace");
    
    AnyWindow = PLMakeString("*");
    EmptyString = PLMakeString("");
    Yes = PLMakeString("Yes");
    No = PLMakeString("No");
}



static void
destroyInspector(WCoreWindow *foo, void *data, XEvent *event)
{
    InspectorPanel *panel;
    InspectorPanel *tmp;
    
    panel = panelList;
    while (panel->frame!=data)
	panel = panel->nextPtr;

    WMDestroyWidget(panel->win);
    wUnmanageWindow(panel->frame, False);

    panel->inspected->flags.inspector_open = 0;
    panel->inspected->inspector = NULL;

    if (panelList == panel) 
	panelList = panel->nextPtr;
    else {
	tmp = panelList;
	while (tmp->nextPtr!=panel) {
	    tmp = tmp->nextPtr;
	}
	tmp->nextPtr = panel->nextPtr;
    }

    free(panel);
}


void
wDestroyInspectorPanels()
{
    InspectorPanel *panel;

    while (panelList != NULL) {
        panel = panelList;
        panelList = panelList->nextPtr;
        WMDestroyWidget(panel->win);
        wUnmanageWindow(panel->frame, False);

        panel->inspected->flags.inspector_open = 0;
        panel->inspected->inspector = NULL;

        free(panel);
    }
}


static void
changePage(WMButton *button, InspectorPanel *panel)
{
    if (button == panel->pageBtn1) {
	WMMapWidget(panel->specFrm);
	WMMapWidget(panel->specLbl);
    } else if (button == panel->pageBtn2) {
	WMMapWidget(panel->attrFrm);	
    } else if (button == panel->pageBtn3) {	
	WMMapWidget(panel->moreFrm);
    } else {
	WMMapWidget(panel->iconFrm);
	WMMapWidget(panel->wsFrm);
    }
    
    if (button!=panel->pageBtn1) {
	WMUnmapWidget(panel->specFrm);
	WMUnmapWidget(panel->specLbl);
    }
    if (button!=panel->pageBtn2)
	WMUnmapWidget(panel->attrFrm);
    if (button!=panel->pageBtn3)
	WMUnmapWidget(panel->moreFrm);
    if (button!=panel->pageBtn4) {
	WMUnmapWidget(panel->iconFrm);
	WMUnmapWidget(panel->wsFrm);
    }
}


#define USE_TEXT_FIELD          1
#define UPDATE_TEXT_FIELD       2
#define REVERT_TO_DEFAULT       4


static void
showIconFor(WMScreen *scrPtr, InspectorPanel *panel,
            char *wm_instance, char *wm_class, int flags)
{
    WMPixmap *pixmap = (WMPixmap*) NULL;
    WMArgument arg;
    char *file=NULL, *path=NULL;
    char *db_icon=NULL;

    if ((flags & USE_TEXT_FIELD) != 0) {
        file = WMGetTextFromTextField(panel->fileText);
    }
    else {
        db_icon = wDefaultGetIconFile(panel->inspected->screen_ptr,
                                      wm_instance, wm_class);
        if(db_icon != NULL)
            file = strdup(db_icon);
    }
    if (file==NULL && db_icon!=NULL && (flags & REVERT_TO_DEFAULT)!=0) {
        file = strdup(db_icon);
        flags |= UPDATE_TEXT_FIELD;
    }

    if ((flags & UPDATE_TEXT_FIELD) != 0) {
        fileArgs[0].value = (WMValue)file;
        WMConfigureTextField(panel->fileText, fileArgs, COUNTARG(fileArgs));
    }

    if (file) {
        path = FindImage(wPreferences.pixmap_path, file);
        free(file);

        if (!path) {
            wMessageDialog(panel->frame->screen_ptr, _("Error"),
                           _("Could not find specified icon file"), WD_ERROR);
            return;
        }
        pixmap = WMCreatePixmapFromFile(scrPtr, path);
        free(path);

        if (!pixmap) {
            wMessageDialog(panel->frame->screen_ptr, _("Error"),
                           _("Could not open specified icon file"), WD_ERROR);
            /*
             WMRunAlertPanel(scrPtr, _("Error"),
             _("Could not open icon file"), _("OK"), NULL, NULL);
             */
            return;
        }
    }

    arg.argument = WARG_IMAGE;
    arg.value = (WMValue)pixmap;
    WMConfigureLabel(panel->iconLbl, &arg, 1);
    if (pixmap)
        WMReleasePixmap(pixmap);
}


static void
updateIcon(WMButton *button, InspectorPanel *panel)
{
    showIconFor(W_SCREEN(button), panel, NULL, NULL, USE_TEXT_FIELD);
}


#if 0
static void
defaultIcon(WMButton *button, InspectorPanel *panel)
{
    char *icon_name=NULL;
    WDDomain *db = panel->frame->screen_ptr->dwindow;
    proplist_t attr, value, icon_value;
    proplist_t dict = db->dictionary;

    showIconFor(W_SCREEN(button), panel, NULL, NULL, USE_TEXT_FIELD);

    if (!dict) {
        dict = PLMakeDictionaryFromEntries(NULL, NULL, NULL);
        if (dict) {
            db->dictionary = dict;
            value = PLMakeString(db->path);
            PLSetFilename(dict, value);
            PLRelease(value);
        }
        else
            return;
    }

    PLSetStringCmpHook(NULL);

    if ((icon_name = WMGetTextFromTextField(panel->fileText)) != NULL) {
        value = PLMakeString(icon_name);
        icon_value = PLMakeDictionaryFromEntries(AIcon, value, NULL);
        PLRelease(value);
        free(icon_name);
    }
    else
        icon_value = NULL;

    if((attr = PLGetDictionaryEntry(dict, AnyWindow)) != NULL) {
        if (PLIsDictionary(attr)) {
            if (icon_value!=NULL)
                PLMergeDictionaries(attr, icon_value);
            else
                PLRemoveDictionaryEntry(attr, AIcon);
        }
    }
    else if (icon_value!=NULL) {
        PLInsertDictionaryEntry(dict, AnyWindow, icon_value);
    }

    if(icon_value)
        PLRelease(icon_value);

    PLSave(dict, YES);
    PLSetStringCmpHook(StringCompareHook);
}
#endif


static int 
getBool(proplist_t value)
{
    char *val;

    if (!PLIsString(value)) {
        return 0;
    }
    if (!(val = PLGetString(value))) {
        return 0;
    }

    if ((val[1]=='\0' && (val[0]=='y' || val[0]=='Y' || val[0]=='T'
                          || val[0]=='t' || val[0]=='1'))
        || (strcasecmp(val, "YES")==0 || strcasecmp(val, "TRUE")==0)) {

        return 1;
    } else if ((val[1]=='\0'
                && (val[0]=='n' || val[0]=='N' || val[0]=='F'
                    || val[0]=='f' || val[0]=='0'))
               || (strcasecmp(val, "NO")==0 || strcasecmp(val, "FALSE")==0)) {

        return 0;
    } else {
        wWarning(_("can't convert \"%s\" to boolean"), val);
        return 0;
    }
}


#define UPDATE_DEFAULTS    1
#define IS_BOOLEAN         2


/*
 *  Will insert the attribute = value; pair in window's list,
 * if it's different from the defaults.
 *  Defaults means either defaults database, or attributes saved
 * for the default window "*". This is to let one revert options that are
 * global because they were saved for all windows ("*").
 *
 */


static void
insertAttribute(proplist_t dict, proplist_t window, proplist_t attr,
                proplist_t value, int *modified, int flags)
{
    proplist_t def_win, def_value=NULL;
    int update = 0;

    if (!(flags & UPDATE_DEFAULTS) && dict) {
        if ((def_win = PLGetDictionaryEntry(dict, AnyWindow)) != NULL) {
            def_value = PLGetDictionaryEntry(def_win, attr);
        }
    }

    /* If we could not find defaults in database, fall to hardcoded values.
     * Also this is true if we save defaults for all windows
     */
    if (!def_value)
        def_value = ((flags & IS_BOOLEAN) != 0) ? No : EmptyString;

    if ((flags & IS_BOOLEAN))
        update = (getBool(value) != getBool(def_value));
    else {
        update = !PLIsEqual(value, def_value);
    }

    if (update) {
        PLInsertDictionaryEntry(window, attr, value);
        *modified = 1;
    }
}


static void
saveSettings(WMButton *button, InspectorPanel *panel)
{
    WWindow *wwin = panel->inspected;
    WDDomain *db = panel->frame->screen_ptr->dwindow;
    proplist_t dict = db->dictionary;
    proplist_t window, value, key;
    char buffer[256], *icon_file;
    int flags = 0;
    int different = 0;

    if (WMGetButtonState(panel->instRb) != 0)
        key = PLMakeString(wwin->wm_instance);
    else if (WMGetButtonState(panel->clsRb) != 0)
        key = PLMakeString(wwin->wm_class);
    else if (WMGetButtonState(panel->bothRb) != 0) {
        strcat(strcat(strcpy(buffer, wwin->wm_instance), "."), wwin->wm_class);
        key = PLMakeString(buffer);
    }
    else if (WMGetButtonState(panel->defaultRb) != 0) {
        key = PLRetain(AnyWindow);
        flags = UPDATE_DEFAULTS;
    }
    else
        key = NULL;

    if (!key)
        return;

    if (!dict) {
        dict = PLMakeDictionaryFromEntries(NULL, NULL, NULL);
        if (dict) {
            db->dictionary = dict;
            value = PLMakeString(db->path);
            PLSetFilename(dict, value);
            PLRelease(value);
        }
        else {
            PLRelease(key);
            return;
        }
    }

    showIconFor(W_SCREEN(button), panel, NULL, NULL, USE_TEXT_FIELD);

    PLSetStringCmpHook(NULL);

    window = PLMakeDictionaryFromEntries(NULL, NULL, NULL);

    /* Update icon for window */
    icon_file = WMGetTextFromTextField(panel->fileText);
    if (icon_file) {
        value = PLMakeString(icon_file);
        insertAttribute(dict, window, AIcon, value, &different, flags);
        PLRelease(value);
        free(icon_file);
    }

    if (WMGetButtonState(panel->curRb) != 0) {
        int n = panel->inspected->screen_ptr->current_workspace;
        WWorkspace *workspace = panel->inspected->screen_ptr->workspaces[n];
        if (workspace->name) {
            value = PLMakeString(workspace->name);
            insertAttribute(dict, window, AStartWorkspace, value, &different, flags);
            PLRelease(value);
        }
    }
    else if (WMGetButtonState(panel->setRb) != 0) {
        char *ws_name = WMGetTextFromTextField(panel->wsText);
        if (ws_name) {
            value = PLMakeString(ws_name);
            insertAttribute(dict, window, AStartWorkspace, value, &different, flags);
            PLRelease(value);
            free(ws_name);
        }
    }

    flags |= IS_BOOLEAN;

    value = (WMGetButtonState(panel->alwChk)!=0) ? Yes : No;
    insertAttribute(dict, window, AAlwaysUserIcon,    value, &different, flags);

    value = (WMGetButtonState(panel->attrChk[0])!=0) ? Yes : No;
    insertAttribute(dict, window, ANoTitlebar,        value, &different, flags);
    value = (WMGetButtonState(panel->attrChk[1])!=0) ? Yes : No;
    insertAttribute(dict, window, ANoResizebar,       value, &different, flags);
    value = (WMGetButtonState(panel->attrChk[2])!=0) ? Yes : No;
    insertAttribute(dict, window, ANotClosable,       value, &different, flags);
    value = (WMGetButtonState(panel->attrChk[3])!=0) ? Yes : No;
    insertAttribute(dict, window, ANotMiniaturizable, value, &different, flags);
    value = (WMGetButtonState(panel->attrChk[4])!=0) ? Yes : No;
    insertAttribute(dict, window, AKeepOnTop,         value, &different, flags);
    value = (WMGetButtonState(panel->attrChk[5])!=0) ? Yes : No;
    insertAttribute(dict, window, AOmnipresent,       value, &different, flags);
    value = (WMGetButtonState(panel->attrChk[6])!=0) ? Yes : No;
    insertAttribute(dict, window, AUnfocusable,       value, &different, flags);
    value = (WMGetButtonState(panel->attrChk[7])!=0) ? Yes : No;
    insertAttribute(dict, window, ASkipWindowList,    value, &different, flags);
    value = (WMGetButtonState(panel->moreChk[0])!=0) ? Yes : No;
    insertAttribute(dict, window, ANoHideOthers,      value, &different, flags);
    value = (WMGetButtonState(panel->moreChk[1])!=0) ? Yes : No;
    insertAttribute(dict, window, ANoKeyBindings,     value, &different, flags);
    value = (WMGetButtonState(panel->moreChk[2])!=0) ? Yes : No;
    insertAttribute(dict, window, ANoMouseBindings,   value, &different, flags);
    value = (WMGetButtonState(panel->moreChk[3])!=0) ? Yes : No;
    insertAttribute(dict, window, AKeepInsideScreen,  value, &different, flags);
    value = (WMGetButtonState(panel->moreChk[4])!=0) ? Yes : No;
    insertAttribute(dict, window, ANoAppIcon,         value, &different, flags);

    PLRemoveDictionaryEntry(dict, key);
    if (different) {
        PLInsertDictionaryEntry(dict, key, window);
    }
    PLSave(dict, YES);

    PLRelease(key);
    PLRelease(window);

    /* clean up */
    PLSetStringCmpHook(StringCompareHook);
}


static void
makeAppIconFor(WApplication *wapp)
{
    WScreen *scr = wapp->main_window_desc->screen_ptr;

    if (wapp->app_icon)
        return;

    if (wapp->main_window_desc->window_flags.application)
        wapp->app_icon = wAppIconCreate(wapp->main_window_desc);
    else
        wapp->app_icon = NULL;

    if (wapp->app_icon) {
        WIcon *icon = wapp->app_icon->icon;

        wapp->app_icon->main_window = wapp->main_window;

        PlaceIcon(scr, &wapp->app_icon->x_pos, &wapp->app_icon->y_pos);
        XMoveWindow(dpy, icon->core->window, wapp->app_icon->x_pos,
                    wapp->app_icon->y_pos);
#ifndef STRICTNS
        wLowerFrame(icon->core);
#endif
        XMapWindow(dpy, icon->core->window);
        if (wPreferences.auto_arrange_icons)
            wArrangeIcons(wapp->main_window_desc->screen_ptr);
    }
}


static void
removeAppIconFor(WApplication *wapp)
{
    if (!wapp->app_icon)
        return;

    if (wapp->app_icon->docked) {
        wapp->app_icon->running = 0;
        wapp->app_icon->main_window = None;
        wapp->app_icon->pid = 0;
        wapp->app_icon->icon->owner = NULL;
        wapp->app_icon->icon->icon_win = None;
        wapp->app_icon->icon->force_paint = 1;
        wIconUpdate(wapp->app_icon->icon);
        wAppIconPaint(wapp->app_icon);
    } else {
        wAppIconDestroy(wapp->app_icon);
    }
    wapp->app_icon = NULL;
    if (wPreferences.auto_arrange_icons)
        wArrangeIcons(wapp->main_window_desc->screen_ptr);
}


static void
changeAppIconImageFor(WApplication *wapp, char *file)
{
    WWindow *wwin = wapp->main_window_desc;
    RImage *image, *tmp;
    char *path;

    if (!file)
        return;

    if ((path = FindImage(wPreferences.pixmap_path, file))) {
        image = RLoadImage(wwin->screen_ptr->rcontext, path, 0);
        free(path);
        if (image) {
            if (wPreferences.icon_size!=ICON_WIDTH) {
                int w = image->width*wPreferences.icon_size/ICON_WIDTH;
                int h = image->height*wPreferences.icon_size/ICON_HEIGHT;

                tmp = RScaleImage(image, w, h);
                RDestroyImage(image);
                image = tmp;
            }
            wIconChangeImage(wapp->app_icon->icon, image);
            wAppIconPaint(wapp->app_icon);
        }
    }
}


static void
applySettings(WMButton *button, InspectorPanel *panel)
{
    WWindow *wwin = panel->inspected;
    WWindowAttributes *wflags = &wwin->window_flags;
    WApplication *wapp = wApplicationOf(wwin->main_window);
    int floating, skip_window_list;

    showIconFor(W_SCREEN(button), panel, NULL, NULL, USE_TEXT_FIELD);

    wflags->titlebar         = !WMGetButtonState(panel->attrChk[0]);
    wflags->resizable        = !WMGetButtonState(panel->attrChk[1]);
    wflags->closable         = !WMGetButtonState(panel->attrChk[2]);
    wflags->miniaturizable   = !WMGetButtonState(panel->attrChk[3]);
    floating                 =  WMGetButtonState(panel->attrChk[4]);
    wflags->omnipresent      =  WMGetButtonState(panel->attrChk[5]);
    wflags->focusable        = !WMGetButtonState(panel->attrChk[6]);
    skip_window_list         =  WMGetButtonState(panel->attrChk[7]);
    wflags->hide_others      = !WMGetButtonState(panel->moreChk[0]);
    wflags->bind_keys        = !WMGetButtonState(panel->moreChk[1]);
    wflags->bind_mouse       = !WMGetButtonState(panel->moreChk[2]);
    wflags->dont_move_off    =  WMGetButtonState(panel->moreChk[3]);
    wflags->application      = !WMGetButtonState(panel->moreChk[4]);
    wflags->always_user_icon =  WMGetButtonState(panel->alwChk);
    
    if (wflags->floating != floating) {
        int wlevel = ((wflags->floating = floating))
            ? NSFloatingWindowLevel : NSNormalWindowLevel;
        ChangeStackingLevel(wwin->frame->core, wlevel);
    }

    if (wflags->skip_window_list != skip_window_list) {
        int action = ((wflags->skip_window_list = skip_window_list))
            ? ACTION_REMOVE : ACTION_ADD;
        UpdateSwitchMenu(wwin->screen_ptr, wwin, action);
    }

    wwin->frame->flags.need_texture_change = 1;
    wWindowConfigureBorders(wwin);
    wFrameWindowPaint(wwin->frame);

    if (wapp) {
        if (wflags->application)
            makeAppIconFor(wapp);
        else
            removeAppIconFor(wapp);

        /* hack */
        wapp->main_window_desc->window_flags.always_user_icon = wflags->always_user_icon;
        if (wapp->app_icon) {
            char *file = WMGetTextFromTextField(panel->fileText);

            if (file) {
                changeAppIconImageFor(wapp, file);
                free(file);
            }
        }
    }
}


static void
revertSettings(WMButton *button, InspectorPanel *panel)
{
    WWindow *wwin = panel->inspected;
    WMArgument chkArg = {WARG_SELECTED, (WMValue)0};
    WApplication *wapp = wApplicationOf(wwin->main_window);
    int i, n, floating, skip_window_list;
    char *wm_instance = NULL;
    char *wm_class = NULL;
    int old_shadeable = wwin->window_flags.shadeable;

    if (WMGetButtonState(panel->instRb) != 0)
        wm_instance = wwin->wm_instance;
    else if (WMGetButtonState(panel->clsRb) != 0)
        wm_class = wwin->wm_class;
    else if (WMGetButtonState(panel->bothRb) != 0) {
        wm_instance = wwin->wm_instance;
        wm_class = wwin->wm_class;
    }

    wDefaultFillAttributes(wwin->screen_ptr, wm_instance, wm_class,
                           &wwin->window_flags);

    wwin->window_flags.broken_close = (wwin->protocols.DELETE_WINDOW) ? 0 : 1;
    /* transients can't be iconified or maximized */
    if (wwin->transient_for)
        wwin->window_flags.miniaturizable=0;
    /* if the window can't be resized, remove the resizebar */
    if (wwin->normal_hints->flags & (PMinSize|PMaxSize)
        && (wwin->normal_hints->min_width==wwin->normal_hints->max_width)
        && (wwin->normal_hints->min_height==wwin->normal_hints->max_height)) {
        wwin->window_flags.resizable=0;
    }

    wwin->window_flags.shadeable = old_shadeable;

    for (i=0; i < 8; i++) {
	switch (i) {
	 case 0:
            chkArg.value = (WMValue)!wwin->window_flags.titlebar;
	    break;
	 case 1:
	    chkArg.value = (WMValue)!wwin->window_flags.resizable;
	    break;
	 case 2:
	    chkArg.value = (WMValue)!wwin->window_flags.closable;
	    break;
	 case 3:
	    chkArg.value = (WMValue)!wwin->window_flags.miniaturizable;
	    break;
	 case 4:
            floating = WMGetButtonState(panel->attrChk[4]);
            if (wwin->window_flags.floating != floating) {
                int wlevel = (wwin->window_flags.floating != 0)
                    ? NSFloatingWindowLevel : NSNormalWindowLevel;
                ChangeStackingLevel(wwin->frame->core, wlevel);
            }
            chkArg.value = (WMValue)wwin->window_flags.floating;
	    break;
	 case 5:
	    chkArg.value = (WMValue)wwin->window_flags.omnipresent;
	    break;
	 case 6:
	    chkArg.value = (WMValue)!wwin->window_flags.focusable;
	    break;
	 case 7:
            skip_window_list = WMGetButtonState(panel->attrChk[7]);
            if (wwin->window_flags.skip_window_list != skip_window_list) {
                int action = (wwin->window_flags.skip_window_list != 0)
                    ? ACTION_REMOVE : ACTION_ADD;
                UpdateSwitchMenu(wwin->screen_ptr, wwin, action);
            }
	    chkArg.value = (WMValue)wwin->window_flags.skip_window_list;
	    break;
	}
        WMConfigureButton(panel->attrChk[i], &chkArg, 1);
    }
    for (i=0; i < 5; i++) {
	switch (i) {
	 case 0:
	    chkArg.value = (WMValue)!wwin->window_flags.hide_others;
	    break;
	 case 1:
	    chkArg.value = (WMValue)!wwin->window_flags.bind_keys;
	    break;
	 case 2:
	    chkArg.value = (WMValue)!wwin->window_flags.bind_mouse;
	    break;
	 case 3:
	    chkArg.value = (WMValue)wwin->window_flags.dont_move_off;
	    break;
	 case 4:
	    chkArg.value = (WMValue)!wwin->window_flags.application;
	    break;
	}
	WMConfigureButton(panel->moreChk[i], &chkArg, 1);
    }
    chkArg.value = (WMValue)wwin->window_flags.always_user_icon;
    WMConfigureButton(panel->alwChk, &chkArg, 1);

    wwin->frame->flags.need_texture_change = 1;
    wWindowConfigureBorders(wwin);
    wFrameWindowPaint(wwin->frame);

    showIconFor(W_SCREEN(button), panel, wm_instance, wm_class,
                UPDATE_TEXT_FIELD);

    workspaceArgs[0].value = (WMValue)NULL;
    n = wDefaultGetStartWorkspace(wwin->screen_ptr, wm_instance, wm_class);
    if (n >= 0 && n <= wwin->screen_ptr->workspace_count) {
        if (n == wwin->screen_ptr->current_workspace) {
            WMPerformButtonClick(panel->curRb);;
        }
        else {
            WMPerformButtonClick(panel->setRb);;
            workspaceArgs[0].value = (WMValue)wwin->screen_ptr->workspaces[n]->name;
        }
    }
    else {
        chkArg.value = (WMValue)0;
        WMConfigureButton(panel->curRb, &chkArg, 1);
        WMConfigureButton(panel->setRb, &chkArg, 1);
    }
    
    WMConfigureTextField(panel->wsText, workspaceArgs, COUNTARG(workspaceArgs));

    if (wapp) {
        if (wwin->window_flags.application)
            makeAppIconFor(wapp);
        else
            removeAppIconFor(wapp);

        /* same hack */
        wapp->main_window_desc->window_flags.always_user_icon = wwin->window_flags.always_user_icon;
        if (wapp->app_icon) {
            char *file = wDefaultGetIconFile(wwin->screen_ptr,
                                             wm_instance, wm_class);

            if (file) {
                changeAppIconImageFor(wapp, file);
            }
        }
    }
}


static InspectorPanel*
createInspectorForWindow(WWindow *wwin)
{
    WScreen *scr = wwin->screen_ptr;

    InspectorPanel *panel;
    Window parent;
    char charbuf[128];
    int i, sel;

    panel = malloc(sizeof(InspectorPanel));
    if (!panel) {
	wWarning(_("not enough memory to open window inspector panel"));
	return NULL;
    }

    
    panel->inspected = wwin;

    panel->nextPtr = panelList;
    panelList = panel;
    

    sprintf(charbuf, "Inspecting  %s.%s", 
	    wwin->wm_instance ? wwin->wm_instance : "?",
	    wwin->wm_class ? wwin->wm_class : "?");

    parent = XCreateSimpleWindow(dpy, scr->root_win, 0, 0, PWIDTH, PHEIGHT, 
				 0, 0, 0);


    XSetTransientForHint(dpy, parent, wwin->client_win);

    panel->win = WMCreateWindow(scr->wmscreen, parent, panelArgs,
				COUNTARG(panelArgs));

    /**** create common stuff ****/
    
    /* command buttons */
    buttonArgs[0].value = (WMValue)30;
    buttonArgs[1].value = (WMValue)_("Save");
    panel->saveBtn = WMCreateCommandButton(panel->win, buttonArgs, 
					    COUNTARG(buttonArgs));
    WMBindEvent(panel->saveBtn, WEV_CLICK, saveSettings, panel);

    buttonArgs[0].value = (WMValue)100;
    buttonArgs[1].value = (WMValue)_("Apply");
    panel->applyBtn = WMCreateCommandButton(panel->win, buttonArgs, 
                                            COUNTARG(buttonArgs));
    WMBindEvent(panel->applyBtn, WEV_CLICK, applySettings, panel);


    buttonArgs[0].value = (WMValue)170;
    buttonArgs[1].value = (WMValue)_("Revert");
    panel->revertBtn = WMCreateCommandButton(panel->win, buttonArgs,
					     COUNTARG(buttonArgs));
    WMBindEvent(panel->revertBtn, WEV_CLICK, revertSettings, panel);
    
    /* page selection buttons */
    pageBtnArgs[0].value = (WMValue)70;
    pageBtnArgs[1].value = (WMValue)"1";
    panel->pageBtn1 = WMCreateButton(panel->win, WBTOnOff, pageBtnArgs,
					    COUNTARG(pageBtnArgs));
    WMBindEvent(panel->pageBtn1, WEV_CLICK, changePage, panel);
    
    pageBtnArgs[0].value = (WMValue)100;
    pageBtnArgs[1].value = (WMValue)"2";
    panel->pageBtn2 = WMCreateButton(panel->win, WBTOnOff, pageBtnArgs,
					    COUNTARG(pageBtnArgs));
    WMBindEvent(panel->pageBtn2, WEV_CLICK, changePage, panel);
    
    pageBtnArgs[0].value = (WMValue)130;
    pageBtnArgs[1].value = (WMValue)"3";
    panel->pageBtn3 = WMCreateButton(panel->win, WBTOnOff, pageBtnArgs,
					    COUNTARG(pageBtnArgs));
    WMBindEvent(panel->pageBtn3, WEV_CLICK, changePage, panel);
    
    pageBtnArgs[0].value = (WMValue)160;
    pageBtnArgs[1].value = (WMValue)"4";
    panel->pageBtn4 = WMCreateButton(panel->win, WBTOnOff, pageBtnArgs,
					    COUNTARG(pageBtnArgs));
    WMBindEvent(panel->pageBtn4, WEV_CLICK, changePage, panel);

    /**** window spec ****/
    specFrmArgs[0].value = (WMValue)_("Window Specification");
    panel->specFrm = WMCreateFrame(panel->win, specFrmArgs, 
				   COUNTARG(specFrmArgs));

    sel = 1;
    if (wwin->wm_class && wwin->wm_instance) {
	sprintf(charbuf, "%s.%s", wwin->wm_instance, wwin->wm_class);
	specRadArgs[0].value = (WMValue)18;
	specRadArgs[1].value = (WMValue)charbuf;
	specRadArgs[2].value = (WMValue)sel; sel=0;
	panel->bothRb = WMCreateRadioButton(panel->specFrm, specRadArgs, 
					    COUNTARG(specRadArgs));
    }

    if (wwin->wm_instance) {
	specRadArgs[0].value = (WMValue)38;
	specRadArgs[1].value = (WMValue)wwin->wm_instance;
	specRadArgs[2].value = (WMValue)sel; sel=0;
	panel->instRb = WMCreateRadioButton(panel->specFrm, specRadArgs, 
					    COUNTARG(specRadArgs));
    }

    if (wwin->wm_class) {
	specRadArgs[0].value = (WMValue)58;
	specRadArgs[1].value = (WMValue)wwin->wm_class;
	specRadArgs[2].value = (WMValue)sel; sel=0;
	panel->clsRb = WMCreateRadioButton(panel->specFrm, specRadArgs, 
					   COUNTARG(specRadArgs));
    }

    specRadArgs[0].value = (WMValue)78;
    specRadArgs[1].value = (WMValue)_("Defaults for all windows");
    specRadArgs[2].value = (WMValue)sel; sel=0;
    panel->defaultRb = WMCreateRadioButton(panel->specFrm, specRadArgs,
                                           COUNTARG(specRadArgs));

    specLblArgs[0].value = (WMValue)_("The configuration will apply to all\n"
                                      "windows that have their WM_CLASS property"
                                      " set to the above selected\nname, when saved.");

    panel->specLbl = WMCreateLabel(panel->win, specLblArgs, 
				   COUNTARG(specLblArgs));
    
    /**** attributes ****/
    attrFrmArgs[0].value = (WMValue)_("Attributes");
    panel->attrFrm = WMCreateFrame(panel->win, attrFrmArgs,
				   COUNTARG(attrFrmArgs));

    for (i=0; i < 8; i++) {
	switch (i) {
	 case 0:
	    attrChkArgs[0].value = (WMValue)_("Disable titlebar");
            attrChkArgs[2].value = (WMValue)!wwin->window_flags.titlebar;
	    break;
	 case 1:
	    attrChkArgs[0].value = (WMValue)_("Disable resizebar");
	    attrChkArgs[2].value = (WMValue)!wwin->window_flags.resizable;
	    break;
	 case 2:
	    attrChkArgs[0].value = (WMValue)_("Not closable");
	    attrChkArgs[2].value = (WMValue)!wwin->window_flags.closable;
	    break;
	 case 3:
	    attrChkArgs[0].value = (WMValue)_("Not miniaturizable");
	    attrChkArgs[2].value = (WMValue)!wwin->window_flags.miniaturizable;
	    break;
	 case 4:
	    attrChkArgs[0].value = (WMValue)_("Keep on top");
	    attrChkArgs[2].value = (WMValue)wwin->window_flags.floating;
	    break;
	 case 5:
	    attrChkArgs[0].value = (WMValue)_("Omnipresent");
	    attrChkArgs[2].value = (WMValue)wwin->window_flags.omnipresent;
	    break;
	 case 6:
	    attrChkArgs[0].value = (WMValue)_("Don't let it take focus");
	    attrChkArgs[2].value = (WMValue)!wwin->window_flags.focusable;
	    break;
	 case 7:
	    attrChkArgs[0].value = (WMValue)_("Skip window list");
	    attrChkArgs[2].value = (WMValue)wwin->window_flags.skip_window_list;
	    break;
	}
	attrChkArgs[1].value = (WMValue)(20*(i+1));
	panel->attrChk[i] = WMCreateSwitchButton(panel->attrFrm, attrChkArgs,
                                                 COUNTARG(attrChkArgs));
    }
    

    
    /**** more attributes ****/
    attrFrmArgs[0].value = (WMValue)_("Advanced");
    panel->moreFrm = WMCreateFrame(panel->win, attrFrmArgs,
				   COUNTARG(attrFrmArgs));

    for (i=0; i < 5; i++) {
	switch (i) {
	 case 0:
	    attrChkArgs[0].value = (WMValue)_("Ignore HideOthers");
	    attrChkArgs[2].value = (WMValue)!wwin->window_flags.hide_others;
	    break;
	 case 1:
	    attrChkArgs[0].value = (WMValue)_("Don't bind keyboard shortcuts");
	    attrChkArgs[2].value = (WMValue)!wwin->window_flags.bind_keys;
	    break;
	 case 2:
	    attrChkArgs[0].value = (WMValue)_("Don't bind mouse clicks");
	    attrChkArgs[2].value = (WMValue)!wwin->window_flags.bind_mouse;
	    break;
	 case 3:
	    attrChkArgs[0].value = (WMValue)_("Keep inside screen");
	    attrChkArgs[2].value = (WMValue)wwin->window_flags.dont_move_off;
	    break;
	 case 4:
	    attrChkArgs[0].value = (WMValue)_("No application icon");
	    attrChkArgs[2].value = (WMValue)!wwin->window_flags.application;
	    break;
	}
	attrChkArgs[1].value = (WMValue)(20*(i+1));
	panel->moreChk[i] = WMCreateSwitchButton(panel->moreFrm, attrChkArgs,
						 COUNTARG(attrChkArgs));
    }

    
    /* miniwindow/workspace */
    
    iconFrmArgs[0].value = (WMValue)_("Miniwindow Image");
    panel->iconFrm = WMCreateFrame(panel->win, iconFrmArgs,
				   COUNTARG(iconFrmArgs));
    panel->iconLbl = WMCreateLabel(panel->iconFrm, iconLblArgs,
                                   COUNTARG(iconLblArgs));
    
    updateIconBtnArgs[0].value = (WMValue)_("Browse");
    updateIconBtnArgs[1].value = (WMValue)30;
    panel->defaultIconBtn = WMCreateCommandButton(panel->iconFrm,
                                                  updateIconBtnArgs,
                                                  COUNTARG(updateIconBtnArgs));
    /* WMBindEvent(panel->defaultIconBtn, WEV_CLICK, defaultIcon, panel); */

    updateIconBtnArgs[0].value = (WMValue)_("Update");
    updateIconBtnArgs[1].value = (WMValue)65;
    panel->updateIconBtn = WMCreateCommandButton(panel->iconFrm, 
						 updateIconBtnArgs,
						 COUNTARG(updateIconBtnArgs));
    WMBindEvent(panel->updateIconBtn, WEV_CLICK, updateIcon, panel);    
    
    fileLblArgs[0].value = (WMValue)_("Icon file name:");
    panel->fileLbl = WMCreateLabel(panel->iconFrm, fileLblArgs,
				   COUNTARG(fileLblArgs));

    fileArgs[0].value = (WMValue)NULL;
    panel->fileText = WMCreateTextField(panel->iconFrm, fileArgs,
                                        COUNTARG(fileArgs));

    alwChkArgs[0].value = (WMValue)_("Ignore client supplied icon");
    alwChkArgs[1].value = (WMValue)wwin->window_flags.always_user_icon;
    panel->alwChk = WMCreateSwitchButton(panel->iconFrm, alwChkArgs,
					 COUNTARG(alwChkArgs));

    wsFrmArgs[0].value = (WMValue)_("Initial Workspace");
    panel->wsFrm = WMCreateFrame(panel->win, wsFrmArgs, COUNTARG(wsFrmArgs));

    workspaceArgs[0].value = (WMValue)NULL;
    i = wDefaultGetStartWorkspace(wwin->screen_ptr, wwin->wm_instance,
                                  wwin->wm_class);
    if (i >= 0 && i <= wwin->screen_ptr->workspace_count) {
        if (i == wwin->screen_ptr->current_workspace) {
            wsRad1Args[1].value = (WMValue)1;
            wsRad2Args[1].value = (WMValue)0;
        }
        else {
            wsRad1Args[1].value = (WMValue)0;
            wsRad2Args[1].value = (WMValue)1;
            workspaceArgs[0].value = (WMValue)wwin->screen_ptr->workspaces[i]->name;
        }
    }
    else {
        wsRad1Args[1].value = (WMValue)0;
        wsRad2Args[1].value = (WMValue)0;
    }

    wsRad1Args[0].value = (WMValue)_("Current workspace");
    panel->curRb = WMCreateRadioButton(panel->wsFrm, wsRad1Args,
				       COUNTARG(wsRad1Args));
    
    panel->setRb = WMCreateRadioButton(panel->wsFrm, wsRad2Args,
				       COUNTARG(wsRad2Args));

    panel->wsText = WMCreateTextField(panel->wsFrm, workspaceArgs,
				      COUNTARG(workspaceArgs));



    panel->frame = wManageInternalWindow(scr, parent, wwin->client_win, charbuf,
					 wwin->frame_x+wwin->frame->core->width/2,
					 wwin->frame_y+wwin->frame->top_width*2, 
					 PWIDTH, PHEIGHT);
    
    panel->frame->window_flags.closable = 1;
    wWindowUpdateButtonImages(panel->frame);
    wFrameWindowShowButton(panel->frame->frame, WFF_RIGHT_BUTTON);
    panel->frame->frame->on_click_right = destroyInspector;

    
    WMMapSubwidgets(panel->win);
    WMMapSubwidgets(panel->specFrm);
    WMMapSubwidgets(panel->attrFrm);
    WMMapSubwidgets(panel->moreFrm);
    WMMapSubwidgets(panel->iconFrm);
    WMMapSubwidgets(panel->wsFrm);

    WMPerformButtonClick(panel->pageBtn1);

    WMMapWidget(panel->win);

    showIconFor(W_SCREEN(panel->alwChk), panel, wwin->wm_instance,
                wwin->wm_class, UPDATE_TEXT_FIELD);

    return panel;
}


void
wShowInspectorForWindow(WWindow *wwin)
{
    if (wwin->flags.inspector_open)
      return;

    make_keys();
    wwin->flags.inspector_open = 1;
    wwin->inspector = createInspectorForWindow(wwin);;
}


