/*  options.c -- Eterm options module
 *            -- 25 July 1997, mej
 *
 * This file is original work by Michael Jennings <mej@tcserv.com> and
 * Tuomo Venalainen <vendu@cc.hut.fi>.  This file, and any other file
 * bearing this same message or a similar one, is distributed under
 * the GNU Public License (GPL) as outlined in the COPYING file.
 *
 * Copyright (C) 1997, Michael Jennings and Tuomo Venalainen
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 * 
 */

#include "main.h"
#include "feature.h"

#include <stdlib.h>
#include <unistd.h>
#include <sys/types.h>
#include <errno.h>
#include <limits.h>
#include <X11/keysym.h>
#include "debug.h"
#include "mem.h"
#include "strings.h"
#include "grkelot.h"
#include "options.h"

const char *true_vals[]  = { "1", "on",  "true"  };
const char *false_vals[] = { "0", "off", "false" };

      char **rs_execArgs = NULL;        /* Args to exec (-e or --exec) */
      char  *rs_title = NULL;		/* Window title */
      char  *rs_iconName = NULL;	/* Icon name */
      char  *rs_geometry = NULL;	/* Geometry string */
      int    rs_saveLines = SAVELINES;	/* Lines in the scrollback buffer */
#ifdef KEYSYM_ATTRIBUTE
unsigned char *KeySym_map[256];         /* probably mostly empty */
#endif
#if defined (HOTKEY_CTRL) || defined (HOTKEY_META)
/* recognized when combined with HOTKEY */
KeySym ks_bigfont = XK_greater;
KeySym ks_smallfont = XK_less;
#endif
#ifdef PIXMAP_SUPPORT
const char *rs_pixmapScale = NULL;
      char *rs_path = NULL;
      char *rs_pixmaps[6] = { NULL, NULL, NULL, NULL, NULL, NULL };
#endif
#ifdef USE_THEMES
char *rs_theme = NULL;
char *rs_config_file = NULL;
#endif

/* local functions referenced */
char *chomp(char *);
void parse_main(char *);
void parse_color(char *);
void parse_attributes(char *);
void parse_toggles(char *);
void parse_keyboard(char *);
void parse_misc(char *);
void parse_pixmaps(char *);
void parse_kanji(char *);
void parse_undef(char *);
char *builtin_random(char *);
char *builtin_exec(char *);
char *builtin_version(char *);
char *builtin_appname(char *);

/* local variables */
const char *rs_loginShell = NULL;
const char *rs_utmpLogging = NULL;
const char *rs_scrollBar = NULL;
const char *rs_homeOnEcho = NULL;
const char *rs_homeOnRefresh = NULL;
const char *rs_scrollBar_right = NULL;
const char *rs_scrollBar_floating = NULL;
const char *rs_borderless = NULL;

const char *rs_term_name = NULL;
const char *rs_menu = NULL;

#if defined (HOTKEY_CTRL) || defined (HOTKEY_META)
static char *rs_bigfont_key = NULL;
static char *rs_smallfont_key = NULL;
#endif

#ifndef NO_MAPALERT
# ifdef MAPALERT_OPTION
static const char *rs_mapAlert = NULL;
# endif
#endif
static const char *rs_visualBell = NULL;
static const char *rs_reverseVideo = NULL;
#ifdef META8_OPTION
static const char *rs_meta8 = NULL;
#endif
#ifdef KANJI
static char *rs_kanji_encoding = NULL;
#endif
#ifdef GREEK_SUPPORT
static char *rs_greek_keyboard = NULL;
#endif

extern char initial_dir[PATH_MAX+1];

/* Options structure */

#define OPT_BOOLEAN          0x0001
#define OPT_INTEGER          0x0002
#define OPT_STRING           0x0004
#define OPT_ARGUMENT         0x0008

#define OPT_STR(s, l, d, p)      { s, l, d, OPT_STRING,   (const char **)  p, 0 }
#define OPT_INT(s, l, d, p)      { s, l, d, OPT_INTEGER,  (const int *)    p, 0 }
#define OPT_BOOL(s, l, d, p, m)  { s, l, d, OPT_BOOLEAN,  (const int *)    p, m }
#define OPT_LONG(l, d, p)        { 0, l, d, OPT_STRING,   (const char **)  p, 0 }
#define OPT_ARGS(s, l, d, p)     { s, l, d, OPT_ARGUMENT, (const char ***) p, 0 }
#define OPT_BLONG(l, d, p, m)    { 0, l, d, OPT_BOOLEAN,  (const int *)    p, m }

static const struct {
  char short_opt;
  char *long_opt;
  const char * const description;
  unsigned short flag;
  const void *pval;
  int mask;
} optList[] = {

#define optList_numoptions()    (sizeof(optList)/sizeof(optList[0]))

#ifdef USE_THEMES /* These two MUST be first! */
OPT_STR('t', "theme", "select a theme", &rs_theme),
OPT_STR('X', "config-file", "choose an alternate config file", &rs_config_file),
#endif /* USE_THEMES */
OPT_BOOL('h', "help", "display usage information", NULL, 0),

/* =======[ Color options ]======= */
OPT_BOOL('r', "reverse-video", "reverse video", &rs_reverseVideo, Opt_reverseVideo),
OPT_STR('b', "background-color", "background color", &rs_color[bgColor]),
OPT_STR('f', "foreground-color", "foreground color", &rs_color[fgColor]),
OPT_LONG("color0", "color 0", &rs_color[minColor]),
OPT_LONG("color1", "color 1", &rs_color[minColor+1]),
OPT_LONG("color2", "color 2", &rs_color[minColor+2]),
OPT_LONG("color3", "color 3", &rs_color[minColor+3]),
OPT_LONG("color4", "color 4", &rs_color[minColor+4]),
OPT_LONG("color5", "color 5", &rs_color[minColor+5]),
OPT_LONG("color6", "color 6", &rs_color[minColor+6]),
OPT_LONG("color7", "color 7", &rs_color[minColor+7]),
#ifndef NO_BRIGHTCOLOR
OPT_LONG("color8", "color 8", &rs_color[minBright]),
OPT_LONG("color9", "color 9", &rs_color[minBright+1]),
OPT_LONG("color10", "color 10", &rs_color[minBright+2]),
OPT_LONG("color11", "color 11", &rs_color[minBright+3]),
OPT_LONG("color12", "color 12", &rs_color[minBright+4]),
OPT_LONG("color13", "color 13", &rs_color[minBright+5]),
OPT_LONG("color14", "color 14", &rs_color[minBright+6]),
OPT_LONG("color15", "color 15", &rs_color[minBright+7]),
#endif	/* NO_BRIGHTCOLOR */
#ifndef NO_BOLDUNDERLINE
OPT_LONG("colorBD", "bold color", &rs_color[colorBD]),
OPT_LONG("colorUL", "underline color", &rs_color[colorUL]),
#endif	/* NO_BOLDUNDERLINE */
#if !defined(XTERM_SCROLLBAR) && defined(KEEP_SCROLLCOLOR)
OPT_STR('S', "scrollbar-color", "scrollbar color", &rs_color[scrollColor]),
#endif	/* XTERM_SCROLLBAR */
OPT_LONG("pointer-color", "mouse pointer color", &rs_color[pointerColor]),
#ifndef NO_CURSORCOLOR
OPT_STR('c', "cursor-color", "cursor color", &rs_color[cursorColor]),
OPT_LONG("cursor-text-color", "cursor text color", &rs_color[cursorColor2]),
#endif	/* NO_CURSORCOLOR */

/* =======[ X11 options ]======= */
OPT_STR('d', "display", "X server to connect to", &display_name),
OPT_STR('g', "geometry", "WxH+X+Y = size and position", &rs_geometry),
OPT_BOOL('i', "iconic", "start iconified", NULL, Opt_iconic),
OPT_STR('n', "name", "client instance, icon, and title strings", &rs_name),
OPT_STR('T', "title", "title string", &rs_title),
OPT_LONG("icon-name", "icon name", &rs_iconName),
#ifndef NO_BOLDFONT
OPT_LONG("bold-font", "bold text font", &rs_boldFont),
#endif
OPT_STR('F', "font", "normal text font", &rs_font[0]),
OPT_LONG("font1", "font 1", &rs_font[1]),
OPT_LONG("font2", "font 2", &rs_font[2]),
OPT_LONG("font3", "font 3", &rs_font[3]),
OPT_LONG("font4", "font 4", &rs_font[4]),

/* =======[ Pixmap options ]======= */
#ifdef PIXMAP_SUPPORT
OPT_STR('P', "background-pixmap", "background pixmap [scaling optional]", &rs_pixmaps[pixmap_bg]),
OPT_STR('U', "up-arrow-pixmap", "up arrow pixmap [scaling optional]", &rs_pixmaps[pixmap_up]),
OPT_STR('D', "down-arrow-pixmap", "down arrow pixmap [scaling optional]", &rs_pixmaps[pixmap_dn]),
OPT_STR('B', "trough-pixmap", "scrollbar background (trough) pixmap [scaling optional]", &rs_pixmaps[pixmap_sb]),
OPT_STR('A', "anchor-pixmap", "scrollbar anchor pixmap [scaling optional]", &rs_pixmaps[pixmap_sa]),
OPT_BOOL('@', "scale", "scale rather than tile", &rs_pixmapScale, Opt_pixmapScale),
OPT_STR('p', "path", "pixmap file search path", &rs_path),
#endif /* PIXMAP_SUPPORT */

/* =======[ Kanji options ]======= */
#ifdef KANJI
OPT_STR('K', "kanji-font", "normal text kanji font", &rs_kfont[0]),
OPT_LONG("kanji-font1", "kanji font 1", &rs_kfont[1]),
OPT_LONG("kanji-font2", "kanji font 2", &rs_kfont[2]),
OPT_LONG("kanji-font3", "kanji font 3", &rs_kfont[3]),
OPT_LONG("kanji-font4", "kanji font 4", &rs_kfont[4]),
OPT_LONG("kanji-encoding", "kanji encoding mode (eucj or sjis)", &rs_kanji_encoding),
#endif	/* KANJI */

/* =======[ Toggles ]======= */
OPT_BOOL('l', "login-shell", "login shell, prepend - to shell name", &rs_loginShell, Opt_loginShell),
OPT_BOOL('s', "scrollbar", "display scrollbar", &rs_scrollBar, Opt_scrollBar),
OPT_BOOL('u', "utmp-logging", "make a utmp entry", &rs_utmpLogging, Opt_utmpLogging),
OPT_BOOL('v', "visual-bell", "visual bell", &rs_visualBell, Opt_visualBell),
OPT_BOOL('H', "home-on-echo", "jump to bottom on output", &rs_homeOnEcho, Opt_homeOnEcho),
OPT_BOOL('E', "home-on-refresh", "jump to bottom on refresh (^L)", &rs_homeOnRefresh, Opt_homeOnRefresh),
OPT_BLONG("scrollbar-right", "display the scrollbar on the right", &rs_scrollBar_right, Opt_scrollBar_right),
OPT_BLONG("scrollbar-floating", "display the scrollbar with no trough", &rs_scrollBar_floating, Opt_scrollBar_floating),
OPT_BOOL('x', "borderless", "force Eterm to have no borders", &rs_borderless, Opt_borderless),
#ifndef NO_MAPALERT
# ifdef MAPALERT_OPTION
OPT_BOOL('m', "map-alert", "uniconify on beep", &rs_mapAlert, Opt_mapAlert),
# endif
#endif
#ifdef META8_OPTION
OPT_BOOL('8', "meta-8", NULL, &rs_meta8, Opt_meta8),
#endif

/* =======[ Keyboard options ]======= */
#if defined (HOTKEY_CTRL) || defined (HOTKEY_META)
OPT_LONG("big-font-key", NULL, &rs_bigfont_key),
OPT_LONG("small-font-key", NULL, &rs_smallfont_key),
#endif
#ifdef GREEK_SUPPORT
OPT_LONG("greek-keyboard", "greek keyboard mapping (iso or ibm)", &rs_greek_keyboard),
#endif

/* =======[ Misc options ]======= */
OPT_INT('L', "save-lines", "lines to save in scrollback buffer", &rs_saveLines),
#ifdef BORDER_WIDTH_OPTION
OPT_INT('w', "border-width", "term window border width", &(TermWin.internalBorder)),
#endif
#ifdef PRINTPIPE
OPT_STR('I', "print-pipe", NULL, &rs_print_pipe),
#endif
#ifdef CUTCHAR_OPTION
OPT_LONG("cut-chars", "seperators for double-click selection", &rs_cutchars),
#endif	/* CUTCHAR_OPTION */
OPT_STR('M', "menu", "Default menubar file", &rs_menu),
OPT_LONG("term-name", "value to use for setting $TERM", &rs_term_name),
OPT_BOOL('C', "console", "grab console messages", NULL, Opt_console),
OPT_ARGS('e', "exec", "execute a command rather than a shell", &rs_execArgs)
};

#if 0 /* Use libmej */

#define DEFAULT_DELIM " \r\n\f\t\v"

/* Find first occurance of string needle in string haystack.  Ignore case. */
char *
StrCaseStr(char *haystack, register const char *needle) {

  register char *t;
  register size_t len = strlen(needle);

  for (t = haystack; t && *t; t++) {
    if (!strncasecmp(t, needle, len)) {
      return(t);
    }
  }
  return(NULL);
}

/* Return malloc'd pointer to index-th word in str.  "..." counts as 1 word. */
char *
Word(unsigned long index, char *str) {

  char *tmpstr;
  char *delim = DEFAULT_DELIM;
  register unsigned long i, j, k;

  k = strlen(str) + 1;
  if ((tmpstr = (char *) malloc(k)) == NULL) {
    fprintf(stderr, "Unable to allocate memory -- %s.  Aborting.\n",
            strerror(errno));
    return ((char *) NULL);
  }

  *tmpstr = 0;
  for(i=0, j=0; j < index && str[i]; j++) {
    for(; isspace(str[i]); i++);
    switch (str[i]) {
    case '\"':
      delim = "\"";
      i++;
      break;
    case '\'':
      delim = "\'";
      i++;
      break;
    default:
      delim = DEFAULT_DELIM;
    }
    for(k=0; str[i] && !strchr(delim, str[i]); ) {
      tmpstr[k++] = str[i++];
    }
    switch (str[i]) {
    case '\"':
    case '\'':
      i++;
      break;
    }
    tmpstr[k] = 0;
  }

  if (j != index) {
    free(tmpstr);
#ifdef DEBUG_OPTIONS
    fprintf(stderr, "[debug] Word(%lu, %s) returning NULL.\n", index, str);
#endif
    return ((char *) NULL);
  } else {
    tmpstr = (char *) realloc(tmpstr, strlen(tmpstr)+1);
#ifdef DEBUG_OPTIONS
    fprintf(stderr, "[debug] Word(%lu, %s) returning \"%s\".\n", index, str, tmpstr);
#endif
    return (tmpstr);
  }
}

/* Return pointer into str to index-th word in str.  "..." counts as 1 word. */
char *
PWord(unsigned long index, char *str) {

  register char *tmpstr = str;
  register unsigned long j;

  if (!str) return ((char *) NULL);
  for (; isspace(*tmpstr) && *tmpstr; tmpstr++);
  for (j=1; j < index && *tmpstr; j++) {
    for (; !isspace(*tmpstr) && *tmpstr; tmpstr++);
    for (; isspace(*tmpstr) && *tmpstr; tmpstr++);
  }

  if (*tmpstr == '\"' || *tmpstr == '\'') {
    tmpstr++;
  }
  if (*tmpstr == '\0') {
#ifdef DEBUG_OPTIONS
  fprintf(stderr, "[debug] PWord(%lu, %s) returning NULL.\n", index, str);
#endif
    return ((char *) NULL);
  } else {
#ifdef DEBUG_OPTIONS
  fprintf(stderr, "[debug] PWord(%lu, %s) returning \"%s\"\n", index, str, tmpstr);
#endif
    return tmpstr;
  }
}

/* Returns the number of words in str, for use with Word() and PWord().  "..." counts as 1 word. */
unsigned long
NumWords(const char *str) {

  register unsigned long cnt=0;
  register const char *s = str;
  char *delim = DEFAULT_DELIM;
  register unsigned long i;

  for(i=0; str[i] && strchr(delim, str[i]); i++);
  for(; str[i]; cnt++) {
    switch (str[i]) {
    case '\"':
      delim = "\"";
      i++;
      break;
    case '\'':
      delim = "\'";
      i++;
      break;
    default:
      delim = DEFAULT_DELIM;
    }
    for(; str[i] && !strchr(delim, str[i]); i++);
    switch (str[i]) {
    case '\"':
    case '\'':
      i++;
      break;
    }
    for(; str[i] && isspace(str[i]); i++);
  }

#if 0
  if (!str) return (0);
  for (; *s && isspace(*s); s++);
  for (; *s; cnt++) {
    for (; *s && !isspace(*s); s++);
    for (; *s && isspace(*s); s++);
  }
#endif
#ifdef DEBUG_OPTIONS
  fprintf(stderr, "NumWords() returning %lu\n", cnt);
#endif
  return (cnt);
}

#endif /* 0 */

/* Print usage information */
#define INDENT "5"
static void
usage(void) {

   int i, col;

   fprintf(stderr, "Eterm Enlightened Terminal Emulator for X Windows\n");
   fprintf(stderr, "Copyright (c) 1997, Tuomo Venalainen and Michael Jennings\n\n");
   fprintf(stderr, "Usage for " APL_NAME " " VERSION ":\n\n");
   fprintf(stderr, "%7s %12s %45s\n", "POSIX", "GNU", "Description");
   fprintf(stderr, "%8s %10s %52s\n", "=======", "====================", 
	   "=========================================");
   for (i=0; i < optList_numoptions(); i++) {
     if (optList[i].description) {
       fprintf(stderr, "%" INDENT "s", " ");
       if (optList[i].short_opt) {
	 fprintf(stderr, "-%c, ", optList[i].short_opt);
       } else {
	 fprintf(stderr, "    ");
       }
       fprintf(stderr, "--%s", optList[i].long_opt);
       for (col = strlen(optList[i].long_opt); col < 30; col++) {
	 fprintf(stderr, " ");
       }
       fprintf(stderr, "%s\n", optList[i].description);
     }
   }
   fprintf(stderr, "\nPlease consult the Eterm(1) man page for more detailed\n");
   fprintf(stderr, "information on command line options.\n\n");
   exit (EXIT_FAILURE);
}

/* This defines how many mistakes to allow before giving up
   and printing the usage                          -- mej   */
#define BAD_THRESHOLD 3
#define CHECK_BAD()  do { \
	               if (++bad_opts >= BAD_THRESHOLD) { \
			 print_error("error threshold exceeded, giving up"); \
			 usage(); \
		       } else { \
			 print_error("attempting to continue, but performance may be unpredictable"); \
		       } \
                     } while(0)

void 
get_options(int argc, char *argv[]) {

  register unsigned long i, j, l;
  unsigned char bad_opts = 0;

  for (i=1; i < argc; i++) {

    register char *opt = argv[i];
    char *val_ptr = NULL;
    unsigned char islong = 0, hasequal = 0;

#ifdef DEBUG_OPTIONS
    fprintf(stderr, "[debug] argv[%d] == \"%s\"\n", i, argv[i]);
    fflush(stderr);
#endif

    if (*opt != '-') {
      print_error("unexpected argument %s -- expected option", opt);
      CHECK_BAD();
      continue;
    }
    if (*(opt+1) == '-') {
      islong = 1;
#ifdef DEBUG_OPTIONS
      fprintf(stderr, "[debug] Long option detected\n");
      fflush(stderr);
#endif
    }
    if (islong) {
      opt += 2;
      for (j=0; j < optList_numoptions(); j++) {
	if (!strncasecmp(optList[j].long_opt, opt, (l = strlen(optList[j].long_opt))) && 
	    (opt[l] == '=' || !opt[l])) {
#ifdef DEBUG_OPTIONS
	  fprintf(stderr, "[debug] Match found at %d:  %s == %s\n", j,
		  optList[j].long_opt, opt);
	  fflush(stderr);
#endif
	  break;
	}
      }
      if (j == optList_numoptions()) {
	print_error("unrecognized long option --%s", opt);
	CHECK_BAD();
	continue;
      }

      /* Put option-specific warnings here -- mej */
#if 0    /* No longer needed, since it works :) */
      if (optList[j].short_opt == 'w') {
	print_error("warning:  Use of the -w / --border-width option is discouraged and unsupported.");
      }
#endif

      if ((val_ptr = strchr(opt, '=')) != NULL) {
	*val_ptr = 0;
	val_ptr++;
	hasequal = 1;
      } else {
	if (argv[i+1]) {
	  if (*argv[i+1] != '-' || StrCaseStr(optList[j].long_opt, "font")) {
	    val_ptr = argv[++i];
	  }
	}
      }
#ifdef DEBUG_OPTIONS
      fprintf(stderr, "[debug] hasequal == %d  val_ptr == %10.8p \"%s\"\n",
	      hasequal, val_ptr, val_ptr);
      fflush(stderr);
#endif
      if (j == 0 || j == 1) {
	continue;
      }

      if (!(optList[j].flag & OPT_BOOLEAN) && (val_ptr == NULL)) {
	print_error("long option --%s requires a%s value", opt,
		    (optList[j].flag & OPT_INTEGER ? "n integer" : " string"));
	CHECK_BAD();
	continue;
      }
      if (!strcasecmp(opt, "exec")) {
#ifdef DEBUG_OPTIONS
	fprintf(stderr, "[debug] --exec option detected\n");
#endif
	Options |= Opt_exec;
	if (!hasequal) {

	  register unsigned short k, len = argc - i;

	  rs_execArgs = (char **) malloc(sizeof(char *) * (argc-i+1));
	  for (k=0; k < len; k++) {
	    rs_execArgs[k] = strdup(argv[k+i]);
#ifdef DEBUG_OPTIONS
	    fprintf(stderr, "[debug] rs_execArgs[%d] == %s\n",
		    k, rs_execArgs[k]);
#endif
	  }
	  rs_execArgs[k] = (char *) NULL;
	  return;
	} else {

	  register unsigned short k;

	  rs_execArgs = (char **) malloc(sizeof(char *) * (NumWords(val_ptr) + 1));
	  for (k=0; val_ptr; k++) {
	    rs_execArgs[k] = Word(1, val_ptr);
	    val_ptr = PWord(2, val_ptr);
#ifdef DEBUG_OPTIONS
	    fprintf(stderr, "[debug] rs_execArgs[%d] == %s\n", k, rs_execArgs[k]);
#endif
	  }
	  rs_execArgs[k] = (char *) NULL;
	}
      } else if (!strcasecmp(opt, "help")) {
	usage();
      } else {  /* It's not --exec */
	if (optList[j].flag & OPT_BOOLEAN) { /* Boolean value */
#ifdef DEBUG_OPTIONS
	  fprintf(stderr, "[debug] Boolean option detected\n");
#endif
	  if (val_ptr) {
	    if (!strcasecmp(val_ptr, true_vals[0])
		|| !strcasecmp(val_ptr, true_vals[1])
		|| !strcasecmp(val_ptr, true_vals[2])) {
#ifdef DEBUG_OPTIONS
	      fprintf(stderr, "[debug] \"%s\" == TRUE\n", val_ptr);
#endif
	      Options |= optList[j].mask;
	      if (optList[j].pval) {
		*((const char **) optList[j].pval) = *true_vals;
	      }
	    } else if (!strcasecmp(val_ptr, false_vals[0])
		       || !strcasecmp(val_ptr, false_vals[1])
		       || !strcasecmp(val_ptr, false_vals[2])) {
#ifdef DEBUG_OPTIONS
	      fprintf(stderr, "[debug] \"%s\" == FALSE\n", val_ptr);
#endif
	      Options &= ~(optList[j].mask);
	      if (optList[j].pval) {
		*((const char **) optList[j].pval) = *false_vals;
	      }
	    } else {
	      print_error("unrecognized boolean value \"%s\" for option --%s",
			  val_ptr, optList[j].long_opt);
	      CHECK_BAD();
	    }
	  } else { /* No value, so force it on */
#ifdef DEBUG_OPTIONS
	    fprintf(stderr, "[debug] Forcing option --%s to TRUE\n", opt);
#endif
	    Options |= optList[j].mask;
	    if (optList[j].pval) {
	      *((const char **) optList[j].pval) = *true_vals;
	    }
	  }
	} else if (optList[j].flag & OPT_INTEGER) {  /* Integer value */
#ifdef DEBUG_OPTIONS
	  fprintf(stderr, "[debug] Integer option detected\n");
#endif
	  *((int *) optList[j].pval) = strtol(val_ptr, (char **) NULL, 0);
	} else {  /* String value */
#ifdef DEBUG_OPTIONS
	  fprintf(stderr, "[debug] String option detected\n");
#endif
	  if (val_ptr && optList[j].pval) {
	    *((const char **) optList[j].pval) = strdup(val_ptr);
	  }
	}
      }
    } else { /* It's a POSIX option */

      register unsigned short pos;
      unsigned char done = 0;

      for (pos = 1; opt[pos] && !done; pos++) {
	for (j=0; j < optList_numoptions(); j++) {
	  if (optList[j].short_opt == opt[pos]) {
#ifdef DEBUG_OPTIONS
	    fprintf(stderr, "[debug] Match found at %d:  %c == %c\n", j,
		    optList[j].short_opt, opt[pos]);
	    fflush(stderr);
#endif
	    break;
	  }
	}
	if (j == optList_numoptions()) {
	  print_error("unrecognized option -%c", opt[pos]);
	  CHECK_BAD();
	  continue;
	}

      /* Put option-specific warnings here -- mej */
#if 0  /* No longer needed, since it works :) */
	if (optList[j].short_opt == 'w') {
	  print_error("warning:  Use of the -w / --border-width option is discouraged and unsupported.");
	}
#endif

	if (!(optList[j].flag & OPT_BOOLEAN)) {
	  if (opt[pos+1]) {
	    val_ptr = opt + pos + 1;
	    done = 1;
	  } else if ((val_ptr = argv[++i]) != NULL) {
	    done = 1;
	  }
#ifdef DEBUG_OPTIONS
	  fprintf(stderr, "[debug] val_ptr == %s  done == %d\n", val_ptr, done);
#endif
	  if (j == 0 || j == 1) {
	    continue;
	  }

	  if ((val_ptr == NULL) || ((*val_ptr == '-') && (optList[j].short_opt != 'F'))) {
	    print_error("option -%c requires a%s value", opt[pos],
			(optList[j].flag & OPT_INTEGER ? "n integer" : " string"));
	    CHECK_BAD();
	    if (val_ptr) { /* If the "arg" was actually an option, don't skip it */
	      i--;
	    }
	    continue;
	  }
	}
	if (opt[pos] == 'e') {  /* It's an exec */

	  register unsigned short k, len;

#ifdef DEBUG_OPTIONS
	  fprintf(stderr, "[debug] -e option detected\n");
#endif
	  Options |= Opt_exec;

	  if (opt[pos+1]) {
	    len = argc - i + 2;
	    k = i;
	  } else {
	    len = argc - i + 1;
	    k = i + 1;
	  }
#ifdef DEBUG_OPTIONS
	  fprintf(stderr, "[debug] len == %d  k == %d\n", len, k);
#endif
	  rs_execArgs = (char **) malloc(sizeof(char *) * len);
	  if (k == i) {
	    rs_execArgs[0] = strdup((char *) (val_ptr));
#ifdef DEBUG_OPTIONS
	    fprintf(stderr, "[debug] rs_execArgs[0] == %s\n", rs_execArgs[0]);
#endif
	    k++;
	  } else {
	    rs_execArgs[0] = strdup(argv[k-1]);
#ifdef DEBUG_OPTIONS
	    fprintf(stderr, "[debug] rs_execArgs[0] == %s\n", rs_execArgs[0]);
#endif
	  }
	  for (; k < argc; k++) {
	    rs_execArgs[k-i] = strdup(argv[k]);
#ifdef DEBUG_OPTIONS
	    fprintf(stderr, "[debug] rs_execArgs[%d] == %s\n", k-i, rs_execArgs[k-i]);
#endif
	  }
	  rs_execArgs[len-1] = (char *) NULL;
	  return;
	} else if (opt[pos] == 'h') {
	  usage();

	} else {
	  if (optList[j].flag & OPT_BOOLEAN) { /* Boolean value */
#ifdef DEBUG_OPTIONS
	    fprintf(stderr, "[debug] Boolean option detected\n");
#endif
	    Options |= optList[j].mask;
	    if (optList[j].pval) {
	      *((const char **) optList[j].pval) = *true_vals;
	    }
	  } else if (optList[j].flag & OPT_INTEGER) {  /* Integer value */
#ifdef DEBUG_OPTIONS
	    fprintf(stderr, "[debug] Integer option detected\n");
#endif
	    *((int *) optList[j].pval) = strtol(val_ptr, (char **) NULL, 0);
#ifdef DEBUG_OPTIONS
	    fprintf(stderr, "[debug] Got value %d\n", *((int *) optList[j].pval));
#endif
	  } else {  /* String value */
#ifdef DEBUG_OPTIONS
	    fprintf(stderr, "[debug] String option detected\n");
#endif
	    if (optList[j].pval) {
	      *((const char **) optList[j].pval) = strdup(val_ptr);
	    }
	  } /* End if value type */
	} /* End if option type */
      } /* End if (exec or help or other) */
    } /* End if (islong) */
  } /* End main for loop */
}

#ifdef USE_THEMES
void 
get_initial_options(int argc, char *argv[]) {

  register unsigned long i, j;

  for (i=1; i < argc; i++) {

    register char *opt = argv[i];
    char *val_ptr = NULL;
    unsigned char islong = 0, hasequal = 0;

#ifdef DEBUG_OPTIONS
    fprintf(stderr, "[debug] argv[%d] == \"%s\"\n", i, argv[i]);
    fflush(stderr);
#endif

    if (*opt != '-') {
      continue;
    }
    if (*(opt+1) == '-') {
      islong = 1;
#ifdef DEBUG_OPTIONS
      fprintf(stderr, "[debug] Long option detected\n");
      fflush(stderr);
#endif
    }
    if (islong) {
      opt += 2;
      if (!strncasecmp(opt, "theme", 5)) {
	j = 0;
      } else if (!strncasecmp(opt, "config-file", 11)) {
	j = 1;
      } else continue;

      if ((val_ptr = strchr(opt, '=')) != NULL) {
	*val_ptr = 0;
	val_ptr++;
	hasequal = 1;
      } else {
	if (argv[i+1]) {
	  if (*argv[i+1] != '-') {
	    val_ptr = argv[++i];
	  }
	}
      }
#ifdef DEBUG_OPTIONS
      fprintf(stderr, "[debug] hasequal == %d  val_ptr == %10.8p \"%s\"\n",
	      hasequal, val_ptr, val_ptr);
      fflush(stderr);
#endif
      if (val_ptr == NULL) {
	print_error("long option --%s requires a string value", opt);
	continue;
      }
#ifdef DEBUG_OPTIONS
      fprintf(stderr, "[debug] String option detected\n");
#endif
      if (val_ptr && optList[j].pval) {
	*((const char **) optList[j].pval) = strdup(val_ptr);
      }
    } else { /* It's a POSIX option */

      register unsigned short pos;
      unsigned char done = 0;

      for (pos = 1; opt[pos] && !done; pos++) {
	if (opt[pos] == 't') {
	  j = 0;
	} else if (opt[pos] == 'X') {
	  j = 1;
	} else continue;

	if (opt[pos+1]) {
	  val_ptr = opt + pos + 1;
	  done = 1;
	} else if ((val_ptr = argv[++i]) != NULL) {
	  done = 1;
	}
#ifdef DEBUG_OPTIONS
	fprintf(stderr, "[debug] val_ptr == %s  done == %d\n", val_ptr, done);
#endif
	if ((val_ptr == NULL) || (*val_ptr == '-')) {
	  print_error("option -%c requires a string value", opt[pos]);
	  if (val_ptr) { /* If the "arg" was actually an option, don't skip it */
	    i--;
	  }
	  continue;
	}
#ifdef DEBUG_OPTIONS
	fprintf(stderr, "[debug] String option detected\n");
#endif
	if (optList[j].pval) {
	  *((const char **) optList[j].pval) = strdup(val_ptr);
	}
      } /* End for loop */
    } /* End if (islong) */
  } /* End main for loop */
}
#endif

/***** The Config File Section *****/

/* Max length of a line in the config file */
#define CONFIG_BUFF 20480

/* The context identifier.  This tells us what section of the config file
   we're in, for syntax checking purposes and the like.            -- mej */

#define CTX_NULL         0
#define CTX_MAIN         1
#define CTX_COLOR        2
#define CTX_ATTRIBUTES   3
#define CTX_TOGGLES      4
#define CTX_KEYBOARD     5
#define CTX_MISC         6
#define CTX_PIXMAPS      7
#define CTX_KANJI        8
#define CTX_UNDEF        ((unsigned char) -1)
#define CTX_MAX          8

/* This structure defines a context and its attributes */

struct context_struct {

  const unsigned char id;
  const char *description;
  void (*ctx_handler)(char *);
  const unsigned char valid_sub_contexts[CTX_MAX];

} contexts[] = {

  { CTX_NULL, "none", NULL, { CTX_MAIN, 0, 0, 0, 0, 0, 0, 0 } },
  { CTX_MAIN, "main", parse_main, { CTX_COLOR, CTX_ATTRIBUTES, CTX_TOGGLES,
                                    CTX_KEYBOARD, CTX_MISC, CTX_PIXMAPS,
                                    CTX_KANJI, 0 }
  },
  { CTX_COLOR, "color", parse_color, { 0, 0, 0, 0, 0, 0, 0, 0 } },
  { CTX_ATTRIBUTES, "attributes", parse_attributes, { 0, 0, 0, 0, 0, 0, 0, 0 } },
  { CTX_TOGGLES, "toggles", parse_toggles, { 0, 0, 0, 0, 0, 0, 0, 0 } },
  { CTX_KEYBOARD, "keyboard", parse_keyboard, { 0, 0, 0, 0, 0, 0, 0, 0 } },
  { CTX_MISC, "misc", parse_misc, { 0, 0, 0, 0, 0, 0, 0, 0 } },
  { CTX_PIXMAPS, "pixmaps", parse_pixmaps, { 0, 0, 0, 0, 0, 0, 0, 0 } },
  { CTX_KANJI, "kanji", parse_kanji, { 0, 0, 0, 0, 0, 0, 0, 0 } },
  { CTX_UNDEF, NULL, parse_undef, { 0, 0, 0, 0, 0, 0, 0, 0 } }

};

#define ctx_name_to_id(the_id, n, i) do { \
                                       for ((i)=0; (i) <= CTX_MAX; (i)++) { \
                                         if (!strcasecmp((n), contexts[(i)].description)) { \
		                           (the_id) = contexts[(i)].id; \
					   break; \
					 } \
			               } \
                                       if ((i) > CTX_MAX) (the_id) = CTX_UNDEF; \
                                     } while (0)

#define ctx_id_to_name(id) (contexts[(id)].description)
#define ctx_id_to_func(id) (contexts[(id)].ctx_handler)

/* The context stack.  This keeps track of the current context and each
   previous one.  You MUST define MAX_CTX_DEPTH to the absolute maximum
   number of context levels deep your contexts go, or the results can be
   Very Bad.  I recommend erring on the side of caution.          -- mej */

#define MAX_CTX_DEPTH 10
#define ctx_push(ctx) id_stack[++cur_ctx] = (ctx)
#define ctx_pop()  (id_stack[cur_ctx--])
#define ctx_peek() (id_stack[cur_ctx])

unsigned char id_stack[MAX_CTX_DEPTH];
unsigned short cur_ctx = 0;

/* The file state stack.  This keeps track of the file currently being
   parsed.  This allows for %include directives.                  -- mej */

typedef struct file_state_struct {

  FILE *fp;
  char *path;
  unsigned long line;
  unsigned char skip_to_end;

} file_state;

#define MAX_FILE_DEPTH 10
#define file_push(fs) do { \
                        cur_file++; \
                        file_stack[cur_file].fp = (fs).fp; \
                        file_stack[cur_file].path = (fs).path; \
                        file_stack[cur_file].line = (fs).line; \
			file_stack[cur_file].skip_to_end = (fs).skip_to_end; \
                      } while (0)

#define file_pop()    (cur_file--)
#define file_peek(fs) do { \
                        (fs).fp = file_stack[cur_file].fp; \
                        (fs).path = file_stack[cur_file].path; \
                        (fs).line = file_stack[cur_file].line; \
			(fs).skip_to_end = file_stack[cur_file].skip_to_end; \
                      } while (0)
#define file_peek_fp()   (file_stack[cur_file].fp)
#define file_peek_path() (file_stack[cur_file].path)
#define file_peek_line() (file_stack[cur_file].line)
#define file_peek_skip() (file_stack[cur_file].skip_to_end)

#define file_poke_fp(f)    ((file_stack[cur_file].fp) = (f))
#define file_poke_path(p)  ((file_stack[cur_file].path) = (p))
#define file_poke_line(l)  ((file_stack[cur_file].line) = (l))
#define file_poke_skip(s)  ((file_stack[cur_file].skip_to_end) = (s))

#define file_inc_line()     (file_stack[cur_file].line++)

file_state file_stack[MAX_FILE_DEPTH];
unsigned short cur_file = 0;

/* The function structures */

typedef char *(*eterm_function_ptr)(char *);
typedef struct eterm_function_struct {

  char *name;
  eterm_function_ptr ptr;
  int params;

} eterm_func;

eterm_func builtins[] = {
  { "random",      builtin_random,            -1 },
  { "exec",        builtin_exec,              -1 },
  { "version",     builtin_version,            0 },
  { "appname",     builtin_appname,            0 },
  { (char *) NULL, (eterm_function_ptr) NULL,  0 }
};

char *
builtin_random(char *param) {

  unsigned long n, index;

#ifdef DEBUG_OPTIONS
  fprintf(stderr, "[debug] builtin_random(%s) called\n", param);
#endif

  srand(getpid());
  n = NumWords(param);
  index = (int) (n*((float)rand())/(RAND_MAX+1.0)) + 1;
#ifdef DEBUG_OPTIONS
  fprintf(stderr, "[debug] random index == %lu\n", index);
#endif

  return (Word(index, param));
}

char *
builtin_exec(char *param) {

#ifdef DEBUG_OPTIONS
  fprintf(stderr, "[debug] builtin_exec(%s) called\n", param);
#endif

  return (param);
}

char *
builtin_version(char *param) {

#ifdef DEBUG_OPTIONS
  fprintf(stderr, "[debug] builtin_version(%s) called\n", param);
#endif

  return (Word(1, VERSION_NUM));
}

char *
builtin_appname(char *param) {

#ifdef DEBUG_OPTIONS
  fprintf(stderr, "[debug] builtin_appname(%s) called\n", param);
#endif

  return (Word(1, APL_NAME "-" VERSION));
}

/* chomp() removes leading and trailing whitespace/quotes from a string */
char *
chomp(char *s) {

  register char *front, *back;

  for (front = s; *front && isspace(*front); front++);
  /*
  if (*front == '\"') front++;
  */
  for (back = s + strlen(s) - 1; *back && isspace(*back) && back > front; back--);
  /*
  if (*back == '\"') back--;
  */

  *(++back) = 0;
  if (front != s) memmove(s, front, back - front + 1);
  return (s);
}

/* shell_expand() takes care of shell variable expansion, quote conventions,
   calling of built-in functions, etc.                                -- mej */
char *
shell_expand(char *s) {

  register char *tmp;
  register char *new;
  register char *pbuff = s, *tmp1;
  register unsigned long j,k,l;
  unsigned char eval_escape = 1, eval_var = 1, eval_exec = 1, eval_func = 1, in_single = 0, in_double = 0;
  unsigned long fsize;
  char *Command, *Output, *EnvVar, *OutFile;
  FILE *fp;

  new = (char *) MALLOC(CONFIG_BUFF);

  for (j=0; *pbuff && j < CONFIG_BUFF; pbuff++, j++) {
    switch (*pbuff) {
    case '~':
#ifdef DEBUG_OPTIONS
      fprintf(stderr, "mejstr: [debug] Tilde detected.\n");
#endif
      if (eval_var) {
	strncpy(new+j, getenv("HOME"), CONFIG_BUFF - j);
	j += strlen(getenv("HOME")) - 1;
      } else {
	new[j] = *pbuff;
      }
      break;
    case '\\':           
#ifdef DEBUG_OPTIONS
      fprintf(stderr, "mejstr: [debug] Escape sequence detected.\n");
#endif
      if (eval_escape) {
	switch (tolower(*(++pbuff))) {
	case 'n':  new[j] = '\n'; break;
	case 'r':  new[j] = '\r'; break;
	case 't':  new[j] = '\t'; break;
	case 'b':  j -= 2; break;
	case 'f':  new[j] = '\f'; break;
	case 'a':  new[j] = '\a'; break;
	case 'v':  new[j] = '\v'; break;
	case 'e':  new[j] = '\033'; break;
	default:   new[j] = *pbuff; break;
	}
      } else {
	new[j++] = *(pbuff++);
	new[j] = *pbuff;
      }
      break;
    case '%':
#ifdef DEBUG_OPTIONS
      fprintf(stderr, "mejstr: [debug] %% detected.\n");
#endif
      for (k=0, pbuff++; builtins[k].name != NULL; k++) {
#ifdef DEBUG_OPTIONS
	fprintf(stderr, "mejstr: [debug] Checking for function %%%s\n", builtins[k].name);
	fprintf(stderr, "mejstr: [debug] pbuff == \"%s\"\n", pbuff);
#endif
	l = strlen(builtins[k].name);
	if (!strncasecmp(builtins[k].name, pbuff, l) && 
	    ((pbuff[l] == '(') || (pbuff[l] == ' ' && pbuff[l+1] == ')'))) {
          break;
	}
      }
      if (builtins[k].name == NULL) {
	new[j] = *pbuff;
      } else {
#ifdef DEBUG_OPTIONS
	fprintf(stderr, "mejstr: [debug] Call to built-in function %s detected.\n", builtins[k].name);
#endif
	Command = (char *) MALLOC(CONFIG_BUFF);
	pbuff += l;
	if (*pbuff != '(') pbuff++;
	for (tmp1 = Command, pbuff++, l=1; l && *pbuff; pbuff++, tmp1++) {
	  switch (*pbuff) {
	  case '(':
	    l++;
	    *tmp1 = *pbuff;
	    break;
	  case ')':
	    l--;
	  default:
	    *tmp1 = *pbuff;
	    break;
	  }
	}
	*(--tmp1) = 0;
	if (l) {
	  print_error("parse error in file %s, line %lu:  Mismatched parentheses",
		      file_peek_path(), file_peek_line());
	  return ((char *) NULL);
	}
	Command = shell_expand(Command);
	Output = (builtins[k].ptr)(Command);
	FREE(Command);
	l = strlen(Output);
	strncpy(new+j, Output, CONFIG_BUFF - j);
	j += strlen(Output) - 1;
	FREE(Output);
	pbuff--;
      }
      break;
    case '`':
#ifdef ALLOW_BACKQUOTE_EXEC
#  ifdef DEBUG_OPTIONS
      fprintf(stderr, "mejstr: [debug] Backquotes detected.  Evaluating expression.\n");
#  endif
      if (eval_exec) {
	Command = (char *) MALLOC(CONFIG_BUFF);
	l = 0;
	for (pbuff++; *pbuff && *pbuff != '`' && l < CONFIG_BUFF; pbuff++, l++) {
	  switch (*pbuff) {
	  case '$':
#  ifdef DEBUG_OPTIONS
	    fprintf(stderr, "mejstr: [debug] Environment variable detected.  Evaluating.\n");
#  endif
	    EnvVar = (char *) MALLOC(128);
	    switch (*(++pbuff)) {
	    case '{':
	      for (pbuff++, k=0; *pbuff != '}' && k < 127; k++, pbuff++) EnvVar[k] = *pbuff;
	      break;
	    case '(':
	      for (pbuff++, k=0; *pbuff != ')' && k < 127; k++, pbuff++) EnvVar[k] = *pbuff;
	      break;
	    default:
	      for (k=0; (isalnum(*pbuff) || *pbuff == '_') && k < 127; k++, pbuff++) 
		EnvVar[k] = *pbuff;
	      break;
	    }
	    EnvVar[k] = 0;
	    if ((tmp = getenv(EnvVar))) {
	      strncpy(Command+l, tmp, CONFIG_BUFF - l);
	      l = strlen(Command) - 1;
	    }
	    pbuff--;
	    break;
	  default:
	    Command[l] = *pbuff;
	  }
	}
	OutFile = tmpnam(NULL);
	if (l + strlen(OutFile) + 8 > CONFIG_BUFF) {
	  print_error("parse error in file %s, line %lu:  Cannot execute command, line too long",
		      file_peek_path(), file_peek_line());
	  return ((char *) NULL);
	}
	strcat(Command, " >");
	strcat(Command, OutFile);
	strcat(Command, " 2>&1");
	system(Command);
	fp = fopen(OutFile, "rb");
	fseek(fp, 0, SEEK_END);
	fsize = ftell(fp);
	rewind(fp);
	Output = (char *) MALLOC(fsize+1);
	fread(Output, fsize, 1, fp);
	Output[fsize] = 0;
#  ifdef DEBUG_OPTIONS
	fprintf(stderr, "mejstr: [debug] Command returned \"%s\"\n", Output);
#  endif
	fclose(fp);
	remove(OutFile);
	Output = CondenseWhitespace(Output);
	strncpy(new+j, Output, CONFIG_BUFF - j);
	j += strlen(Output) - 1;
	FREE(Output);
	FREE(Command);
      } else {
	new[j] = *pbuff;
      }
#else
      print_error("warning:  backquote execution support not compiled in, ignoring");
      new[j] = *pbuff;
#endif
      break;
    case '$':
#ifdef DEBUG_OPTIONS
      fprintf(stderr, "mejstr: [debug] Environment variable detected.  Evaluating.\n");
#endif
      if (eval_var) {
	EnvVar = (char *) MALLOC(128);
	switch (*(++pbuff)) {
	case '{':
	  for (pbuff++, k=0; *pbuff != '}' && k < 127; k++, pbuff++) EnvVar[k] = *pbuff;
	  break;
	case '(':
	  for (pbuff++, k=0; *pbuff != ')' && k < 127; k++, pbuff++) EnvVar[k] = *pbuff;
	  break;
	default:
	  for (k=0; (isalnum(*pbuff) || *pbuff == '_') && k < 127; k++, pbuff++) 
	    EnvVar[k] = *pbuff;
	  break;
	}
	EnvVar[k] = 0;
	if ((tmp = getenv(EnvVar))) {
	  strncpy(new, tmp, CONFIG_BUFF - j);
	  j += strlen(tmp) - 1;
	}
	pbuff--;
      } else {
	new[j] = *pbuff;
      }
      break;
    case '\"':
#ifdef DEBUG_OPTIONS
      fprintf(stderr, "mejstr: [debug] Double quotes detected.\n");
#endif
      if (!in_single) {
	if (in_double) {
	  in_double = 0;
	} else {
	  in_double = 1;
	}
      }
      new[j] = *pbuff;
      break;

    case '\'':
#ifdef DEBUG_OPTIONS
      fprintf(stderr, "mejstr: [debug] Single quotes detected.\n");
#endif
      if (in_single) {
	eval_var = 1;
	eval_exec = 1;
	eval_func = 1;
	eval_escape = 1;
	in_single = 0;
      } else {
	eval_var = 0;
	eval_exec = 0;
	eval_func = 0;
	eval_escape = 0;
	in_single = 1;
      }
      new[j] = *pbuff;
      break;

    default:
      new[j] = *pbuff;
    }
  }
  new[j] = 0;

#ifdef DEBUG_OPTIONS
  fprintf(stderr, "[debug] shell_expand(%s) returning \"%s\"\n", s, new);
#endif

  strcpy(s, new);
  FREE(new);
  return (s);
}

/* The config file parsers.  Each function handles a given context. */

void 
parse_main(char *buff) {

  print_error("parse error in file %s, line %lu:  Attribute "
	      "\"%s\" is not valid within context main\n",
	      file_peek_path(), file_peek_line(), buff);
}

void 
parse_color(char *buff) {

  if (!strncasecmp(buff, "foreground ", 11)) {
    rs_color[fgColor] = Word(2, buff);
  } else if (!strncasecmp(buff, "background ", 11)) {
    rs_color[bgColor] = Word(2, buff);
  } else if (!strncasecmp(buff, "cursor ", 7)) {

#ifndef NO_CURSORCOLOR
    rs_color[cursorColor] = Word(2, buff);
#else
    print_error("warning:  support for the cursor attribute was not compiled in, ignoring");
#endif

  } else if (!strncasecmp(buff, "cursor_text ", 12)) {
#ifndef NO_CURSORCOLOR
    rs_color[cursorColor2] = Word(2, buff);
#else
    print_error("warning:  support for the cursor_text attribute was not compiled in, ignoring");
#endif

  } else if (!strncasecmp(buff, "scrollbar ", 10)) {
#if !defined(XTERM_SCROLLBAR) && defined(KEEP_SCROLLCOLOR)
    rs_color[scrollColor] = Word(2, buff);
#else
    print_error("warning:  support for the scrollbar color attribute was not compiled in, ignoring");
#endif

  } else if (!strncasecmp(buff, "pointer ", 8)) {
    rs_color[pointerColor] = Word(2, buff);

  } else if (!strncasecmp(buff, "video ", 6)) {

    char *tmp = PWord(2, buff);

    if (!strncasecmp(tmp, "reverse", 7)) {
      Options |= Opt_reverseVideo;
      rs_reverseVideo = *true_vals;
    } else if (strncasecmp(tmp, "normal", 6)) {
      print_error("parse error in file %s, line %lu:  Invalid value \"%s\" for attribute video",
		  file_peek_path(), file_peek_line(), tmp);
    }
  } else if (!strncasecmp(buff, "color ", 6)) {

    char *tmp, *r1, *g1, *b1;
    unsigned long n, r, g, b, index;

    n = NumWords(buff);
    if (n < 3) {
      print_error("parse error in file %s, line %lu:  Invalid parameter list \"%s\" for "
		  "attribute color", file_peek_path(), file_peek_line(), tmp);
      return;
    }

    tmp = PWord(2, buff);
    r1 = PWord(3, buff);
    if (!isdigit(*r1)) {
      if (isdigit(*tmp)) {
	n = strtoul(tmp, (char **) NULL, 0);
	if (n >= 0 && n <= 7) {
	  index = minColor+n;
	} else if (n >= 8 && n <= 15) {
	  index = minBright + n - 8;
	}
	rs_color[index] = Word(1, r1);
	return;
      } else {
	if (!strncasecmp(tmp, "bd ", 3)) {
#ifndef NO_BOLDUNDERLINE
	  rs_color[colorBD] = Word(1, r1);
#else
	  print_error("warning:  support for the color bd attribute was not compiled in, ignoring");
#endif
	  return;
	} else if (!strncasecmp(tmp, "ul ", 3)) {
#ifndef NO_BOLDUNDERLINE
	  rs_color[colorUL] = Word(1, r1);
#else
	  print_error("warning:  support for the color ul attribute was not compiled in, ignoring");
#endif
	  return;
	} else {
	  tmp = Word(1, tmp);
	  print_error("parse error in file %s, line %lu:  Invalid color index \"%s\"",
		      file_peek_path(), file_peek_line(), tmp);
	  free(tmp);
	}
      }
    }
      
    if (n != 5) {
      print_error("parse error in file %s, line %lu:  Invalid parameter list \"%s\" for "
		  "attribute color", file_peek_path(), file_peek_line(), tmp);
      return;
    }

    g1 = PWord(4, buff);
    b1 = PWord(5, buff);
    if (isdigit(*tmp)) {
      n = strtoul(tmp, (char **) NULL, 0);
      r = strtoul(r1, (char **) NULL, 0);
      g = strtoul(g1, (char **) NULL, 0);
      b = strtoul(b1, (char **) NULL, 0);
      if (n >= 0 && n <= 7) {
	index = minColor+n;
	rs_color[index] = MALLOC(14);
	sprintf(rs_color[index], "#%02x%02x%02x", r, g, b);
      } else if (n >= 8 && n <= 15) {
	index = minBright + n - 8;
	rs_color[index] = MALLOC(14);
	sprintf(rs_color[index], "#%02x%02x%02x", r, g, b);
      } else {
	print_error("parse error in file %s, line %lu:  Invalid color index %lu",
		    file_peek_path(), file_peek_line(), n);
      }

    } else if (!strncasecmp(tmp, "bd ", 3)) {
#ifndef NO_BOLDUNDERLINE
      rs_color[colorBD] = MALLOC(14);
      r = strtoul(r1, (char **) NULL, 0);
      g = strtoul(g1, (char **) NULL, 0);
      b = strtoul(b1, (char **) NULL, 0);
      sprintf(rs_color[colorBD], "#%02x%02x%02x", r, g, b);
#else
      print_error("warning:  support for the color bd attribute was not compiled in, ignoring");
#endif

    } else if (!strncasecmp(tmp, "ul ", 3)) {
#ifndef NO_BOLDUNDERLINE
      rs_color[colorUL] = MALLOC(14);
      r = strtoul(r1, (char **) NULL, 0);
      g = strtoul(g1, (char **) NULL, 0);
      b = strtoul(b1, (char **) NULL, 0);
      sprintf(rs_color[colorUL], "#%02x%02x%02x", r, g, b);
#else
      print_error("warning:  support for the color ul attribute was not compiled in, ignoring");
#endif

    } else {
      tmp = Word(1, tmp);
      print_error("parse error in file %s, line %lu:  Invalid color index \"%s\"",
		    file_peek_path(), file_peek_line(), tmp);
      free(tmp);
    }
  } else {
    print_error("parse error in file %s, line %lu:  Attribute \"%s\" is not valid "
		"within context color", file_peek_path(), file_peek_line(), buff);
  }
}

void
parse_attributes(char *buff) {

  if (!strncasecmp(buff, "geometry ", 9)) {
    rs_geometry = Word(2, buff);

  } else if (!strncasecmp(buff, "title ", 6)) {
    rs_title = Word(2, buff);

  } else if (!strncasecmp(buff, "name ", 5)) {
    rs_name = Word(2, buff);

  } else if (!strncasecmp(buff, "iconname ", 9)) {
    rs_iconName = Word(2, buff);

  } else if (!strncasecmp(buff, "font ", 5)) {

    char *tmp = PWord(2, buff);
    unsigned char n;

    if (NumWords(buff) != 3) {
      print_error("parse error in file %s, line %lu:  Invalid parameter list \"%s\" for "
		  "attribute font", file_peek_path(), file_peek_line(), tmp);
      return;
    }
    if (isdigit(*tmp)) {
      n = (unsigned char) strtoul(tmp, (char **) NULL, 0);
      if (n <= 4) {
	rs_font[n] = Word(2, tmp);
      } else {
	print_error("parse error in file %s, line %lu:  Invalid font index %d",
		    file_peek_path(), file_peek_line(), n);
      }
    } else if (!strncasecmp(tmp, "bold ", 5)) {
#ifndef NO_BOLDFONT
      rs_boldFont = Word(2, tmp);
#else
      print_error("warning:  support for the bold font attribute was not compiled in, ignoring");
#endif

    } else {
      tmp = Word(1, tmp);
      print_error("parse error in file %s, line %lu:  Invalid font index \"%s\"",
		    file_peek_path(), file_peek_line(), tmp);
      free(tmp);
    }

  } else {
    print_error("parse error in file %s, line %lu:  Attribute \"%s\" is not valid "
		"within context attributes", file_peek_path(), file_peek_line(), buff);
  }
}

void
parse_toggles(char *buff) {

  char *tmp = PWord(2, buff);
  unsigned char bool_val;

  if (!tmp) {
    print_error("parse error in file %s, line %lu:  Missing boolean value in "
		"context toggles", 
		file_peek_path(), file_peek_line());
    return;
  }
  if (!strcasecmp(tmp, true_vals[0]) || !strcasecmp(tmp, true_vals[1]) 
      || !strcasecmp(tmp, true_vals[2])) {
    bool_val = 1;
  } else if (!strcasecmp(tmp, false_vals[0]) || !strcasecmp(tmp, false_vals[1])
	     || !strcasecmp(tmp, false_vals[2])) {
    bool_val = 0;
  } else {
    print_error("parse error in file %s, line %lu:  Invalid boolean value \"%s\" "
		"in context toggles", 
		file_peek_path(), file_peek_line(), tmp, buff);
    return;
  }
    
  if (!strncasecmp(buff, "map_alert ", 10)) {
#if !defined(NO_MAPALERT) && defined(MAPALERT_OPTION)
    if (bool_val) {
      Options |= Opt_mapAlert;
      rs_mapAlert = *true_vals;
    } else {
      Options &= ~(Opt_mapAlert);
      rs_mapAlert = *false_vals;
    }
#else
    print_error("warning:  support for the map_alert attribute was not compiled in, ignoring");
#endif

  } else if (!strncasecmp(buff, "visual_bell ", 12)) {
    if (bool_val) {
      Options |= Opt_visualBell;
      rs_visualBell = *true_vals;
    } else {
      Options &= ~(Opt_visualBell);
      rs_visualBell = *false_vals;
    }
  } else if (!strncasecmp(buff, "login_shell ", 12)) {
    if (bool_val) {
      Options |= Opt_loginShell;
      rs_loginShell = *true_vals;
    } else {
      Options &= ~(Opt_loginShell);
      rs_loginShell = *false_vals;
    }
  } else if (!strncasecmp(buff, "scrollbar ", 10)) {
    if (bool_val) {
      Options |= Opt_scrollBar;
      rs_scrollBar = *true_vals;
    } else {
      Options &= ~(Opt_scrollBar);
      rs_scrollBar = *false_vals;
    }
  } else if (!strncasecmp(buff, "utmp_logging ", 13)) {
#ifdef UTMP_SUPPORT
    if (bool_val) {
      Options |= Opt_utmpLogging;
      rs_utmpLogging = *true_vals;
    } else {
      Options &= ~(Opt_utmpLogging);
      rs_utmpLogging = *false_vals;
    }
#else
    print_error("warning:  support for the utmp_logging attribute was not compiled in, ignoring");
#endif

  } else if (!strncasecmp(buff, "meta8 ", 6)) {
#ifdef META8_OPTION
    if (bool_val) {
      Options |= Opt_meta8;
      rs_meta8 = *true_vals;
    } else {
      Options &= ~(Opt_meta8);
      rs_meta8 = *false_vals;
    }
#else
    print_error("warning:  support for the meta8 attribute was not compiled in, ignoring");
#endif

  } else if (!strncasecmp(buff, "iconic ", 7)) {
    if (bool_val) {
      Options |= Opt_iconic;
    } else {
      Options &= ~(Opt_iconic);
    }

  } else if (!strncasecmp(buff, "home_on_echo ", 13)) {
    if (bool_val) {
      Options |= Opt_homeOnEcho;
      rs_homeOnEcho = *true_vals;
    } else {
      Options &= ~(Opt_homeOnEcho);
      rs_homeOnEcho = *false_vals;
    }

  } else if (!strncasecmp(buff, "home_on_refresh ", 16)) {
    if (bool_val) {
      Options |= Opt_homeOnRefresh;
      rs_homeOnRefresh = *true_vals;
    } else {
      Options &= ~(Opt_homeOnRefresh);
      rs_homeOnRefresh = *false_vals;
    }

  } else if (!strncasecmp(buff, "scrollbar_floating ", 19)) {
    if (bool_val) {
      Options |= Opt_scrollBar_floating;
      rs_scrollBar_floating = *true_vals;
    } else {
      Options &= ~(Opt_scrollBar_floating);
      rs_scrollBar_floating = *false_vals;
    }

  } else if (!strncasecmp(buff, "scrollbar_right ", 16)) {
    if (bool_val) {
      Options |= Opt_scrollBar_right;
      rs_scrollBar_right = *true_vals;
    } else {
      Options &= ~(Opt_scrollBar_right);
      rs_scrollBar_right = *false_vals;
    }

  } else if (!strncasecmp(buff, "borderless ", 11)) {
    if (bool_val) {
      Options |= Opt_borderless;
      rs_borderless = *true_vals;
    } else {
      Options &= ~(Opt_borderless);
      rs_borderless = *false_vals;
    }

  } else {
    print_error("parse error in file %s, line %lu:  Attribute \"%s\" is not valid "
		"within context toggles", file_peek_path(), file_peek_line(), buff);
  }
}

#define to_keysym(p,s) do { KeySym sym; \
                             if (s && ((sym = XStringToKeysym(s)) != 0)) *p = sym; \
                           } while (0)

void
parse_keyboard(char *buff) {

  if (!strncasecmp(buff, "smallfont_key ", 14)) {
#if defined (HOTKEY_CTRL) || defined (HOTKEY_META)
    rs_smallfont_key = Word(2, buff);
    to_keysym(&ks_smallfont, rs_smallfont_key);
#else
    print_error("warning:  support for the smallfont_key attribute was not compiled in, ignoring");
#endif

  } else if (!strncasecmp(buff, "bigfont_key ", 12)) {
#if defined (HOTKEY_CTRL) || defined (HOTKEY_META)
    rs_bigfont_key = Word(2, buff);
    to_keysym(&ks_bigfont, rs_bigfont_key);
#else
    print_error("warning:  support for the bigfont_key attribute was not compiled in, ignoring");
#endif

  } else if (!strncasecmp(buff, "keysym.", 7)) {
#ifdef KEYSYM_ATTRIBUTE

    int sym, len;
    char *str = buff + 7;

    sym = (int) strtol(str, (char **) NULL, 0);
    if (sym != (int) 2147483647L) {

      if (sym >= 0xff00) sym -= 0xff00;
      if (sym < 0 || sym > 0xff) {
	print_error("parse error in file %s, line %lu:  Keysym 0x%x out of range 0xff00-0xffff", 
		    file_peek_path(), file_peek_line(), sym+0xff00);
	return;
      }
      str = Word(2, buff);
      chomp(str);
      len = Str_escaped(str);
      if (len > 255) len = 255; /* We can only handle lengths that will fit in a char */
      if (len && KeySym_map[sym] == NULL) {

	char *p = malloc(len+1);

	*p = len;
	strncpy(p+1, str, len);
	KeySym_map[sym] = p;
      }
    }
#else
    print_error("warning:  support for the keysym.* attributes was not compiled in, ignoring");
#endif

  } else if (!strncasecmp(buff, "greek ", 6)) {
#ifdef GREEK_SUPPORT

    char *tmp = PWord(2, buff);

    if (!tmp) {
      print_error("parse error in file %s, line %lu:  Missing boolean value for attribute greek", 
		  file_peek_path(), file_peek_line());
      return;
    }
    if (!strcasecmp(tmp, true_vals[0]) || !strcasecmp(tmp, true_vals[1]) 
	|| !strcasecmp(tmp, true_vals[2])) {
      rs_greek_keyboard = Word(3, buff);
      if (strncasecmp(rs_greek_keyboard, "iso", 3)) {
	greek_setmode(GREEK_ELOT928);
      } else if (strncasecmp(rs_greek_keyboard, "ibm", 3)) {
	greek_setmode(GREEK_IBM437);
      } else {
	print_error("parse error in file %s, line %lu:  Invalid greek keyboard mode \"%s\"",
		    file_peek_path(), file_peek_line(), rs_greek_keyboard);
      }
    } else if (!strcasecmp(tmp, false_vals[0]) || !strcasecmp(tmp, false_vals[1])
	       || !strcasecmp(tmp, false_vals[2])) {
      /* This space intentionally left no longer blank =^) */
    } else {
      print_error("parse error in file %s, line %lu:  Invalid boolean value \"%s\" for "
		  "attribute %s", file_peek_path(), file_peek_line(), tmp, buff);
      return;
    }
#else
    print_error("warning:  support for the greek attribute was not compiled in, ignoring");
#endif
    
  } else {
    print_error("parse error in file %s, line %lu:  Attribute \"%s\" is not valid "
		"within context keyboard", file_peek_path(), file_peek_line(), buff);
  }
}
#undef to_keysym

void
parse_misc(char *buff) {

  if (!strncasecmp(buff, "print_pipe ", 11)) {
#ifdef PRINTPIPE
    rs_print_pipe = strdup(PWord(2, buff));
    chomp(rs_print_pipe);
#else
    print_error("warning:  support for the print_pipe attribute was not compiled in, ignoring");
#endif

  } else if (!strncasecmp(buff, "save_lines ", 11)) {
    rs_saveLines = strtol(PWord(2, buff), (char **) NULL, 0);

  } else if (!strncasecmp(buff, "border_width ", 13)) {
#ifdef BORDER_WIDTH_OPTION
    TermWin.internalBorder = (short) strtol(PWord(2, buff), (char **) NULL, 0);
#else
    print_error("warning:  support for the border_width attribute was not compiled in, ignoring");
#endif

  } else if (!strncasecmp(buff, "menu ", 5)) {
    rs_menu = Word(2, buff);

  } else if (!strncasecmp(buff, "term_name ", 10)) {
    rs_term_name = Word(2, buff);

  } else if (!strncasecmp(buff, "exec ", 5)) {

    register unsigned short k, n;

    Options |= Opt_exec;

    rs_execArgs = (char **) malloc(sizeof(char *) * ((n = NumWords(PWord(2, buff))) + 1));
    for (k = 0; k < n; k++) {
      rs_execArgs[k] = Word(k+2, buff);
#ifdef DEBUG_OPTIONS
      fprintf(stderr, "[debug] rs_execArgs[%d] == %s\n", k, rs_execArgs[k]);
#endif
    }
    rs_execArgs[n] = (char *) NULL;

  } else if (!strncasecmp(buff, "cut_chars ", 10)) {
#ifdef CUTCHAR_OPTION
    rs_cutchars = Word(2, buff);
    chomp(rs_cutchars);
#else
    print_error("warning:  support for the cut_chars attribute was not compiled in, ignoring");
#endif

  } else {
    print_error("parse error in file %s, line %lu:  Attribute \"%s\" is not valid "
		"within context misc", file_peek_path(), file_peek_line(), buff);
  }
}

void
parse_pixmaps(char *buff) {

  long w, h;
  char *w1, *h1;
  unsigned long n;

#ifdef PIXMAP_SUPPORT
  if (!strncasecmp(buff, "background ", 11)) {

    if ((n = NumWords(buff) < 4) || (n > 6)) {
      print_error("parse error in file %s, line %lu:  Invalid parameter list \"%s\" for attribute "
		"background", file_peek_path(), file_peek_line(), PWord(2, buff));
      return;
    }
    w1 = PWord(2, buff);
    h1 = PWord(3, buff);
    w = strtol(w1, (char **) NULL, 0);
    h = strtol(h1, (char **) NULL, 0);
    if (w || h) {
      rs_pixmapScale = *true_vals;
      Options |= Opt_pixmapScale;
      rs_pixmaps[pixmap_bg] = Word(4, buff);
      rs_pixmaps[pixmap_bg] = (char *) realloc(rs_pixmaps[pixmap_bg], strlen(rs_pixmaps[pixmap_bg]) + 9);
      strcat(rs_pixmaps[pixmap_bg], "@100x100");
    } else {
      rs_pixmaps[pixmap_bg] = Word(4, buff);
    }

  } else if (!strncasecmp(buff, "scroll_up ", 10)) {
    print_error("warning:  support for scroll_up attribute not yet implemented, ignoring");
  } else if (!strncasecmp(buff, "scroll_down ", 12)) {
    print_error("warning:  support for scroll_down attribute not yet implemented, ignoring");
  } else if (!strncasecmp(buff, "scroll_background ", 18)) {
    print_error("warning:  support for scroll_background attribute not yet implemented, ignoring");
  } else if (!strncasecmp(buff, "scroll_anchor ", 14)) {
    print_error("warning:  support for scroll_anchor attribute not yet implemented, ignoring");
  } else if (!strncasecmp(buff, "path ", 5)) {
    rs_path = Word(2, buff);
  } else {
    print_error("parse error in file %s, line %lu:  Attribute \"%s\" is not valid "
		"within context pixmaps", file_peek_path(), file_peek_line(), buff);
  }
#else
  print_error("warning:  pixmap support was not compiled in, ignoring entire context");
#endif
}

void
parse_kanji(char *buff) {

#ifdef KANJI
  if (!strncasecmp(buff, "encoding ", 9)) {
    rs_kanji_encoding = Word(2, buff);
    if (strncasecmp(rs_kanji_encoding, "eucj", 4) && strncasecmp(rs_kanji_encoding, "sjis", 4)) {
      print_error("parse error in file %s, line %lu:  Invalid kanji encoding mode \"%s\"",
		  file_peek_path(), file_peek_line(), rs_kanji_encoding);
      return;
    }
    set_kanji_encoding(rs_kanji_encoding);
  } else if (!strncasecmp(buff, "font ", 5)) {

    char *tmp = PWord(2, buff);
    unsigned char n;

    if (NumWords(buff) != 3) {
      print_error("parse error in file %s, line %lu:  Invalid parameter list \"%s\" for "
		  "attribute font", file_peek_path(), file_peek_line(), tmp);
      return;
    }
    if (isdigit(*tmp)) {
      n = (unsigned char) strtoul(tmp, (char **) NULL, 0);
      if (n <= 4) {
	rs_kfont[n] = Word(2, tmp);
      } else {
	print_error("parse error in file %s, line %lu:  Invalid font index %d",
		    file_peek_path(), file_peek_line(), n);
      }
    } else {
      tmp = Word(1, tmp);
      print_error("parse error in file %s, line %lu:  Invalid font index \"%s\"",
		    file_peek_path(), file_peek_line(), tmp);
      free(tmp);
    }
  } else {
    print_error("parse error in file %s, line %lu:  Attribute \"%s\" is not valid "
		"within context kanji", file_peek_path(), file_peek_line(), buff);
  }
#else
  print_error("warning:  kanji support was not compiled in, ignoring entire context");
  file_poke_skip(1);
#endif
}

void
parse_undef(char *buff) {

  print_error("parse error in file %s, line %lu:  Undefined subcontext \"%s\" within context %s",
	      file_peek_path(), file_peek_line(), PWord(2, buff),
	      ctx_id_to_name(ctx_peek()));
  file_poke_skip(1);

}

/* The config file reader.  This loads a file called MAIN, which is
   located by searching CONFIG_SEARCH_PATH, PATH_ENV, and $PATH.  If 
   it can't find a config file, it displays a warning but continues.
                                                               -- mej */

#ifdef USE_THEMES
int
find_theme(char *path, char *name) {

  register char *search_path = strdup(path);
  register char *cur_path, *conf_name;
  int ver;
  char buff[256];
#ifdef DEBUG_OPTIONS
  char *pwd2;
#endif

  if (!name) return (0);

#ifdef DEBUG_OPTIONS
  fprintf(stderr, "[debug] Searching for theme %s\n", name);
#endif

  for (cur_path = strtok(search_path, ":"); !file_peek_fp() && cur_path; 
       cur_path = strtok((char *) NULL, ":")) {

#ifdef DEBUG_OPTIONS
    fprintf(stderr, "[debug] cur_path == %s\n", cur_path);
#endif
    if (!strncasecmp(cur_path, "~/", 2)) {
      chdir(getenv("HOME"));
      cur_path += 2;
    }
    chdir(cur_path);
#ifdef DEBUG_OPTIONS
    pwd2 = (char *) malloc(2048);
    getcwd(pwd2, 2048);
    fprintf(stderr, "[debug] cur_path == %s   wd == %s\n", cur_path, pwd2);
    free(pwd2);
#endif
    if (chdir(name)) {
	continue;
    }
    if (rs_config_file) {
      conf_name = rs_config_file;
    } else {
      conf_name = CONFIG_FILE_NAME;
    }
    file_poke_fp(fopen(conf_name, "rt"));
    if (file_peek_fp()) {
      fgets(buff, 256, file_peek_fp());
#ifdef DEBUG_OPTIONS
      fprintf(stderr, "[debug] Magic line \"%s\" [%s]  VERSION_NUM == \"%s\"  size == %lu\n",
	      buff, buff+7, VERSION_NUM, sizeof(VERSION_NUM)-1);
#endif
      if (strncasecmp(buff, "<Eterm-", 7)) {
	file_poke_fp(NULL);
      } else if ((ver = strncasecmp(buff+7, VERSION_NUM, sizeof(VERSION_NUM)-1)) > 0) {
	print_error("warning:  config file is designed for a newer version of Eterm");
#ifdef WARN_OLDER
      } else if (ver < 0) {
	print_error("warning:  config file is designed for an older version of Eterm");
#endif
      }
    }
  }
  return ((int) file_peek_fp());
}
#endif

int
find_config_file(char *path, char *name) {

  register char *search_path = strdup(path);
  register char *cur_path;
  int ver;
  char buff[256];
#ifdef DEBUG_OPTIONS
  char *pwd2;
#endif

  if (!name) return (0);

#ifdef DEBUG_OPTIONS
  fprintf(stderr, "[debug] Searching for config file %s\n", name);
#endif

  for (cur_path = strtok(search_path, ":"); !file_peek_fp() && cur_path; 
       cur_path = strtok((char *) NULL, ":")) {

#ifdef DEBUG_OPTIONS
    fprintf(stderr, "[debug] cur_path == %s\n", cur_path);
#endif
    if (!strncasecmp(cur_path, "~/", 2)) {
      chdir(getenv("HOME"));
      cur_path += 2;
    }
    chdir(cur_path);
#ifdef DEBUG_OPTIONS
    pwd2 = (char *) malloc(2048);
    getcwd(pwd2, 2048);
    fprintf(stderr, "[debug] cur_path == %s   wd == %s\n", cur_path, pwd2);
    free(pwd2);
#endif

    file_poke_fp(fopen(name, "rt"));
    if (file_peek_fp()) {
      fgets(buff, 256, file_peek_fp());
#ifdef DEBUG_OPTIONS
      fprintf(stderr, "[debug] Magic line \"%s\" [%s]  VERSION_NUM == \"%s\"  size == %lu\n",
	      buff, buff+7, VERSION_NUM, sizeof(VERSION_NUM)-1);
#endif
      if (strncasecmp(buff, "<Eterm-", 7)) {
	file_poke_fp(NULL);
      } else if ((ver = strncasecmp(buff+7, VERSION_NUM, sizeof(VERSION_NUM)-1)) > 0) {
	print_error("warning:  config file is designed for a newer version of Eterm");
#ifdef WARN_OLDER
      } else if (ver < 0) {
	print_error("warning:  config file is designed for an older version of Eterm");
#endif
      }
    }
  }
  return ((int) file_peek_fp());
}

void
read_config(void) {

  char *search_path, *env_path, *path_env, *desc;
  char buff[CONFIG_BUFF];
  char *conf_name;
  register unsigned long i=0;
  int ver;
  unsigned char id;
  file_state fs = { NULL, CONFIG_FILE_NAME, 1, 0 };

  env_path = getenv("PATH");
  if (env_path) i += strlen(env_path);
  path_env = getenv(PATH_ENV);
  if (path_env) i += strlen(path_env);
  i += sizeof(CONFIG_SEARCH_PATH) + 3;
  search_path = (char *) MALLOC(i);
  strcpy(search_path, CONFIG_SEARCH_PATH);
  if (path_env) {
    strcat(search_path, ":");
    strcat(search_path, path_env);
  }
  if (env_path) {
    strcat(search_path, ":");
    strcat(search_path, env_path);
  }
  file_push(fs);
  
  getcwd(initial_dir, PATH_MAX);

#ifdef USE_THEMES
  if (rs_config_file) {
    conf_name = rs_config_file;
  } else {
    conf_name = CONFIG_FILE_NAME;
  }
  if (!find_theme(search_path, rs_theme)) {
    if (!find_theme(search_path, "Eterm")) {
      if (!find_theme(search_path, "DEFAULT")) {
#else
	conf_name = CONFIG_FILE_NAME;
#endif
	find_config_file(search_path, conf_name);
#ifdef USE_THEMES
	FREE(rs_theme);
	rs_theme = NULL;
      }
    }
  }
#endif

  FREE(search_path);
  if (!file_peek_fp()) {
    print_error("unable to find/open config file %s, continuing with defaults", conf_name);
    return;
  }

  file_poke_path(conf_name);
  ctx_peek() = 0;  /* <== This looks weird, but works -- mej */
  for (; fgets(buff, CONFIG_BUFF, file_peek_fp()); ) {
    file_inc_line();
    if (!strchr(buff, '\n')) {
      print_error("parse error in file %s, line %lu:  line too long",
		  file_peek_path(), file_peek_line());
      for (; fgets(buff, CONFIG_BUFF, file_peek_fp()) && !strrchr(buff, '\n'); );
      continue;
    }
    chomp(buff);
    switch (*buff) {
    case '#':
    case 0:
      break;
    case '%':
      if (!strncasecmp(PWord(1, buff+1), "include ", 8)) {
	shell_expand(buff);
	fs.path = Word(2, buff+1);
	fs.line = 1;
	fs.skip_to_end = 0;
	if ((fs.fp = fopen(fs.path, "rt")) == NULL) {
	  print_error("I/O error in file %s, line %lu:  Unable to open %include file %s "
		      "(%s), continuing", file_peek_path(), file_peek_line(), fs.path, 
		      strerror(errno));
	} else {
	  file_push(fs);
	  fgets(buff, CONFIG_BUFF, file_peek_fp());
#ifdef DEBUG_OPTIONS
	  fprintf(stderr, "[debug] Magic line \"%s\" [%s]  VERSION_NUM == \"%s\" "
		  "size == %lu\n", buff, buff+7, VERSION_NUM, sizeof(VERSION_NUM)-1);
#endif
	  if (strncasecmp(buff, "<Eterm-", 7)) {
	    print_error("parse error in file %s:  Not a valid Eterm config file.  "
			"This file will be ignored.", file_peek_path());
	    file_pop();
	  } else if ((ver = strncasecmp(buff+7, VERSION_NUM, sizeof(VERSION_NUM)-1)) > 0) {
	    print_error("warning:  %included file %s is designed for a newer version"
			" of Eterm", file_peek_path());
#ifdef WARN_OLDER
	  } else if (ver < 0) {
	    print_error("warning:  %included file %s is designed for an older version of "
			"Eterm", file_peek_path());
#endif
	  }
	}
      } else {
	print_error("parse error in file %s, line %lu:  Undefined macro \"%s\"",
		    file_peek_path(), file_peek_line(), buff);
      }
      break;
    case 'b':
      if (file_peek_skip()) continue;
      if (!strncasecmp(buff, "begin ", 6)) {
	desc = PWord(2, buff);
	ctx_name_to_id(id, desc, i);
	if (id == CTX_UNDEF) {
	  parse_undef(buff);
	} else {
	  if (ctx_peek()) {
	    for (i=0; i <= CTX_MAX; i++) {
	      if (contexts[ctx_peek()].valid_sub_contexts[i]) {
		if (contexts[ctx_peek()].valid_sub_contexts[i] == id) {
		  ctx_push(id);
		  break;
		}
	      } else {
		print_error("parse error in file %s, line %lu:  subcontext %s is not valid "
			    "within context %s", file_peek_path(), file_peek_line(),
			    contexts[id].description, contexts[ctx_peek()].description);
		break;
	      }
	    }
	  } else if (id != CTX_MAIN) {
	    print_error("parse error in file %s, line %lu:  subcontext %s is not valid "
			"outside context main", file_peek_path(), file_peek_line(),
			contexts[id].description);
	    break;
	  } else {
	    ctx_push(id);
	  }
	}
	break;
      } 
    case 'e':
      if (!strncasecmp(buff, "end ", 4) || !strcasecmp(buff, "end")) {
	ctx_pop();
	file_poke_skip(0);
	break;
      }
    default:
      if (file_peek_skip()) continue;
      if (id = ctx_peek()) {
	shell_expand(buff);
	(*ctx_id_to_func(id))(buff);
      } else {
	print_error("parse error in file %s, line %lu:  No established context to parse \"%s\"",
		    file_peek_path(), file_peek_line(), buff);
      }
    }
  }
}

