/* spawn.c
 * Routines to setup execute other programs as children.
 */

#include <stdio.h>
#include <stdlib.h>
#include <limits.h>
#include <signal.h>
#include <stdarg.h>
#include <ctype.h>

#include "arena.h"
#ifdef ARENA_DEBUG
#  include "debug.h"
#endif
#include "error.h"
#include "main.h"
#include "spawn.h"
#include "util.h"

#include "WWWApp.h"


/* Keep a very simple list of running children...   */
static SPAWNChildInfo *Children[SPAWN_MAX_CHILDREN] = { NULL };

/* Count number of "active" children.  Used as a flag to indicate to Gui
 * and Event Handlers that children are playing!  Also we have a flag to
 * indicate that ONLY "basic X" Gui events should be honored during WAIT_GUI.
 */
static int SPAWNChildrenPlaying = 0;
static Bool SPAWNChildGui = False;



/* Return T/F if "child" is still active (according to our tables).
 * If child is NULL, will return True if ANY children are running!
 */
Bool SpawnIsChildPlaying(SPAWNChildInfo *child)
{
 int cnum;

 if (child == NULL)
   return (Bool)SPAWNChildrenPlaying;

 for (cnum = 0; cnum < SPAWN_MAX_CHILDREN; cnum++)
   {
    if (Children[cnum] &&
	Children[cnum] == child &&
	Children[cnum]->pid == child->pid &&
	Children[cnum]->exit_status == SPAWN_NO_EXIT_YET)
      return True;
   }
 return False;
}

/* Quickly see if the child is ALIVE!!!
 */
Bool SpawnIsChildAlive(SPAWNChildInfo *child)
{
 if (child != NULL &&
     child->pid > 0 &&
     child->exit_status == SPAWN_NO_EXIT_YET)
   return True;

 return False;
}


/* Return T/F if we have a waiting child running in WAIT_GUI mode!
 * This keeps us from having a "global" variable hanging around.
 * Gui must test this (could/should be macro?) to keep us safe.
 */
Bool SpawnIsWaitGui()
{
 return SPAWNChildGui;
}

/* routine to "remember" a Child... We keep its wait type, kill type and the
 * function to call when it delivers a SIGCHLD to us... That way we can get
 * rid of it correctly
 */
SPAWNChildInfo *SpawnRememberChild(pid_t pid,
				   SPAWNWaitMode wait,
				   SPAWNKillMode kill,
				   char *args[],
				   SPAWNChildHandler chandler )
{
 int cnum;
 SPAWNChildInfo *nc = NULL;

#ifdef ARENA_DEBUG
 char Iam[] = "SpawnRememberChild";
 if (SPAWN_TRACE && VERBOSE) Arena_TracePrint(Iam, " Entering... %d\n", pid);
#endif

 /* make space to remember this child with */
 nc = (SPAWNChildInfo *)Arena_MAlloc(sizeof(SPAWNChildInfo), True);
 nc->pid = pid;
 nc->kill_how = kill;
 nc->wait_how = wait;
 nc->exit_status = SPAWN_NO_EXIT_YET;
 nc->exit_handler = chandler;
 nc->exit_handler_done = False;
 nc->file = NULL;

 /* Now we need to make space to keep OUR OWN copy of the arguments!
  * We want to be sure this arg list exists when and if the exit_handler
  * for this child needs it.
  */
 nc->argc = 0;
 nc->args = NULL;
 if (args != NULL)
    nc->args = SpawnDupRagged(&nc->argc, args);

 /* Find room in our table of children to store this new one!
  * Notice, it is COMPLETLY ready to be "slotted".
  */
 for (cnum = 0; cnum < SPAWN_MAX_CHILDREN; cnum++)
   {
    if (Children[cnum] == NULL)
      {
       Children[cnum] = nc;
#ifdef ARENA_DEBUG
       if (SPAWN_TRACE)
	 {
	  Arena_TracePrint(Iam, " Child @"POINTER_FORMAT"\n"
			   "\t\t Child pid = %d\n"
			   "\t\t Child cnum= %d\n", nc, pid, cnum);
	 }
#endif
       SPAWNChildrenPlaying++;
       return nc;
      }
   }
 Arena_PrintError(_("NO ROOM to remember child pid = %d\n"), pid);
 return NULL;
}


/* Forget about this one child!
 * This routine does NOT free file streams, or run exit handlers!
 * You will need (must) use this routine to remove the child from the tables.
 * BUT only for WAIT mode...  i.e. you Spawn a child with WAIT... you get
 * back the ChildInfo* and can use it to test exit status, etc. Then Forget IT!
 *
 * NOWAIT (aka ASYNC) mode will take care of doing the ForgetChild for you
 * in  SpawnSIGCHLDHandler().
 */
void SpawnForgetChild(SPAWNChildInfo *c)
{
 int cnum;
#ifdef ARENA_DEBUG
 char Iam[] = "SpawnForgetChild";
 if (SPAWN_TRACE) Arena_TracePrint(Iam, " Forgetting "POINTER_FORMAT"\n", c);
#endif

 if (c == NULL) return;

 for (cnum = 0; cnum < SPAWN_MAX_CHILDREN; cnum++)
   {
    if (Children[cnum] == c)
      {
#ifdef ARENA_DEBUG
       if (c->file != NULL && SPAWN_TRACE)
	 {
	  Arena_TracePrint(Iam, " Files not yet freed!!! MEMORY LOSS!\n");
	  Arena_TracePrint(Iam, " Child @"POINTER_FORMAT"\n", c);
	  Arena_TracePrint(Iam, " Child pid = %d\n", c->pid);
	  Arena_TracePrint(Iam, " Child cnum= %d\n", cnum);
	 }
       if (c->exit_handler && c->exit_handler_done != True && SPAWN_TRACE)
	 {
	  Arena_TracePrint(Iam, " Exit Handler NOT DONE!!! MEMORY LOSS!\n");
	  Arena_TracePrint(Iam, " Child @"POINTER_FORMAT"\n", c);
	  Arena_TracePrint(Iam, " Child pid = %d\n", c->pid);
	  Arena_TracePrint(Iam, " Child cnum= %d\n", cnum);
	 }
#endif	   
       SpawnFreeRagged(&c->argc, &c->args);
       c->args = NULL;
       c->kill_how = 0;
       c->wait_how = 0;
       c->exit_status = 0;
       c->exit_handler = NULL;
       c->exit_handler_done = True;
       c->file = NULL;

       SPAWNChildrenPlaying--;
       if (SPAWNChildrenPlaying < 0) SPAWNChildrenPlaying = 0;

       c->pid = 0;
       Free(c);
       c = NULL;
       Children[cnum] = NULL;         /* Last thing!   Now it's free for use */
       return;
      }
   }
#ifdef ARENA_DEBUG
    Arena_TracePrint(Iam, " No Child to Forget @"POINTER_FORMAT"\n", c);
    Arena_TracePrint(Iam, " Child pid = %d\n", c->pid);
#endif
}

/* Remember file stream for easy cleanups.
 * You specify the child with which to associate this file, and
 * whether or not to Flush the file's buffers prior to closing!
 * SpawnChild will automatically "remember" I/O/E Streams along with Files.
 */
void SpawnRememberFile(SPAWNChildInfo *c, FILE *f, Bool flushit)
{
 SPAWNFileInfo *nfi = NULL;

 if (c != NULL)
   {
    nfi = (SPAWNFileInfo *)Arena_MAlloc(sizeof(SPAWNFileInfo), True);
    nfi->stream = f;
    nfi->flush_file = flushit;
    nfi->next = c->file;
    c->file = nfi;
   }
}

/* Handler used internally by SpawnChild.
 * Used to close up files and release file resources when a SIGCHLD signal
 * is received!  It will call the exit_handler specified to SpawnChild.
 */
void SpawnSIGCHLDHandler()
{
 int cnum, exiting_child_pid, status;
 SPAWNChildInfo *c;

#ifdef ARENA_DEBUG
 char Iam[] = "SpawnSIGCHLDHandler";
#endif


 /* Do we have any child we can do anything about? */
 if ((exiting_child_pid = wait(&status)) == -1)
   status = -1;

#ifdef ARENA_DEBUG
 if (SPAWN_TRACE)
   {
    Arena_TracePrint(Iam, " Enter SIGCHLD Handler-------------------------> \n"
		     "\t\t\texiting_child_pid = %d\n"
		     "\t\t\tstatus = %d\n", exiting_child_pid, status);
   }
#endif


 /* Yes, find it in our table */
 c = NULL;
 for (cnum = 0; cnum < SPAWN_MAX_CHILDREN; cnum++)
   {
    if (Children[cnum] && (Children[cnum]->pid == exiting_child_pid))
      {
       c = Children[cnum];
       break;
      }
   }
 if (c == NULL)
   {
    signal(SIGCHLD, SpawnSIGCHLDHandler);
#ifdef ARENA_DEBUG
    if (SPAWN_TRACE)
      Arena_TracePrint(Iam, " NOT FOUND pid %d\n", exiting_child_pid);
#endif
    return;
   }

#ifdef ARENA_DEBUG
 if (SPAWN_TRACE)
   Arena_TracePrint(Iam, " Child @"POINTER_FORMAT" \n"
		    "\t\t\tChild pid == %d\n", c, c->pid);
#endif

 /* close any files (WITHOUT FLUSHING) & free any file info structs */
 SpawnCleanUpChildFiles(c);

 /* Set the correct status for this child... so exit handler can see */
 c->exit_status = status;
 if (status == -1)
   {
    Arena_PrintError(_("Child NOT terminated normally\n"));
    c->exit_status = 1;
   }
 else if (WIFEXITED(status))
   {
    c->exit_status = WEXITSTATUS(status);
    if (c->exit_status != 0)
      Arena_PrintError(_("Child abnormal exit%d\n"), status);
#ifdef ARENA_DEBUG
    else if (SPAWN_TRACE)
      Arena_TracePrint(Iam, " Child "POINTER_FORMAT" exited normally\n", c);
#endif
   }


 /* Call the exit handler that was specified during the SpawnChild
  * The exit_handlers will get only the child* and a copy of the original
  * "arguments" given to the child itself... which is contained within the
  * child structure.
  */
 if (c->exit_handler)
   (c->exit_handler) (c);
 

 /* If we (the parent) are NOT WAITING on this child, clean up!
  * If we ARE WAITING, FINALLY set handler done flag...
  * this allows WAIT to finish up and return to you from SpawnChild()
  */
 c->exit_handler_done = True;
 if (c->wait_how == WAIT_NOWAIT)
    SpawnForgetChild(c);
 
 /* Always reinstate self as the handler... Linux resets to default
  * when signal is raised.  MAKE SURE ALL RETURNS in here DO THIS!
  */
 signal(SIGCHLD, SpawnSIGCHLDHandler);


#ifdef ARENA_DEBUG
 if (SPAWN_TRACE) Arena_TracePrint(Iam, " DONE!\n");
#endif
}

/* SpawnChild
 * Ipipe is the CHILD's INPUT pipe set to STDIN! Then our end of the pipe
 * is returned as the stream!
 * If Iname != NULL, then Child's STDIN will be set to this input file stream
 * and Ipipe will get that stream returned to parent ** NOT YET IMPLEMENTED **
 *
 * Opipe refers to the CHILD's OUTPUT pipe, back to us, on stream Opipe.
 * Epipe is the CHILD's ERROR stream pipe, back to us, on Epipe.
 */
SPAWNChildInfo *SpawnChild(void (*func) (SPAWNChildInfo *, char *, char **, char **),
			    char *path,
			    char **args,
			    char **envp,
			    SPAWNWaitMode wait,
			    SPAWNKillMode kill,
			    char *Iname, FILE **Ipipe,
			    char *Oname, FILE **Opipe, 
			    char *Ename, FILE **Epipe,
			    SPAWNChildHandler chandler)
{
 int pid, SaveErrno;
 int Ifd[2], Ofd[2], Efd[2];

 SPAWNChildInfo *child = NULL;

#ifdef ARENA_DEBUG
 char Iam[] = "SpawnChild";
 if (SPAWN_TRACE) Arena_TracePrint(Iam, " Entering...\n");
#endif

 /* Set up the child's Input Pipe...
  * Die if any error on pipe.
  */
 if (Ipipe != NULL && pipe(Ifd) < 0) return 0;

 /* Set up the child's Output Pipe...
  * If any problem, clean up all the file descriptors.
  */
 if (Opipe != NULL && pipe(Ofd) < 0)
   {
    SaveErrno = errno;
    if (Ipipe != NULL)
      {
       close(Ifd[0]);
       close(Ifd[1]);
      }

    errno = SaveErrno;
    return 0;
   }

 /* Set up the Error stream Pipe
  * If any problems, clean up good
  */
 if (Epipe && pipe(Efd) < 0)
   {
    SaveErrno = errno;
    if (Ipipe != NULL)
      {
       close(Ifd[0]);
       close(Ifd[1]);
      }

    if (Opipe != NULL)
      {
       close(Ofd[0]);
       close(Ofd[1]);
      }

    errno = SaveErrno;
    return 0;
   }

 /* set up a handler to take care of SIGCHLD signals
  * It will find out which child signaled, and clean up for us
  */
 signal(SIGCHLD, SpawnSIGCHLDHandler);

 /* Create the child process...
  * Clean up if die!
  */
 if ((pid = fork()) < 0)
   {
    SaveErrno = errno;
    if (Ipipe != NULL)
      {
       close(Ifd[0]);
       close(Ifd[1]);
      }

    if (Opipe != NULL)
      {
       close(Ofd[0]);
       close(Ofd[1]);
      }

    if (Epipe != NULL)
      {
       close(Efd[0]);
       close(Efd[1]);
      }

    errno = SaveErrno;
    return 0;
   }



 /* CHILD Process!!!
  * Stop a lot of extraneous signals...
  * Make our Opipe use STDOUT...
  * Ipipe should use STDIN...
  *
  * Epipe should use STDERR... NOTE: If Epipe is NULL and ARENA_DEBUG is 
  * not defined (hopefully that will become the norm) Errors will go to
  * /dev/null... you will get back the exit code only!
  *
  * Execute the "function" specified...
  */
 if (pid == 0)
   {
    signal(SIGHUP, SIG_IGN);
    signal(SIGINT, SIG_IGN);
    signal(SIGQUIT, SIG_IGN);
    signal(SIGTTIN, SIG_IGN);
    signal(SIGTTOU, SIG_IGN);

    if (Opipe != NULL)
      {
       close(Ofd[0]);
       dup2(Ofd[1], STDOUT_FILENO);
       close(Ofd[1]);
      }

    if (Ipipe != NULL)
      {
       close(Ifd[1]);
       dup2(Ifd[0], STDIN_FILENO);
       close(Ifd[0]);
      }

    if (Epipe != NULL)
      {
       close(Efd[0]);
       dup2(Efd[1], STDERR_FILENO);
       close(Efd[1]);
      }

#ifndef ARENA_DEBUG
    if (Epipe == NULL)
      {
       int nfd;

       nfd = open("/dev/null", O_WRONLY);
       dup2(nfd, STDOUT_FILENO);
      }
#endif

    func(child, path, args, envp);
    Exit(1);
   }


 /* PARENT process!
  * There's a bit of a race condition here... If the child dies faster than
  * we can "remember" it... there WILL be lots of trouble!
  */
 child = SpawnRememberChild(pid, wait, kill, args, chandler);
 if (child == NULL) return 0;

 /* Rememeber, Opipe is the CHILDS's output pipe...
  * so, set up our end as input.
  */
 if (Opipe != NULL) 
   {
    close(Ofd[1]);
    *Opipe = fdopen(Ofd[0], "r");
    if (*Opipe != NULL) SpawnRememberFile(child, *Opipe, False);
   }

 /* Remember, Ipipe is the CHILD's input,
  * so set up OUR end as our output stream
  */
 if (Ipipe != NULL)
   {
    close(Ifd[0]);
    *Ipipe = fdopen(Ifd[1], "w");
    if (*Ipipe != NULL) SpawnRememberFile(child, *Ipipe, False);
   }

 /* read from the childs error pipe */
 if (Epipe != NULL)
   {
    close(Efd[1]);
    *Epipe = fdopen(Efd[0], "r");
    if (*Epipe != NULL) SpawnRememberFile(child, *Epipe, False);
   }

 /* Use a generalize Waiting routine...
  * This way you can use it in other ways later...
  * DO NOT fail to ForgetChild() if you have created this process with 
  * a mode OTHER THAN ASYNC. Only ASYNC does the Forget for you!
  */
 SpawnWaitForChild(child, child->wait_how);
 return child;
}


/* Get rid of all subprocesses we have created...
 * This may involve killing some of them off...
 * Should probably be called from Exit!
 *
 * NOTE: Since we are Killing these children, NO SIGNAL will be received.
 * Therefore: we must do everything here we expected SIGCHLDHandler() to do.
 * ALSO: We do NOT restore the SIGCHLDHandler() when we finish...
 */
void SpawnCleanUpChildren(void)
{
 int cnum;
 SPAWNChildInfo *c;
 int NeedSleeper = FALSE;

#ifdef ARENA_DEBUG
 char Iam[] = "SpawnCleanUpChildren";
 if (SPAWN_TRACE) Arena_TracePrint(Iam, " Beginning cleanup\n");
#endif

 /* Let's make sure that we don't get any child signals from here on out!
  * Actually, we could only get one from a child that dies prior to our
  * killing it!   But, since we are here to KILL 'EM... ingnore signals!
  */
 signal(SIGCHLD, SIG_IGN);


 /* Pick up any processes that have already exited on us...
  * Flag them so we don't try to kill them later!
  * These SHOULD already have the SIGCHLHandler() and exit_handler
  * done!  We do NOT want to do it here again!
  */
 for (cnum = 0; cnum < SPAWN_MAX_CHILDREN; cnum++)
   {
    if ((c = Children[cnum]) != NULL)
      {
       if (waitpid(c->pid, &c->exit_status, WNOHANG) > 0)
	 c->kill_how = KILL_NEVER;
      }
   }

 /* Ok, give the SIGTTERM signal to any subprocess... if we need to
  * We will sleep here for 3 seconds if we are killing any processes
  * that are "timeout" kills.
  */
 for (cnum = 0; cnum < SPAWN_MAX_CHILDREN; cnum++)
   {
    if ((c = Children[cnum]) != NULL)
      {
       if (c->kill_how == KILL_TIMEOUT)
	 {
	  /* Only need timeout if not already dead */
	  if (kill(c->pid, SIGTERM) != -1)
	    NeedSleeper = TRUE;
	 }
       else if (c->kill_how == KILL_ALWAYS)
	    kill(c->pid, SIGKILL);
      }
   }

 /* Do we have to wait for the KILLs above?
  * Do all this looping so we will only sleep 1 x 3 seconds.
  * We never need to sleep 32 * 3 seconds!
  */
 if (NeedSleeper) sleep(3);


 /* The subprocesses should now be done.
  * Now... get rid of them.  Do our own version of file and exit handler
  * cleanup.  Each process will be taken care of separately!
  */
 for (cnum = 0; cnum < SPAWN_MAX_CHILDREN; cnum++)
   {
    if ((c = Children[cnum]) != NULL)
      {
       if (c->kill_how == KILL_TIMEOUT)  kill(c->pid, SIGKILL);
       if (c->kill_how != KILL_NEVER)
	 {
	  waitpid(c->pid, &c->exit_status, 0);

	  /* Close any "remembered" files                                    */
	  SpawnCleanUpChildFiles(c);

	  /* Call the exit handler that was specified during the SpawnChild  */
	  if (c->exit_handler)
	    (c->exit_handler) (c);

	  /* NOW, forget everything about the children!                      */
	  c->exit_handler_done = True;
	  SpawnForgetChild(c);

	 }  /* end of != KILL_NEVER */
      }  /* end of c != NULL     */
   }  /* end of for()         */
}

/* CleanUpChildFiles
 * That's right, close this childs remembered files.
 */
void SpawnCleanUpChildFiles(SPAWNChildInfo *c)
{
 SPAWNFileInfo *f = NULL;

 /* Close any files (WITHOUT FLUSHING) & free any file info structs */
 if (c != NULL)
   {
    while (c->file != NULL)
      {
       f = c->file;
       if (f->flush_file)
	 fclose(f->stream);
       else
	 close(fileno(f->stream));
       c->file = f->next;
       Free(f);
       f = NULL;
      }
   }
}

/* FIXME */
/* This routine will kill and clean up "One Specified" child!
 * As of 27.9.97  NOT TESTED!
 */
void SpawnKillOneChild(SPAWNChildInfo *c)
{
 /* Don't kill any kids if we don't know which one to do... or
  * if it has been flagged at a NEVER KILL ME kid!
  */
 if (c == NULL) return;
 if (c->kill_how == KILL_NEVER) return;


 /* See if we have a child to get rid of?
  * If child has already exited, just let the normal stuff happen!
  * If the exit handler has not finished, we will proceed.
  * Possible RACE Condition?
  */
 if (waitpid(c->pid, &c->exit_status, WNOHANG) > 0)
   {
    if (c->exit_handler_done == True) 
      {
       signal(SIGCHLD, SpawnSIGCHLDHandler);
       return;
      }
   }

 /* Make sure we don't get interrupted by any dying children.
  * This needs to temporarily Block signals... and allow them 
  * through later... not just ignore them!
  */
 signal(SIGCHLD, SIG_IGN);


 /* if necessary, KILL it */
 if (c->kill_how == KILL_TIMEOUT)
   {
    if (kill(c->pid, SIGTERM) != -1)
      sleep(4);
   }
 else if (c->kill_how == KILL_ALWAYS)
   kill(c->pid, SIGKILL);


 /* It should be done dying by now... get rid of it! */
 if (c->kill_how == KILL_TIMEOUT) kill(c->pid, SIGKILL);
 if (c->kill_how != KILL_NEVER)
   {
     /* Hang here waiting till exit */
    waitpid(c->pid, &c->exit_status, 0);

    /* Close any files (WITHOUT FLUSHING) & free any file info structs */
    SpawnCleanUpChildFiles(c);
 
    /* Yes, call the exit handler he specified */
    if (c->exit_handler)
      (c->exit_handler) (c);
 
    /* It is now kool to FORGET the kid! */
    c->exit_handler_done = True;
    SpawnForgetChild(c);
   }

 /* Now, set up our normal signal handling to catch the death of children
  * We really need to UNBlock signals here!
  */
 signal(SIGCHLD, SpawnSIGCHLDHandler);
 return;
}



/* SpawnExecve is the simple version of a function to pass to SpawnChild
 * to be used as THE CHILD function!  All it does is execve using the
 * path, arguments, and environment you specify.  It is expected that
 * this routine will NEVER return to you.  In other words... you are here
 * since you are the child, and are about to overlay yourself with another
 * process... If that dies, so do you!
 */
void SpawnExecve(SPAWNChildInfo *c, char *path, char *argv[], char *envp[])
{
  int istat;

#ifdef ARENA_DEBUG
  char Iam[] = "SpawnExecve";
  if (SPAWN_TRACE) Arena_TracePrint(Iam, " execve about to happen\n");
#endif

  istat = execve(path, argv, envp);

#ifdef ARENA_DEBUG
  Arena_TracePrint(Iam, " WHY are we back here?\n");
#endif
  Exit(istat);
}

/* SpawnExecvp is the simple version of a function to pass to SpawnChild
 * to be used as THE CHILD function!  All it does is execvp.
 * This routine will NEVER return to you.  In other words... you are here
 * since you are the child, and are about to overlay yourself with another
 * process... If that dies, so do you!
 */
void SpawnExecvp(SPAWNChildInfo *c, char *path, char *argv[], char *envp[])
{
  int i;

#ifdef ARENA_DEBUG
  char Iam[] = "SpawnExecvp";
  if (SPAWN_TRACE) Arena_TracePrint(Iam, " execvp about to happen...\n");
#endif

  i = execvp(path, argv);
  Exit(i);
}

/* The ultimate Exit Handler...
 * When supplying your own exit handler...
 *   ALWAYS see if you've already done it!  (very unusual to want > 1).
 *   NEVER clear the function pointer c->exit_handler.
 *   AND, NEVER set c->exit_handler_done to True...
 *   NEVER, NEVER call ForgetChild() in your Exit Handler!
 * There are several things that need to occur outside this routine before
 * allowing WAIT to proceed! SpawnSIGCHLDHandler and/or SpawnChild will
 * clear these flags "at the last" second.  Then, if WAITING, do your own
 * tesing of the child* returned to you and then ForgetChild()...
 * but NOT within your exit handler!
 */
void SpawnExitHandler(SPAWNChildInfo *c)
{
 if (c == NULL) return;

 /* YOU should make this test before doing anything in here...
  * HELP out... don't do yourself multi times... unless you want to!
  */
 if (c->exit_handler_done == True)
   {
    Arena_PrintError(_("SpawnExitHandler called too many times...\n"
		       "\t\tProgrammers forgot to SpawnForgetChild() ? \n"
		       "\t\tPlease contact %s\n"), DEVELOPERS_MAILTO);
    return;
   }

#if ARENA_DEBUG
 if (SPAWN_TRACE)
   Arena_TracePrint("SpawnExitHandler", " Called "POINTER_FORMAT"\n"
		    "\t\t\tWith status of %d\n", c, c->exit_status);
#endif
 return;
}


/* SpawnVE
 * The simple spawn a child routine...
 * The child must get all its info from the path, argv and environment you
 * pass to it... i.e. this routine expects the child to need no other info
 * from us.
 *
 * If path == NULL, use argv[0] as filespec to run.
 * If env == NULL, us a near nothing environment.
 */
SPAWNChildInfo *SpawnVE(SPAWNWaitMode wait, SPAWNKillMode kill,
			char *path, char *argv[], char *env[],
			SPAWNChildHandler chandler)
{
 extern Display *display;

 char *p = path;
 char ***e = &env;
 SPAWNChildInfo *child = NULL;
 SPAWNChildHandler handler = chandler;


 /* This seems to be the absolute minimum environment to get anything
  * to run under X.  Make sure we get the correct DISPLAY value... It
  * may have been specified via the command line options.
  */
 static char *default_envp[] = { "PATH=::",    /* Specify it!           */
				 "IFS=' '",    /* Don't allow anything  */
				 NULL,         /* used for DISPLAY=     */
				 NULL,         /* used for TERM=        */
				 NULL };       /* you're right! THE END */
#ifdef ARENA_DEBUG
 char Iam[] = "SpawnVE";
 if (SPAWN_TRACE) Arena_TracePrint(Iam, " Entering... \n");
#endif

 /* Don't do anyting if we don't have any arguments! */
 if (argv == NULL) return NULL;
 
 /* Take care of fixing up our "default" environment.  THESE ARE HARDWIRED !
  * Get OUR display... it may be different than environment!
  */
 StrAllocCopy(default_envp[2], "DISPLAY=");
 StrAllocCat(default_envp[2], DisplayString(display));

 if (getenv("TERM") != NULL)
   {
    StrAllocCopy(default_envp[3], "TERM=");
    StrAllocCat(default_envp[3], getenv("TERM"));
   }

 /* Make life easy... assume the executable is specified by first argument
  * Iff the path to it was NOT specified
  */
 if (p == NULL) StrAllocCopy(p, argv[0]);

 /* If the user specified the environment, USE It!
  * If not, use the simple (but safe) default.
  */
 if (env == NULL) *e = default_envp;

 /* If debugging, and user didn't specify an exit handler, USE ULTIMATE...
  * This will help trace what's going on!
  */
#if ARENA_DEBUG
 if (chandler == NULL) handler = SpawnExitHandler;
#endif

 if (access(p, X_OK) != -1)
   {
    child = SpawnChild (SpawnExecve, p, argv, *e,
			wait, kill,
			NULL, NULL,
			NULL, NULL,
			NULL, NULL,
			handler);
   }
 if (path == NULL) Free(p);

#ifdef ARENA_DEBUG
 if (SPAWN_TRACE) Arena_TracePrint(Iam, " Returning... \n");
#endif
 return child;
}


/* SpawnVP
 * The simple spawn a child routine...
 * The child must get all its info from the path, argv.
 * However, the  User's environment will be used (PATH=, etc...)
 *
 * If path == NULL, use argv[0] as filespec to run.
 */
SPAWNChildInfo *SpawnVP(SPAWNWaitMode wait, SPAWNKillMode kill,
			char *path, char *argv[], SPAWNChildHandler chandler)
{
 char *p = path;
 SPAWNChildInfo *child = NULL;
 SPAWNChildHandler handler = chandler;

#ifdef ARENA_DEBUG
 char Iam[] = "SpawnVP";
 if (SPAWN_TRACE) Arena_TracePrint(Iam, " Entering... \n");
#endif

 /* Don't do anyting if we don't have any arguments! */
 if (argv == NULL) return NULL;
 
 /* Make life easy... assume the executable is specified by first argument
  * Iff the path to it was NOT specified
  */
 if (p == NULL) StrAllocCopy(p, argv[0]);

 /* If debugging, and user didn't specify an exit handler, USE ULTIMATE...
  * This will help trace what's going on!
  */
#if ARENA_DEBUG
 if (chandler == NULL) handler = SpawnExitHandler;
#endif

 child = SpawnChild (SpawnExecvp, p, argv, NULL,
		     wait, kill,
		     NULL, NULL,
		     NULL, NULL,
		     NULL, NULL,
		     handler);
 if (path == NULL) Free(p);

#ifdef ARENA_DEBUG
 if (SPAWN_TRACE) Arena_TracePrint(Iam, " Returning... \n");
#endif
 return child;
}

/* SpawnSystemCommand
 * Duplicate the system() function (well, sort of anyway!).
 */
int SpawnSystemCommand(char *cmd, char *envp[],
		       SPAWNWaitMode wait, SPAWNKillMode kill,
		       SPAWNChildHandler chandler)
{
 int argc = 0;
 char **argv = NULL;
 SPAWNChildInfo *child = NULL;
 SPAWNChildHandler handler = chandler;
 void(*func) (SPAWNChildInfo *, char *, char **, char **);

#ifdef ARENA_DEBUG
 char Iam[] = "SpawnSystemCommand";
 if (SPAWN_TRACE)
   Arena_TracePrint(Iam, " Entering... cmd:%s\n", cmd ? cmd : "(null)");
#endif

 if (cmd == NULL) return EXIT_FAILURE;

 /* Convert the command string into an argc/argv list */
 argv = SpawnStringToRagged(cmd, &argc);
 if (argv == NULL || argc == 0) return EXIT_FAILURE;
 
 /* If debugging, and user didn't specify an exit handler, USE ULTIMATE...
  * This will help trace what's going on!
  */
#if ARENA_DEBUG
 if (chandler == NULL) handler = SpawnExitHandler;
#endif

 /* Which "child function" will we use?
  * Execvp will be used if no environment is specified... thereby getting
  * the user's environment.  Execve is used if an environment IS specified...
  * That will mean of course that the command string MUST have FULL path
  * specifications to the executable.
  */
 if (envp == NULL)
   func = SpawnExecvp;
 else
   func = SpawnExecve;

 /* Spawn the child and execute the "correct" function depending only on
  * whether or not we are allow the use of the user's environment!
  */
 child = SpawnChild (func, argv[0], argv, envp,
		     wait, kill,
		     NULL, NULL,
		     NULL, NULL,
		     NULL, NULL,
		     handler);

 SpawnFreeRagged(&argc, &argv);
 if (child == NULL) return EXIT_FAILURE;

 /* Temporarily store the exit status so we can give it back to the caller */
 argc = child->exit_status;

 /* Forget the child if we were here and Waiting till done! */
 if (wait != WAIT_ASYNC)
   SpawnForgetChild(child);
 
#ifdef ARENA_DEBUG
 if (SPAWN_TRACE) Arena_TracePrint(Iam, " Returning... \n");
#endif
 return argc;
}


/* SpawnEventHandler
 * A routine you must supply for different applications!
 * This one will use the Arena libwww event handler!
 * glk 15.10.97 added libwww-5.1 support!
 */
int SpawnEventHandler(SPAWNChildInfo *child)
{
 extern Doc* CurrentDoc;

 /*
  * Suggested 14-10-97 by Steffen Zahn.
  */
#if defined(LibWWW_version) && (LibWWW_version == 501)
 HTEventList_loop(CurrentDoc->request); /* libwww-5.1 API */
# else
 HTEventrg_loop(CurrentDoc->request);   /* libwww-5.0 API */
#endif

 return EXIT_SUCCESS;
}

/* SpawnGuiEventHandler
 * A routine you must supply for different applications!
 * Again, we will use the Arena GuiEvents handler to take care of X based
 * GUI events.  It had to be modified to NOT allow some events!
 */
int SpawnGuiEventHandler(SPAWNChildInfo *child)
{
 GuiEvents(0, NULL, 0);
 return EXIT_SUCCESS;
}

/* We MAY need some simple method of marking a block for pieces of code.
 * This will simply allow a variable of your choice to be used!
 */
void SpawnBlock(int *blocking_flag)
{
 *blocking_flag += 1;
}

void SpawnUnBlock(int *blocking_flag)
{
 *blocking_flag -= 1;
 if (*blocking_flag < 0) *blocking_flag = 0;
}

Bool SpawnIsBlocked(int blocking_flag)
{
 return blocking_flag ? True : False;
}

void SpawnWaitForBlock(int blocking_flag)
{
 while (blocking_flag)
   {
    struct timeval timer;
    timer.tv_sec = 0;
    timer.tv_usec = 10000;
    select(0, 0, 0, 0, &timer);  /* we don't care why select returned    */
   }
}

/* We may want to wait for a child to exit... but not want to get involved
 * in its death... if we use wait or waitpid here... we MUST be involved!
 * Even waitpid() with NOHANG! If it did exit and we caught it with the wait
 * the we have to do the cleanup... the SIGCHLD would NOT be taken care of!
 * Use of select() with long wait times will be interrupted by the SIGCHLD
 */
void SpawnWaitForChild(SPAWNChildInfo *child, SPAWNWaitMode myWait)
{
#ifdef ARENA_DEBUG
 char Iam[] = "SpawnWaitForChild";
 if (SPAWN_TRACE)
   Arena_TracePrint(Iam, " Entering to wait %d on child @"POINTER_FORMAT"\n",
		    myWait, child);
#endif

 /* Do we have ANYThing to do?! */
 if (child == NULL) return;
 if (!SpawnIsChildPlaying(child)) return;

 /* OK, we got it created and know EVERYThing about it... now decide
  * if we are waiting (and the type of wait) for it to finish or not.
  */
 switch (myWait)
   {
   case WAIT_ASYNC:    /* aka WAIT_NOWAIT */
     /* Return to caller...
      * Caller continues on its own with the child running!
      * Remember, do NOT Forget(child) since ASYNC was used...
      * SIGCHLDHandler() will do it for you when the child dies!
      */
     break;

   default:
     Arena_PrintError(_("SPAWN Invalid wait mode %d\n"),child->wait_how);
     /* Fall through and ASSUME WAIT_WAIT */

   case WAIT_WAIT:
     /* NOW... if we are to WAIT for exit of the child, we have to stay in here
      * until the SIGCHLD handler and the exit_hander have EACH been completed!
      * If we were to return too quickly, we will be working with a DEAD
      * SPAWNChildInfo *     THAT would be BAD!
      */

     /* If no exit_handler was specified... it has to be done by NOW! */
     if (child->exit_handler == NULL)
       child->exit_handler_done = True;

     /* Now, we are waiting for exit_status to be set by SIGCHLDHandler()
      * and for the exit_handler (if any) to finish.  Since select() is only
      * going to return to us if an interrupt occurs (maybe our child dying),
      * or our timer runs out.  We really just loop here to reinstate the
      * timer if some other child signals its death and sleep long enought to
      * not consume too much CPU time. We know our child finishes by the flags.
      */
     while (child->exit_status == SPAWN_NO_EXIT_YET ||
	    child->exit_handler_done == False)
       {
	struct timeval timer;
	timer.tv_sec = 10;           /* pause long enough to not consume CPU */
	timer.tv_usec = 0;
	select(0, 0, 0, 0, &timer);  /* we don't care why select returned    */
       }

     /* DONE!  Return the child * to the user...
      * HE must ForgetChild() to clean up our tables for us!
      */
     break;


   case WAIT_GUI:
     /* We will "wait" for the child to exit... We will allow only the
      * "basic" X Gui events to respond to the user (e.g. Expose, etc ).
      * This flag MUST be tested within the Gui code so as to ONLY allow
      * Gui events... Cannot run off and do TOO much!
      */
     SPAWNChildGui = True;

     /* If no exit_handler was specified... it has to be done by NOW! */
     if (child->exit_handler == NULL)
       child->exit_handler_done = True;

     /* Now, we are waiting for exit_status to be set by SIGCHLDHandler()
      * and for the exit_handler (if any) to finish.
      */
     while (child->exit_status == SPAWN_NO_EXIT_YET ||
	    child->exit_handler_done == False)
       {
	struct timeval timer;
	timer.tv_sec = 0;
	timer.tv_usec = 100000;
	select(0, 0, 0, 0, &timer);  /* we don't care why select returned    */
	SpawnGuiEventHandler(child);
       }

     /* DONE!  Return the child * to the user... Also allow Gui to work!
      * HE must ForgetChild() to clean up our tables for us!
      */
     SPAWNChildGui = False;
     break;


   case WAIT_BACKGROUND:
     /* We will "wait" for the child to exit... BUT, we will allow ALL program
      * events to continue to respond to the user.  Great care should be used
      * with this... Other events can change a LOT of things!
      */

     /* If no exit_handler was specified... it has to be done by NOW! */
     if (child->exit_handler == NULL)
       child->exit_handler_done = True;

     /* Now, we are waiting for exit_status to be set by SIGCHLDHandler()
      * and for the exit_handler (if any) to finish.
      */
     while (child->exit_status == SPAWN_NO_EXIT_YET ||
	    child->exit_handler_done == False)
       {
	SpawnEventHandler(child);
       }

     /* DONE!  Return the child * to the user...
      * HE must ForgetChild() to clean up our tables for us!
      */
     break;
   } /* end switch */

 return;
}


/* SpawnCmdLineToRagged
 * Simply take a string a generate Argc/Argv 
 * YOU must use FreeRagged() to cleanup the argv... 
 */
char **SpawnStringToRagged(char *string, int *argc)
{
 char c;
 int slen;
 int np = 1;
 char *b = NULL;
 char *t = NULL;
 char *s = string;
 char **argv = NULL;
 char **targ = NULL;

#ifdef ARENA_DEBUG
 char Iam[] = "SpawnStringToRagged";
 if (SPAWN_TRACE && VERBOSE) Arena_TracePrint(Iam, " Entering...\n");
#endif

 /* Don't do anything if we don't have anything to DO! */
 *argc = 0;
 if (s == NULL) return NULL;
 if ((slen = Arena_StrLen(s)) == 0) return NULL;

 /* Allocate buffer for temporary work, and "t" will remember its beginning */
 b = t = (char *)Arena_MAlloc(slen+2, True);
 if (t == NULL) return NULL;

 /* skip leading whitespace */
 while (*s == ' ')
   s++;

 /* Copy input string (s) to our buffer (b)
  * Place NULL char after each "word" of the input.
  */
 while (*s != '\0')
   {
    c = *s;
    switch (c)
      {
      case ' ':
	*b++ = '\0';
	np++;
	s++;
	while (*s == ' ')
	  s++;
	break;

      case '\\':
	s++;
	if (*s == '\\' || *s == '"' || *s == '\'')
	  {
	   *b++ = '\\';
	   s++;
	   break;
	  }
	*b++ = c;
	break;

      case '\'':
      case '"':
	*b++ = *s++;
	if (strchr(s, c) == NULL)
	  break;

	while (*s != c)
	   *b++ = *s++;

	*b++ = *s++;
	break;

      default:
	*b++ = *s++;
	break;
      } /* end switch(c) */
   } /* end while(*s != '\0') */
 *b++ = '\0';
 *b++ = '\0';

 /* Calculate the number of args + 2!
  * This give the user 1 extra slot to use (if he wants it)
  * The second one is for a NULL... guess why!
  */
 argv = targ = (char **)Arena_MAlloc((np+2) * sizeof(char *), True);

 /* Fill in our argv with pointers */
 s = b = t;
 while (*s != '\0')
   {
    *targ++ = strdup(s);   /*    *targ++ = s; */
    s += Arena_StrLen(s);
    s++;
   }
 *targ = NULL;
 Free(t);

 /* Give our user back the argc and argv! */
 *argc = np;
 return argv;
}


/* SpawnFreeRagged
 * Free entire structure of argv, argc... Each argv[] will be freed also.
 */
void SpawnFreeRagged(int *argc, char ***argv)
{
 int i;
 int cnt = *argc;
 char **av = *argv;

#ifdef ARENA_DEBUG
 char Iam[] = "SpawnFreeRagged";
 if (SPAWN_TRACE && VERBOSE)
   {
    Arena_TracePrint(Iam, " Entering...\n"
		     "\t\t cnt = %d\n"
		     "\t\t %s\n", cnt, av[0]);
    Arena_TracePrint(Iam, " Argv @"POINTER_FORMAT"\n", av);
   }
#endif

 /* Do we need to determine the number of args */
 if (cnt <= 0)
   {
    cnt = 0;
    while (av[cnt] != NULL)
      cnt++;

    *argc = cnt;
    if (cnt == 0) return;
   }

 /* release each arg */
 for (i = 0; i < cnt; i++)
   {
    if (av[i] != NULL) Free(av[i]);
    av[i] = NULL;
   }

 /* release the pointers */
 if (*argv != NULL) Free(*argv);
 *argv = NULL;
 *argc = 0;
}


/* SpawnDupRagged
 * Duplicate a ragged array and add an extra NULL pointer at end.
 */
char **SpawnDupRagged(int *argc, char *argv[])
{
 int i;
 char **ov = NULL;
 int cnt = *argc;

#ifdef ARENA_DEBUG
 char Iam[] = "SpawnDupRagged";
 if (SPAWN_TRACE && VERBOSE)
   Arena_TracePrint(Iam, " Entering...\n"
		    "\t\t cnt = %d\n"
		    "\t\t %s\n", cnt, argv[0]);
#endif

 /* Do we need to determine the number of args */
 if (cnt <= 0)
   {
    cnt = 0;
    while (argv[cnt] != NULL)
      cnt++;

    *argc = cnt;
    if (cnt == 0) return NULL;
   }

 /* create a new list */
 ov = (char **)Arena_MAlloc( (cnt+2) * sizeof(char *), True);

 /* and fill it */
 for (i = 0; i < cnt; i++)
   {
    ov[i] = NULL;
    if (argv[i] != NULL)
      StrAllocCopy(ov[i], argv[i]);
   }
 ov[cnt] = NULL;
 ov[cnt+1] = NULL;
 return ov;
}
#if ARENA_DEBUG
void SpawnPrintRagged(int argc, char **argv)
{
 int i;
 int cnt = argc;
 char Iam[] = "SpawnPrintRagged";
 
 /* Do we need to determine the number of args */
 if (cnt <= 0 && argv != NULL)
   {
    cnt = 0;
    while (argv[cnt] != NULL)
      cnt++;
   }
 if (SPAWN_TRACE) Arena_TracePrint(Iam, "COUNTER = %d\n", cnt);

 if (cnt == 0 || argv == NULL)
   {
    Arena_TracePrint(Iam, " No Data! %d "POINTER_FORMAT"\n", cnt, argv);
    return;
   }

  for (i = 0; i < cnt; i++)
    {
     Arena_TracePrint(Iam," argv[%d]=%s\n", i, argv[i] ? argv[i] : "(null)");
     if (argv[i] == NULL) break;
    }
}

/* Routine to plug into Arena for testing the entire Spawning system.
 * USED to create trouble!
 */
int SPAWNTesting(int which_test)
{
 SPAWNChildInfo *c1, *c2, *c3, *c4, *c5;

 char Iam[] = "SPAWNTesting";

#define TstCMD "/usr/X11R6/bin/xterm", "-e", "/usr/bin/less"
 char *TstC1[] = { TstCMD, "README", NULL };
 char *TstC2[] = { TstCMD, "THANKS", NULL };
 char *TstC3[] = { TstCMD, "INSTALL", NULL };
 char *TstC4[] = { TstCMD, "KNOWNBUGS", NULL };
 char *TstC5[] = { TstCMD, "TODO", NULL };

 switch (which_test)
   {
   default:
     /* Fall into "standard" spawn, wait! */

   case 1:
     c1 = SpawnVE(WAIT_WAIT, KILL_ALWAYS, NULL, TstC1, NULL, NULL);
     if (c1 == NULL || c1->exit_status != SPAWN_NO_EXIT_YET)
       Arena_TracePrint(Iam, " Spawning Troubles (1)... \n"
			"1) "POINTER_FORMAT" %d\n", c1, c1->exit_status);
     SpawnForgetChild(c1);
     break;

   case 2:
    c1 = SpawnVE(WAIT_ASYNC, KILL_ALWAYS, NULL, TstC1, NULL, NULL);
    c2 = SpawnVE(WAIT_ASYNC, KILL_ALWAYS, NULL, TstC2, NULL, NULL);
    c3 = SpawnVE(WAIT_ASYNC, KILL_ALWAYS, NULL, TstC3, NULL, NULL);
    c4 = SpawnVE(WAIT_ASYNC, KILL_ALWAYS, NULL, TstC4, NULL, NULL);
    c5 = SpawnVE(WAIT_ASYNC, KILL_ALWAYS, NULL, TstC5, NULL, NULL);

    if (!c1 || !c2 || !c3 || !c4 || !c5)
      {
       Arena_TracePrint(Iam, " Spawning Troubles (2)... \n"
			"1) "POINTER_FORMAT" %d\n"
			"2) "POINTER_FORMAT" %d\n"
			"3) "POINTER_FORMAT" %d\n"
			"4) "POINTER_FORMAT" %d\n"
			"5) "POINTER_FORMAT" %d\n",
			c1,c1->exit_status,
			c2,c2->exit_status,
			c3,c3->exit_status,
			c4,c4->exit_status,
			c5,c5->exit_status);
       return 1;
      }
    break;

   case 3:
    c1 = SpawnVE(WAIT_ASYNC, KILL_ALWAYS, NULL, TstC1, NULL, NULL);
    c2 = SpawnVE(WAIT_ASYNC, KILL_ALWAYS, NULL, TstC2, NULL, NULL);
    c3 = SpawnVE(WAIT_ASYNC, KILL_ALWAYS, NULL, TstC3, NULL, NULL);
    c4 = SpawnVE(WAIT_ASYNC, KILL_ALWAYS, NULL, TstC4, NULL, NULL);
    c5 = SpawnVE(WAIT_GUI, KILL_ALWAYS, NULL, TstC5, NULL, NULL);

    if (!c1 || !c2 || !c3 || !c4 || !c5)
      {
       Arena_TracePrint(Iam, " Spawning Troubles (2)... \n"
			"1) "POINTER_FORMAT" %d\n"
			"2) "POINTER_FORMAT" %d\n"
			"3) "POINTER_FORMAT" %d\n"
			"4) "POINTER_FORMAT" %d\n"
			"5) "POINTER_FORMAT" %d\n",
			c1,c1->exit_status,
			c2,c2->exit_status,
			c3,c3->exit_status,
			c4,c4->exit_status,
			c5,c5->exit_status);
       SpawnForgetChild(c5);
       return 1;
      }
    SpawnForgetChild(c5);
    break;

   } /* end switch (which_test) */
return 0;
}
#undef TstCMD
#endif /* ARENA_DEBUG  around SPAWNTesting */
