#include <stdio.h>
#include <sys/stat.h>
#include <pwd.h>
#include <grp.h>
#include <unistd.h>
#include <time.h>

/* stat the file and print what we find */
void do_stat(char *filename, int link)
{
    struct stat statbuf;
    int i = (link == 1) ? stat(filename, &statbuf) : lstat(filename, &statbuf);
    char access[10];
    struct passwd *pw_ent;
    struct group *gw_ent;
    char linkname[256];

    if (i == -1) {
	perror(filename);
	return;
    }
    if ((statbuf.st_mode & S_IFMT) == S_IFLNK) {
	if ((i = readlink(filename, linkname, 256)) == -1)
	    perror(filename);

	linkname[(i >= 256) ? 255 : i] = '\0';
	printf("  File: \"%s\" -> \"%s\"\n", filename, linkname);
    } else
	printf("  File: \"%s\"\n", filename);

    printf("  Size: %-10d Blocks: %-10d", (int)statbuf.st_size, (int)statbuf.st_blocks);

    switch (statbuf.st_mode & S_IFMT) {
    case S_IFDIR:
	(void) printf("Directory\n");
	break;
    case S_IFCHR:
	(void) printf("Character Device\n");
	break;
    case S_IFBLK:
	(void) printf("Block Device\n");
	break;
    case S_IFREG:
	(void) printf("Regular File\n");
	break;
    case S_IFLNK:
	(void) printf("Symbolic Link\n");
	break;
    case S_IFSOCK:
	(void) printf("Socket\n");
	break;
    case S_IFIFO:
	(void) printf("Fifo File\n");
	break;
    default:
	(void) printf("Unknown\n");
    }

    access[9] = (statbuf.st_mode & S_IXOTH) ? ((statbuf.st_mode & S_ISVTX) ? 't' : 'x') : '-';
    access[8] = (statbuf.st_mode & S_IWOTH) ? 'w' : '-';
    access[7] = (statbuf.st_mode & S_IROTH) ? 'r' : '-';
    access[6] = (statbuf.st_mode & S_IXGRP) ? ((statbuf.st_mode & S_ISGID) ? 's' : 'x') : '-';
    access[5] = (statbuf.st_mode & S_IWGRP) ? 'w' : '-';
    access[4] = (statbuf.st_mode & S_IRGRP) ? 'r' : '-';
    access[3] = (statbuf.st_mode & S_IXUSR) ? ((statbuf.st_mode & S_ISUID) ? 's' : 'x') : '-';
    access[2] = (statbuf.st_mode & S_IWUSR) ? 'w' : '-';
    access[1] = (statbuf.st_mode & S_IRUSR) ? 'r' : '-';

    switch (statbuf.st_mode & S_IFMT) {
    case S_IFDIR:
	access[0] = 'd';
	break;
    case S_IFCHR:
	access[0] = 'c';
	break;
    case S_IFBLK:
	access[0] = 'b';
	break;
    case S_IFREG:
	access[0] = '-';
	break;
    case S_IFLNK:
	access[0] = 'l';
	break;
    case S_IFSOCK:
	access[0] = 's';
	break;
    case S_IFIFO:
	access[0] = 'p';
	break;
    default:
	access[0] = '?';
    }
    (void) printf("Access: (%04o/%10.10s)", statbuf.st_mode & 07777, access);

    setpwent();
    setgrent();
    pw_ent = getpwuid(statbuf.st_uid);
    gw_ent = getgrgid(statbuf.st_gid);
    printf("         Uid: (%5d/%8s)  Gid: (%5d/%8s)\n", statbuf.st_uid,
    		(pw_ent != 0L) ? pw_ent->pw_name : "UNKNOWN", statbuf.st_gid,
    		(gw_ent != 0L) ? gw_ent->gr_name : "UNKNOWN");

    printf("Device: %-10d Inode: %-10d Links: %-5d", (int)statbuf.st_dev,
    	 (int)statbuf.st_ino, (int)statbuf.st_nlink);

    i = statbuf.st_mode & S_IFMT;
    if (i == S_IFCHR || i == S_IFBLK)
	printf(" Device type: %d\n", (int)statbuf.st_rdev);
    else
	(void) printf("\n");

    printf("Access: %s", ctime(&statbuf.st_atime));
    printf("Modify: %s", ctime(&statbuf.st_mtime));
    printf("Change: %s\n", ctime(&statbuf.st_ctime));
}

void usage(char *progname)
{
    fprintf(stderr, "Usage: %s [-l] file1 [file2 ...]\n", progname);
    exit(1);
}

int main(int argc, char *argv[])
{
    int i, link = 0;
    char c;

    while ((c = getopt(argc, argv, "l")) != EOF) {
	switch (c) {
	case 'l':
	    link = 1;
	    break;
	case 'v':
	    printf("stat version: 1.1\n");
	default:
	    usage(argv[0]);
	}
    }
    if (argc == 1)
	usage(argv[0]);

    for (i = link + 1; i < argc; i++)
	do_stat(argv[i], link);

    return (0);
}
