// interpolater.h

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


// This class  perform a second-order parabolic interpolation upon the input, 
// storing the result into the output.  Note that this class, as with all
// InputOutputFunction subclasses, does not directly modify its input.

#ifndef INTERPOLATER_H
#ifdef __GNUG__
#pragma interface
#endif
#define INTERPOLATER_H

#include "localdefs.h"
#include "iofunction.h"

struct QueryInfo;

class Interpolater : public InputOutputFunction {
	typedef InputOutputFunction Super;
	friend class InterpolateRequester;
	friend class TransposeRequester;
public:
	Interpolater(Data* input, Data* output, double factor, boolean filt=false);
	Interpolater(Data* input, Data* output, int newlen, boolean filt=false);
	Interpolater(Data* input, Data* output, boolean tranposing=false);
	virtual ~Interpolater();
	static Modifier* create(class DataEditor *);
	redefined void restoreState();
	redefined const char* message();
protected:
	redefined Requester* createRequester();
	redefined int newLength(int oldLength);
	redefined boolean areCompatibleLengths(int, int);
	redefined void initialize();
	double preProcess(double input);
	double postProcess(double input);
	redefined int processInput(boolean flushing);
	virtual double interpFactor() { return theInterpFactor; }
	void setInterpFactor(double factor) { theInterpFactor = factor; }
	void setFilter(boolean flag) { useFilter = flag; }
	void setGain(double gn) { gain = gn; }
	void increment() { incount++; }
	boolean filtering() { return (processMode != None); }
	boolean transposing() { return transpose; }
	enum QLen { InputQLen = 1024, OutputQLen = 1024 };
	enum ProcessMode { None, Pre, Post } processMode;
private:
	void clear();
	static double interpolate(double, double, double, double fraction);
private:
	int theNewLength;
	double theInterpFactor;
	double gain;
	ChoiceValue useFilter;
	boolean transpose;
	double voldsig;
	double oldsig;
	double sig;
	int incount;
	double fractionCounter;
	SimpleFunction* filter;
};

// note:  this will need to be carefully redefined in variable Interpolater

inline int
Interpolater::newLength(int oldLength) {
	return (theInterpFactor != 0.0) ? roundUp(oldLength/theInterpFactor) : 0;
}

inline void
Interpolater::clear() {
	voldsig = oldsig = sig = 0; incount = -1; fractionCounter = 0;
}


#endif
