// framedata.h

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


// FrameData is an abstract "intermediate" base class for all Data subclasses
// which represent "framed" data, i.e., those where the data always exists in
// groups of more than one sample per frame.  These all have a frame rate
// attribute which may likely differ from the sampling rate attribute (which
// represents the sampling rate of the sound from which the FrameData instance
// was generated, via FFT, Phase Vocoder, or LPC analysis.

#ifndef FRAMEDATA_H
#ifdef __GNUG__
#pragma interface
#endif
#define FRAMEDATA_H

#include "localdefs.h"
#include "data.h"

class Range;
class Header;

class FrameData : public Data {
	typedef Data Super;
public:
	virtual ~FrameData() {}
	redefined Data *newData();
	redefined Data *newData(int nwlen);
	redefined Data *clone(const Range &r);
	redefined Data *clone(const Range &r, const Range &c);
	redefined int sRate() const { return sr; }
	redefined int changeSRate(int srate, boolean) {
		sr = srate; Notify(); return true;
	}
	virtual int nFrames() const { return length(); }
	virtual int frameOffset() const { return int(sr/framerate); }
	redefined double frameRate() const { return framerate; }
	virtual int frameSize() const { return 2 * frameOffset(); }
	virtual double duration() const { return length() / framerate; }
	redefined Range frameRange(RangeUnit units) const;
	redefined const char* frameRangeLabel(RangeUnit units) const;
	redefined Scale::NumberDisplay frameRangeDisplay(RangeUnit units) const;
protected:
	FrameData(DataType dt, int length, int nchans, int srate, double frate)
		: Data(dt, length, nchans), sr(srate), framerate(frate) {}
	// protected ctors for virtual ctors
	FrameData(const FrameData *f) : Data(f), sr(f->sr),
		framerate(f->framerate) {}
	FrameData(const FrameData *f, int newlen) : Data(f, newlen), sr(f->sr),
		framerate(f->framerate) {}
	// protected ctors for cloning
	FrameData(const FrameData *fd, const Range &selection)
		: Data(fd, selection), sr(fd->sr), framerate(fd->framerate) {}
	FrameData(const FrameData *fd, const Range &selection,
		const Range &chans) : Data(fd, selection, chans), sr(fd->sr),
		framerate(fd->framerate) {}
	redefined void readFromHeader(Header *);
	redefined void checkValues();
	redefined double scanForMaxValue() const;
#if defined(NX_COMPILER_RELEASE_3_0)
	Data::setFromArray(double *, int, int, int);
	Data::getArray (double *, int, int, int) const;
#endif
protected:
	int sr;
	double framerate;
};

#endif
