// dialogbox.C

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


#ifdef __GNUG__
#pragma implementation
#endif

#include <InterViews/box.h>
#include <InterViews/button.h>
#include <InterViews/canvas.h>
#include <InterViews/event.h>
#include <InterViews/font.h>
#include <InterViews/frame.h>
#include <InterViews/glue.h>
#include <InterViews/message.h>
#include <InterViews/painter.h>
#include <InterViews/sensor.h>
#include <InterViews/shape.h>
#include <InterViews/world.h>

#include "dialogbox.h"
#include "fileselector.h"
#include "textinput.h"

boolean DialogBox::override_WindowManager = false;
int DialogBox::beep_Level = 50;

DialogBox::DialogBox(ButtonState* s, Interactor* u, Response r) 
		: Dialog(s, nil), defaultResponse(r), underlying(u), useBell(false) {
	SetClassName("DialogBox");
	SetCanvasType(CanvasSaveUnder); 
	input = new Sensor(updownEvents);
	input->Catch(KeyEvent);
}

Interactor*
DialogBox::Wrap(Interactor *interior) {
	return new Frame(interior, 2);
}

boolean
DialogBox::Popup(Event&, boolean) {
	appear();
	boolean accepted = Accept();
	disappear();
	return accepted;
}

int
DialogBox::display() {
	reset();
	if(useBell) bell();
	setBell(false);				// only ring when first displayed
	Event e;
	Popup(e);
	return Status();
}

void
DialogBox::reset() {
	state->SetValue(NullResponse);
}

void
DialogBox::bell() { 
	World::current()->RingBell(beepLevel());
}

// Dialogs are positioned in the center of whichever window received the most
// recent command, if any.  Otherwise, centered on the screen.  Class attribute
// determines whether dialog window can be adjusted via the WM.

void
DialogBox::appear() {
	World* world = World::current();
	Coord x, y;
	if(underlying != nil) {
		underlying->Align(Center, 0, 0, x, y);
		underlying->GetRelative(x, y, world);
	}
	else {
		x = world->Width() / 2;
		y = world->Height() / 2;
	}
	if(overrideWindowManager())
		world->InsertPopup(this, x, y, Center);
	else
		world->InsertTransient(this, this, x, y, Center);
}

boolean
DialogBox::Accept() {
	int answer = NullResponse;
	do {
		answer = getAnswer();
	} while (!acceptible(answer));
	return (answer == Yes);
}

// This filters out any events that were not directed at an element of the
// dialog box, and causes program to quit if null-target event recieved

void
DialogBox::eventLoop(Event& e) {
	if(e.target == nil) {
		fprintf(stderr, "DialogBox::eventLoop received null-target event.  Exiting program.\n");
		exit(0);
	}
    else if (isChild(e.target)) {
        e.target->Handle(e);
    } else {
        Handle(e);
    }
}

void
DialogBox::disappear() {
	GetWorld()->Remove(this);
	Sync();
}

boolean
DialogBox::isChild (Interactor* i) {
    Scene* parent = i->Parent();
    while (parent != nil) {
        if (parent == this) {
            return true;
        }
        parent = parent->Parent();
    }
    return false;
}

///////////

Alert::Alert(ButtonState* s, Interactor* u, Response r) : DialogBox(s, u, r) {
	SetClassName("Alert");
	SetInstance("AlertPanel");
}

void
Alert::Reconfig () {
	Super::Reconfig();
	const Font* font = output->GetFont();
	shape->width += 2 * font->Width("mmmm");
	shape->height += 4 * font->Height();
}

boolean
Alert::acceptible(int answer) { return (answer == Yes); }

int
Alert::getAnswer() {
	Event e;
	Read(e);
	if (e.eventType == KeyEvent && e.len > 0) {
		switch (e.keystring[0]) {
		case '\r':	/* CR */
		case '\007':	/* ^G */
			state->SetValue(Yes);
			break;
		default:
			break;
		}
	}
	else eventLoop(e);
	int answer = NullResponse;
	state->GetValue(answer);
	return answer;
}

///////////

Confirmer::Confirmer(ButtonState* s, Interactor* u, Response r)
		: DialogBox(s, u, r) {
	SetClassName("Confirmer");
	SetInstance("ConfirmPanel");
}

void
Confirmer::Reconfig () {
	Super::Reconfig();
	const Font* font = output->GetFont();
	shape->width += 2 * font->Width("mmmm");
	shape->height += 4 * font->Height();
}

boolean
Confirmer::acceptible(int answer) {
	return (answer == Yes || answer == Cancel);
}

int
Confirmer::getAnswer() {
	Event e;
	Read(e);
	if (e.eventType == KeyEvent && e.len > 0) {
		switch (e.keystring[0]) {
		case 'y':
			state->SetValue(Yes);
			break;
		case '\r':	/* CR */
			state->SetValue(defaultResponse);
			break;
		case 'n':
		case 'c':
		case '\007':	/* ^G */
			state->SetValue(Cancel);
			break;
		default:
			break;
		}
	}
	else eventLoop(e);
	int answer = NullResponse;
	state->GetValue(answer);
	return answer;
}

///////////

ChoiceDialog::ChoiceDialog(ButtonState* s, Interactor* u, Response r)
		: DialogBox(s, u, r) {
	SetClassName("ChoiceDialog");
	SetInstance("ChoicePanel");
}

void
ChoiceDialog::Reconfig () {
	Super::Reconfig();
	const Font* font = output->GetFont();
	shape->width += 2 * font->Width("mmmm");
	shape->height += 4 * font->Height();
}

boolean
ChoiceDialog::acceptible(int answer) {
	return (answer != NullResponse);
}

int
ChoiceDialog::getAnswer() {
	Event e;
	Read(e);
	if (e.eventType == KeyEvent && e.len > 0) {
		switch (e.keystring[0]) {
		case 'y':
			state->SetValue(Yes);
			break;
		case '\r':	/* CR */
			state->SetValue(defaultResponse);
			break;
		case 'n':
			state->SetValue(No);
			break;
		case 'c':
		case '\007':	/* ^G */
			state->SetValue(Cancel);
			break;
		default:
			break;
		}
	}
	else eventLoop(e);
	int answer;
	state->GetValue(answer);
	return answer;
}

InputDialog::InputDialog(ButtonState* s, Interactor* u, Response r)
		: Confirmer(s, u, r), textInput(nil), fileSelector(nil) {
	SetClassName("InputDialog");
	SetInstance("InputPanel");
}

void
InputDialog::Reconfig () {
	Super::Reconfig();
}

void
InputDialog::reset() {
	Super::reset();
	editing = false;
}

// input dialogs automatically set the keyboard focus to the top text-entry
// item when they are first displayed.  checkInput() is called at the top of
// the wait() loop to do this

int 
InputDialog::checkInput() {
 	if(fileSelector != nil)
		fileSelector->Accept();
	else if(textInput != nil && !editing) {
 		textInput->Edit();
 		editing = true;
 	}
	else {			// no textInputs or fileSelectors
	}
	int val = NullResponse;
	state->GetValue(val);
	return val != NullResponse;
}

boolean
InputDialog::acceptible(int answer) {
	return (answer == Yes || answer == Cancel);
}

int
InputDialog::getAnswer() {
	int answer = NullResponse;
	if(checkInput() == false) {
		Event e;
		Read(e);
		eventLoop(e);
	}	
	state->GetValue(answer);		// retrieve state set by TextInputs
	if(answer == '\007')
		state->SetValue(Cancel);
	else if(answer == '\013' || answer == '\015')
		state->SetValue(Yes);
	state->GetValue(answer);
	return answer;
}

void
InputDialog::addTextInput(TextInput* ti) {
	textInput = ti;
}
