/*----------------------------------------------------------------
 * virtual keyboard for AWE driver
 *	written by Takashi Iwai
 *----------------------------------------------------------------*/

#include <stdio.h>
#include <stdarg.h>
#include <ctype.h>
#include <sys/ioctl.h>
#include <fcntl.h>
#include <tcl.h>
#ifdef __FreeBSD__
#  include <machine/soundcard.h>
#  include <awe_voice.h>
#elif defined(linux)
#  include <linux/soundcard.h>
#  include <linux/awe_voice.h>
#endif
#include "util.h"
#include "sffile.h"

/*----------------------------------------------------------------*/

static int seq_init(Tcl_Interp *ip);
static void seq_end(Tcl_Interp *ip);
static void set_program(void);
static int SeqOn(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[]);
static int SeqOff(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[]);
static int SeqStartNote(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[]);
static int SeqStopNote(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[]);
static int SeqControl(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[]);
static int SeqProgram(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[]);
static int SeqBender(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[]);
static int SeqChorusMode(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[]);
static int SeqReverbMode(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[]);
static int ReadSF(ClientData clientData, Tcl_Interp *interp, int argc, char *argv[]);
static void v_error(Tcl_Interp *ip, char *str);
static char *v_eval(Tcl_Interp *ip, char *fmt, ...);
static int AppInit(Tcl_Interp *interp);



#define SEQUENCER_DEV	"/dev/sequencer"

SEQ_DEFINEBUF(128);
int seqfd;

void seqbuf_dump()
{
	if (_seqbufptr)
		if (write(seqfd, _seqbuf, _seqbufptr) == -1) {
			perror("write " SEQUENCER_DEV);
			exit(-1);
		}
	_seqbufptr = 0;
}

int nrsynths;
struct synth_info card_info;
int awe_dev;
int max_synth_voices;

static int seq_opened = 0;
static int seq_bank = 0, seq_preset = 0;
static int seq_bend = 0x2000;

static int seq_init(Tcl_Interp *ip)
{
	int i;

	if (seq_opened) return  1;

	if ((seqfd = open(SEQUENCER_DEV, O_WRONLY, 0)) < 0) {
		v_error(ip, "can't open sequencer device");
		return 0;
	}

	if (ioctl(seqfd, SNDCTL_SEQ_NRSYNTHS, &nrsynths) == -1) {
		v_error(ip, "there is no soundcard installed");
		close(seqfd);
		return 0;
	}
	awe_dev = -1;
	for (i = 0; i < nrsynths; i++) {
		card_info.device = i;
		if (ioctl(seqfd, SNDCTL_SYNTH_INFO, &card_info) == -1) {
			v_error(ip, "cannot get info on soundcard");
			close(seqfd);
			return 0;
		}
		if (card_info.synth_type == SYNTH_TYPE_SAMPLE
		    && card_info.synth_subtype == SAMPLE_TYPE_AWE32)
			awe_dev = i;
	}

	if (awe_dev < 0) {
		v_error(ip, "No AWE synth device is found\n");
		close(seqfd);
		return 0;
	}

	/* use MIDI channel mode */
	AWE_SET_CHANNEL_MODE(awe_dev, AWE_PLAY_MULTI);
	/* toggle drum flag if bank #128 is received */
	AWE_MISC_MODE(awe_dev, AWE_MD_TOGGLE_DRUM_BANK, TRUE);
	set_program();
	
	seq_opened = 1;
	return 1;
}

static void seq_end(Tcl_Interp *ip)
{
	if (seq_opened) {
		close(seqfd);
		seq_opened = 0;
	}
}

/*----------------------------------------------------------------
 * Tcl commands to sequencer device
 *----------------------------------------------------------------*/

static void set_program(void)
{
	SEQ_CONTROL(awe_dev, 0, CTL_BANK_SELECT, seq_bank);
	SEQ_SET_PATCH(awe_dev, 0, seq_preset);
	SEQ_BENDER(awe_dev, 0, seq_bend);
}

static int SeqOn(ClientData clientData, Tcl_Interp *interp,
		 int argc, char *argv[])
{
	if (seq_init(interp)) {
		Tcl_Eval(interp, "ToggleSwitch normal");
	}
	return TCL_OK;
}

static int SeqOff(ClientData clientData, Tcl_Interp *interp,
		 int argc, char *argv[])
{
	seq_end(interp);
	Tcl_Eval(interp, "ToggleSwitch disabled");
	return TCL_OK;
}

static int SeqStartNote(ClientData clientData, Tcl_Interp *interp,
			int argc, char *argv[])
{
	int note, vel;
	if (argc < 3)
		return TCL_ERROR;
	if (! seq_opened) return TCL_OK;
	note = atoi(argv[1]);
	vel = atoi(argv[2]);
	SEQ_START_NOTE(awe_dev, 0, note, vel);
	SEQ_DUMPBUF();
	return TCL_OK;
}

static int SeqStopNote(ClientData clientData, Tcl_Interp *interp,
		       int argc, char *argv[])
{
	int note, vel;
	if (argc < 3)
		return TCL_ERROR;
	if (! seq_opened) return TCL_OK;
	note = atoi(argv[1]);
	vel = atoi(argv[2]);
	SEQ_STOP_NOTE(awe_dev, 0, note, vel);
	SEQ_DUMPBUF();
	return TCL_OK;
}

static int SeqControl(ClientData clientData, Tcl_Interp *interp,
		      int argc, char *argv[])
{
	int type, val;
	if (argc < 3)
		return TCL_ERROR;
	if (! seq_opened) return TCL_OK;
	type = atoi(argv[1]);
	val = atoi(argv[2]);
	SEQ_CONTROL(awe_dev, 0, type, val);
	SEQ_DUMPBUF();
	return TCL_OK;
}

static int SeqProgram(ClientData clientData, Tcl_Interp *interp,
		      int argc, char *argv[])
{
	if (argc < 3)
		return TCL_ERROR;
	seq_bank = atoi(argv[1]);
	seq_preset = atoi(argv[2]);
	if (! seq_opened) return TCL_OK;
	set_program();
	SEQ_DUMPBUF();
	return TCL_OK;
}

static int SeqBender(ClientData clientData, Tcl_Interp *interp,
		     int argc, char *argv[])
{
	if (argc < 2)
		return TCL_ERROR;
	seq_bend = atoi(argv[1]) + 8192;
	if (! seq_opened) return TCL_OK;
	SEQ_BENDER(awe_dev, 0, seq_bend);
	SEQ_DUMPBUF();
	return TCL_OK;
}

static int SeqChorusMode(ClientData clientData, Tcl_Interp *interp,
			 int argc, char *argv[])
{
	if (argc < 2)
		return TCL_ERROR;
	if (seq_init(interp)) {
		int mode = atoi(argv[1]);
		Tcl_Eval(interp, "ToggleSwitch normal");
		AWE_CHORUS_MODE(awe_dev, mode);
		SEQ_DUMPBUF();
	}
	return TCL_OK;
}

static int SeqReverbMode(ClientData clientData, Tcl_Interp *interp,
		     int argc, char *argv[])
{
	if (argc < 2)
		return TCL_ERROR;
	if (seq_init(interp)) {
		int mode = atoi(argv[1]);
		Tcl_Eval(interp, "ToggleSwitch normal");
		AWE_REVERB_MODE(awe_dev, mode);
		SEQ_DUMPBUF();
	}
	return TCL_OK;
}

/*----------------------------------------------------------------*/

/* parse soundfont file and give the preset list */
static int ReadSF(ClientData clientData, Tcl_Interp *interp,
		  int argc, char *argv[])
{
	FILE *fp;
	SFInfo sf;
	char *varname;
	int i;

	/*fprintf(stderr, "opening %s..\n", argv[1]);*/
	if ((fp = fopen(argv[1], "r")) == NULL) {
		v_error(interp, "can't open soundfont file");
		return TCL_OK;
	}
	if (load_soundfont(&sf, fp, TRUE)) {
		v_error(interp, "fail to read soundfont file");
		fclose(fp);
		return TCL_OK;
	}
	fclose(fp);

	varname = argv[2];
	for (i = 0; i < sf.npresets-1; i++) {
		char *p;
		/* convert illegal characters */
		for (p = sf.preset[i].hdr.name; *p; p++) {
			if (!isprint(*p) || *p == '{' || *p == '}')
				*p = ' ';
			else if (*p == '[')
				*p = '(';
			else if (*p == ']')
				*p = ')';
		}
		v_eval(interp, "lappend %s [list %d %d \"%s\"]",
		       varname,
		       sf.preset[i].bank,
		       sf.preset[i].preset,
		       sf.preset[i].hdr.name);
	}
	free_soundfont(&sf);
	return TCL_OK;
}

/*----------------------------------------------------------------
 * Tcl/Tk handlers
 *----------------------------------------------------------------*/

static void v_error(Tcl_Interp *ip, char *str)
{
	fputs("ERROR: ", stderr);
	fputs(str, stderr);
	putc('\n', stderr);
}

static char *v_eval(Tcl_Interp *ip, char *fmt, ...)
{
	char buf[256];
	va_list ap;
	va_start(ap, fmt);
	vsprintf(buf, fmt, ap);
	Tcl_Eval(ip, buf);
	va_end(ap);
	return ip->result;
}

static int AppInit(Tcl_Interp *interp)
{
	if (Tcl_Init(interp) == TCL_ERROR) {
		return TCL_ERROR;
	}
	if (Tk_Init(interp) == TCL_ERROR) {
		return TCL_ERROR;
	}

	Tcl_CreateCommand(interp, "SeqOn", SeqOn,
			  (ClientData)NULL, (Tcl_CmdDeleteProc*)NULL);
	Tcl_CreateCommand(interp, "SeqOff", SeqOff,
			  (ClientData)NULL, (Tcl_CmdDeleteProc*)NULL);
	Tcl_CreateCommand(interp, "SeqStartNote", SeqStartNote,
			  (ClientData)NULL, (Tcl_CmdDeleteProc*)NULL);
	Tcl_CreateCommand(interp, "SeqStopNote", SeqStopNote,
			  (ClientData)NULL, (Tcl_CmdDeleteProc*)NULL);
	Tcl_CreateCommand(interp, "SeqControl", SeqControl,
			  (ClientData)NULL, (Tcl_CmdDeleteProc*)NULL);
	Tcl_CreateCommand(interp, "SeqProgram", SeqProgram,
			  (ClientData)NULL, (Tcl_CmdDeleteProc*)NULL);
	Tcl_CreateCommand(interp, "SeqBender", SeqBender,
			  (ClientData)NULL, (Tcl_CmdDeleteProc*)NULL);
	Tcl_CreateCommand(interp, "SeqChorusMode", SeqChorusMode,
			  (ClientData)NULL, (Tcl_CmdDeleteProc*)NULL);
	Tcl_CreateCommand(interp, "SeqReverbMode", SeqReverbMode,
			  (ClientData)NULL, (Tcl_CmdDeleteProc*)NULL);
	Tcl_CreateCommand(interp, "ReadSF", ReadSF,
			  (ClientData)NULL, (Tcl_CmdDeleteProc*)NULL);
	return TCL_OK;
}

/*----------------------------------------------------------------*/

void main(int argc, char **argv)
{
	char **nargv;
	int c, nargc;

	nargc = argc + 1;
	if ((nargv = (char**)malloc(sizeof(char*) * nargc)) == NULL) {
		fprintf(stderr, "can't malloc\n");
		exit(1);
	}
	nargv[0] = "-f";
	nargv[1] = TCLFILE;
	for (c = 1; c < argc; c++)
		nargv[c+1] = argv[c];

	/* call Tk main routine */
	Tk_Main(nargc, nargv, AppInit);

	exit(0);
}
