/*
 *  ntcp.c
 *
 *  Copyright (C) 1995 Martin von Lwis
 *  Copyright (C) 1997 Rgis Duchesne
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <errno.h>
#include <fcntl.h>
#ifdef HAVE_GETOPT_H
#include <getopt.h>
#else
#define getopt_long(a,v,o,ol,x)        getopt(a,v,o)
#endif
#include <unistd.h>
#include <string.h>
#include <stdlib.h>
#include "types.h"
#include "struct.h"
#include "util.h"
#include "nttools.h"
#include "inode.h"

char *short_opts="Vh";
#ifdef HAVE_GETOPT_H
struct option options[]={
	{"version",0,0,'V'},
	{"help",0,0,'h'},
	{0,0,0,0}
};
#endif

void usage(void)
{
  fprintf(stderr,"ntcp [-V|h] <src> <dst>\n"
	  "Files on the NTFS volume are accessed as //<device>/<path>\n");
}

struct ntfs_file{
  ntfs_inode ino;
  ntfs_attribute *data;
  ntfs_offset_t offset;
};

void
write_unix(void *file,void *buf,size_t len)
{
  if(write(*(int*)file,buf,len)!=len){
    perror("cp2unix");
    exit(1);
  }
}

int
read_unix(void *file,void *buf,size_t len)
{
  int result;
  result=read(*(int*)file,buf,len);
  if(result==-1){
    perror("read_unix");
    exit(1);
  }
  return result;
}

void
write_nt(void *file,void* buf,size_t len)
{
  struct ntfs_file* f=file;
  struct ntfs_io io;
  int error;
  io.fn_put=ntfs_put;
  io.param=buf;
  io.size=len;
  io.do_read=0;
  error=ntfs_readwrite_attr(&f->ino,f->data,f->offset,&io);
  if(error){
    fprintf(stderr,"Error %d writing to ntfs\n",error);
    exit(1);
  }
  f->offset+=len;
}

int
read_nt(void *file,void *buf,size_t len)
{
  struct ntfs_file* f=file;
  struct ntfs_io io;
  int error;
  io.fn_get=ntfs_get;
  io.param=buf;
  io.size=len;
  io.do_read=1;
  error=ntfs_readwrite_attr(&f->ino,f->data,f->offset,&io);
  if(error){
    fprintf(stderr,"Error %d reading from ntfs\n",error);
    exit(1);
  }
  f->offset+=io.size;
  return io.size;
}
  

typedef void (*write_func)(void*,void*,size_t);
typedef int (*read_func)(void*,void*,size_t);
void copy(read_func r,void *rf,write_func w,void *wf)
{
  void *buf=malloc(8192);
  int len;
  do{
    len=r(rf,buf,8192);
    w(wf,buf,len);
  }while(len==8192);
}

int 
ntfs_open(char *name,read_func *r,write_func* w,void **f,int flags)
{
	ntfs_volume *vol;
	char device[256];
	char *it;
	int inum;
	struct ntfs_file *file;

	for(it=name+2;*it && *it!='/';it++)
		/*nothing*/;
	strcpy(device,"/dev/");
	strncpy(device+5,name+2,(it-name)-2);
	device[(it-name)+3]='\0';
	vol=ntfs_open_volume(device,0,1,0);
	if(!vol)return ENODEV;

	file=malloc(sizeof(struct ntfs_file));
	inum=5;
	/* walk the directory tree */
	name=it;
	if(*name=='/')name++;
	do{
		char *next;
		if(ntfs_init_inode(&file->ino,vol,inum)){
			fprintf(stderr,"error finding %s\n",it);
			return 1;
		}
		if(!name || !*name)break;
		next=strchr(name,'/');
		if(next){
			*next='\0';
			next++;
		}
		inum=ntfs_find_file(&file->ino,name);
		if(inum==-1){
			fprintf(stderr,"%s not found\n",name);
			return 1;
		}
		name=next;
	}while(1);
	if(ntfs_init_inode(&file->ino,vol,inum)){
		fprintf(stderr,"error opening %s\n",name);
		return ENOENT;
	}
	file->data=ntfs_find_attr(&file->ino,vol->at_data,0);
	if(!file->data)
		return EISDIR;
	*r=read_nt;
	*w=write_nt;
	*f=file;
	return 0;
}

int
unix_open(char *name,read_func *r,write_func* w,void **f,int flags)
{
	int *file=malloc(sizeof(int));
	*file=open(name,flags,0777);
	if(*file==-1)return errno;
	*r=read_unix;
	*w=write_unix;
	*f=file;
	return 0;
}

int
do_open(char *name,read_func *r,write_func* w,void **f,int flags)
{
	int error;
	if(name[0]=='/' && name[1]=='/')
		error=ntfs_open(name,r,w,f,flags);
	else
		error=unix_open(name,r,w,f,flags);
	
	if(error){
		fprintf(stderr,"%s:%s\n",name,strerror(error));
		exit(1);
	}
	return 0;
}

int main(int argc,char *argv[])
{
	int c;
	extern int opterr,optind;
	extern char* optarg;
	read_func r1,r2;
	write_func w1,w2;
	void *f1,*f2;

	opterr=1;
	while((c=getopt_long(argc,argv,short_opts,options,NULL))>0)
		switch(c)
		{
		case 'V': printf("ntcat " NTFS_VERSION "\n");exit(0);break;
		case 'h': usage();exit(0);break;
		}
	if(optind+2!=argc){
		usage();
		exit(1);
	}
	do_open(argv[optind],&r1,&w1,&f1,O_RDONLY);
	do_open(argv[optind+1],&r2,&w2,&f2,O_WRONLY|O_CREAT);
	copy(r1,f1,w2,f2);
	return 0;
}

/*
 * Local variables:
 * c-file-style: "linux"
 * End:
 */
