/*
 *  fs.c
 *  NTFS driver for Linux 2.0
 *
 *  Copyright (C) 1995-1997 Martin von Lwis
 *  Copyright (C) 1996 Richard Russon
 *  Copyright (C) 1996-1997 Rgis Duchesne
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "types.h"
#include "struct.h"
#include "util.h"
#include "inode.h"
#include "super.h"
#include "dir.h"
#include "support.h"
#include "macros.h"
#include "sysctl.h"
#include <linux/module.h>
#include <asm/segment.h>
#include <linux/fs.h>
#include <linux/string.h>
#include <linux/locks.h>
#include <linux/sysctl.h>
/* for proc_dointvec using isspace */
#include <linux/ctype.h>

#define ITEM_SIZE 2040
#define NAME_SIZE 300

/* io functions to user space */
static void ntfs_putuser(ntfs_io* dest,void *src,ntfs_size_t len)
{
	memcpy_tofs(dest->param,src,len);
	dest->param+=len;
}

#ifdef CONFIG_NTFS_RW
static void ntfs_getuser(void *dest,ntfs_io *src,ntfs_size_t len)
{
	memcpy_fromfs(dest,src->param,len);
	src->param+=len;
}
#endif

static int ntfs_read(struct inode* inode, struct file * filp, char *buf, int count)
{
	int error;
	ntfs_io io;
	ntfs_volume *vol=NTFS_INO2VOL(inode);
	
	ntfs_debug(DEBUG_OTHER, "ntfs_read %x,%x,%x ->",
	       (unsigned)inode->i_ino,(unsigned)filp->f_pos,(unsigned)count);
	/* inode is not properly initialized */
	if(!inode || !inode->u.generic_ip)return -EINVAL;
	/* inode has no unnamed data attribute */
	if(!ntfs_find_attr((ntfs_inode*)inode->u.generic_ip,vol->at_data,NULL))
		return -EINVAL;
	
	/* read the data */
	io.fn_put=ntfs_putuser;
	io.fn_get=0;
	io.param=buf;
	io.size=count;
	error=ntfs_read_attr((ntfs_inode*)inode->u.generic_ip,
			   vol->at_data,NULL,filp->f_pos,&io);
	if(error)return -error;
	
	filp->f_pos+=io.size;
	return io.size;
}

#ifdef CONFIG_NTFS_RW
static int ntfs_write(struct inode* inode, struct file *filp,
	const char* buf, int count)
{
	struct super_block* sb;
	int ret;
	ntfs_io io;
	ntfs_volume* vol=NTFS_INO2VOL(inode);

	ntfs_debug(DEBUG_OTHER, "ntfs_write %x,%x,%x ->",
	       (unsigned)inode->i_ino,(unsigned)filp->f_pos,(unsigned)count);
	if(!inode || !inode->u.generic_ip)
		return -EINVAL;
	sb = inode->i_sb;
	/* Allows to lock fs ro at any time */
	if(sb->s_flags & MS_RDONLY)
		return -ENOSPC;
	if(!ntfs_find_attr(inode->u.generic_ip,vol->at_data,NULL))
		return -EINVAL;

	io.fn_put=0;
	io.fn_get=ntfs_getuser;
	io.param=(void*)buf; /* to get rid of the const */
	io.size=count;
	ret = ntfs_write_attr((ntfs_inode*)inode->u.generic_ip,
			      vol->at_data,NULL,filp->f_pos,&io);
	ntfs_debug(DEBUG_OTHER, "%x\n",ret);
	if(ret<0)return -EINVAL;

	filp->f_pos+=ret;
	return ret;
}
#endif

struct ntfs_filldir{
	struct inode *dir;
	filldir_t filldir;
	unsigned int type;
	ntfs_u32 ph,pl;
	void *dirent;
	char *name;
	int namelen;
};
	
static int ntfs_printcb(ntfs_u8 *entry,void *param)
{
	struct ntfs_filldir* nf=param;
	int flags=NTFS_GETU8(entry+0x51);
	int show_hidden=0,to_lower=0;
	int length=min(NTFS_GETU8(entry+0x50),NAME_SIZE);
	int inum=NTFS_GETU32(entry);
	int i,error;
	switch(nf->type){
	case ngt_dos:
		/* Don't display long names */
		if((flags & 2)==0)
			return 0;
		break;
	case ngt_nt:
		/* Don't display short-only names */
		switch(flags&3){
		case 2: return 0;
		case 3: to_lower=1;
		}
		break;
	case ngt_posix:
		break;
	case ngt_full:
		show_hidden=1;
		break;
	}
	if(!show_hidden && ((NTFS_GETU8(entry+0x48) & 2)==2))
		return 0;
	nf->name=0;
	if(ntfs_encodeuni(NTFS_INO2VOL(nf->dir),(ntfs_u16*)(entry+0x52),
			  length,&nf->name,&nf->namelen))
		return 0;
	/* Do not return ".", as this is faked */
	if(length==1 && *nf->name=='.')
		return 0;
	if(to_lower)
		for(i=0;i<nf->namelen;i++)
			/* This supports ASCII only. Since only DOS-only
			   names get converted, and since those are restricted
			   to ASCII, this should be correct */
			if(nf->name[i]>='A' && nf->name[i]<='Z')
				nf->name[i]+='a'-'A';
	nf->name[nf->namelen]=0;
	ntfs_debug(DEBUG_OTHER, "readdir got %s,len %d\n",nf->name,nf->namelen);
	/* filldir expects an off_t rather than an loff_t.
	   Hope we don't have more than 65535 index records */
	error=nf->filldir(nf->dirent,nf->name,nf->namelen,
			(nf->ph<<16)|nf->pl,inum);
	ntfs_free(nf->name);
	/* Linux filldir errors are negative, other errors positive */
	return error;
}

/* readdir returns '..', then '.', then the directory entries in sequence
   As the root directory contains a entry for itself, '.' is not emulated
   for the root directory */
static int ntfs_readdir(struct inode* dir, struct file *filp, 
	void *dirent, filldir_t filldir)
{
	struct ntfs_filldir cb;
	int error;
	ntfs_debug(DEBUG_OTHER, "ntfs_readdir ino %x mode %x\n",
	       (unsigned)dir->i_ino,(unsigned int)dir->i_mode);
	if(!dir || (dir->i_ino==0) || !S_ISDIR(dir->i_mode))return -EBADF;

	ntfs_debug(DEBUG_OTHER, "readdir: Looking for file %x dircount %d\n",
	       (unsigned)filp->f_pos,dir->i_count);
	cb.pl=filp->f_pos & 0xFFFF;
	cb.ph=filp->f_pos >> 16;
	/* end of directory */
	if(cb.ph==0xFFFF){
		/* FIXME: Maybe we can return those with the previous call */
		switch(cb.pl){
		case 0: filldir(dirent,".",1,filp->f_pos,dir->i_ino);
			filp->f_pos=0xFFFF0001;
			return 0;
			/* FIXME: parent directory */
		case 1: filldir(dirent,"..",2,filp->f_pos,0);
			filp->f_pos=0xFFFF0002;
			return 0;
		}
		ntfs_debug(DEBUG_OTHER, "readdir: EOD\n");
		return 0;
	}
	cb.dir=dir;
	cb.filldir=filldir;
	cb.dirent=dirent;
	cb.type=NTFS_INO2VOL(dir)->ngt;
	do{
		error=ntfs_getdir_unsorted(dir->u.generic_ip,&cb.ph,&cb.pl,
				   ntfs_printcb,&cb);
	}while(!error && cb.ph!=0xFFFFFFFF);
	filp->f_pos=(cb.ph<<16)|cb.pl;
	ntfs_debug(DEBUG_OTHER, "new position %x\n",(unsigned)filp->f_pos);
        /* -EINVAL is on user buffer full. This is not considered 
	   as an error by sys_getdents */
	if(error<0) 
		error=0;
	/* Otherwise (device error, inconsistent data), switch the sign */
	return -error;
}

static int parse_options(struct super_block *sb,ntfs_volume* vol,char *opt)
{
	char *value;

	vol->uid=vol->gid=0;
	vol->umask=0777;
	vol->ngt=ngt_nt;
	vol->nct=nct_utf8;
	if(!opt)return 1;

	for(opt = strtok(opt,",");opt;opt=strtok(NULL,","))
	{
		if ((value = strchr(opt, '=')) != NULL)
			*value++='\0';
		if(strcmp(opt,"uid")==0)
		{
			if(!value || !*value){
				printk(KERN_ERR "NTFS: uid needs an argument\n");
				return 0;
			}
			vol->uid=simple_strtoul(value,&value,0);
			if(*value){
				printk(KERN_ERR "NTFS: uid invalid argument\n");
				return 0;
			}
		}else if(strcmp(opt, "gid") == 0)
		{
			if(!value || !*value){
				printk(KERN_ERR "NTFS: gid needs an argument\n");
				return 0;
			}
			vol->gid=simple_strtoul(value,&value,0);
			if(*value){
				printk(KERN_ERR "gid invalid argument\n");
				return 0;
			}
		}else if(strcmp(opt, "umask") == 0)
		{
			if(!value || !*value){
				printk(KERN_ERR "NTFS: umask needs an argument\n");
				return 0;
			}
			vol->umask=simple_strtoul(value,&value,0);
			if(*value){
				printk(KERN_ERR "umask invalid argument\n");
				return 0;
			}
		}else if(strcmp(opt, "charset") == 0){
			if(!value || !*value){
				printk(KERN_ERR "NTFS: charset needs an argument");
				return 0;
			}
			if(strcmp(value,"iso8859-1") == 0)
				vol->nct = nct_iso8859_1;
			else{
				printk(KERN_ERR "NTFS: unknown charset");
				return 0;
			}
		}else if(strcmp(opt, "posix") == 0)
			vol->ngt=ngt_posix;
		else if(strcmp(opt, "nt") == 0)
			vol->ngt=ngt_nt;
		else if(strcmp(opt, "dos") == 0)
			vol->ngt=ngt_dos;
		else if(strcmp(opt, "full") == 0)
			vol->ngt=ngt_full;
		else if(strcmp(opt, "ntrw") == 0)
		{
			if(value){
				printk(KERN_ERR "NTFS: rw doesn't need an argument\n");
				return 0;
			}
			sb->s_flags &= ~MS_RDONLY;
		}else{
			printk(KERN_ERR "NTFS: unkown option '%s'\n", opt);
			return 0;
		}
	}
	return 1;
}
			
static int ntfs_lookup(struct inode *dir, const char *name, int len,
	struct inode **result)
{
	struct inode *res=0;
	unsigned long ino;
	char *item=0;
	ntfs_iterate_s walk;
	ntfs_volume* vol=NTFS_INO2VOL(dir);
	int error;
	ntfs_debug(DEBUG_OTHER, "Looking up %s in %x\n",name,(unsigned)dir->i_ino);
	if(strncmp(name,".",len)==0){
		*result=dir;
		return 0;
	}
	if(dcache_lookup(dir,name,len,&ino))
	{
		if(!(*result = iget(dir->i_sb,ino)))
		{
			iput(dir);
			return -EACCES;
		}
		iput(dir);
		return 0;
	}
	if(strncmp(name,"..",len)==0)
	{
		/*ntfs_debug(DEBUG_OTHER, ".. not supported\n");*/
		/* What if a directory has more than one name? */
		/* FIXME: support for segments */
		ntfs_attribute *attr=ntfs_find_attr((ntfs_inode*)dir->u.generic_ip,
						    vol->at_file_name,NULL);
		if(!attr){
			ntfs_debug(DEBUG_OTHER, ".. not found\n");
			iput(dir);
			return -ENOENT;
		}
		if(!(*result = iget(dir->i_sb,NTFS_GETU32(attr->d.data))))
		{
			iput(dir);
			return -EACCES;
		}
		iput(dir);
		return 0;
	}
	/* convert to wide string */
	error=ntfs_decodeuni(NTFS_INO2VOL(dir),(char*)name,len,
			     &walk.name,&walk.namelen);
	if(error)
		return error;
	item=ntfs_malloc(ITEM_SIZE);
	/* ntfs_getdir will place the directory entry into item,
	   and the first long long is the MFT record number */
	walk.type=BY_NAME;
	walk.dir=(ntfs_inode*)dir->u.generic_ip;
	walk.result=item;
	if(ntfs_getdir_byname(&walk))
	{
		res=iget(dir->i_sb,NTFS_GETU32(item));
		dcache_add(dir, name, len, NTFS_GETU32(item));
	}
	iput(dir);
	*result=res;
	ntfs_free(item);
	ntfs_free(walk.name);
	return res?0:-ENOENT;
}

struct file_operations ntfs_file_operations_nommap = {
	NULL, /* lseek */
	ntfs_read,
#ifdef CONFIG_NTFS_RW
	ntfs_write,
#else
	NULL,
#endif
	NULL, /* readdir */
	NULL, /* select */
	NULL, /* ioctl */
	NULL, /* mmap */
	NULL, /* open */
	NULL, /* release */
	NULL, /* fsync */
	NULL, /* fasync */
	NULL, /* check_media_change */
	NULL  /* revalidate */
};

struct inode_operations ntfs_inode_operations_nobmap = {
	&ntfs_file_operations_nommap,
	NULL, /* create */
	NULL, /* lookup */
	NULL, /* link */
	NULL, /* unlink */
	NULL, /* symlink */
	NULL, /* mkdir */
	NULL, /* rmdir */
	NULL, /* mknod */
	NULL, /* rename */
	NULL, /* readlink */
	NULL, /* follow_link */
	NULL, /* readpage */
	NULL, /* writepage */
	NULL, /* bmap */
	NULL, /* truncate */
	NULL, /* permission */
	NULL, /* smap */
};

#ifdef CONFIG_NTFS_RW
static int
ntfs_create(struct inode* dir,const char* name,int namelen,
	    int mode,struct inode** result)
{
	struct inode *r=0;
	ntfs_inode *ino=0;
	ntfs_volume *vol;
	int error=0;
	ntfs_attribute *si;

	r=get_empty_inode();
	if(!r){
		error=ENOMEM;
		goto fail;
	}

	ntfs_debug(DEBUG_OTHER, "ntfs_create %s\n",name);
	vol=NTFS_INO2VOL(dir);
	ino=ntfs_malloc(sizeof(ntfs_inode));
	if(!ino){
		error=ENOMEM;
		goto fail;
	}
	error=ntfs_alloc_inode(dir->u.generic_ip,ino,(char*)name,namelen);
	if(error)goto fail;
	error=ntfs_update_inode(ino);
	if(error)goto fail;
	error=ntfs_update_inode(dir->u.generic_ip);
	if(error)goto fail;

	r->u.generic_ip=ino;
	r->i_uid=vol->uid;
	r->i_gid=vol->gid;
	r->i_nlink=1;
	r->i_sb=dir->i_sb;
	/* FIXME: dirty? dev? */
	/* get the file modification times from the standard information */
	si=ntfs_find_attr(ino,vol->at_standard_information,NULL);
	if(si){
		char *attr=si->d.data;
		r->i_atime=ntfs_ntutc2unixutc(NTFS_GETU64(attr+0x18));
		r->i_ctime=ntfs_ntutc2unixutc(NTFS_GETU64(attr));
		r->i_mtime=ntfs_ntutc2unixutc(NTFS_GETU64(attr+8));
	}
	/* It's not a directory */
	r->i_op=&ntfs_inode_operations_nobmap;
	r->i_mode=S_IFREG|S_IRUGO;
	r->i_mode &= ~vol->umask;

	*result=r;
	iput(dir);
	return 0;
 fail:
	if(ino)ntfs_free(ino);
	if(r)iput(r);
	iput(dir);
	return -error;
}
#endif

static int 
ntfs_bmap(struct inode *ino,int block)
{
	int ret=ntfs_vcn_to_lcn((ntfs_inode*)ino->u.generic_ip,block);
	ntfs_debug(DEBUG_OTHER, "bmap of %lx,block %x is %x\n",
	       ino->i_ino,block,ret);
	return (ret==-1) ? 0:ret;
}

struct file_operations ntfs_file_operations = {
	NULL, /* lseek */
	ntfs_read,
#ifdef CONFIG_NTFS_RW
	ntfs_write,
#else
	NULL,
#endif
	NULL, /* readdir */
	NULL, /* select */
	NULL, /* ioctl */
	generic_file_mmap,
	NULL, /* open */
	NULL, /* release */
	NULL, /* fsync */
	NULL, /* fasync */
	NULL, /* check_media_change */
	NULL  /* revalidate */
};

struct inode_operations ntfs_inode_operations = {
	&ntfs_file_operations,
	NULL, /* create */
	NULL, /* lookup */
	NULL, /* link */
	NULL, /* unlink */
	NULL, /* symlink */
	NULL, /* mkdir */
	NULL, /* rmdir */
	NULL, /* mknod */
	NULL, /* rename */
	NULL, /* readlink */
	NULL, /* follow_link */
	NULL, /* readpage */
	NULL, /* writepage */
	ntfs_bmap,
	NULL, /* truncate */
	NULL, /* permission */
	NULL, /* smap */
};

struct file_operations ntfs_dir_operations = {
	NULL, /* lseek */
	NULL, /* read */
	NULL, /* write */
	ntfs_readdir, /* readdir */
	NULL, /* select */
	NULL, /* ioctl */
	NULL, /* mmap */
	NULL, /* open */
	NULL, /* release */
	NULL, /* fsync */
	NULL, /* fasync */
	NULL, /* check_media_change */
	NULL  /* revalidate */
};

struct inode_operations ntfs_dir_inode_operations = {
	&ntfs_dir_operations,
#ifdef CONFIG_NTFS_RW
	ntfs_create, /* create */
#else
	NULL,
#endif
	ntfs_lookup, /* lookup */
	NULL, /* link */
	NULL, /* unlink */
	NULL, /* symlink */
	NULL, /* mkdir */
	NULL, /* rmdir */
	NULL, /* mknod */
	NULL, /* rename */
	NULL, /* readlink */
	NULL, /* follow_link */
	NULL, /* readpage */
	NULL, /* writepage */
	NULL, /* bmap */
	NULL, /* truncate */
	NULL, /* permission */
	NULL, /* smap */
};

/* ntfs_read_inode is called by the Virtual File System (the kernel layer that
 * deals with filesystems) when iget is called requesting an inode not already
 * present in the inode table. Typically filesystems have separate
 * inode_operations for directories, files and symlinks.
 */
static void ntfs_read_inode(struct inode* inode)
{
	ntfs_volume *vol;
	int can_mmap=0;
	ntfs_inode *ino;
	ntfs_attribute *data;
	ntfs_attribute *si;

	vol=NTFS_INO2VOL(inode);
	inode->i_op=NULL;
	inode->i_mode=0;
	inode->u.generic_ip=NULL;
	ntfs_debug(DEBUG_OTHER, "ntfs_read_inode %x\n",(unsigned)inode->i_ino);

	switch(inode->i_ino)
	{
		/* those are loaded special files */
	case FILE_MFT:
		ino=NTFS_INO2VOL(inode)->mft_ino;break;
	default:
		ino=(ntfs_inode*)ntfs_malloc(sizeof(ntfs_inode));
		if(!ino || ntfs_init_inode(ino,
					   NTFS_INO2VOL(inode),inode->i_ino))
		{
			ntfs_debug(DEBUG_OTHER, "NTFS:Error loading inode %x\n",
			       (unsigned int)inode->i_ino);
			return;
		}
	}
	inode->u.generic_ip=ino;
	/* Set uid/gid from mount options */
	inode->i_uid=vol->uid;
	inode->i_gid=vol->gid;
	inode->i_nlink=1;
	/* Use the size of the data attribute as file size */
	data = ntfs_find_attr(ino,vol->at_data,NULL);
	if(!data)
	{
		inode->i_size=0;
		can_mmap=0;
	}
	else
	{
		inode->i_size=data->size;
		can_mmap=!data->resident && !data->compressed;
	}
	/* get the file modification times from the standard information */
	si=ntfs_find_attr(ino,vol->at_standard_information,NULL);
	if(si){
		char *attr=si->d.data;
		inode->i_atime=ntfs_ntutc2unixutc(NTFS_GETU64(attr+0x18));
		inode->i_ctime=ntfs_ntutc2unixutc(NTFS_GETU64(attr));
		inode->i_mtime=ntfs_ntutc2unixutc(NTFS_GETU64(attr+8));
	}
	/* if it has an index root, it's a directory */
	if(ntfs_find_attr(ino,vol->at_index_root,"$I30"))
	{
		ntfs_attribute *at;
		at = ntfs_find_attr (ino, vol->at_index_allocation, "$I30");
		inode->i_size = at ? at->size : 0;
	  
		inode->i_op=&ntfs_dir_inode_operations;
		inode->i_mode=S_IFDIR|S_IRUGO|S_IXUGO;
	}
	else
	{
		inode->i_op=can_mmap ? &ntfs_inode_operations : 
			&ntfs_inode_operations_nobmap;
		inode->i_mode=S_IFREG|S_IRUGO|S_IMMUTABLE;
		inode->i_mode=S_IFREG|S_IRUGO;
	}
	inode->i_mode &= ~vol->umask;
}

static void ntfs_put_inode(struct inode *ino)
{
	ntfs_debug(DEBUG_OTHER, "ntfs_put_inode %lx\n",ino->i_ino);
	if(ino->i_ino!=FILE_MFT && ino->u.generic_ip)
	{
		ntfs_clear_inode(ino->u.generic_ip);
		ntfs_free(ino->u.generic_ip);
		ino->u.generic_ip=0;
	}
	clear_inode(ino);
}

static void ntfs_put_super(struct super_block *sb)
{
	ntfs_volume *vol=NTFS_SB2VOL(sb);
	ntfs_debug(DEBUG_OTHER, "ntfs_put_super\n");
	lock_super(sb);
	ntfs_release_volume(vol);
	sb->s_dev=0;
	ntfs_free(sb->u.generic_sbp);
	sb->u.generic_sbp=0;
	unlock_super(sb);
	MOD_DEC_USE_COUNT;
}

static void ntfs_statfs(struct super_block *sb, struct statfs *sf, int bufsize)
{
	struct statfs fs;
	struct inode *mft;
	ntfs_volume *vol;
	ntfs_debug(DEBUG_OTHER, "ntfs_statfs\n");
	vol=NTFS_SB2VOL(sb);
	memset(&fs,0,sizeof(fs));
	fs.f_type=NTFS_SUPER_MAGIC;
	fs.f_bsize=vol->clustersize;

	fs.f_blocks=ntfs_get_volumesize(NTFS_SB2VOL(sb));
	fs.f_bfree=ntfs_get_free_cluster_count(vol->bitmap);
	fs.f_bavail=fs.f_bfree;

	/* Number of files is limited by free space only, so we lie here */
	fs.f_ffree=0;
	mft=iget(sb,FILE_MFT);
	fs.f_files=mft->i_size/vol->mft_recordsize;
	iput(mft);

	/* should be read from volume */
	fs.f_namelen=255;
	memcpy_tofs(sf,&fs,bufsize);
}

static int ntfs_remount_fs(struct super_block *sb, int* flags, char* options)
{
	if(parse_options(sb,(ntfs_volume*)sb->u.generic_sbp, options))
	{
		if((*flags & MS_RDONLY) && (sb->s_flags && MS_RDONLY))
			return 0;
		if(*flags & MS_RDONLY)
			/* changing from rw to ro */
			sb->s_flags |= MS_RDONLY;
		else if(sb->s_flags & MS_RDONLY)
			/* changing from rw to rw */
			sb->s_flags &= ~MS_RDONLY;
		return 0;
	}
	else
		return -EINVAL;
}

struct super_operations ntfs_super_operations = {
	ntfs_read_inode,
	NULL, /* notify_change */
	NULL, /* write_inode */
	ntfs_put_inode,
	ntfs_put_super,
	NULL, /* write_super */
	ntfs_statfs,
	ntfs_remount_fs, /* remount */
};

/* Entry point of the filesystem. It is called after when the filesystem is
 * mounted. Don't use any pointer into options, as the page containing is is
 * freed after ntfs_read_super returns.
 */
struct super_block * ntfs_read_super(struct super_block *sb, 
				     void *options, int silent)
{
	struct buffer_head *bh;
	int dev=sb->s_dev;
	int i;
	ntfs_volume *vol;
#if defined(DEBUG) && !defined(MODULE)
	extern int console_loglevel;
	console_loglevel=15;
#endif
	ntfs_debug(DEBUG_OTHER, "ntfs_read_super\n");
	MOD_INC_USE_COUNT;
	lock_super(sb);
	ntfs_debug(DEBUG_OTHER, "lock_super\n");

	/* set to read only, user option might reset it */
	sb->s_flags |= MS_RDONLY;

	vol=ntfs_malloc(sizeof(ntfs_volume));
	if(!parse_options(sb,vol,(char*)options)){
		unlock_super(sb);
		ntfs_free(vol);
		MOD_DEC_USE_COUNT;
		return 0;
	}

	/* Read the superblock assuming 512 byte blocks for now */
	set_blocksize(dev,512);
	/* Read the boot block to get the location of the MFT */
	if(!(bh=bread(dev,0,512)))
	{
		printk("Reading super block failed\n");
		sb->s_dev=0;
		ntfs_free(vol);
		unlock_super(sb);
		MOD_DEC_USE_COUNT;
		return NULL;
	}
	ntfs_debug(DEBUG_OTHER, "Done reading boot block\n");

	/* Check for 'NTFS' magic at offset 3 */
	if(!IS_NTFS_VOLUME(bh->b_data)){
		ntfs_debug(DEBUG_OTHER, "Not a NTFS volume\n");
		ntfs_free(vol);
		sb->s_dev=0;
		brelse(bh);
		unlock_super(sb);
		MOD_DEC_USE_COUNT;
		return NULL;
	}

	ntfs_debug(DEBUG_OTHER, "Going to init volume\n");
	ntfs_init_volume(vol,bh->b_data);
	NTFS_SB(vol)=sb;
	NTFS_SB2VOL(sb)=vol;
	ntfs_debug(DEBUG_OTHER, "Done to init volume\n");

	sb->s_blocksize=vol->clustersize;

	for(i=sb->s_blocksize,sb->s_blocksize_bits=0;i;i>>=1)
		sb->s_blocksize_bits++;

	/* inform the kernel that a device block is a NTFS cluster */
	set_blocksize(dev,sb->s_blocksize);
	ntfs_debug(DEBUG_OTHER, "set_blocksize\n");

	ntfs_debug(DEBUG_OTHER, "MFT record at cluster 0x%X\n",vol->mft_cluster);
	brelse(bh);

	/* Allocate a MFT record */
	/* Handle cases where mft record is smaller than a cluster */
	vol->mft=ntfs_malloc(max(vol->mft_recordsize,vol->clustersize));

	/* read the MFT record 0 */
	/* If there is more than one MFT record per cluster, 
	   read just one cluster */
	for(i=0;i<max(vol->mft_clusters_per_record,1);i++){
		if(!(bh=bread(dev,vol->mft_cluster+i,vol->clustersize)))
		{
			ntfs_error("Could not read MFT record 0\n");
			sb->s_dev=0;
			unlock_super(sb);
			MOD_DEC_USE_COUNT;
			return NULL;
		}
		ntfs_memcpy(vol->mft+i*vol->clustersize,bh->b_data,vol->clustersize);
		brelse(bh);
		ntfs_debug(DEBUG_OTHER, "Read cluster %x\n",vol->mft_cluster+i);
	}

	/* Check and fixup MFT record 0 */
	if(!ntfs_check_mft_record(vol,vol->mft)){
		ntfs_error("Invalid MFT record 0\n");
		sb->s_dev=0;
		unlock_super(sb);
		MOD_DEC_USE_COUNT;
		return NULL;
	}
	/* inform the kernel about which super operations are available */
	sb->s_op = &ntfs_super_operations;
	sb->s_magic = NTFS_SUPER_MAGIC;
	
	ntfs_debug(DEBUG_OTHER, "Reading master and upcase files\n");
	if(ntfs_load_special_files(vol)){
		ntfs_error("Error loading special files\n");
		sb->s_dev=0;
		unlock_super(sb);
		MOD_DEC_USE_COUNT;
		return NULL;
	}

	unlock_super(sb);

	ntfs_debug(DEBUG_OTHER, "Getting RootDir\n");
	/* get the root directory */
	if(!(sb->s_mounted=iget(sb,FILE_ROOT))){
		sb->s_dev=0;
		ntfs_debug(DEBUG_OTHER, "Could not get root dir inode\n");
		MOD_DEC_USE_COUNT;
		return NULL;
	}
	ntfs_debug(DEBUG_OTHER, "read_super: done\n");
	return sb;
}

/* This structure defines the filesystem
 *
 * Define SECOND if you cannot unload ntfs, and want to avoid rebooting
 * for just one more test
 */
struct file_system_type ntfs_fs_type = {
	ntfs_read_super,/* Entry point of the filesystem */
#ifndef SECOND
	"ntfs", /* Name of the filesystem, as used in mount -t ntfs */
#else
	"ntfs2",
#endif
	1,      /* The filesystem requires a block device to be mounted on */
	NULL    /* Will point to the next filesystem in the kernel table */
};

int init_ntfs_fs(void)
{
	/* Uncomment this if you don't trust klogd. There are reasons
	   not to trust it. */
#if 0
	extern int console_loglevel;
	console_loglevel=15;
#endif
	printk(KERN_NOTICE "NTFS version " NTFS_VERSION "\n");
	SYSCTL(1);
	ntfs_debug(DEBUG_OTHER, "registering %s\n",ntfs_fs_type.name);
	/* add this filesystem to the kernel table of filesystems */
	return register_filesystem(&ntfs_fs_type);
}

#ifdef MODULE
/* Every module must contain 2 particular functions whose names are predefined :
 * . init_module() is called by the kernel when kerneld loads the module in
 *     memory. Here is the real beginning of the code.
 * . cleanup_module() is called by the kernel when kerneld removes the module
 */

int init_module(void)
{
	return init_ntfs_fs();
}

void cleanup_module(void)
{
        SYSCTL(0);
	ntfs_debug(DEBUG_OTHER, "unregistering %s\n",ntfs_fs_type.name);
	unregister_filesystem(&ntfs_fs_type);
}
#endif

/* this is copied from linux/kernel/sysctl.c, because Linux 2.0 fails
   to export it */
int proc_dointvec(ctl_table *table, int write, struct file *filp,
		  void *buffer, size_t *lenp)
{
	int *i, vleft, first=1, len, left, neg, val;
	#define TMPBUFLEN 20
	char buf[TMPBUFLEN], *p;
	
	if (!table->data || !table->maxlen || !*lenp ||
	    (filp->f_pos && !write)) {
		*lenp = 0;
		return 0;
	}
	
	i = (int *) table->data;
	vleft = table->maxlen / sizeof(int);
	left = *lenp;
	
	for (; left && vleft--; i++, first=0) {
		if (write) {
			while (left && isspace(get_user((char *) buffer)))
				left--, ((char *) buffer)++;
			if (!left)
				break;
			neg = 0;
			len = left;
			if (len > TMPBUFLEN-1)
				len = TMPBUFLEN-1;
			memcpy_fromfs(buf, buffer, len);
			buf[len] = 0;
			p = buf;
			if (*p == '-' && left > 1) {
				neg = 1;
				left--, p++;
			}
			if (*p < '0' || *p > '9')
				break;
			val = simple_strtoul(p, &p, 0);
			len = p-buf;
			if ((len < left) && *p && !isspace(*p))
				break;
			if (neg)
				val = -val;
			buffer += len;
			left -= len;
			*i = val;
		} else {
			p = buf;
			if (!first)
				*p++ = '\t';
			sprintf(p, "%d", *i);
			len = strlen(buf);
			if (len > left)
				len = left;
			memcpy_tofs(buffer, buf, len);
			left -= len;
			buffer += len;
		}
	}

	if (!write && !first && left) {
		put_user('\n', (char *) buffer);
		left--, buffer++;
	}
	if (write) {
		p = (char *) buffer;
		while (left && isspace(get_user(p++)))
			left--;
	}
	if (write && first)
		return -EINVAL;
	*lenp -= left;
	filp->f_pos += *lenp;
	return 0;
}

/*
 * Local variables:
 *  c-file-style: "linux"
 * End:
 */
