/*
 * pftp -- sends files from host to host through free choosable ports
 *
 * Copyright (C) 1996, 1997 Ben Schluricke
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the emplied warranty of MERCHANT-
 * ABILITY OF FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
 * Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 * 
 *    Written by Ben Schluricke
 *    E-Mail:    bhor0533@lehr.chem.TU-Berlin.DE
 *
 * This program is dedicated to my girl-friend, Heather O'Rourke.
 *
 *
 */
#ifdef USE_POSIX_THREAD
#define _REENTRANT
#include <pthread.h>
#endif
#ifdef FreeBSD
#include <sys/errno.h>
#endif
#if !defined HP_UX
#include <fcntl.h>
#endif
#include <sys/wait.h>
#include "connect.h"
#include "main.h"

extern void set_tty(int);
extern void accs(void);
extern void handler(int);
extern void server_filter_programm(void *);
extern void set_filter(short);
extern void receive_data(void *);
extern void free_memory(void);
extern int init_struct(void);

/*
 * Server--This is the server part of pftp.
 */
void Server(int portn)
{
   char hostname[64];
   FILE *fp=NULL, *ld=NULL;
   int ns, setopt=1, optval=BUFSIZE, size=0;
   struct hostent *hp;
   struct sockaddr_in sin, fsin;
   int sinlen, fsinlen;
   int pid=0;
   int strnum=0;
#ifdef USE_POSIX_THREAD
   pthread_t thr;
#endif

   /*
    * Set signal SIGINT.
    */
   signal(SIGINT, handler);
   signal(SIGQUIT, handler);
   signal(SIGCHLD, SIG_IGN);

   /*
    * Before we can do anything, we need to know
    * our hostname.
    */
   gethostname(hostname, sizeof(hostname));
   fprintf(stderr, "\n*** %s is waiting for %s on port %d ***\n\n", hostname, (*statstr)->_STANDARD_INPUT_ ? "a stream" : "file(s)", portn);

   /*
    * Set server filter if '-f' was given on cmdline.
    */
   if ((*statstr)->usefilter) set_filter(1);

   /*
    * Now we look up our host to get
    * its network number.
    */
   if ((hp = (struct hostent *)gethostbyname(hostname)) != NULL) {
      memcpy(&sin.sin_addr, hp->h_addr, hp->h_length);
   }
   else if (!(sin.sin_addr.s_addr = inet_addr(hostname))) {
      fprintf(stderr, "%s: unknown host.\n", hostname);
      exit(1);
   }

   /*
    * Create the address we will be connected to.
    */
   sinlen = sizeof(sin);
   bzero((char *)&sin, sinlen);
   sin.sin_family = AF_INET;
   sin.sin_port = htons(portn);
   sin.sin_addr.s_addr = htonl(INADDR_ANY);

   /*
    * Get a socket and set options on it.
    */
   accs();
   if (setsockopt(s, SOL_SOCKET, SO_REUSEADDR, (char *)&setopt, sizeof(setopt)) < 0) {
         fprintf(stderr, "** setsockopt: %s\n", _PFTP_ERROR_ARRAY_);
         exit(1);
   }
   if (setsockopt(s, SOL_SOCKET, SO_RCVBUF, (char *)&optval, sizeof(optval)) < 0) {
         fprintf(stderr, "** setsockopt: %s\n", _PFTP_ERROR_ARRAY_);
         exit(1);
   }

   /*
    * Try to bind the address to the socket.
    */
   if (bind(s, (struct sockaddr *)&sin, sinlen) < 0) {
      fprintf(stderr, "** bind: %s\n", _PFTP_ERROR_ARRAY_);
      if (pftplog) {
         if ((ld = fopen(pftplog, "a")) != NULL) {
            fprintf(ld, "** bind: %s\n", _PFTP_ERROR_ARRAY_);
            fclose(ld);
         }
      }
      exit(1);
   }

   while (s) {
      if ((*statstr)->OVERWRITE==1) (*statstr)->OVERWRITE=0;
      /*
       * Listen on the socket.
       */
      if (listen(s, (*statstr)->_STANDARD_INPUT_ ? 1: MAXCLIENTS) < 0) {
         fprintf(stderr, "** listen: %s\n", _PFTP_ERROR_ARRAY_);
         if (pftplog) {
            if ((ld = fopen(pftplog, "a")) != NULL) {
               fprintf(ld, "** listen: %s\n", _PFTP_ERROR_ARRAY_);
               fclose(ld);
            }
         }
         exit(1);
      }

      /*
       * Accept connections.  When we accept one, ns
       * will be connected to the client.
       */
      fsinlen = sizeof(fsin);
      if ((ns = accept(s, (struct sockaddr *)&fsin, &fsinlen)) < 0) {
         fprintf(stderr, "** accept: %s\n", _PFTP_ERROR_ARRAY_);
         if (pftplog) {
            if ((ld = fopen(pftplog, "a")) != NULL) {
               fprintf(ld, "** accept: %s\n", _PFTP_ERROR_ARRAY_);
               fclose(ld);
            }
         }
         exit(1);
      }

      /*
       * Get client's host name.
       */
      if ((hp = gethostbyaddr((char *)&fsin.sin_addr, sizeof(fsin.sin_addr), AF_INET)) == 0) {
         if (*_CLIENTHOSTNAME_) {
            fprintf(stderr, "** Can't get client's hostname.\n");
            shutdown(ns, 2);
            if ((*statstr)->_STANDARD_INPUT_) {
               shutdown(s, 2);
               s = 0;
               break;
            }
            continue;
         }
         (*statstr)->REMOTEHOSTNAME = NULL;
      }
      else {
         (*statstr)->size = size = strlen(hp->h_name) + 1;
         if ((*statstr)->REMOTEHOSTNAME) free((*statstr)->REMOTEHOSTNAME);
         MEM_CHECK(((*statstr)->REMOTEHOSTNAME = (char *) calloc(size, sizeof(char))));
         strcpy((*statstr)->REMOTEHOSTNAME, hp->h_name);
      }

      /*
       * Check if remote server has a valid host name.
       */
      if (*_CLIENTHOSTNAME_) {
         int i, loc=0, cl, isok=0, length = strlen((*statstr)->REMOTEHOSTNAME);

         for (cl=0; _CLIENTHOSTNAME_[cl] && !isok; cl++) {
            loc = strlen(*(_CLIENTHOSTNAME_+cl));

            if (length < loc) continue;
            else if (*_CLIENTHOSTNAME_[cl] == '.') {
               loc--;
               for (i=length-1; loc; loc--, i--) {
                  if (*(_CLIENTHOSTNAME_[cl]+loc) != tolower(*((*statstr)->REMOTEHOSTNAME+i))) break;
               }
               if (!loc) isok = 1;
            }
            else if (length == loc) {
               isok=1;
               for (i=0; i < length; i++) {
                  if (*(_CLIENTHOSTNAME_[cl]+i) != tolower(*((*statstr)->REMOTEHOSTNAME+i))) {
                     isok = 0;
                     break;
                  }
               }
            }
         }
         if (!isok) {
            fprintf(stderr, "** Connection to %s refused.\n", (*statstr)->REMOTEHOSTNAME);
            if (pftplog) {
               if ((ld = fopen(pftplog, "a")) != NULL) {
                  fprintf(ld, "** Connection to %s refused.\n", (*statstr)->REMOTEHOSTNAME);
                  fclose(ld);
               }
            }
            if (shutdown(ns, 2) < 0) {
               fprintf(stderr, "** shutdown: %s\n", _PFTP_ERROR_ARRAY_);
               exit(1);
            }
            if ((*statstr)->_STANDARD_INPUT_) {
               shutdown(s, 2);
               s = 0;
               break;
            }
            continue;
         }
      }
      if (setsockopt(ns, SOL_SOCKET, SO_RCVBUF, (char *)&optval, sizeof(optval)) < 0) {
         fprintf(stderr, "** setsockopt: %s\n", _PFTP_ERROR_ARRAY_);
         exit(1);
      }
#if !defined HP_UX
      fcntl(ns, F_SETFL, FASYNC);
      fcntl(2, F_SETFL, FASYNC);
#endif
#ifdef USE_POSIX_THREAD
      strnum = init_struct();
      if (!strnum) {
         fprintf(stderr, "\n** Connection to %s refused.\n", (*statstr)->REMOTEHOSTNAME);
         if (pftplog) {
            if ((ld = fopen(pftplog, "a")) != NULL) {
               fprintf(ld, "\n** Connection to %s refused.\n", (*statstr)->REMOTEHOSTNAME);
               fclose(ld);
            }
         }
         if (shutdown(ns, 2) < 0) {
            fprintf(stderr, "** shutdown: %s\n", _PFTP_ERROR_ARRAY_);
            exit(1);
         }
         continue;
      }
#else
      strnum = 0;
#endif
      (*(statstr+strnum))->ns = ns;
      if (*filter) {
         (*statstr)->REMOTEHOSTNAME[size] = '\0';
         fprintf(stderr, "\n* Filter is receiving data from %s ...\n", (*statstr)->REMOTEHOSTNAME);
#ifdef USE_POSIX_THREAD
         /*
          * Create a new thread for the filter routine.
          */
         if ((pid = pthread_create(&thr, NULL, (void *)&server_filter_programm, (void *)&strnum)) != 0) {
            fprintf(stderr, "** pthread_create: %s\n", _PFTP_ERROR_ARRAY_PID_);
            exit(1);
         }
         if ((*statstr)->_STANDARD_INPUT_) {
            /*
             * Wait for the thread if stdin.
             */
            if ((pid = pthread_join(thr, (void **)NULL)) != 0) {
               fprintf(stderr, "** pthread_join: %s\n", _PFTP_ERROR_ARRAY_PID_);
               exit(1);
            }
            s = 0;
         }
#else
         /*
          * Spawn a new process for the filter routine.
          */
         if ((pid = fork()) < 0) {
            fprintf(stderr, "** fork: %s\n", _PFTP_ERROR_ARRAY_);
            exit(1);
         }

         /*
          * Here starts the filter process.
          */
         if (!pid) {
            server_filter_programm((void *)&strnum);
         }
         if ((*statstr)->_STANDARD_INPUT_) {
            /*
             * Wait for the filter process if stdin.
             */
#ifdef NEXTSTEP
            while (wait((union wait *)0) != pid); 
            wait3((union wait *)0, WNOHANG|WUNTRACED, (struct rusage *)NULL); 
#else
            while (wait((int *)0) != pid);
            waitpid(-2, (int *)0, 0);
#endif
            s = 0;
         }
#endif /* USE_POSIX_THREAD */
      }
      else {
#ifdef USE_POSIX_THREAD
         fp = fdopen(ns, "r");
         *((*(statstr+strnum))->REMOTEHOSTNAME+size) = '\0';
         if ((*statstr)->_STANDARD_INPUT_) {
            setvbuf(fp, \
            (*statstr)->_STDIN_BUFFER_, \
            (*statstr)->_STDIN_BUFFER_ ?  _IOFBF : _IONBF, \
            (*statstr)->_SET_STDIN_BUF_ ? (*statstr)->_STDIN_BUFSIZ_: BUFSIZ);
         }
         (*(statstr+strnum))->fp = fp;
         if ((pid = pthread_create(&thr, NULL, (void *)&receive_data, (void *)&strnum)) != 0) {
            fprintf(stderr, "** pthread_create: %s\n", _PFTP_ERROR_ARRAY_PID_);
            exit(1);
         }
         if ((*statstr)->_STANDARD_INPUT_) {
            if ((pid = pthread_join(thr, (void **)NULL)) != 0) {
               fprintf(stderr, "** pthread_join: %s\n", _PFTP_ERROR_ARRAY_PID_);
               exit(1);
            }
            s = 0;
         }
#else
         if ((pid = fork()) < 0) {
            fprintf(stderr, "** fork: %s\n", _PFTP_ERROR_ARRAY_);
            exit(1);
         }

         if (!pid) {
            signal(SIGCHLD, SIG_DFL);
            signal(SIGINT, SIG_DFL);
            signal(SIGQUIT, SIG_DFL);
            fp = fdopen(ns, "r");
            *((*(statstr+strnum))->REMOTEHOSTNAME+size) = '\0';
            if ((*statstr)->_STANDARD_INPUT_) {
               setvbuf(fp, \
               (*statstr)->_STDIN_BUFFER_, \
               (*statstr)->_STDIN_BUFFER_ ?  _IOFBF : _IONBF, \
               (*statstr)->_SET_STDIN_BUF_ ? (*statstr)->_STDIN_BUFSIZ_: BUFSIZ);
               s = 0;
            }
            (*(statstr+strnum))->fp = fp;
            receive_data((void *)&strnum);
            fclose(fp);
         }
         else {
            if ((*statstr)->_STANDARD_INPUT_) {
#ifdef NEXTSTEP
               while (wait((union wait *)0) != pid); 
               wait3((union wait *)0, WNOHANG|WUNTRACED, (struct rusage *)NULL); 
#else
               while (wait((int *)0) != pid);
               waitpid(-2, (int *)0, 0);
#endif
               s = 0;
            }
         }
#endif /* USE_POSIX_THREAD */
      }
   }
   shutdown(s, 2);
   free_memory();
   if (*statstr) free(*statstr);
   *statstr = NULL;

   exit(0);
}
