/***

options.c - implements the configuration section of the utility
Written by Gerard Paul Java
Copyright (c) Gerard Paul Java 1997, 1998

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

***/

#include <curses.h>
#include <panel.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include "serv.h"
#include "options.h"
#include "deskman.h"
#include "attrs.h"
#include "stdwinset.h"
#include "menurt.h"
#include "input.h"
#include "error.h"
#include "dirs.h"

extern void srpromisc(int mode);

void makeoptionmenu(struct MENU *menu)
{
    initmenu(menu, 11, 25, 7, 15);
    additem(menu, " ^R^everse DNS lookups", "Toggles lookup of IP addresses for host names");
    additem(menu, " ^P^romiscuous operation", "Toggles capture of all packets by Ethernet interfaces");
    additem(menu, " ^C^olor", "Turns color on or off (restart to effect change)");
    additem(menu, " ^L^ogging", "Toggles logging of traffic to a data file");
    additem(menu, " TCP ^t^imeout...", "Determines how long TCP entries may remain idle");
    additem(menu, " Logging ^i^nterval...", "Sets the time between loggings for interface, host, and service stats");
    additem(menu, " ^A^dditional port...", "Allows you to add custom ports for the service stats");
    additem(menu, " ^D^elete port...", "Deletes a port earlier added");
    additem(menu, " E^x^it menu", "Return to main menu");
}

void indicatesetting(int row, struct OPTIONS *options, WINDOW * win)
{
    char *blankpad = "              ";

    wmove(win, row, 2);

    switch (row) {
    case 1:
	if (options->revlook)
	    wprintw(win, "Reverse lookup");
	else
	    wprintw(win, blankpad);

	break;
    case 2:
	if (options->promisc)
	    wprintw(win, "Promiscuous");
	else
	    wprintw(win, blankpad);

	break;
    case 3:
	if (options->color)
	    wprintw(win, "Color");
	else
	    wprintw(win, blankpad);

	break;
    case 4:
	if (options->logging)
	    wprintw(win, "Logging");
	else
	    wprintw(win, blankpad);

	break;
    }
}

void saveoptions(struct OPTIONS *options)
{
    int fd;
    int bw;
    int response;

    fd = open(CONFIGFILE, O_CREAT | O_TRUNC | O_WRONLY, S_IRUSR | S_IWUSR);

    if (fd < 0) {
	errbox("Cannot create config file", ANYKEY_MSG, &response);
	return;
    }
    bw = write(fd, options, sizeof(struct OPTIONS));

    if (bw < 0)
	errbox("Unable to write config file", ANYKEY_MSG, &response);

    close(fd);
}

void setdefaultopts(struct OPTIONS *options)
{
    options->revlook = 0;
    options->promisc = 0;
    options->color = 1;
    options->logging = 0;
    options->timeout = 15;
    options->logspan = 3600;
}

void loadoptions(struct OPTIONS *options)
{
    int fd;
    int br;

    fd = open(CONFIGFILE, O_RDONLY);

    if (fd < 0) {
	setdefaultopts(options);
	return;
    }
    br = read(fd, options, sizeof(struct OPTIONS));

    if (br < 0)
	setdefaultopts(options);

    close(fd);
}

void updatetimes(struct OPTIONS *options, WINDOW * win)
{
    wmove(win, 6, 20);
    wprintw(win, "%3u mins", options->timeout);
    wmove(win, 7, 20);
    wprintw(win, "%3u mins", options->logspan / 60);
}

void showoptions(struct OPTIONS *options, WINDOW * win)
{
    int i;

    for (i = 1; i <= 4; i++)
	indicatesetting(i, options, win);

    updatetimes(options, win);
}

void settimeout(unsigned int *value)
{
    WINDOW *dlgwin;
    PANEL *dlgpanel;
    struct FIELDLIST field;
    int aborted;
    int resp;
    unsigned int tmval = 0;

    dlgwin = newwin(7, 40, 10, 10);
    dlgpanel = new_panel(dlgwin);

    wattrset(dlgwin, BOXATTR);
    colorwin(dlgwin);
    box(dlgwin, ACS_VLINE, ACS_HLINE);

    wattrset(dlgwin, STDATTR);
    wmove(dlgwin, 2, 2);
    wprintw(dlgwin, "Enter value in minutes");
    wmove(dlgwin, 5, 2);
    stdkeyhelp(dlgwin);

    initfields(&field, 1, 10, 13, 12);
    addfield(&field, 3, 0, 0, "");

    do {
	fillfields(&field, &aborted);

	if (!aborted) {
	    tmval = atoi(field.list->buf);
	    if (tmval == 0)
		errbox("Invalid timeout value", ANYKEY_MSG, &resp);
	}
    } while ((tmval == 0) && (!aborted));

    if (!aborted)
	*value = tmval;

    del_panel(dlgpanel);
    delwin(dlgwin);

    destroyfields(&field);
    update_panels();
    doupdate();
}

void setoptions(struct OPTIONS *options, struct porttab **ports)
{
    int row = 1;
    int aborted;

    struct MENU menu;
    WINDOW *statwin;
    PANEL *statpanel;

    makeoptionmenu(&menu);

    statwin = newwin(9, 30, 7, 40);
    statpanel = new_panel(statwin);

    wattrset(statwin, BOXATTR);
    colorwin(statwin);
    box(statwin, ACS_VLINE, ACS_HLINE);
    wmove(statwin, 0, 1);
    wprintw(statwin, " Enabled Options ");
    wattrset(statwin, STDATTR);
    wmove(statwin, 6, 2);
    wprintw(statwin, "TCP timeout:");
    wmove(statwin, 7, 2);
    wprintw(statwin, "Log interval:");
    showoptions(options, statwin);

    do {
	showmenu(&menu);
	operatemenu(&menu, &row, &aborted);

	switch (row) {
	case 1:
	    options->revlook = ~(options->revlook);
	    break;
	case 2:
	    options->promisc = ~(options->promisc);
	    srpromisc(options->promisc);
	    break;
	case 3:
	    options->color = ~(options->color);
	    break;
	case 4:
	    options->logging = ~(options->logging);
	    break;
	case 5:
	    settimeout(&(options->timeout));
	    updatetimes(options, statwin);
	    break;
	case 6:
	    settimeout((unsigned int *) &(options->logspan));
	    options->logspan = options->logspan * 60;
	    updatetimes(options, statwin);
	    break;
	case 7:
	    addmoreports(ports);
	    break;
	case 8:
	    removeaport(ports);
	    break;
	}

	indicatesetting(row, options, statwin);
    } while (row != 9);

    destroymenu(&menu);
    del_panel(statpanel);
    delwin(statwin);
    update_panels();
    doupdate();
}
