/* sane - Scanner Access Now Easy.
   Copyright (C) 1997 David Mosberger-Tang
   This file is part of the SANE package.

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston,
   MA 02111-1307, USA.

   As a special exception, the authors of SANE give permission for
   additional uses of the libraries contained in this release of SANE.

   The exception is that, if you link a SANE library with other files
   to produce an executable, this does not by itself cause the
   resulting executable to be covered by the GNU General Public
   License.  Your use of that executable is in no way restricted on
   account of linking the SANE library code into it.

   This exception does not, however, invalidate any other reasons why
   the executable file might be covered by the GNU General Public
   License.

   If you submit changes to SANE to the maintainers to be included in
   a subsequent release, you agree by submitting the changes that
   those changes may be distributed with this exception intact.

   If you write modifications of your own for SANE, it is your choice
   whether to permit this exception to apply to your modifications.
   If you do not wish that, delete this exception notice.

   This file implements a SANE backend for the Ultima/Artec AT3 and
   A6000C scanners.

   Copyright (C) 1998, Chris Pinkham
   Released under the terms of the GPL.
   *NO WARRANTY*   */

#include <sane/config.h>

#include <limits.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>

#include <sane/sane.h>
#include <sane/saneopts.h>
#include <sane/sanei_scsi.h>
#include <artec.h>

#define BACKEND_NAME	artec
#include <sane/sanei_backend.h>

#ifndef PATH_MAX
# define PATH_MAX	1024
#endif

#include <sane/sanei_config.h>
#define ARTEC_CONFIG_FILE "artec.conf"

static int num_devices;
static ARTEC_Device *first_dev;
static ARTEC_Scanner *first_handle;

static const SANE_String_Const mode_list[] =
  {
    "Lineart", "Halftone", "Gray", "Color",
    0
  };

#define INQ_LEN	0x60
static const u_int8_t inquiry[] =
  {
    0x12, 0x00, 0x00, 0x00, INQ_LEN, 0x00
  };

static const u_int8_t test_unit_ready[] =
  {
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00
  };

static size_t
max_string_size (const SANE_String_Const strings[])
{
  size_t size, max_size = 0;
  int i;

  for (i = 0; strings[i]; ++i)
    {
      size = strlen (strings[i]) + 1;
      if (size > max_size)
	max_size = size;
    }
  return max_size;
}

static SANE_Status
read_data (int fd, int data_type_code, u_char * dest, size_t *len)
{
  static u_char read_6[10];

  DBG( 3, "read_data: len = %lu\n", (u_long) *len );

  memset (read_6, 0, sizeof (read_6));
  read_6[0] = 0x28;
  read_6[2] = data_type_code;
  read_6[6] = *len >> 16;
  read_6[7] = *len >> 8;
  read_6[8] = *len;
  return sanei_scsi_cmd (fd, read_6, sizeof (read_6), dest, len);
}

static SANE_Status
write_data (int fd, void *data, size_t len)
{
  char write_6[4096];

  memset (write_6, 0, 6);
  write_6[0] = 0xa;
  write_6[2] = len >> 16;
  write_6[3] = len >> 8;
  write_6[4] = len;
  memcpy (write_6 + 6, data, len);

  return sanei_scsi_cmd (fd, write_6, 6 + len, 0, 0);
}

int artec_get_status (int fd)
{
  u_char write_10[10];
  u_char read_12[12];
  size_t nread;

  nread = 12;

  memset (write_10, 0, 10);
  write_10[0] = 0x34;
  write_10[8] = 0x0c;

  sanei_scsi_cmd (fd, write_10, 10, read_12, &nread );

  nread = ( read_12[9] << 16 ) + ( read_12[10] << 8 ) + read_12[11];
  DBG(3, "artec_status: %lu\n", (u_long) nread );

  return( nread );
}

static SANE_Status
artec_calibrate_white_shading (SANE_Handle handle)
{
  ARTEC_Scanner *s = handle;
  u_char buf[10368];
  size_t len;

  len = 10368;

  read_data( s->fd, DATA_RED_SHADING, buf, &len );
  read_data( s->fd, DATA_GREEN_SHADING, buf, &len );
  read_data( s->fd, DATA_BLUE_SHADING, buf, &len );

  return( 0 );
}

static SANE_Status
artec_line_rgb_to_byte_rgb ( SANE_Byte *data, SANE_Int len )
{
  SANE_Byte tmp_buf[ 8192 ]; /* Artec max dpi 300 * 8.5 inches * 3 = 7650 */
  int count, to;

  /* copy the rgb data to our temp buffer */
  memcpy( tmp_buf, data, len * 3 );

  /* now copy back to *data in RGB format */
  for ( count = 0, to = 0; count < len ; count++, to += 3 )
  {
    data[ to ] = tmp_buf[ count ];                    /* R byte */
    data[ to + 1 ] = tmp_buf[ count + len ];          /* G byte */
    data[ to + 2 ] = tmp_buf[ count + ( len * 2 ) ];  /* B byte */
  }

  return( SANE_STATUS_GOOD );
}

static SANE_Status
artec_buffer_line_offset( SANE_Int line_offset, SANE_Byte *data, size_t *len )
{
  static SANE_Byte *tmp_line_buf = NULL;
  static SANE_Byte **r_line_buf = NULL;
  static SANE_Byte **g_line_buf = NULL;
  static SANE_Int  r_buf_lines;
  static SANE_Int  g_buf_lines;
  static SANE_Int  width;
  static SANE_Int  cur_line = 0;
  SANE_Byte        *tmp_r_buf_ptr;
  SANE_Byte        *tmp_g_buf_ptr;
  int              count;

  if ( *len == 0 )
    return( SANE_STATUS_GOOD );

  if ( tmp_line_buf == NULL )
  {
    width = *len / 3;

    DBG( 5, "buffer_line_offset: offset = %d, len = %lu, width = %d\n",
	line_offset, (u_long) *len, width );

    tmp_line_buf = malloc( *len );
    if ( tmp_line_buf == NULL )
    {
      DBG( 5, "couldn't allocate memory for temp line buffer\n" );
      return( SANE_STATUS_NO_MEM );
    }

    r_buf_lines = line_offset * 2;
    r_line_buf = malloc( r_buf_lines * sizeof( SANE_Byte * ));
    if ( r_line_buf == NULL )
    {
      DBG( 5, "couldn't allocate memory for red line buffer pointers\n" );
      return( SANE_STATUS_NO_MEM );
    }

    for ( count = 0; count < r_buf_lines; count++ )
    {
      r_line_buf[ count ] = malloc( width * sizeof( SANE_Byte ));
      if ( r_line_buf[ count ] == NULL )
      {
        DBG( 5, "couldn't allocate memory for red line buffer %d\n", count );
        return( SANE_STATUS_NO_MEM );
      }
    }

    g_buf_lines = line_offset;
    g_line_buf = malloc( g_buf_lines * sizeof( SANE_Byte * ));
    if ( g_line_buf == NULL )
    {
      DBG( 5, "couldn't allocate memory for green line buffer pointers\n" );
      return( SANE_STATUS_NO_MEM );
    }

    for ( count = 0; count < g_buf_lines; count++ )
    {
      g_line_buf[ count ] = malloc( width * sizeof( SANE_Byte ));
      if ( g_line_buf[ count ] == NULL )
      {
        DBG( 5, "couldn't allocate memory for green line buffer %d\n", count );
        return( SANE_STATUS_NO_MEM );
      }
    }

    DBG( 5, "buffer_line_offset: r lines = %d, g lines = %d\n",
	r_buf_lines, g_buf_lines );
  }

  cur_line++;

  if ( cur_line > r_buf_lines )
  {
    /* get the red line info from r_buf_lines ago */
    memcpy( tmp_line_buf, r_line_buf[ 0 ], width );

    /* get the green line info from g_buf_lines ago */
    memcpy( tmp_line_buf + width, g_line_buf[ 0 ], width );
  }

  /* move all the buffered red lines down (just move the ptrs for speed) */
  tmp_r_buf_ptr = r_line_buf[ 0 ];
  for ( count = 0; count < ( r_buf_lines - 1 ); count++ )
  {
    r_line_buf[ count ] = r_line_buf[ count + 1 ];
  }
  r_line_buf[ r_buf_lines - 1 ] = tmp_r_buf_ptr;

  /* insert the supplied red line data at the end of our FIFO */
  memcpy( r_line_buf[ r_buf_lines - 1 ], data, width );

  /* move all the buffered green lines down (just move the ptrs for speed) */
  tmp_g_buf_ptr = g_line_buf[ 0 ];
  for ( count = 0; count < ( g_buf_lines - 1 ); count++ )
  {
    g_line_buf[ count ] = g_line_buf[ count + 1 ];
  }
  g_line_buf[ g_buf_lines - 1 ] = tmp_g_buf_ptr;

  /* insert the supplied green line data at the end of our FIFO */
  memcpy( g_line_buf[ g_buf_lines - 1 ], data + width, width );

  if ( cur_line > r_buf_lines )
  {
    /* copy the red and green data in with the original blue */
    memcpy( data, tmp_line_buf, width * 2 );
  }
  else
  {
    /* if we are in the first r_buf_lines, then we don't return anything */
    *len = 0;
  }

  return( SANE_STATUS_GOOD );
}

/* CTP FIXME */
static SANE_Status
init_and_start_scan (SANE_Handle handle)
{
  ARTEC_Scanner *s = handle;
  char write_6[4096];
  char *data;

  artec_calibrate_white_shading( s );

  data = write_6 + 10;

  DBG( 20, "Scanning info:\n" );
  DBG( 20, "  X resolution: %5d (%d-%d)\n",
	s->val[OPT_RESOLUTION].w,
	s->hw->dpi_range.min, s->hw->dpi_range.max );
  DBG( 20, "  Y resolution: %5d (%d-%d)\n",
	s->val[OPT_RESOLUTION].w,
	s->hw->dpi_range.min, s->hw->dpi_range.max );
  DBG( 20, "  Width       : %5d (%d-%d)\n",
	s->params.pixels_per_line,
	s->hw->x_range.min, s->hw->x_range.max );
  DBG( 20, "  Height      : %5d (%d-%d)\n",
	s->params.lines,
	s->hw->y_range.min, s->hw->y_range.max );
  DBG( 20, "  Image Comp. : %s\n",
	s->val[OPT_MODE].s );
  DBG( 20, "  Line Offset : %lu\n",
       (u_long) s->line_offset );

  memset (write_6, 0, 62);
  write_6[0] = 0x24;
  write_6[8] = 52;
  write_6[9] = 0x00;

  /* beginning of set window data header */
  data[7] = 53 - 8;
  data[10] = s->val[OPT_RESOLUTION].w >> 8;  /* x resolution */
  data[11] = s->val[OPT_RESOLUTION].w;
  data[12] = s->val[OPT_RESOLUTION].w >> 8;  /* y resolution */
  data[13] = s->val[OPT_RESOLUTION].w;
  data[22] = s->params.pixels_per_line >> 24; /* width */
  data[23] = s->params.pixels_per_line >> 16;
  data[24] = s->params.pixels_per_line >> 8;
  data[25] = s->params.pixels_per_line;
  data[26] = ( s->params.lines + ( s->line_offset * 2 )) >> 24; /* height */
  data[27] = ( s->params.lines + ( s->line_offset * 2 )) >> 16;
  data[28] = ( s->params.lines + ( s->line_offset * 2 )) >> 8;
  data[29] = ( s->params.lines + ( s->line_offset * 2 ));
  data[31] = 0x80; /* threshold */
  data[32] = s->val[OPT_CONTRAST].w; /* contrast */

  if ( strcmp( s->val[OPT_MODE].s, "Lineart" ) == 0 )
  {
    data[33] = COMP_LINEART;
  }
  else if ( strcmp( s->val[OPT_MODE].s, "Halftone" ) == 0 )
  {
    data[33] = COMP_HALFTONE;
  }
  else if ( strcmp( s->val[OPT_MODE].s, "Gray" ) == 0 )
  {
    data[33] = COMP_GRAY;
  }
  else if ( strcmp( s->val[OPT_MODE].s, "Color" ) == 0 )
  {
    data[33] = COMP_COLOR;
  }

  data[34] = s->params.depth; /* bits per pixel */
  data[35] = 0x04; /* halftone pattern */

  /* set the window */
  sanei_scsi_cmd (s->fd, write_6, 62, 0, 0);

  memset( write_6, 0, 7 );
  write_6[0] = 0x1b;
  write_6[4] = 0x01;

  /* start the scan */
  return sanei_scsi_cmd (s->fd, write_6, 7, 0, 0);
}

static SANE_Status
write_string (int fd, const char *format, ...)
{
  va_list args;
  char buf[1024];

  va_start (args, format);
  vsprintf (buf, format, args);
  return write_data (fd, buf, strlen (buf));
}

static SANE_Status
inquire (int fd, int request_code, char request_letter,
	 char reply_letter, int *valp)
{
  u_char buf[128], letter;
  SANE_Status status;
  int code;
  size_t len;

  *valp = 0;

  /* CTP FIXME */
  return SANE_STATUS_UNSUPPORTED;

  sprintf ((char *) buf, "\033*s%d%c", request_code, request_letter);
  write_string (fd, (char *) buf);
  len = sizeof (buf);
/* CTP FIXME */
/*
  status = read_data (fd, buf, &len);
*/
  if (status != SANE_STATUS_GOOD)
    {
      DBG(1, "inquire: read_data failed (%s)\n", sane_strstatus (status));
      return status;
    }

  if (sscanf ((char *) buf, "\033*s%d%c%dV", &code, &letter, valp) != 3)
    {
      DBG(1, "inquire: returned string is malformed (%s)\n", buf);
      return SANE_STATUS_IO_ERROR;
    }
  if (code != request_code || letter != reply_letter)
    {
      DBG(1, "inquire: inquired %d/%c but received %d/%c\n",
	  request_code, reply_letter, code, letter);
      return SANE_STATUS_IO_ERROR;
    }
  return SANE_STATUS_GOOD;
}

static SANE_Status
attach (const char *devname, ARTEC_Device **devp)
{
  char result[INQ_LEN], *str, *t;
  int i, fd, artec_scanner;
  SANE_Status status;
  ARTEC_Device *dev;
  size_t size;
  struct
    {
      int code;
      char request_letter;
      char reply_letter;
      int default_value;
      int value;
    }
  inq_params[] =
    {
      {10489, 'L', 'k',     0, 0},	/* min_x */
      {10489, 'H', 'g',  2550, 0},	/* max_x */
      {10490, 'L', 'k',     0, 0},	/* min_y */
      {10490, 'H', 'g',  3300, 0},	/* max_y */
      {10323, 'L', 'k',    12, 0},	/* min_dpi */
      {10323, 'H', 'g',   300, 0},	/* max_dpi */
      {10316, 'L', 'k',     0, 0},	/* min_contrast */
      {10316, 'H', 'g',  0xff, 0},	/* max_contrast */
    };

  for (dev = first_dev; dev; dev = dev->next)
    if (strcmp (dev->sane.name, devname) == 0)
      {
	if (devp)
	  *devp = dev;
	return SANE_STATUS_GOOD;
      }

  DBG(3, "attach: opening %s\n", devname);
  status = sanei_scsi_open (devname, &fd, 0, 0);
  if (status != SANE_STATUS_GOOD)
    {
      DBG(1, "attach: open failed (%s)\n", sane_strstatus (status));
      return SANE_STATUS_INVAL;
    }

  DBG(3, "attach: sending INQUIRY\n");
  size = sizeof (result);
  status = sanei_scsi_cmd (fd, inquiry, sizeof (inquiry), result, &size);
  if (status != SANE_STATUS_GOOD || size < 16)
    {
      DBG(1, "attach: inquiry failed (%s)\n", sane_strstatus (status));
      sanei_scsi_close (fd);
      return status;
    }

  DBG(3, "attach: sending TEST_UNIT_READY\n");
  status = sanei_scsi_cmd (fd, test_unit_ready, sizeof (test_unit_ready),
			   0, 0);
  if (status != SANE_STATUS_GOOD)
    {
      DBG(1, "attach: test unit ready failed (%s)\n",
	  sane_strstatus (status));
      sanei_scsi_close (fd);
      return status;
    }

/* CTP FIXME */
/* have artec_get_capability() function */
  for (i = 0; i < NELEMS (inq_params); ++i)
    if (inquire (fd, inq_params[i].code, inq_params[i].request_letter,
		 inq_params[i].reply_letter, &inq_params[i].value)
	!= SANE_STATUS_GOOD)
    {
      inq_params[i].value = inq_params[i].default_value;
      DBG(1, "using default value of %d\n", inq_params[i].value);
    }

  sanei_scsi_close (fd);

  /* are we really dealing with a scanner by ULTIMA/ARTEC? */
  artec_scanner = ((result[0] == 0x03 || result[0] == 0x06)
		&& strncmp (result + 8, "ULTIMA", 6) == 0);
  if (!artec_scanner)
    {
      DBG(1, "attach: device doesn't look like a ULTIMA scanner\n");
      return SANE_STATUS_INVAL;
    }

  dev = malloc (sizeof (*dev));
  if (!dev)
    return SANE_STATUS_NO_MEM;

  memset (dev, 0, sizeof (*dev));
  dev->x_range.min = inq_params[0].value;
  dev->x_range.max = inq_params[1].value;
  dev->x_range.quant = 1;

  dev->y_range.min = inq_params[2].value;
  dev->y_range.max = inq_params[3].value;
  dev->y_range.quant = 1;

  dev->dpi_range.min = inq_params[4].value;
  dev->dpi_range.max = inq_params[5].value;
  dev->dpi_range.quant = 1;

  dev->contrast_range.min   = inq_params[6].value;
  dev->contrast_range.max   = inq_params[7].value;
  dev->contrast_range.quant = 1;

  str = malloc (17);
  memcpy (str, result + 16, 16);
  str[16] = ' ';
  t = strchr (str, ' ');
  *t = '\0';
  dev->sane.model  = str;

  str = malloc (9);
  memcpy( str, result + 8, 8 );
  str[8] = ' ';
  t = strchr( str, ' ' );
  *t = '\0';
  dev->sane.vendor = str;

  dev->sane.name   = strdup (devname);
  dev->sane.type   = "flatbed scanner";

  DBG(1, "attach: found %s model %s, x=%g-%gmm, y=%g-%gmm, "
      "resolution=%d-%ddpi\n",
      dev->sane.vendor, dev->sane.model,
      SANE_UNFIX (dev->x_range.min), SANE_UNFIX (dev->x_range.max),
      SANE_UNFIX (dev->y_range.min), SANE_UNFIX (dev->y_range.max),
      dev->dpi_range.min, dev->dpi_range.max);

  ++num_devices;
  dev->next = first_dev;
  first_dev = dev;

  if (devp)
    *devp = dev;
  return SANE_STATUS_GOOD;
}

static SANE_Status
init_options (ARTEC_Scanner *s)
{
  int i;

  memset (s->opt, 0, sizeof (s->opt));
  memset (s->val, 0, sizeof (s->val));

  for (i = 0; i < NUM_OPTIONS; ++i)
    {
      s->opt[i].size = sizeof (SANE_Word);
      s->opt[i].cap = SANE_CAP_SOFT_SELECT | SANE_CAP_SOFT_DETECT;
    }

  s->opt[OPT_NUM_OPTS].title = SANE_TITLE_NUM_OPTIONS;
  s->opt[OPT_NUM_OPTS].desc = SANE_DESC_NUM_OPTIONS;
  s->opt[OPT_NUM_OPTS].cap = SANE_CAP_SOFT_DETECT;
  s->val[OPT_NUM_OPTS].w = NUM_OPTIONS;

  /* "Mode" group: */

  s->opt[OPT_MODE_GROUP].title = "Scan Mode";
  s->opt[OPT_MODE_GROUP].desc = "";
  s->opt[OPT_MODE_GROUP].type = SANE_TYPE_GROUP;
  s->opt[OPT_MODE_GROUP].cap = 0;
  s->opt[OPT_MODE_GROUP].constraint_type = SANE_CONSTRAINT_NONE;

  /* scan mode */
  s->opt[OPT_MODE].name = SANE_NAME_SCAN_MODE;
  s->opt[OPT_MODE].title = SANE_TITLE_SCAN_MODE;
  s->opt[OPT_MODE].desc = SANE_DESC_SCAN_MODE;
  s->opt[OPT_MODE].type = SANE_TYPE_STRING;
  s->opt[OPT_MODE].size = max_string_size (mode_list);
  s->opt[OPT_MODE].constraint_type = SANE_CONSTRAINT_STRING_LIST;
  s->opt[OPT_MODE].constraint.string_list = mode_list;
  s->val[OPT_MODE].s = strdup (mode_list[2]);

  /* resolution */
  s->opt[OPT_RESOLUTION].name = SANE_NAME_SCAN_RESOLUTION;
  s->opt[OPT_RESOLUTION].title = SANE_TITLE_SCAN_RESOLUTION;
  s->opt[OPT_RESOLUTION].desc = SANE_DESC_SCAN_RESOLUTION;
  s->opt[OPT_RESOLUTION].type = SANE_TYPE_INT;
  s->opt[OPT_RESOLUTION].unit = SANE_UNIT_DPI;
  s->opt[OPT_RESOLUTION].constraint_type = SANE_CONSTRAINT_RANGE;
  s->opt[OPT_RESOLUTION].constraint.range = &s->hw->dpi_range;
  s->val[OPT_RESOLUTION].w = 300;

  /* "Geometry" group: */

  s->opt[OPT_GEOMETRY_GROUP].title = "Geometry";
  s->opt[OPT_GEOMETRY_GROUP].desc = "";
  s->opt[OPT_GEOMETRY_GROUP].type = SANE_TYPE_GROUP;
  s->opt[OPT_GEOMETRY_GROUP].cap = SANE_CAP_ADVANCED;
  s->opt[OPT_GEOMETRY_GROUP].constraint_type = SANE_CONSTRAINT_NONE;

  /* top-left x */
  s->opt[OPT_TL_X].name = SANE_NAME_SCAN_TL_X;
  s->opt[OPT_TL_X].title = SANE_TITLE_SCAN_TL_X;
  s->opt[OPT_TL_X].desc = SANE_DESC_SCAN_TL_X;
  s->opt[OPT_TL_X].type = SANE_TYPE_INT;
  s->opt[OPT_TL_X].unit = SANE_UNIT_PIXEL;
  s->opt[OPT_TL_X].constraint_type = SANE_CONSTRAINT_RANGE;
  s->opt[OPT_TL_X].constraint.range = &s->hw->x_range;
  s->val[OPT_TL_X].w = 0;

  /* top-left y */
  s->opt[OPT_TL_Y].name = SANE_NAME_SCAN_TL_Y;
  s->opt[OPT_TL_Y].title = SANE_TITLE_SCAN_TL_Y;
  s->opt[OPT_TL_Y].desc = SANE_DESC_SCAN_TL_Y;
  s->opt[OPT_TL_Y].type = SANE_TYPE_INT;
  s->opt[OPT_TL_Y].unit = SANE_UNIT_PIXEL;
  s->opt[OPT_TL_Y].constraint_type = SANE_CONSTRAINT_RANGE;
  s->opt[OPT_TL_Y].constraint.range = &s->hw->y_range;
  s->val[OPT_TL_Y].w = 0;

  /* bottom-right x */
  s->opt[OPT_BR_X].name = SANE_NAME_SCAN_BR_X;
  s->opt[OPT_BR_X].title = SANE_TITLE_SCAN_BR_X;
  s->opt[OPT_BR_X].desc = SANE_DESC_SCAN_BR_X;
  s->opt[OPT_BR_X].type = SANE_TYPE_INT;
  s->opt[OPT_BR_X].unit = SANE_UNIT_PIXEL;
  s->opt[OPT_BR_X].constraint_type = SANE_CONSTRAINT_RANGE;
  s->opt[OPT_BR_X].constraint.range = &s->hw->x_range;
  s->val[OPT_BR_X].w = 300;

  /* bottom-right y */
  s->opt[OPT_BR_Y].name = SANE_NAME_SCAN_BR_Y;
  s->opt[OPT_BR_Y].title = SANE_TITLE_SCAN_BR_Y;
  s->opt[OPT_BR_Y].desc = SANE_DESC_SCAN_BR_Y;
  s->opt[OPT_BR_Y].type = SANE_TYPE_INT;
  s->opt[OPT_BR_Y].unit = SANE_UNIT_PIXEL;
  s->opt[OPT_BR_Y].constraint_type = SANE_CONSTRAINT_RANGE;
  s->opt[OPT_BR_Y].constraint.range = &s->hw->y_range;
  s->val[OPT_BR_Y].w = 300;

  /* contrast */
  s->opt[OPT_CONTRAST].name = SANE_NAME_CONTRAST;
  s->opt[OPT_CONTRAST].title = SANE_TITLE_CONTRAST;
  s->opt[OPT_CONTRAST].desc = SANE_DESC_CONTRAST;
  s->opt[OPT_CONTRAST].type = SANE_TYPE_INT;
  s->opt[OPT_CONTRAST].unit = SANE_UNIT_NONE;
  s->opt[OPT_CONTRAST].constraint_type = SANE_CONSTRAINT_RANGE;
  s->opt[OPT_CONTRAST].constraint.range = &s->hw->contrast_range;
  s->val[OPT_CONTRAST].w = 0x80;

  return SANE_STATUS_GOOD;
}

static SANE_Status
do_cancel (ARTEC_Scanner *s)
{
  s->scanning = SANE_FALSE;

  if (s->fd >= 0)
    {
      sanei_scsi_close (s->fd);
      s->fd = -1;
    }
  return SANE_STATUS_CANCELLED;
}

SANE_Status
sane_init (SANE_Int *version_code, SANE_Auth_Callback authorize)
{
  char dev_name[PATH_MAX];
  size_t len;
  FILE *fp;

  DBG_INIT();

  if (version_code)
    *version_code = SANE_VERSION_CODE (V_MAJOR, V_MINOR, 0);

  fp = sanei_config_open (ARTEC_CONFIG_FILE);
  if (!fp)
    {
      /* default to /dev/scanner instead of insisting on config file */
      attach ("/dev/scanner", 0);
      return SANE_STATUS_GOOD;
    }

  while (fgets (dev_name, sizeof (dev_name), fp))
    {
      if (dev_name[0] == '#')		/* ignore line comments */
	continue;
      len = strlen (dev_name);
      if (dev_name[len - 1] == '\n')
	dev_name[--len] = '\0';

      if (!len)
	continue;			/* ignore empty lines */

      attach (dev_name, 0);
    }
  fclose (fp);
  return SANE_STATUS_GOOD;
}

void
sane_exit (void)
{
  ARTEC_Device *dev, *next;

  for (dev = first_dev; dev; dev = next)
    {
      next = dev->next;
      free ((void *) dev->sane.name);
      free ((void *) dev->sane.model);
      free (dev);
    }
}

SANE_Status
sane_get_devices (const SANE_Device ***device_list, SANE_Bool local_only)
{
  static const SANE_Device **devlist = 0;
  ARTEC_Device *dev;
  int i;

  if (devlist)
    free (devlist);

  devlist = malloc ((num_devices + 1) * sizeof (devlist[0]));
  if (!devlist)
    return SANE_STATUS_NO_MEM;

  i = 0;
  for (dev = first_dev; i < num_devices; dev = dev->next)
    devlist[i++] = &dev->sane;
  devlist[i++] = 0;

  *device_list = devlist;
  return SANE_STATUS_GOOD;
}

SANE_Status
sane_open (SANE_String_Const devicename, SANE_Handle *handle)
{
  SANE_Status status;
  ARTEC_Device *dev;
  ARTEC_Scanner *s;

  if (devicename[0])
    {
      for (dev = first_dev; dev; dev = dev->next)
	if (strcmp (dev->sane.name, devicename) == 0)
	  break;

      if (!dev)
	{
	  status = attach (devicename, &dev);
	  if (status != SANE_STATUS_GOOD)
	    return status;
	}
    }
  else
    /* empty devicname -> use first device */
    dev = first_dev;

  if (!dev)
    return SANE_STATUS_INVAL;

  s = malloc (sizeof (*s));
  if (!s)
    return SANE_STATUS_NO_MEM;
  memset (s, 0, sizeof (*s));
  s->fd = -1;
  s->hw = dev;

  init_options (s);

  /* insert newly opened handle into list of open handles: */
  s->next = first_handle;
  first_handle = s;

  *handle = s;
  return SANE_STATUS_GOOD;
}

void
sane_close (SANE_Handle handle)
{
  ARTEC_Scanner *prev, *s;

  /* remove handle from list of open handles: */
  prev = 0;
  for (s = first_handle; s; s = s->next)
    {
      if (s == handle)
	break;
      prev = s;
    }
  if (!s)
    {
      DBG(1, "close: invalid handle %p\n", handle);
      return;		/* oops, not a handle we know about */
    }

  if (s->scanning)
    do_cancel (handle);

  if (prev)
    prev->next = s->next;
  else
    first_handle = s;

  free (handle);
}

const SANE_Option_Descriptor *
sane_get_option_descriptor (SANE_Handle handle, SANE_Int option)
{
  ARTEC_Scanner *s = handle;

  if ((unsigned) option >= NUM_OPTIONS)
    return 0;
  return s->opt + option;
}

SANE_Status
sane_control_option (SANE_Handle handle, SANE_Int option,
		     SANE_Action action, void *val, SANE_Int *info)
{
  ARTEC_Scanner *s = handle;
  SANE_Status status;
  SANE_Word cap, temp_word;

  if (info)
    *info = 0;

  if (s->scanning)
    return SANE_STATUS_DEVICE_BUSY;

  if (option >= NUM_OPTIONS)
    return SANE_STATUS_INVAL;

  cap = s->opt[option].cap;

  if (!SANE_OPTION_IS_ACTIVE (cap))
    return SANE_STATUS_INVAL;

  if (action == SANE_ACTION_GET_VALUE)
    {
      switch (option)
	{
	  /* word options: */
	case OPT_RESOLUTION:
	case OPT_TL_X:
	case OPT_TL_Y:
	case OPT_BR_X:
	case OPT_BR_Y:
	case OPT_NUM_OPTS:
	case OPT_CONTRAST:
	  *(SANE_Word *) val = s->val[option].w;
	  return SANE_STATUS_GOOD;

	  /* string options: */
	case OPT_MODE:
	  strcpy (val, s->val[option].s);
	  return SANE_STATUS_GOOD;
	}
    }
  else if (action == SANE_ACTION_SET_VALUE)
    {
      if (!SANE_OPTION_IS_SETTABLE (cap))
	return SANE_STATUS_INVAL;

      status = sanei_constrain_value (s->opt + option, val, info);
      if (status != SANE_STATUS_GOOD)
	return status;

      switch (option)
	{
	  /* (mostly) side-effect-free word options: */
	case OPT_BR_X:
	case OPT_BR_Y:
	  temp_word = *(SANE_Word *) val;
          *(SANE_Word *)val = temp_word + 1;
	case OPT_RESOLUTION:
	case OPT_TL_X:
	case OPT_TL_Y:
	  if (info && s->val[option].w != *(SANE_Word *) val)
	    *info |= SANE_INFO_RELOAD_PARAMS;
	  /* fall through */
	case OPT_NUM_OPTS:
	case OPT_CONTRAST:
	  s->val[option].w = *(SANE_Word *) val;
	  return SANE_STATUS_GOOD;

	case OPT_MODE:
	  if (s->val[option].s)
	    free (s->val[option].s);
	  s->val[option].s = strdup (val);
	  return SANE_STATUS_GOOD;
	}
    }
  return SANE_STATUS_INVAL;
}

SANE_Status
sane_get_parameters (SANE_Handle handle, SANE_Parameters *params)
{
  ARTEC_Scanner *s = handle;

  if (!s->scanning)
    {
      double width, height, dpi;
      const char *mode;

      memset (&s->params, 0, sizeof (s->params));

      width  = s->val[OPT_BR_X].w - s->val[OPT_TL_X].w;
      height = s->val[OPT_BR_Y].w - s->val[OPT_TL_Y].w;
      dpi    = s->val[OPT_RESOLUTION].w;

      /* make best-effort guess at what parameters will look like once
         scanning starts.  */
      if (dpi > 0.0 && width > 0.0 && height > 0.0)
	{
	  s->params.pixels_per_line = width;
	  s->params.lines = height;

          DBG( 3, "ppl: %d, l: %d\n",
            s->params.pixels_per_line, s->params.lines );
	}

      mode = s->val[OPT_MODE].s;
      if (strcmp (mode, "Lineart") == 0 || strcmp (mode, "Halftone") == 0)
	{
	  s->params.format = SANE_FRAME_GRAY;
	  s->params.bytes_per_line = (s->params.pixels_per_line + 7) / 8;
	  s->params.depth = 1;
	  s->line_offset = 0;
	}
      else if (strcmp (mode, "Gray") == 0)
	{
	  s->params.format = SANE_FRAME_GRAY;
	  s->params.bytes_per_line = s->params.pixels_per_line;
	  s->params.depth = 8;
	  s->line_offset = 0;
	}
      else
	{
	  /* XXX assume single pass--are there any one-pass ARTECs? */
	  s->params.format = SANE_FRAME_RGB;
	  s->params.bytes_per_line = 3 * s->params.pixels_per_line;
	  s->params.depth = 8;

          switch( s->val[OPT_RESOLUTION].w )
          {
            case 600: s->line_offset = 16;
                      break;
            case 300: s->line_offset = 8;
                      break;
            case 200: s->line_offset = 5;
                      break;
            case 100: s->line_offset = 2;
                      break;
            case  50: s->line_offset = 1;
                      break;
            default : s->line_offset =
                        (int)( 8 * ( s->val[OPT_RESOLUTION].w / 300.0 ));
                      break;
          }
	}
      s->params.last_frame = SANE_TRUE;
    }
  if (params)
    *params = s->params;
  return SANE_STATUS_GOOD;
}

SANE_Status
sane_start (SANE_Handle handle)
{
  ARTEC_Scanner *s = handle;
  SANE_Status status;

  /* First make sure we have a current parameter set.  Some of the
     parameters will be overwritten below, but that's OK.  */
  status = sane_get_parameters (s, 0);
  if (status != SANE_STATUS_GOOD)
    return status;

  status = sanei_scsi_open (s->hw->sane.name, &s->fd, 0, 0);
  if (status != SANE_STATUS_GOOD)
    {
      DBG(1, "open of %s failed: %s\n",
	  s->hw->sane.name, sane_strstatus (status));
      return status;
    }

  s->bytes_to_read = s->params.bytes_per_line * s->params.lines;

  DBG(1, "%d pixels per line, %d bytes, %d lines high, dpi=%d, btr = %lu\n",
      s->params.pixels_per_line, s->params.bytes_per_line, s->params.lines,
      s->val[OPT_RESOLUTION].w, (u_long) s->bytes_to_read );

  init_and_start_scan( s );

  s->scanning = SANE_TRUE;
  return SANE_STATUS_GOOD;
}

SANE_Status
sane_read (SANE_Handle handle, SANE_Byte *buf, SANE_Int max_len, SANE_Int *len)
{
  ARTEC_Scanner *s = handle;
  SANE_Status status;
  size_t nread;
  static int first_time_in = 1;

  *len = 0;

  if (s->bytes_to_read == 0)
    {
      do_cancel (s);
      return SANE_STATUS_EOF;
    }

  if (!s->scanning)
    return do_cancel (s);

/* CTP FIXME */
  if ( first_time_in == 1 )
  {
    first_time_in = 0;

    while( artec_get_status( s->fd ) == 0 )
    {
      DBG( 3, "sleeping till data available\n" );
/*
      sleep( 1 );
*/
    }
  }

  if ( artec_get_status( s->fd ) == 0 )
	return SANE_STATUS_GOOD;

  nread = max_len;
  nread = s->params.bytes_per_line;

  if (nread > s->bytes_to_read)
    nread = s->bytes_to_read;

  DBG( 5, "sane_read: nread = %lu\n", (u_long) nread );

  status = read_data (s->fd, DATA_IMAGE, buf, &nread);
  if (status != SANE_STATUS_GOOD)
    {
      do_cancel (s);
      return SANE_STATUS_IO_ERROR;
    }

  if ( strcmp( s->val[OPT_MODE].s, "Color" ) == 0 )
  {
    artec_buffer_line_offset( s->line_offset, buf, &nread );

    if ( nread > 0 )
    {
	  artec_line_rgb_to_byte_rgb( buf, s->params.pixels_per_line );
    }
  }

  *len = nread;
  s->bytes_to_read -= nread;

  return SANE_STATUS_GOOD;
}

void
sane_cancel (SANE_Handle handle)
{
  ARTEC_Scanner *s = handle;

  s->scanning = SANE_FALSE;
}

SANE_Status
sane_set_io_mode (SANE_Handle handle, SANE_Bool non_blocking)
{
  return SANE_STATUS_UNSUPPORTED;
}

SANE_Status
sane_get_select_fd (SANE_Handle handle, SANE_Int *fd)
{
  return SANE_STATUS_UNSUPPORTED;
}
