// This file is part of Moonlight Creator
//   Copyright (C) 1996-1998  Stephane Rehel
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
//
// You should have received a copy of the GNU Library General Public
// License along with this library; if not, write to the Free
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/*
   MLPolygon.C

   November 10 1996
   Stephane Rehel
*/

#include <stdio.h>
#include <assert.h>

#include "MLVertex.h"
#include "MLEdge.h"
#include "MLPolygon.h"
#include "MeshRep.h"
#include "MLRay.h"

/////////////////////////////////////////////////////////////////////////////

int MLPolygon::getSizeOf() const
{
  return sizeof(*this) + nVertices * sizeof(elements[0])
                       + nTessTriangles * sizeof(tesselation[0]);
}

/////////////////////////////////////////////////////////////////////////////

void MLPolygon::alloc( short _nVertices )
{
  delete elements;

  nVertices= max( 1, _nVertices );

  elements= new MLPolygon::Element [ nVertices ];

  for( int i= 0; i < nVertices; ++i )
    {
    Element& e= elements[i];
    e.vertex= 0;
    e.edge= 0;
    e.svertex= 0;
    }

  flags |= Flag(ALLOCATED);
}

/////////////////////////////////////////////////////////////////////////////

void MLPolygon::allocTriangle( int v1, int v2, int v3 )
{
  nVertices= 3;

  elements= new MLPolygon::Element [ 3 ];

  elements[0].edge=
  elements[1].edge=
  elements[2].edge= 0;

  elements[0].svertex=
  elements[1].svertex=
  elements[2].svertex= 0;

  elements[0].vertex= v1;
  elements[1].vertex= v2;
  elements[2].vertex= v3;

  flags |= Flag(ALLOCATED);
}

/////////////////////////////////////////////////////////////////////////////

void MLPolygon::allocQuad( int v1, int v2, int v3, int v4 )
{
  nVertices= 4;

  elements= new MLPolygon::Element [ 4 ];

  elements[0].edge=
  elements[1].edge=
  elements[2].edge=
  elements[3].edge= 0;

  elements[0].svertex=
  elements[1].svertex=
  elements[2].svertex=
  elements[3].svertex= 0;

  elements[0].vertex= v1;
  elements[1].vertex= v2;
  elements[2].vertex= v3;
  elements[3].vertex= v4;

  flags |= Flag(ALLOCATED);
}

/////////////////////////////////////////////////////////////////////////////

void MLPolygon::updateNormalArea( IBOOL recurse /* = IFALSE */ )
{
  normal= Vector(0,0,0);
  for( int j= 0; j < nVertices; ++j )
    {
    register int vi1= elements[j].vertex;
    register int vi2= elements[ (j+1) % nVertices ].vertex;

    register Point& v1= mesh->vertices[vi1].point;
    register Point& v2= mesh->vertices[vi2].point;
    normal[0]+= (v1[1]-v2[1]) * (v2[2]+v1[2]);
    normal[1]+= (v1[2]-v2[2]) * (v2[0]+v1[0]);
    normal[2]+= (v1[0]-v2[0]) * (v2[1]+v1[1]);
    }

  double norm= normal.norm();
  area= norm * 0.5;

  if( norm < 1e-20 )
    {
    ray_info.X= 0;
    ray_info.Y= 1;
    ray_info.x1= ray_info.y1= ray_info.x2= ray_info.y2= 0.;
    return;
    }

  normal /= norm;

  if( nVertices != 3 )
    {
    ray_info.X= 0;
    ray_info.Y= 1;
    ray_info.x1= ray_info.y1= ray_info.x2= ray_info.y2= 0.;
    }
   else
    {
#define vertex(i) mesh->vertices[ elements[i].vertex ].point
    char& X= ray_info.X;
    char& Y= ray_info.Y;
    if( fabs(normal.x()) > fabs(normal.y()) )
      {
      if( fabs(normal.x()) > fabs(normal.z()) )
        { X= 1; Y= 2; } // max = 0
       else
        { X= 0; Y= 1; } // max = 2
      }
     else
      {
      if( fabs(normal.y()) > fabs(normal.z()) )
        { X= 2; Y= 0; } // max = 1
       else
        { X= 0; Y= 1; } // max = 2
      }
/*
    if( normal[Z] < 0. )
      swap(X,Y);
*/
    Point& p0= vertex(0);
    Point& p1= vertex(1);
    Point& p2= vertex(2);

    ray_info.x1= p1[X] - p0[X];
    ray_info.y1=-p1[Y] + p0[Y];
    ray_info.x2=-p2[X] + p0[X];
    ray_info.y2= p2[Y] - p0[Y];
    float delta= ray_info.x1*ray_info.y2 - ray_info.x2*ray_info.y1;
    if( fabs(delta) > 1e-20 )
      {
      ray_info.x1 /= delta;
      ray_info.y1 /= delta;
      ray_info.x2 /= delta;
      ray_info.y2 /= delta;
      }
#undef vertex
    }

  if( recurse && tesselation != 0 )
    {
    MLPolygonArray& polygons_array= mesh->polygons;
    for( int i= 0; i < nTessTriangles; ++i )
      {
      int ti= tesselation[i];
      if( ti == 0 )
        continue;
      MLPolygon& p= polygons_array[ti];
      p.updateNormalArea(ITRUE);
      }
    }
}

/////////////////////////////////////////////////////////////////////////////

// ray.direction is to be normalized
IBOOL MLPolygon::intersect( MLRay& ray )
{
  if( nVertices > 3 )
    {
    if( tesselation == 0 )
      return IFALSE;

    for( int i= 0; i < nTessTriangles; ++i )
      if( mesh->polygons[ tesselation[i] ].intersect(ray) )
        return ITRUE;

    return IFALSE;
    }

  register double ko= ( ray.direction | normal );
  if( fabs(ko) < 1e-10 )
    return IFALSE;

  if( ray.skip_reversed )
    {
    if( ray.lightRay )
      {
      if( ko <= 0. )
        return IFALSE;
      }
     else
      {
      if( ko >= 0. )
        return IFALSE;
      }
    }

  MLVertexArray& global_vertices= mesh->vertices;
#define vertex(i) global_vertices[ elements[i].vertex ].point
  Point& p0= vertex(0);
  register double k= ( (p0 - ray.origin) | normal ) / ko;
  ray.t= k;
  if( k < ray.min_t || k > ray.max_t )
    return IFALSE;

  Point point= ray.origin + k * ray.direction;

  register float x= point[ray_info.X] - p0[ray_info.X];
  register float y= point[ray_info.Y] - p0[ray_info.Y];

  register float alpha= x * ray_info.y2 + y * ray_info.x2;
  if( alpha < 0. || alpha > 1. )
    return IFALSE;
  register float beta=  y * ray_info.x1 + x * ray_info.y1;
  if( beta < 0. || beta > 1. )
    return IFALSE;
  register float gamma= 1. - alpha - beta;
  if( gamma < 0. || gamma > 1. )
    return IFALSE;

  ray.I= point;
  ray.alpha= alpha;
  ray.beta= beta;
  ray.gamma= gamma;

  ray.polygon= this;
  ray.polygon_k_parameter= ko;

  return ITRUE;
#undef vertex
}

/////////////////////////////////////////////////////////////////////////////

void MLPolygon::gob( const MLPolygon& p )
{
  mesh= p.mesh;
  index= p.index;

  nVertices= p.nVertices;
  elements= new MLPolygon::Element [ nVertices ];

  memcpy( (void*) elements,
          (const void*) p.elements,
          sizeof(MLPolygon::Element) * nVertices );

  nTessTriangles= p.nTessTriangles;
  if( p.tesselation == 0 )
    tesselation= 0;
   else
    {
    tesselation= new int [ nTessTriangles ];
    memcpy( (void*) tesselation,
            (const void*) p.tesselation,
            sizeof(tesselation[0]) * nTessTriangles );
    }
  fatherPolygon= p.fatherPolygon;

  normal= p.normal;
  area= p.area;
  material= p.material;
  flags= p.flags;

  ray_info.x1= p.ray_info.x1;
  ray_info.y1= p.ray_info.y1;
  ray_info.x2= p.ray_info.x2;
  ray_info.y2= p.ray_info.y2;
  ray_info.X= p.ray_info.X;
  ray_info.Y= p.ray_info.Y;
}

/////////////////////////////////////////////////////////////////////////////

void MLPolygon::unallocate()
{
  delete elements; elements= 0;
  nVertices= 0;
  delete tesselation; tesselation= 0;
  nTessTriangles= 0;

  flags &= ~Flag(ALLOCATED);
}

/////////////////////////////////////////////////////////////////////////////

Point MLPolygon::getCenter() const
{
  Point center(0,0,0);
  if( nVertices == 0 )
    return center;

  for( int i= 0; i < nVertices; ++i )
    {
    MLVertex& v= mesh->vertices[ elements[i].vertex ];
    center+= v.point;
    }

  return center / double(nVertices);
}

/////////////////////////////////////////////////////////////////////////////

MLPolygon& MLPolygon::getFatherPolygon()
{
  MLPolygon* p= this;
  for(;;)
    {
    if( p->fatherPolygon == 0 )
      return *p;
    assert( p->fatherPolygon != index );
    p= &(mesh->polygons[ p->fatherPolygon ]);
    }
}

/////////////////////////////////////////////////////////////////////////////

