// This file is part of Moonlight Creator
//   Copyright (C) 1996-1998  Stephane Rehel
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
//
// You should have received a copy of the GNU Library General Public
// License along with this library; if not, write to the Free
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/*
  MLDSavePackaged.C

  Stephane Rehel

  December 29 1997
*/

#include <string.h>
#include <stdlib.h>
#include <errno.h>
#include "GL/opengl.h"

#include "tools/Path.h"
#include "tools/Filename.h"

#include "graphics/mlgl.h"
#include "graphics/MLPicture.h"

#include "koala/VWidget.h"
#include "kw/KWDialog.h"
#include "kw/KWFileDialogManager.h"

#include "MLDSavePackaged.h"
#include "SavePackagedDlg.h"

#include "scene/file/MLDatabase.h"

#include "model/ModelModule.h"
#include "model/ModelFlags.h"
#include "model/file/ModelModuleIO.h"
#include "model/file/MLPackagedIO.h"

/////////////////////////////////////////////////////////////////////////////

static char* current_path= 0;
//static const char* errorString= "*ERROR*";

/////////////////////////////////////////////////////////////////////////////

MLDSavePackaged::MLDSavePackaged( ModelModule* _model ):
  MLGlobalDialog(_model)
{
  model= _model;
  snapshot= 0;
  manager= new KWFileDialogManager;
}

/////////////////////////////////////////////////////////////////////////////

MLDSavePackaged::~MLDSavePackaged()
{
  delete manager;
  manager= 0;

  delete snapshot;
  snapshot= 0;
}

/////////////////////////////////////////////////////////////////////////////

// return the abstract object read
// return 0 if cancelled or error
IBOOL MLDSavePackaged::run()
{
  delete snapshot;
  snapshot= 0;

  if( ! MLGlobalDialog::create(SavePackagedDlg::data) )
    return IFALSE;

  manager->init( kwDialog,
                 SavePackagedDlg::directories,
                 SavePackagedDlg::files,
                 SavePackagedDlg::location,
                 SavePackagedDlg::selection,
                 SavePackagedDlg::ok,
                 SavePackagedDlg::cancel );

  manager->setPattern( OString("*") + MLDatabase::packagedExtension );
  if( current_path == 0 )
    current_path= strdup("~");
  manager->setPath(current_path);
  manager->updateDirectories();

  kwDialog->setChecked( SavePackagedDlg::snapshot,
                        model->getFlags()->writeSnapshot );

  kwDialog->focus( SavePackagedDlg::selection );

  int outcome= kwDialog->run();

  OString filename= manager->getFilename();
  IBOOL include_snapshot= kwDialog->getChecked(SavePackagedDlg::snapshot);

  free( (void*) current_path );
  current_path= strdup( kwDialog->getText(SavePackagedDlg::location).get() );

  MLGlobalDialog::close();

  if( outcome == KWDialog::CANCEL )
    return IFALSE;

  if( filename.length() == 0 )
    return IFALSE;

  return save(filename,include_snapshot);
}

/////////////////////////////////////////////////////////////////////////////

IBOOL MLDSavePackaged::save( const OString& _filename,
                             IBOOL include_snapshot )
{
  if( _filename.length() == 0 )
    return IFALSE;

  OString filename= _filename;
  if( Filename(filename).getExtension().length() == 0 )
    filename += OString(MLDatabase::packagedExtension);

  model->stopLightingProcess();

  model->pushWaitCursor();

  model->flags->writeSnapshot= include_snapshot;

  MLPackagedIO p_io(model);

  IBOOL outcome= p_io.write(filename);

  model->popWaitCursor();

  if( ! outcome )
    {
    KWError( "Error",
               OString("Unable to save:\n")
             + filename + OString("\n")
             + OString(strerror(errno)) );
    return IFALSE;
    }

  return ITRUE;
}

/////////////////////////////////////////////////////////////////////////////

IBOOL MLDSavePackaged::kwCallback( KWEvent& event )
{
  if( manager->handleKWEvent(event) )
    return ITRUE;

  return MLGlobalDialog::kwCallback(event);
}

/////////////////////////////////////////////////////////////////////////////
