// This file is part of Moonlight Creator
//   Copyright (C) 1996-1998  Stephane Rehel
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
//
// You should have received a copy of the GNU Library General Public
// License along with this library; if not, write to the Free
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/*
  VTextList.C

  Stephane Rehel
  Creation: SR, July 23 1996
*/

#include "graphics/mlgl.h"
#include "graphics/MLFont.h"

#include "VWindow.h"
#include "VTextList.h"
#include "VOptions.h"

/////////////////////////////////////////////////////////////////////////////

VTextList::VTextList( VWindow* parent ): VList(parent)
{
  font= vOptions->list.font;
  margin= 3;
}

/////////////////////////////////////////////////////////////////////////////

VTextList::~VTextList()
{}

/////////////////////////////////////////////////////////////////////////////

void VTextList::setFixedFont()
{
  font= vOptions->fixedFont;
}

/////////////////////////////////////////////////////////////////////////////

void VTextList::create( int x, int y,
                        int width, int height, IBOOL scrolling )
{
  VList::create( x, y, width, height, scrolling, 1);//font->getHeight()+1 );
}

/////////////////////////////////////////////////////////////////////////////

// protected:
void VTextList::handleVListItem( VList*, VList::Item* i,
                                 int yBase, IBOOL selected )
{
  if( i == 0 )
    return;
  if( i->data == 0 )
    return;

  if( selected )
    vOptions->gray3();
   else
    vOptions->gray2();
  mlRect( 0, yBase, i->getWidth()-1, yBase+i->getHeight()-1 );

  if( ! isAvailable() )
    vOptions->gray1();
   else
    vOptions->gray0();

  const OString& message= ((VTextListItemData*)i->data)->cutMessage;

  font->print( margin, yBase+font->getDescent()+1, message );
}

/////////////////////////////////////////////////////////////////////////////

IBOOL VTextList::appendMessage( const OString& text )
{
  VTextListItemData* data= new VTextListItemData;
  data->message= text;

  VList::Item* i= VList::appendItem( font->getHeight()+1, this, data );
  if( i == 0 )
    {
    delete data;
    return IFALSE;
    }

  data->cutMessage= VWindow::cutString(font,text,i->getWidth()-2*margin);

  return ITRUE;
}

/////////////////////////////////////////////////////////////////////////////

// 1 <= i <= nElements
IBOOL VTextList::setMessage( int i, const OString& text )
{
  if( i < 1 || i > items.getNElements() )
    return IFALSE;

  Item* item= items.get(i);

  VTextListItemData* data= (VTextListItemData*) item->data;
  if( data == 0 )
    return IFALSE;
  if( data->message == text )
    return ITRUE;

  data->message= text;
  data->cutMessage= VWindow::cutString(font,text,item->getWidth()-2*margin);

  postRedraw();

  return ITRUE;
}

/////////////////////////////////////////////////////////////////////////////

// 1 <= i <= nElements+1
IBOOL VTextList::insertMessage( int i, const OString& text )
{
  if( i < 1 || i > items.getNElements()+1 )
    return IFALSE;

  if( i == items.getNElements()+1 )
    return appendMessage(text);

  VTextListItemData* data= new VTextListItemData;
  data->message= text;

  VList::Item* item= VList::insertItem( i, font->getHeight()+1, this, data );
  if( item == 0 )
    {
    delete data;
    return IFALSE;
    }

  data->cutMessage= VWindow::cutString(font,text,item->getWidth()-2*margin);

  return ITRUE;
}

/////////////////////////////////////////////////////////////////////////////

// 1 <= i <= nElements
IBOOL VTextList::deleteMessage( int i )
{
  if( i < 1 || i > items.getNElements() )
    return IFALSE;

  Item* item= items.get(i);

  VList::deleteItem( item->label );

  return ITRUE;
}

/////////////////////////////////////////////////////////////////////////////

// 1 <= i <= nElements
const OString& VTextList::getMessage( int i )
{
  if( i < 1 || i > items.getNElements() )
    return OString::empty();

  Item* item= items.get(i);
  VTextListItemData* data= (VTextListItemData*) item->data;
  if( data == 0 )
    return OString::empty();

  return data->message;
}

/////////////////////////////////////////////////////////////////////////////

int VTextList::getMessagePosition( const OString& message )
{
  int index= 1;
  IListIterator<Item> li(items);
  while( ! li.eol() )
    {
    Item* item= li.next();
    ++index;

    VTextListItemData* data= (VTextListItemData*) item->data;
    if( data == 0 )
      continue;

    if( data->message == message )
      return index;
    }

  return 0;
}

/////////////////////////////////////////////////////////////////////////////
