// This file is part of Moonlight Creator
//   Copyright (C) 1996-1998  Stephane Rehel
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
//
// You should have received a copy of the GNU Library General Public
// License along with this library; if not, write to the Free
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/*
  XGraphicsSystem.C

  X Graphics System Class

  Stephane Rehel
  June 28, 1996
*/

#include <stdio.h>
#include <stdlib.h>

#include "GL/opengl.h"
#include "GL/glx.h"

#include "tools/Command.h"
#include "tools/Chrono.h"

#include "XGraphicsSystem.h"
#include "XWindowDriver.h"
#include "XMLGLPixmap.h"
#include "XMLCursor.h"

#include "SystemWindow.h"

#include <X11/cursorfont.h>
#include <X11/Xproto.h>
#include <X11/Xatom.h>

/////////////////////////////////////////////////////////////////////////////

//static IBOOL need_overrideRedirect_windows= IFALSE;

/////////////////////////////////////////////////////////////////////////////

static Display* display= 0;
static OString displayName("");
static IVector screenSize(0,0);
static IVector screenDimension(0,0); // in millimeters
static int depth= 0; // just default depth of screen

static Screen* screen= 0;
static int screen_number= 0;
static Colormap colormap= Colormap(0);
static XVisualInfo* visualInfo= 0;
static Window xRootWindow= Window(0);

// for unblocking
//static Display* display2= 0;
//static Atom unblock_protocol= Atom(0);

static OString applicationName;
static OString iconName;
static XTextProperty titleProperty;
static XTextProperty iconnameProperty;

/////////////////////////////////////////////////////////////////////////////

static int ErrorHandler( Display* display, XErrorEvent* event )
{
/*
  char buf[256];

  fprintf( stderr, "\nReceived X error!\n" );
  fprintf( stderr, "\tError code   : %d\n", event->error_code );
  fprintf( stderr, "\tRequest code : %d\n", event->request_code );
  fprintf( stderr, "\tMinor code   : %d\n", event->minor_code );
  XGetErrorText( xDisplay, event->error_code, buf, 255 );
  fprintf( stderr, "\tError text : '%s'\n\n", buf );
  return 0;
*/
  // in case the error occurred during the Grab command...
  XUngrabServer(display);
  XUngrabButton(display, (u_int) AnyButton, 0, xRootWindow );

  int xerrcode = event->error_code;

  // non-fatal errors:   (sets xerrcode and returns)
  //    BadAlloc
  //    BadAccess errors on XFreeColors call
  //    Any error on the 'XKillClient()' call
  //    BadWindow errors (on a GetProperty call) (workaround SGI problem)
  //    BadLength errors on XChangeProperty
  //    BadMatch  errors on XGetImage
  //

  int req= event->request_code;

  if( xerrcode==BadMatch && req==X_SetInputFocus ) return 0;

/*
  if( (xerrcode == BadAlloc)              ||
      (xerrcode == BadAccess && req==88 ) || // X_FreeColors
      (                         req==113) || // X_KillClient
      (xerrcode == BadLength && req==18 ) || // X_ChangeProp
      (xerrcode == BadMatch  && req==73 ) || // X_GetImage
      (xerrcode == BadWindow && req==20 ) )  // X_GetProperty
    return 0;
*/

  // all other errors are 'fatal'
  fprintf( stderr, "\nReceived X error!\n" );
  fprintf( stderr, "\t  Error code : %d\n", xerrcode );
  fprintf( stderr, "\tRequest code : %d\n", req );
  fprintf( stderr, "\t  Minor code : %d\n", event->minor_code );
  char buf[256];
  XGetErrorText( display, xerrcode, buf, 255 );
  fprintf( stderr, "\t  Error text : '%s'\n\n", buf );

#ifndef NDEBUG
  // crash 'n' burn for debugging purposes
  char *str;
  str  = NULL;
  *str = '0';
  exit(-1);
#endif

  return 0;
}

/////////////////////////////////////////////////////////////////////////////

XGraphicsSystem::XGraphicsSystem()
{
  applicationName= "";
  iconName= "";
}

/////////////////////////////////////////////////////////////////////////////

XGraphicsSystem::~XGraphicsSystem()
{
  if( display != 0 )
    {
    XCloseDisplay(display);
    display= 0;
    }
}

/////////////////////////////////////////////////////////////////////////////

/*
static Cursor makeCursor( unsigned int cursorShape,
                          float r, float g, float b )
{
  if( display == 0 || screen == 0 )
    return Cursor(0);

  Cursor cursor= XCreateFontCursor( display, cursorShape );
  if( cursor == Cursor(0) )
    return cursor;

  typedef unsigned short ColorComponent;

  XColor fore;
  fore.flags= DoRed | DoGreen | DoBlue;
  fore.red=   min( ColorComponent(65535), ColorComponent(r * 65535.) );
  fore.green= min( ColorComponent(65535), ColorComponent(g * 65535.) );
  fore.blue=  min( ColorComponent(65535), ColorComponent(b * 65535.) );

  XColor back;
  back.flags= DoRed | DoGreen | DoBlue;
  back.red=   0xffff;
  back.green= 0xffff;
  back.blue=  0xffff;

  XRecolorCursor( display, cursor, &fore, &back );

  return cursor;
}
*/

/////////////////////////////////////////////////////////////////////////////

IBOOL XGraphicsSystem::init( Command& options )
{
  if( display != 0 )
    return ITRUE; // already initialized

/*
  if( options.findOption("-override") )
    {
    // Careful: in this case, our private colormap may not be
    // correctly set by the window manager...
    ::need_overrideRedirect_windows= ITRUE;
    }
*/

  displayName= options.getStringAfter("-display");

  display= XOpenDisplay( (displayName.length() == 0) ? 0 : displayName.get() );

  if( display == 0 )
    {
    if( displayName.length() == 0 )
      fprintf( stderr, "Error: unable to open default display\n" );
     else
      fprintf( stderr, "Error: unable to open display `%s'\n",
               displayName.get() );
    return IFALSE;
    }

  int erb, evb;
  if( !glXQueryExtension( display, &erb, &evb ) )
    {
    fprintf( stderr, "Fatal: no GLX extension in display %s\n",
             displayName.length() ? displayName.get() : ":0" );
    return IFALSE;
    }

  XSetErrorHandler(ErrorHandler);

//  display2= XOpenDisplay( (displayName.length() == 0) ? 0 : displayName.get() );
//  if( display2 != 0 )
//    unblock_protocol= XInternAtom( display2, "unblock_ml", False );

  // set screen
  if( ! initOpenGL() )
    return IFALSE;

  GraphicsSystem::_supportSaveUnder= XDoesSaveUnders(screen);

#ifdef NDEBUG
  reallyFullScreen= ITRUE;
#else
  reallyFullScreen= IFALSE;
#endif
  if( options.findOption("-fs") )
    reallyFullScreen= ITRUE;
  if( options.findOption("-debug") )
    reallyFullScreen= IFALSE;

  screenSize= IVector( WidthOfScreen(screen),
                       HeightOfScreen(screen) );
  screenDimension= IVector( WidthMMOfScreen(screen),
                            HeightMMOfScreen(screen) );

  if( ! reallyFullScreen )
    {
    int height= screenSize.y();
    screenSize[1]= height - 50;
    screenDimension[1]= screenDimension.y() * height / screenSize.y();
    }

  return GraphicsSystem::init(options);
}

/////////////////////////////////////////////////////////////////////////////

// private
IBOOL XGraphicsSystem::initOpenGL()
{
  if( display == 0 )
    return IFALSE;

  int gl_attrib[32];
  int i= 0;

  gl_attrib[i++] = GLX_LEVEL;
  gl_attrib[i++] = 0;

  // ask for double buffer
  gl_attrib[i++]= GLX_DOUBLEBUFFER;

  gl_attrib[i++] = GLX_RGBA;
  gl_attrib[i++] = GLX_RED_SIZE;
  gl_attrib[i++] = 1;
  gl_attrib[i++] = GLX_GREEN_SIZE;
  gl_attrib[i++] = 1;
  gl_attrib[i++] = GLX_BLUE_SIZE;
  gl_attrib[i++] = 1;

  gl_attrib[i++] = GLX_DEPTH_SIZE;
  gl_attrib[i++] = 1;

  // for indexed 8-bits visuals:
  //   gl_attrib[i++] = GLX_BUFFER_SIZE;
  //   gl_attrib[i++] = 1;

  gl_attrib[i] = (int)None;

  visualInfo= glXChooseVisual( display,
                               DefaultScreen(display),
                               gl_attrib );
  if( visualInfo == 0 )
    {
    fprintf( stderr, "Couldn't find visual for OpenGL RGB mode!\n" );
    return IFALSE;
    }

  screen_number= visualInfo->screen;
  screen= ScreenOfDisplay( display, screen_number );
  depth= visualInfo->depth;

  // RGB colormap is AllocNone, share the root colormap if possible
  int scrnum= DefaultScreen(display);
  xRootWindow= RootWindowOfScreen(screen);

  IBOOL private_colormap= ITRUE;

  if( MaxCmapsOfScreen(screen) == 1 &&
      visualInfo->visual == DefaultVisual( display, scrnum ) )
    {
    // the window and root are of the same visual type
    if( private_colormap )
      {
      // user doesn't want to share colormaps
      colormap= XCreateColormap( display, xRootWindow,
                                 visualInfo->visual,
                                 AllocNone );
      }
     else
      {
      // share the root colormap
      colormap= DefaultColormap( display, scrnum );
      }
    }
   else
    {
    // window and root are different visual types, allocate new cmap
    if( visualInfo->c_class == DirectColor )
      {
      colormap= XCreateColormap( display, xRootWindow,
                                 visualInfo->visual,
                                 AllocAll );
      }
     else
      {
      colormap= XCreateColormap( display, xRootWindow,
                                 visualInfo->visual,
                                 AllocNone );
      }
    }

  return ITRUE;
}

/////////////////////////////////////////////////////////////////////////////

void XGraphicsSystem::_iconify( IBOOL yes )
{
  SystemWindow* root= getRootWindow();
  if( root == 0 )
    return;

  XWindowDriver* wd= (XWindowDriver*) (root->getDriver());
  if( wd == 0 )
    return;

  Window rootID= wd->getWindowID();
  if( ! yes )
    {
    XMapWindow( display, rootID );

    root->currentPixel();
    glDrawBuffer(GL_FRONT);
    root->clear(0,0,0);
    glFlush();
    if( GraphicsSystem::doRedrawEvents() )
      GraphicsSystem::refreshScreen();
    }
   else
    {
    XIconifyWindow( display, rootID, screen_number );
    }
}

/////////////////////////////////////////////////////////////////////////////

const OString& XGraphicsSystem::getName()
{
  static OString name("X11");
  return name;
}

/////////////////////////////////////////////////////////////////////////////

IVector XGraphicsSystem::getScreenSize() const
{
  return screenSize;
}

/////////////////////////////////////////////////////////////////////////////

int XGraphicsSystem::getDepth() const
{
  return depth;
}

/////////////////////////////////////////////////////////////////////////////

void XGraphicsSystem::autoRepeat( IBOOL yes /* = ITRUE */ )
{
  if( display == 0 )
    return;

  if( yes )
    XAutoRepeatOn(display);
   else
    XAutoRepeatOff(display);
  XSync(display,False);
}

/////////////////////////////////////////////////////////////////////////////

void XGraphicsSystem::emitBeep()
{
  if( display == 0 )
    return;

  XBell(display,50);
}

/////////////////////////////////////////////////////////////////////////////

IVector XGraphicsSystem::getScreenDimemsion() const
{
  return screenDimension;
}

/////////////////////////////////////////////////////////////////////////////

Display* XGraphicsSystem::getDisplay() const
{
  assert( display != 0 );
  return display;
}

/////////////////////////////////////////////////////////////////////////////

XVisualInfo* XGraphicsSystem::getVisualInfo() const
{
  assert( visualInfo != 0 );
  return visualInfo;
}

/////////////////////////////////////////////////////////////////////////////

// parent can be NULL for root window
Window XGraphicsSystem::createWindow( Window parent,
                                      int x, int y, int width, int height,
                                      IBOOL saveUnder )
{
  if( display == 0 || colormap == Colormap(0) )
    return Window(0);

  IBOOL root= IFALSE;
  if( parent == Window(0) )
    {
    parent= xRootWindow;
    root= ITRUE;
    }

  XSetWindowAttributes attr;
  unsigned long valuemask(0);

  if( ! root ) // for root, this will be done just later
    {
    valuemask |= CWColormap;
    attr.colormap= colormap;
    }

  // required for SGI OpenGL
  valuemask |= CWBorderPixel;
  attr.border_pixel= BlackPixelOfScreen(screen);

  if( saveUnder && GraphicsSystem::_supportSaveUnder )
    {
    valuemask |= CWSaveUnder;
    attr.save_under= True;
    }

/*
  if( need_overrideRedirect_windows )
    {
    valuemask |= CWOverrideRedirect;
    attr.override_redirect= True;
    }
*/

//  if( root )
//    {
//    // it's the application root window
//    valuemask |= CWBackPixel;
//    attr.background_pixel= BlackPixelOfScreen(screen);
//    }

//  valuemask |= CWCursor;
//  attr.cursor= defaultCursor;

  Window windowID;
  windowID= XCreateWindow( display,
                           parent,
                           x, y, width, height,
                           0, // border width
                           depth,
                           InputOutput,
                           visualInfo->visual,
                           valuemask, // valuemask
                           &attr      // XSetWindowAttributes*
                         );

  if( windowID == Window(0) )
    return windowID;

  if( root )
    {
    // Tell mwm to remove decoration
    // No matter if we're not running mwm
    extern void set_mwm_border( Display *dpy, Window w );
    set_mwm_border(display,windowID);

    // force size and position of root window
    XSizeHints hints;
    hints.x= 0;               hints.y= 0;
    hints.width= width;       hints.height= height;
    hints.max_width= width;   hints.max_height = height;
    hints.min_width= width;   hints.min_height = height;
    hints.flags = USPosition | USSize | PPosition | PMinSize | PMaxSize;

    XSetStandardProperties( display, windowID,
                            0,//applicationName.get(), //window_name
                            "Moon Light", //iconName.get(), // icon_name
                            None, // icon_pixmap
                            NULL, 0, // argv, argc
                            &hints );

//XSetWMNormalHints( display, windowID, &hints );
    XMapWindow(display,windowID);

    // SetWindowPos
    XWindowChanges xwc;
    xwc.x= 0; xwc.y= 0;
    xwc.width= width; xwc.height= height;

    XConfigureWindow( display, windowID,
                      CWX | CWY | CWWidth | CWHeight, &xwc );

    XWMHints* wmh= XAllocWMHints();
    wmh->flags= 0;
    XSetWMHints(display,windowID,wmh);
    }

  XSync(display,False);

  if( root )
    {
    // ok, window is black, now set the colormap
    // (this avoids flashing init screen)
    XSetWindowColormap( display, windowID, colormap );
    }

  long event_mask= ExposureMask |
                   ButtonPressMask | ButtonReleaseMask |
                   KeyPressMask | KeyReleaseMask |
                   ButtonMotionMask ;
  if( root )
    event_mask |= StructureNotifyMask;

  XSelectInput( display, windowID, event_mask );

/*
if( ::need_overrideRedirect_windows )
{
// test it...
XSetWindowColormap( display, windowID, colormap );
XSetWMColormapWindows( display, windowID, &windowID, 1 );
}
*/

  // why not?
  Atom atom_list[1];
  atom_list[0]= XInternAtom( getDisplay(), "WM_DELETE_WINDOW", False );
  if( atom_list[0] != None )
    XSetWMProtocols( display, windowID, atom_list, 1 );

  return windowID;
}

/////////////////////////////////////////////////////////////////////////////

Window XGraphicsSystem::getRootWindowID() const
{
  if( rootWindow == 0 )
    return Window(0);

  XWindowDriver* wd= (XWindowDriver*) (rootWindow->getDriver());
  if( wd == 0 )
    return Window(0);

  return wd->getWindowID();
}

/////////////////////////////////////////////////////////////////////////////

Window XGraphicsSystem::getXRootWindowID() const
{
  return xRootWindow;
}

/////////////////////////////////////////////////////////////////////////////

Colormap XGraphicsSystem::getColormap() const
{
  return colormap;
}

/////////////////////////////////////////////////////////////////////////////

WindowDriver* XGraphicsSystem::newWindowDriver( SystemWindow* sw ) const
{
  return new XWindowDriver(sw);
}

/////////////////////////////////////////////////////////////////////////////

MLGLPixmap* XGraphicsSystem::newMLGLPixmap() const
{
  return new XMLGLPixmap;
}

/////////////////////////////////////////////////////////////////////////////

MLCursor* XGraphicsSystem::newMLCursor() const
{
  return new XMLCursor;
}

/////////////////////////////////////////////////////////////////////////////

// for the XEventManager::unblockWaiting() function
/*
Display* XGraphicsSystem::getDisplay2() const
{
  return display2;
}
*/

/////////////////////////////////////////////////////////////////////////////

// for the XEventManager::unblockWaiting() function
/*
Atom XGraphicsSystem::getUnblockProtocol() const
{
  return unblock_protocol;
}
*/

/////////////////////////////////////////////////////////////////////////////

void XGraphicsSystem::setApplicationName( const char* name )
{
  if( name == 0 )
    return;

  ::applicationName= name;
  ::iconName= name;

  Window rootID= ((XWindowDriver*)(getRootWindow()->getDriver()))
                ->getWindowID();

  XSetStandardProperties( display,
                          rootID,
                          name, // window name
                          name, // icon name
                          None,  // icon pixmap
                          0, 0,  // argv, argc
                          0 );   // &hints
//printf("coucou!!! <%s>\n",name);

  static OString __windowName;
  static OString __iconName;
  __windowName= OString(::applicationName);
  __iconName= OString(::iconName);
  char* _windowName= __windowName._get();
  char* _iconName= __iconName._get();

  XClassHint* classhints= XAllocClassHint();
  classhints->res_name= _windowName;
  classhints->res_class= _windowName;

  if( titleProperty.value != 0 )
    {
    XFree( titleProperty.value );
    titleProperty.value= 0;
    }
  if( XStringListToTextProperty( &_windowName, 1, &titleProperty ) == 0 )
    {
    // error
    titleProperty.value= 0;
    return;
    }

  if( iconnameProperty.value != 0 )
    {
    XFree( iconnameProperty.value );
    iconnameProperty.value= 0;
    }
  if( XStringListToTextProperty( &_iconName, 1, &iconnameProperty ) == 0 )
    {
    // error
    iconnameProperty.value= 0;
    return;
    }

  XSetWMProperties( display, rootID,
                    &titleProperty, &iconnameProperty,
                    0, 0, // argv, argc,
                    0, // sizehints
                    0, // wmhints
                    classhints  // classhints
                  );
  XSetWMName( display, rootID, &titleProperty );
  XSetWMIconName( display, rootID, &iconnameProperty );

  XSetStandardProperties( display, rootID,
                          _windowName,
                          _iconName,
                          Pixmap(0), // icon
                          0, 0, // argv, argc
                          0 // XSizeHints
                        );
}

/////////////////////////////////////////////////////////////////////////////

