/* XQF - Quake server browser and launcher
 * Copyright (C) 1998 Roman Pozlevich <roma@botik.ru>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include <stdio.h>	/* FILE, popen, fgets, sprintf, ... */
#include <string.h>	/* strchr, strcmp, strlen, strcpy, ... */
#include <stdlib.h>	/* strtol, qsort */
#include <unistd.h>	/* getpid, unlink */

#include <gtk/gtk.h>

#include "xqf.h"
#include "pref.h"
#include "qstat.h"
#include "utils.h"
#include "server.h"
#include "dialogs.h"


#define MAX_BUF		4096
#define MAX_TOKENS	64


FILE *p = NULL;
static char buf[MAX_BUF];
static char *token[MAX_TOKENS];


/*
 * TODO:
 *   -- automatic buffer size adjustment for long lines
 */

static char *qstat_getline (void) {
  char *ptr;

  if (feof (p))
    return NULL;

  fgets (buf, MAX_BUF, p);
  if (ferror (p))
    return NULL;

  ptr = buf + strlen (buf) - 1;
  if (*ptr == '\n')
    *ptr = '\0';

  return buf;
}


static int tokenize (char *str, char dlm) {
  int tokens = 0;

  if (!str || str[0] == '\0')
    return 0;

  while (str && tokens < MAX_TOKENS) {
    token[tokens++] = str;
    str = strchr (str, dlm);
    if (str)
      *str++ = '\0';
  }

  return tokens;
}


static struct player *parse_player (char *token[], int n, enum server_type type) {
  struct player *player = NULL;
  char *ptr;

  switch (type) {

  case QW_SERVER:
    if (n < 8)
      break;

    player = g_malloc (sizeof (struct player) + 
                               strlen (token[1]) + 1 + strlen (token[7]) + 1);
    player->id    = strtol (token[0], NULL, 10);
    player->frags = strtol (token[2], NULL, 10);
    player->time  = strtol (token[3], NULL, 10);
    player->shirt = strtol (token[4], NULL, 10);
    player->pants = strtol (token[5], NULL, 10);
    player->ping  = strtol (token[6], NULL, 10);

    ptr = (char*) player + sizeof (struct player);
    player->name = strcpy (ptr, token[1]);
    player->skin = strcpy (ptr + strlen (token[1]) + 1, token[7]);

    if (player->shirt < 0 || player->shirt > 13)
      player->shirt = 0;

    if (player->pants < 0 || player->pants > 13)
      player->pants = 0;

    break;

  case Q2_SERVER:
    if (n < 3)
      break;

    player = g_malloc (sizeof (struct player) + strlen (token[0]) + 1);
    player->frags = strtol (token[1], NULL, 10);
    player->ping  = strtol (token[2], NULL, 10);

    ptr = (char*) player + sizeof (struct player);
    player->name = strcpy (ptr, token[0]);

    player->id = 0;
    player->time  = 0;
    player->shirt = 0;
    player->pants = 0;
    player->skin = NULL;

    break;
  }

  return player;
}


int qsort_players (const void *a, const void *b) {
  struct player *p1 = * (struct player **) a;
  struct player *p2 = * (struct player **) b;
  int sortmode = player_clist_columns[psort_column].sortmode;
  int invert = 0;
  int res;

  if (sortmode < 0) {
    sortmode = -sortmode;
    invert = 1;
  }

  switch (sortmode) {
  case SORT_PLAYER_FRAGS:
    res = p1->frags - p2->frags;
    break;

  case SORT_PLAYER_ID:
    res = p1->id - p2->id;
    break;

  case SORT_PLAYER_NAME:
    if (p1->name)
      res = (p2->name)? strcmp (p1->name, p2->name) : 1;
    else
      res = (p2->name)? -1 : 0;
    break;

  case SORT_PLAYER_TIME:
    res = p1->time - p2->time;
    break;

  case SORT_PLAYER_COLOR:
    res = p1->pants - p2->pants;
    if (!res) 
      res = p1->shirt - p2->shirt;
    break;

  case SORT_PLAYER_PING:
    res = p1->ping - p2->ping;
    break;

  case SORT_PLAYER_SKIN:
    if (p1->skin)
      res = (p2->skin)? strcmp (p1->skin, p2->skin) : 1;
    else
      res = (p2->skin)? -1 : 0;
    break;

  case SORT_PLAYER_NONE:
  default:
    res = 0;
    break;
  }

  return (invert) ? -res : res;
}


int qsort_servers (const void *a, const void *b) {
  struct server *s1 = * (struct server **) a;
  struct server *s2 = * (struct server **) b;
  int sortmode = server_clist_columns[ssort_column].sortmode;
  int invert = 0;
  int res;

  if (sortmode < 0) {
    sortmode = -sortmode;
    invert = 1;
  }

  switch (sortmode) {
  case SORT_SERVER_NAME:
    if (s1->name)
      res = (s2->name)? strcmp (s1->name, s2->name) : 1;
    else
      res = (s2->name)? -1 : 0;
    break;

  case SORT_SERVER_ADDRESS:
    if (s1->address) {
      if (s2->address) {
	res = strcmp (s1->address, s2->address);
	if (!res)
	  res = s1->port - s2->port;
      }
      else
	res = 1;
    }
    else
      res = (s2->address)? -1 : 0;
    break;

  case SORT_SERVER_MAP:
    if (s1->map)
      res = (s2->map)? strcmp (s1->map, s2->map) : 1;
    else
      res = (s2->map)? -1 : 0;
    break;

  case SORT_SERVER_PLAYERS:
    res = s1->curplayers - s2->curplayers;
    if (!res) 
      res = s1->maxplayers - s2->maxplayers;
    break;

  case SORT_SERVER_PING:
    res = s1->ping - s2->ping;
    if (!res) 
      res = s1->retries - s2->retries;
    break;

  case SORT_SERVER_TO:
    res = s1->retries - s2->retries;
    if (!res) 
      res = s1->ping - s2->ping;
    break;

  case SORT_SERVER_GAME:
    if (s1->game)
      res = (s2->game)? strcmp (s1->game, s2->game) : 1;
    else
      res = (s2->game)? -1 : 0;
    break;

  case SORT_SERVER_NONE:
  default:
    res = 0;
    break;
  }

  return (invert) ? -res : res;
}


static int qsort_serverinfo (const void *a, const void *b) {
  char *ptr1, *ptr2;

  ptr1 = * (char**) a;
  if (*ptr1 == '*')
    ptr1++;

  ptr2 = * (char**) b;
  if (*ptr2 == '*')
    ptr1++;

  return strcmp (ptr1, ptr2);
}


static int collect_players (struct server *server) {
  GSList *list = NULL;
  GSList *ptr;
  struct player *player;
  int n;
  int num = 0;

  for (;;) {
    if (qstat_getline () == NULL)
      break;

    if (buf[0] == '\0')		/* stop after blank line */
      break;

    n = tokenize (buf, QSTAT_DELIM);
    player = parse_player (token, n, server->type);

    if (!player)
      continue;			/* error, try to recover */

    list = g_slist_prepend (list, player);
    num++;
  }

  server->curplayers = num;

  if (num == 0)
    return 0;
  
  server->players = g_malloc (sizeof (struct player) * num);

  for (ptr = list, n = 0; ptr; n++) {
    server->players[n] = (struct player *) ptr->data;
    ptr = ptr->next;
  }

  g_slist_free (list);

  return num;
}


static void free_players (struct player **p, int n) {
  struct player **ptr = p;

  if (p) {
    while (n-- > 0)
      g_free (*ptr++);
    g_free (p);
  }
}


static char **parse_serverinfo (char *token[], int n) {
  char **info;
  int size = 0;
  int i, j;
  char *ptr, *info_ptr;

  if (n == 0)
    return NULL;

  for (size = 0, i = 0; i < n; i++) {
    size += strlen (token[i]) + 1;
  }

  info = g_malloc (sizeof (gpointer) * (n + 1) * 2 + size);

  info_ptr = (char*) info + sizeof (gpointer) * (n + 1) * 2;

  for (i = 0, j = 0; i < n; i++) {
    ptr = strchr (token[i], '=');

    if (ptr)
      *ptr++ = '\0';

    info [j++] = strcpy (info_ptr, token[i]);
    info_ptr += strlen (token[i]) + 1;

    if (ptr) {
      info [j++] = strcpy (info_ptr, ptr);
      info_ptr += strlen (ptr) + 1;
    }
    else {
      info [j++] = NULL;
    }
  }

  info [j + 1] = info [j] = NULL;

  qsort (info, n, sizeof (char*) * 2, qsort_serverinfo);

  return info;
}


static struct server *parse_server (char *token[], int n) {
  struct server *s;
  enum server_type type;
  char *addr;
  int port;

  if (n < 3)
    return NULL;

  if (strcmp (token[0], "QW") == 0) {
    type = QW_SERVER;
  }
  else if (strcmp (token[0], "Q2") == 0) {
    type = Q2_SERVER;
  }
  else {
    return NULL;
  }

  if (!parse_address (token[1], &addr, &port))
    return NULL;

  if (port == -1)
    port = (type == Q2_SERVER)? Q2_DEFAULT_PORT : QW_DEFAULT_PORT;

  s = server_lookup (addr, port);
  if (!s) {
    s = server_new (addr, port, type);
  }
  else {
    if (s->map) {
      g_free (s->map);
      s->map  = NULL;
    }
    if (s->players) {
      free_players (s->players, s->curplayers);
      s->players = NULL;
    }
    if (s->info) {
      g_free (s->info);
      s->info = NULL;
      s->game = NULL;
    }
  }

  g_free (addr);

  if (n >= 8) {
    s->type = type;    /* QW <--> Q2 */

    if (s->name) 
      g_free (s->name);

    if (*(token[2]))		/* if name is not empty */
      s->name = g_strdup (token[2]);

    if (*(token[3]))            /* if map is not empty */
      s->map  = g_strdup (token[3]);

    s->maxplayers = strtol (token[4], NULL, 10);
    s->curplayers = strtol (token[5], NULL, 10);

    s->ping = strtol (token[6], NULL, 10);
    if (s->ping > MAX_PING)
      s->ping = MAX_PING;

    s->retries = strtol (token[7], NULL, 10);
    if (s->retries > maxretries)
      s->retries = maxretries;
  }
  else {
    s->maxplayers = 0;
    s->curplayers = 0;
    s->retries = MAX_RETRIES;

    if (n == 3 && strcmp (token[2], "DOWN") == 0)
      s->ping = MAX_PING + 1;
    else 
      s->ping = MAX_PING;	/* TIMEOUT */
  }

  return s;
}


void free_servers (GSList *list) {
  if (list) {
    g_slist_foreach (list, (GFunc) server_unref, NULL);
    g_slist_free (list);
  }
}


static GSList *collect_servers (void) {
  GSList *list = NULL;
  struct server *s;
  char **info_ptr;
  int n;
  int i;

  for (;;) {
    if (qstat_getline () == NULL)
      break;

    if ((n = tokenize (buf, QSTAT_DELIM)) < 3)	/* error, try to recover */
      continue;

    s = parse_server (token, n);
    if (s) {
      list = g_slist_prepend (list, s);
      s->ref_count++; 
      
      if (qstat_getline () == NULL)
	break;

      if ((n = tokenize (buf, QSTAT_DELIM)) == 0)
	continue;

      s->flags = 0;

      s->info = parse_serverinfo (token, n);
      if (s->info) {
	for (info_ptr = s->info; info_ptr && *info_ptr; info_ptr += 2) {
	  if ((s->type == QW_SERVER && strcmp (*info_ptr, "*gamedir") == 0) ||
              (s->type == Q2_SERVER && strcmp (*info_ptr, "gamedir") == 0)) {
	    s->game = info_ptr[1];
	    continue;
	  }
	  else if ((s->type == QW_SERVER && strcmp (*info_ptr, "*cheats") == 0) ||
	           (s->type == Q2_SERVER && strcmp (*info_ptr, "cheats") == 0 && 
                                                      info_ptr[1][0] != '0')) {
	    s->flags |= SERVER_CHEATS;
	  }
	  else if (strcmp (*info_ptr, "needpass") == 0) {
	    i = strtol (info_ptr[1], NULL, 10);
	    if (i & 1)
	      s->flags |= SERVER_PASSWORD;
	    if (i & 2)
	      s->flags |= SERVER_SP_PASSWORD;
	  }
	}
	collect_players (s);
      }
      else {
	/* try to recover after error -- skip to the next record */
        while (qstat_getline () && buf[0] != '\0');
      }
    }
  }

  return list;
}


static int list_collect_servers (void) {
  GSList *list;
  GSList *lp;
  struct server *s;
  int res;
  
  list = collect_servers ();
  res = g_slist_length (list);

  if (list) {
    for (lp = list; lp; lp = lp->next) {
      s = (struct server *) lp->data;
      if (s->ref_count == 1)
	fprintf (stderr, "list_collect_servers(): new server is not expected here\n");
    }
    
    free_servers (list);	/* free list and drop unexpected servers */
  }

  print_status ("Done.");

  return res;
}


int stat_master (struct master *m) {
  int n;

  sprintf (buf, "qstat -maxsimultaneous %d -retry %d -R -P -raw \\%c -qw %s ", 
                         maxsimultaneous, maxretries, QSTAT_DELIM, m->address);
#ifdef DEBUG
  fprintf (stderr, "%s\n", buf);
#endif

  p = popen (buf, "r");
  if (!p) {
    dialog_ok (NULL, "Cannot execute \"qstat\"");
    return FALSE;
  }

  if (qstat_getline () == NULL) {
    pclose (p);
    dialog_ok (NULL, "Cannot execute \"qstat\"");
    return FALSE;
  }

  n = tokenize (buf, QSTAT_DELIM);
  if (n < 3 || strcmp (token[0], "QWM") != 0) {
    pclose (p);
    dialog_ok (NULL, "Cannot execute \"qstat\"");
    return FALSE;
  }

  n = strtol (token[2], NULL, 10);
  if (n <= 0 || n > MAX_SERVERS) {
    pclose (p);
    return FALSE;
  }

  qstat_getline ();	/* skip blank line */

  free_servers (m->servers);
  m->servers = collect_servers ();
  print_status ("Done.");

  pclose (p);
  return TRUE;
}


int stat_server_list (GSList *list) {
  GSList *lp;
  int n;
  struct server *s;
  FILE *f = NULL;
  char *fn = NULL;

  if (!list) 
    return TRUE;
  
  n = sprintf (buf, "qstat -maxsimultaneous %d -retry %d -R -P -raw \\%c", 
                                     maxsimultaneous, maxretries, QSTAT_DELIM);
  if (g_slist_length (list) <= 8) {
    for (lp = list; lp; lp = lp->next) {
      s = (struct server *) lp->data;
      n += sprintf (buf + n, " -qws %s:%d", s->address, s->port);
    }
  }
  else {
    fn = g_malloc (128);
    sprintf (fn, "/tmp/xqf-qstt.%d", getpid ());

    sprintf (buf + n, " -f %s", fn);

    f = fopen (fn, "w");

    if (!f) {
      g_free (fn);
      dialog_ok (NULL, "/tmp is not writable");
      return FALSE;
    }

    for (lp = list; lp; lp = lp->next) {
      s = (struct server *) lp->data;
      fprintf (f, "%s %s:%d\n", (s->type == QW_SERVER)? "QW" : "Q2",
                                                          s->address, s->port);
    }

    fclose (f);
  }

#ifdef DEBUG
  fprintf (stderr, "%s\n", buf);
#endif

  p = popen (buf, "r");
  if (!p) {
    if (fn) {
      unlink (fn);
      g_free (fn);
    }	
    dialog_ok (NULL, "Cannot execute \"qstat\"");
    return FALSE;
  }

  if (list_collect_servers () == 0) {
    dialog_ok (NULL, "Cannot execute \"qstat\"");
    return FALSE;
  }

  pclose (p);

  if (fn) {
    unlink (fn);
    g_free (fn);
  }

  return TRUE;
}

