/* XQF - Quake server browser and launcher
 * Copyright (C) 1998 Roman Pozlevich <roma@botik.ru>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include <stdio.h>	/* FILE, sprintf, perror */
#include <unistd.h>	/* execvp, fork, exit, chdir, stat */
#include <string.h>	/* strtok */
#include <signal.h>	/* sigaction */
#include <sys/types.h>	/* waitpid */
#include <sys/wait.h>	/* waitpid */
#include <sys/stat.h>	/* stat */

#include <gtk/gtk.h>

#include "xqf.h"
#include "pref.h"
#include "utils.h"
#include "qrun.h"
#include "dialogs.h"


int quake_config_is_valid (enum server_type type) {
  struct stat stat_buf;
  char *game;
  char *dir;
  char *cmd;
  char *cfgdir;
  char *path;

  if (type == Q2_SERVER) {
    game = "Quake2";
    dir = default_q2_dir;
    cmd = default_q2_cmd;
    cfgdir = "baseq2";
  }
  else {
    game = "QuakeWorld";
    dir = default_quake_dir;
    cmd = default_qw_cmd;
    cfgdir = "id1";
  }

  if (cmd == NULL || cmd[0] == '\0') {
    dialog_ok (NULL, "%s command line is empty.", game);
    return FALSE;
  }

  if (dir != NULL && dir[0] != '\0') {
    if (stat (dir, &stat_buf) != 0 || !S_ISDIR (stat_buf.st_mode)) {
      dialog_ok (NULL, "\"%s\" is not a directory\n"
	              "Please specify correct %s working directory.", 
                       dir, game);
      return FALSE;
    }
  }

  path = file_in_dir (dir, cfgdir);

  if (stat (path, &stat_buf) || !S_ISDIR (stat_buf.st_mode)) {
    if (!dir || dir[0] == '\0') {
      dialog_ok (NULL, "Please specify correct %s working directory.", game);
    }
    else {
      dialog_ok (NULL, "Directory \"%s\" doesn\'t contain \"%s\" subdirectory.\n"
                       "Please specify correct %s working directory.", 
                       dir, cfgdir, game);
    }
    g_free (path);
    return FALSE;
  }

  g_free (path);
  return TRUE;
}


static int write_qw_vars (char *filename, struct server *s) {
  FILE *f;
  int plat;

  if ((f = fopen (filename, "w")) == NULL) 
    return FALSE;

  fprintf (f, "// generated by XQF, do not modify\n");

  fprintf (f, "name \"%s\"\n", (default_name)? default_name : "");
  fprintf (f, "team \"%s\"\n", (default_team)? default_team : "");
  fprintf (f, "skin \"%s\"\n", (default_skin)? default_skin : "");
  fprintf (f, "topcolor    \"%d\"\n", default_top_color);
  fprintf (f, "bottomcolor \"%d\"\n", default_bottom_color);

  /* pushlatency calculation */

  if (s->ping <= 0)
    plat = -50;		/* "min" value */
  else if (s->ping >= 2000)
    plat = -1000;		/* "max" value */
  else {
    plat = s->ping / 2;
    plat = ((plat+49)/50)*50;	/* beautify it */
  }
  fprintf (f, "pushlatency %d\n", -plat);

  fprintf (f, "rate        \"%d\"\n", default_rate);
  fprintf (f, "cl_nodelta  \"%d\"\n", default_cl_nodelta);
  fprintf (f, "cl_predict_players \"%d\"\n", default_cl_predict);
  fprintf (f, "noaim       \"%d\"\n", default_noaim);
  fprintf (f, "noskins     \"%d\"\n", default_noskins);
  if (default_w_switch >= 0)
    fprintf (f, "setinfo w_switch \"%d\"\n", default_w_switch);
  if (default_b_switch >= 0)
    fprintf (f, "setinfo b_switch \"%d\"\n", default_b_switch);
  fprintf (f, "_windowed_mouse \"%d\"\n", default_windowed_mouse);

  fclose (f);
  return TRUE;
}


static int write_q2_vars (char *filename, struct server *s) {
  FILE *f;
  int plat;

  if ((f = fopen (filename, "w")) == NULL) 
    return FALSE;

  fprintf (f, "// generated by XQF, do not modify\n");

  fprintf (f, "set name \"%s\"\n", (default_name)? default_name : "");
  fprintf (f, "set rate        \"%d\"\n", default_rate);
  fprintf (f, "set cl_nodelta  \"%d\"\n", default_cl_nodelta);
  fprintf (f, "set cl_predict  \"%d\"\n", default_cl_predict);
  fprintf (f, "set cl_noskins  \"%d\"\n", default_noskins);
  fprintf (f, "set _windowed_mouse \"%d\"\n", default_windowed_mouse);

  fclose (f);
  return TRUE;
}


void launch_quake (struct condef *con) {
  char *cmd;
  char *argv[32];
  int argi = 0;
  const char delim[] = " \t\n\r";
  char *q_cmd;
  char *q_dir;
  char *q_conf;

  if (!con || !con->server)
    return;

  if (con->s->type == QW_SERVER) {
    q_cmd = default_qw_cmd;
    q_dir = default_quake_dir;
    q_conf = "id1/" EXEC_CFG;
  }
  else {
    q_cmd = default_q2_cmd;
    q_dir = default_q2_dir;
    q_conf = "baseq2/" EXEC_CFG;
  }

  if (!q_cmd || q_cmd[0] == '\0')
    return;

  cmd = strdup_strip (q_cmd);

  if (q_dir != NULL && q_dir[0] != '\0') {
    if (chdir (q_dir) != 0)
      fprintf (stderr, "Cannot change directory to \"%d\"", q_dir);
  }

  if ((con->s->type == QW_SERVER && !write_qw_vars (q_conf, con->s)) ||
      (con->s->type == Q2_SERVER && !write_q2_vars (q_conf, con->s))) {
    fprintf (stderr, "Cannot write to file \"%s\".", q_conf);
    g_free (cmd);
    return;
  }

  argv[argi++] = strtok (cmd, delim);
  while ((argv[argi] = strtok (NULL, delim)) != NULL)
    argi++;

  if (con->gamedir) {
    argv[argi++] = (con->s->type == QW_SERVER)? "-gamedir" : "+set game";
    argv[argi++] = con->gamedir;
  }

  argv[argi++] = "+exec";
  argv[argi++] = EXEC_CFG;

  if (con->observe) {
    argv[argi++] = "+spectator";
    argv[argi++] = con->observe;
  }

  if (con->password) {
    argv[argi++] = "+password";
    argv[argi++] = con->password;
  }

  if (con->server) {
    argv[argi++] = "+connect";
    argv[argi++] = con->server;
  }

  argv[argi] = NULL;

  for (argi = 0; argv[argi]; argi++)
    fprintf (stderr, "%s ", argv[argi]);
  fprintf (stderr, "\n");

  execvp (argv[0], argv);

  /* I'm still alive - bad */

  g_free (cmd);
  fprintf (stderr, "Cannot execute client\n");
}


void launch_quake_fork (struct condef *con) {
  int pid;

  if (!con || !con->server)
    return;

  pid = fork ();

  if (pid == -1) {
    dialog_ok (NULL, "Internal Error: fork() failed.");
    return;
  }

  if (pid == 0) {	/* child */
    launch_quake (con);
    exit (1);
  }
}


struct condef *condef_new (struct server *s) {
  struct condef *con;

  con = g_malloc (sizeof (struct condef));

  con->s = s;
  con->server = NULL;
  con->gamedir = NULL;
  con->observe = NULL;
  con->password = NULL;

  return con;
}


void condef_free (struct condef *con) {
  if (con->server) {
    g_free (con->server);
    con->server = NULL;
  }

  if (con->gamedir) {
    g_free (con->gamedir);
    con->gamedir = NULL;
  }

  if (con->observe) {
    g_free (con->observe);
    con->observe = NULL;
  }

  if (con->password) {
    g_free (con->password);
    con->password = NULL;
  }

  g_free (con);
}


static void sigchild_handler (int signum) {
  int pid;
  int status;

  while ((pid = waitpid (WAIT_ANY, &status, WNOHANG)) > 0);

  /*
  if (pid < 0)
    perror ("waitpid");
  */
}


void set_sigchild (void) {
  struct sigaction action;

  action.sa_handler = sigchild_handler;
  sigemptyset (&action.sa_mask);
  action.sa_flags = SA_NOCLDSTOP | SA_NODEFER | SA_RESTART;
  sigaction (SIGCHLD, &action, NULL);
}



