/*	$Id: undo.c,v 1.7 1997/09/11 18:13:40 sandro Exp $	*/

/*
 * Copyright (c) 1997
 *	Sandro Sigala, Brescia, Italy.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <err.h>

#include "zile.h"
#include "extern.h"

/*
 * This variable is set to TRUE when the `undo_save()' function should not
 * register the undo information.
 */
int undo_nosave = 0;
/*
 * This variable is set to TRUE when an undo is in action.
 */
static int doing_undo = 0;

/*
 * Jump to the specified line number and offset.
 */
static void
goto_point(int pointn, int pointo)
{
	if (cur_wp->pointn > pointn)
		do 
			F_previous_line(1);
		while (cur_wp->pointn > pointn);
	else if (cur_wp->pointn < pointn)
		do
			F_next_line(1);
		while (cur_wp->pointn < pointn);
	if (cur_wp->pointo > pointo)
		do
			F_backward_char(1);
		while (cur_wp->pointo > pointo);
	else if (cur_wp->pointo < pointo)
		do
			F_forward_char(1);
		while (cur_wp->pointo < pointo);
}

/*
 * Save a reverse delta for doing undo.
 */
void
undo_save(int type, int startn, int starto, int arg)
{
	undop up;

	if (cur_bp->flags & BFLAG_NOUNDO || undo_nosave)
		return;

	up = (undop)xmalloc(sizeof(*up));

	up->type = type;

	up->pointn = startn;
	up->pointo = starto;

	switch (type) {
	case UNDO_INSERT_CHAR:
	case UNDO_REPLACE_CHAR:
		up->delta.c = arg;
		break;
	case UNDO_INSERT_BLOCK:
	case UNDO_REPLACE_BLOCK:
		up->delta.block.size = arg;
		up->delta.block.text = copy_text_block(startn, starto, arg);
		break;
	case UNDO_REMOVE_BLOCK:
		up->delta.block.size = arg;
		break;
	}

	if (cur_bp->last_undop == NULL)
		up->next = NULL;
	else
		up->next = cur_bp->last_undop;
	cur_bp->last_undop = up;
	if (!doing_undo)
		cur_bp->next_undop = up;
}

DEFUN("undo", undo)
/*+
Undo some previous changes.
Repeat this command to undo more changes.
+*/
{
	undop up;
	int i;

	if (cur_bp->flags & BFLAG_NOUNDO) {
		minibuf_error("%FCUndo disabled in this buffer%E");
		return FALSE;
	}

	if (warn_if_readonly_buffer())
		return FALSE;

	if ((up = cur_bp->next_undop) == NULL) {
		minibuf_error("%FCNo further undo information%E");
		cur_bp->next_undop = cur_bp->last_undop;
		return FALSE;
	}

	goto_point(up->pointn, up->pointo);

	doing_undo = 1;

	switch (up->type) {
	case UNDO_INSERT_CHAR:
		if (up->delta.c == '\n')
			insert_newline();
		else
			insert_char(up->delta.c);
		break;
	case UNDO_INSERT_BLOCK:
		undo_save(UNDO_REMOVE_BLOCK, up->pointn, up->pointo, up->delta.block.size);
		undo_nosave = 1;
		for (i = 0; i < up->delta.block.size; i++)
			if (up->delta.block.text[i] != '\n')
				insert_char(up->delta.block.text[i]);
			else
				insert_newline();
		undo_nosave = 0;
		break;
	case UNDO_REMOVE_CHAR:
		F_delete_char(1);
		break;
	case UNDO_REMOVE_BLOCK:
		undo_save(UNDO_INSERT_BLOCK, up->pointn, up->pointo, up->delta.block.size);
		undo_nosave = 1;
		for (i = 0; i < up->delta.block.size; i++)
			F_delete_char(1);
		undo_nosave = 0;
		break;
	case UNDO_REPLACE_CHAR:
		undo_save(UNDO_REPLACE_CHAR, up->pointn, up->pointo,
			  cur_wp->pointp->text[up->pointo]);
		cur_wp->pointp->text[up->pointo] = up->delta.c;
		cur_bp->flags |= BFLAG_MODIFIED;
		if (cur_bp->flags & BFLAG_FONTLOCK)
			font_lock_reset_anchors(cur_bp, cur_wp->pointp);
		break;
	case UNDO_REPLACE_BLOCK:
		undo_save(UNDO_REPLACE_BLOCK, up->pointn, up->pointo, up->delta.block.size);
		undo_nosave = 1;
		for (i = 0; i < up->delta.block.size; i++)
			F_delete_char(1);
		for (i = 0; i < up->delta.block.size; i++)
			if (up->delta.block.text[i] != '\n')
				insert_char(up->delta.block.text[i]);
			else
				insert_newline();
		undo_nosave = 0;
		break;
	}

	doing_undo = 0;

	/*
	 * Switch to next undo entry; unset the BFLAG_MODIFIED flag
	 * if all the undo deltas were applied.
	 */
	if ((cur_bp->next_undop = up->next) == NULL)
		cur_bp->flags &= ~BFLAG_MODIFIED;

	minibuf_write("%FCUndo!%E");

	return TRUE;
}
