------------------------------------------------------------------------------
--                                                                          --
--                         GNAT COMPILER COMPONENTS                         --
--                                                                          --
--                              G N A T L B R                               --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                            $Revision: 1.1 $                              --
--                                                                          --
--          Copyright (C) 1996-1997 Free Software Foundation, Inc.          --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNAT;  see file COPYING.  If not, write --
-- to  the Free Software Foundation,  59 Temple Place - Suite 330,  Boston, --
-- MA 02111-1307, USA.                                                      --
--                                                                          --
-- GNAT was originally developed  by the GNAT team at  New York University. --
-- It is now maintained by Ada Core Technologies Inc (http://www.gnat.com). --
--                                                                          --
------------------------------------------------------------------------------

--  This program is currently used only on Alpha/VMS

with Ada.Command_Line;     use Ada.Command_Line;
with Ada.Text_IO;          use Ada.Text_IO;
with GNAT.OS_Lib;          use GNAT.OS_Lib;
with Gnatvsn;              use Gnatvsn;
with Interfaces.C_Streams; use Interfaces.C_Streams;
with Osint;                use Osint;
with Sdefault;             use Sdefault;
with System;

procedure GnatLbr is
   type Lib_Mode is (Create, Set, Delete);
   Next_Arg  : Integer;
   Mode      : Lib_Mode;
   ADC_File  : String_Access := null;
   Lib_Dir   : String_Access := null;
   Make      : constant String := "make";
   Make_Path : String_Access;

   procedure Create_Directory (Name : System.Address; Mode : Integer);
   pragma Import (C, Create_Directory, "mkdir");
begin

   --  Program to create, set, or delete an alternate runtime library.
   --
   --  Works by calling an appropriate target specific Makefile residing
   --  in the default library object (e.g. adalib) directory from the context
   --  of the new library objects directory.
   --
   --  Command line arguments are:
   --  1st:  create | set | delete
   --    create : Build a library
   --    set    : Set environment variables to point to a library
   --    delete : Delete a library
   --
   --  2nd:  /config=<file_spec>
   --  A -gnatg valid file containing desired configuration pragmas
   --
   --  3rd: <directory_spec>
   --  The directory which contains or will contain the alternate library.

   if Argument_Count = 0 then
      Put_Line ("Usage: gnatlib [create|set|delete] ...");
      Exit_Program (E_Fatal);
   end if;

   if Argument (1) = "create" then
      Mode := Create;
   elsif Argument (1) = "set" then
      Mode := Set;
   elsif Argument (1) = "delete" then
      Mode := Delete;
   else
      Put_Line ("Usage: gnatlib [create|set|delete] ...");
      Exit_Program (E_Fatal);
   end if;

   Next_Arg := 2;

   loop
      exit when Next_Arg > Argument_Count;

      Process_One_Arg : declare
         Arg : String := Argument (Next_Arg);

      begin

         if Arg'Length > 8 and then Arg (1 .. 8) = "/config=" then
            if ADC_File /= null then
               Put_Line ("Error: Multiple gnat.adc files specified");
               Exit_Program (E_Fatal);
            end if;

            ADC_File := new String'(Arg (9 .. Arg'Last));
         elsif Arg'Length > 0 and then Arg (1) = '-' then
            null;
         else
            if Lib_Dir /= null then
               Put_Line ("Error: Multiple library directories specified");
               Exit_Program (E_Fatal);
            end if;

            Lib_Dir := new String'(Arg);
         end if;
      end Process_One_Arg;

      Next_Arg := Next_Arg + 1;
   end loop;

   case Mode is
      when Create =>
         --
         --  Validate arguments.
         --
         if Lib_Dir = null then
            Put_Line ("No library directory specified");
            Exit_Program (E_Fatal);
         end if;

         if Is_Directory (Lib_Dir.all) then
            Put_Line (Lib_Dir.all & " already exists");
            Exit_Program (E_Fatal);
         end if;

         if ADC_File = null then
            Put_Line ("No configuration file specified");
            Exit_Program (E_Fatal);
         end if;

         if not Is_Regular_File (ADC_File.all) then
            Put_Line (ADC_File.all & " doesn't exist");
            Exit_Program (E_Fatal);
         end if;

         Create_Block : declare
            Success        : Boolean;
            Make_Args      : Argument_List (1 .. 9);
            C_Lib_Dir      : String := Lib_Dir.all & Ascii.Nul;
            C_ADC_File     : String := ADC_File.all & Ascii.Nul;
            F_ADC_File     : String (1 .. max_path_len);
            F_ADC_File_Len : Integer := max_path_len;
            Include_Dirs   : Integer;
            Object_Dirs    : Integer;
            Include_Dir    : array (Integer range 1 .. 256) of String_Access;
            Object_Dir     : array (Integer range 1 .. 256) of String_Access;
         begin
            --
            --  Create the new top level library directory
            --
            if not Is_Directory (Lib_Dir.all) then
               Create_Directory (C_Lib_Dir'Address, 8#755#);
            end if;
            full_name (C_ADC_File'Address, F_ADC_File'Address);
            for I in 1 .. max_path_len loop
               if F_ADC_File (I) = Ascii.Nul then
                  F_ADC_File_Len := I - 1;
                  exit;
               end if;
            end loop;

            --
            --  Make a list of the default library source and object
            --  directories.  Usually only one, except on VMS where
            --  there are two.
            --
            Include_Dirs := 1;
            Get_Next_Dir_In_Path_Init
              (String_Access (Include_Dir_Default_Name));
            loop
               Include_Dir (Include_Dirs)
                 := Get_Next_Dir_In_Path
                   (String_Access (Include_Dir_Default_Name));
               exit when Include_Dir (Include_Dirs) = null;
               Include_Dirs := Include_Dirs + 1;
            end loop;
            Include_Dirs := Include_Dirs - 1;

            Object_Dirs := 1;
            Get_Next_Dir_In_Path_Init
              (String_Access (Object_Dir_Default_Name));
            loop
               Object_Dir (Object_Dirs)
                 := Get_Next_Dir_In_Path
                   (String_Access (Object_Dir_Default_Name));
               exit when Object_Dir (Object_Dirs) = null;
               Object_Dirs := Object_Dirs + 1;
            end loop;
            Object_Dirs := Object_Dirs - 1;

            --
            --  "Make" an alternate sublibrary for each default sublibrary.
            --
            for Dirs in 1 .. Object_Dirs loop

               Make_Args (1)
                 := new String'("-C");
               Make_Args (2)
                 := new String'(Lib_Dir.all);

               Make_Args (3)
                 := new String'("ADA_INCLUDE_PATH="
                                & Include_Dir (Dirs).all);

               Make_Args (4)
                 := new String'("ADA_OBJECTS_PATH="
                                & Object_Dir (Dirs).all);

               Make_Args (5)
                 := new String'("GNAT_ADC_FILE="
                                & F_ADC_File (1 .. F_ADC_File_Len));

               Make_Args (6)
                 := new String'("LIBRARY_VERSION="
                                 & '"' & Library_Version & '"');

               Make_Args (7)
                 := new String'("-f");
               Make_Args (8)
                 := new String'(Object_Dir (Dirs).all & "Makefile.lib");

               Make_Args (9)
                 := new String'("create");

               Make_Path := Locate_Exec_On_Path (Make);
               Put (Make);
               for I in 1 .. Make_Args'Last loop
                  Put (" ");
                  Put (Make_Args (I).all);
               end loop;
               New_Line;
               Spawn (Make_Path.all, Make_Args, Success);
            end loop;
         end Create_Block;

      when Set =>
         --
         --  Validate arguments.
         --
         if Lib_Dir = null then
            Put_Line ("No library directory specified");
            Exit_Program (E_Fatal);
         end if;

         if not Is_Directory (Lib_Dir.all) then
            Put_Line (Lib_Dir.all & " doesn't exist");
            Exit_Program (E_Fatal);
         end if;

         if ADC_File = null then
            Put_Line ("No configuration file specified");
            Exit_Program (E_Fatal);
         end if;

         if not Is_Regular_File (ADC_File.all) then
            Put_Line (ADC_File.all & " doesn't exist");
            Exit_Program (E_Fatal);
         end if;

         --
         --  Give instructions.
         --
         Put_Line ("GNAT Librarian SET not yet fully implemented, "
           & "to work around:");
         Put_Line ("Copy the contents of "
           & ADC_File.all & " into your GNAT.ADC file");
         Put_Line ("and use GNAT Make qualifier /OBJECT_SEARCH=("
           & Lib_Dir (Lib_Dir'First .. Lib_Dir'Last - 1) & ".DECLIB],"
           & Lib_Dir (Lib_Dir'First .. Lib_Dir'Last - 1) & ".ADALIB])");

      when Delete =>
         --
         --  Give instructions.
         --
         Put_Line ("GNAT Librarian DELETE not yet implemented.");
         Put_Line ("Use appropriate system tools to remove library");

   end case;

end GnatLbr;
