;; hilit-LaTeX.el - Enhancements for LaTeX highlighting w/ hilit19.el or hl319
;;
;; Copyright (C) 1994 Peter S. Galbraith
 
;; Author:    Peter S. Galbraith <rhogee@bathybius.meteo.mcgill.ca>
;; Created:   03 March 1994
;; Version:   1.14 (21 March 95)
;; Keywords:  LaTeX, hilit19, hl319, highlight, auctex 

;; Everyone is granted permission to copy, modify and redistribute this
;; file provided:
;;   1. All copies contain this copyright notice.
;;   2. All modified copies shall carry a prominant notice stating who
;;      made modifications and the date of such modifications.
;;   3. The name of the modified file be changed.
;;   4. No charge is made for this software or works derived from it.
;;      This clause shall not be construed as constraining other software
;;      distributed on the same medium as this software, nor is a
;;      distribution fee considered a charge.

;; ----------------------------------------------------------------------------
;;; Commentary:

;; New versions of this package (if they exist) may be found at:
;;  ftp://bathybius.meteo.mcgill.ca/pub/users/rhogee/elisp/hilit-LaTeX.el

;; Description:
;;  This package enhances hilit19 (or hl319) highlighting patterns for LaTeX.
;;  hl319.el (or hl319-fsfmacs.el) is an enhanced version of hilit19.  
;;  I suggest that you get it:
;;   at ftp.netcom.com in /pub/st/stig/src/elisp/Beta/hl319-fsfmacs.el.gz
;;   (I could never get Mosaic to connect to it...)
;;  It is faster than hilit19 and allows on-the-fly highlighting

;; Installation instructions:
;;
;;  All you need to do is add this line to your .emacs file after you have 
;;  loaded hl319 or hilit19: 
;;
;;    (if window-system
;;        (require 'hilit-LaTeX))
;;
;;  But default, unknown LaTeX commands are highlighted (in light grey when
;;  in light background).  I like this feature, but it might slow down 
;;  hilit-LaTeX too much for you.  To disable this feature, add the following
;;  to your .emacs file *before* you require (or load) hilit-LaTeX.
;;
;;    (setq hilit-LaTeX-commands nil)
;;
;;  Additional highlights are available for AmS-LaTeX commands by setting
;;
;;    (setq hilit-AmSLaTeX-commands t)
;;
;;  And some multi-lingual quoted strings are highlighted by setting:
;;
;;    (setq hilit-multilingual-strings t)
;;
;;  These variables must all ny set *before* hilit-LaTeX is loaded.

;; To add (?)
;;  - optinal highlighting of [options] and {arguments} after unknown commands.

;; (Following LCD Archive Entry included for your later use --- 
;;  Not yet submitted but perhaps you can get your next version there)
;; LCD Archive Entry:
;; hilit-LaTeX|Peter Galbraith|rhogee@bathybius.meteo.mcgill.ca
;; |Enhancements for LaTeX highlighting w/ hilit19.el or hl319.el
;; Mar 21 1995|1.14||
;;
;; ----------------------------------------------------------------------------
;;; Change log:
;; V1.00 01Mar94 Peter S Galbraith - Created
;; V1.01 14Mar94 PSG - Added {\sc } 
;;                   - Added fix for \[ ]\ excluding \\[len]
;; V1.02 15Mar94 PSG - Real dollar sign \$ won't hilit like math mode.
;; V1.03 16Mar94 Jonathan Katz <jkatz@weber.ucsd.edu>
;;                   - lateX2e-mode, \documentclass, \usepackage
;; V1.04 16Mar94 PSG - Added \sc as a declaration.
;; V1.05 17Mar94 Andrew Swann <swann@imada.ou.dk>
;;                   - ams-latex-mode, \eqref , \title[]{}, \numberwithin
;;               PSG - \[any]cite command that user makes up.
;; V1.06 06Apr94 Martin Maechler <maechler@stat.math.ethz.ch>
;;                   - \cite[anyword] commands that user makes up.        
;; V1.07 04Jan95 PSG - Added hilit-inside-bracket-region for LaTeX2e
;;                     commands \textrm \textbf and \emph.
;; V1.08 16Feb95 PSG - and support for many other LaTeX2e font commands.
;;                   - Fixed hl319 on-the-fly highlighting (sort of)  
;; V1.09 27Feb95 Andrew Swann <swann@imada.ou.dk>
;;                   - Fixed bug where slash was always hilighted.
;; V1.10 01Mar95 Andrew Swann <swann@imada.ou.dk> and PSG.
;;                   - Fixed embedded (quoted) dollar sign within formula.
;; V1.11 02Mar95 PSG - Bug report by Richard Staton <stanton@haas.berkeley.edu>
;;                   - Fixed highlighted character before %comment
;;                   - Fixed highlight for % in first column
;; V1.12 03Mar95 PSG - Allow options to cite commands.
;; V1.13 09Mar95 PSG - Suggestions from kevin@rodin.wustl.edu (Kevin Ruland)
;;                      extras for ams-latex-mode, equation environment, 
;;                      newpage-type commands.
;;                   - hilit-LaTeX-commands variable and multilingual strings.
;; V1.14 21Mar95 PSG   bug reported by Martin Maechler 
;;                      <maechler@stat.math.ethz.ch>
;;                     Fixed infinite loop on equation environments that fail
;;                     to have matching end (\begin{equation} \end {equation})
;; ----------------------------------------------------------------------------
;;; Code:

(require 'hilit19)

(defvar hilit-AmSLaTeX-commands nil
  "*Set to t if you want to hightlight AmSLaTeX commands.")

(defvar hilit-multilingual-strings nil
  "*Set to t if you want to hightlight multilingual quoted strings.
Highlights:  \"`german\"', \"< french \">, << french >> and + 8-bit french ;.")

(defvar hilit-LaTeX-commands t
  "*Set to nil if you don't want to highlight unknown LaTeX commands")

(defvar hilit-on-the-fly-in-use nil
  "Used by hilit-LaTeX to determine if on-the-fly hilighting is in use")

;; I need to modify hl319.el's on-the-fly highlighter so that it
;; tells me on-the-fly highlighting is in use.
(defun hilit-rehighlight-changed-lines (st en len)
  "Quietly rehighlight just this line.
Useful as an after change hook in VM/gnus summary buffers and dired buffers."
  (save-match-data
    (let ((hilit-on-the-fly-in-use t))
      ;; (> (- en st) hilit-max-change-rehighlight)
      (hilit-rehighlight-region st en 'quietly))))

 (defun hilit-bracket-region (open)
  "Find region within curly brackets for hilit pattern.
ARG is pattern for beginning of pattern and ends with {.
It cannot be white space.  
Patterns ends simply at the matching closing bracket.

Used for patterns like:
 \\chapter{stuff with matching {} ending with }"
  (if (re-search-forward open nil t)
      (let ((here (point))
            (the-start (match-beginning 0)))
        (backward-char 1)               ; point is on bracket
        (if hilit-on-the-fly-in-use
            ;; if hl319's on-the-fly hilighting is in use then we can't use 
            ;; forward-list because it uses the built-in scan-lists and will 
            ;; return an error if there is no matching bracket yet.
            (re-search-forward "}" 
                               (progn (save-excursion (end-of-line)(point))) 
                               1)
          (forward-list 1))
        (cons the-start (point)))))

(defun hilit-inside-bracket-region (open)
  "Find region within curly brackets for hilit pattern.
ARG is pattern for beginning of pattern and ends with {.
It cannot be white space.  
Patterns ends simply at the matching closing bracket.

Used for patterns like:
 \\textbf{only stuff within bracket is highlited}"
  (if (re-search-forward open nil t)
      (let ((the-start (point)))
        (backward-char 1)               ; point is on bracket
        (if hilit-on-the-fly-in-use
            (re-search-forward "}" 
                               (progn (save-excursion (end-of-line)(point))) 1)
          (forward-list 1))
        (backward-char 1)
        (cons the-start (point)))))

(defun hilit-inside-environment (open)
  "Find region within curly brackets for hilit pattern.
ARG is pattern for \\begin{something}."
  (if (re-search-forward open nil t)
      (let ((the-start (progn (forward-line 1)(point))) ;Start on next line
            (the-end))
        (re-search-backward "\\\\begin{\\([^}]+\\)}" nil t)
        (setq the-end (buffer-substring (match-beginning 1)(match-end 1)))
        (if hilit-on-the-fly-in-use     ;Search no more than 10 lines away
            (if (search-forward (concat "\\end{" the-end "}") 
                                (save-excursion (forward-line 10)(point)) t)
                (cons the-start (progn (forward-line -1)(point)))
              (cons the-start the-start))
          (if (search-forward (concat "\\end{" the-end "}") nil t)
              (cons the-start (match-beginning 0))
            (end-of-line)               ;Move off the \begin{}
            (cons the-start the-start))))))

(defun hilit-bracket-wysiwyg (open)
  "Find region within curly brackets for hilit pattern.
ARG is pattern for beginning of pattern and starts with {.
Patterns ends simply at the matching closing bracket.

Used for patterns like:
 {\em stuff with matching {} ending with }"
  (if (re-search-forward open nil t)
      (let ((the-start (match-beginning 0)))
        (goto-char the-start)
        (if hilit-on-the-fly-in-use
            (re-search-forward "}" 
                               (progn (save-excursion (end-of-line)(point))) 1)
          (forward-list 1))
        (cons the-start (point)))))

(hilit-set-mode-patterns
 '(LaTeX-mode japanese-LaTeX-mode slitex-mode SliTeX-mode japanese-SliTeX-mode 
              FoilTeX-mode latex-mode latex2e-mode ams-latex-mode)
 (append
  '(("\\(^\\|[^\\]\\)\\(%.*\\)$" 2 comment)) ; comments
  (cond 
   (hilit-AmSLaTeX-commands
    '(("\\\\\\(\\(no\\)?pagebreak\\|\\(new\\|clear\\(double\\)?\\)page\\|enlargethispage\\|\\(no\\)?linebreak\\|newline\\|-\\|displaybreak\\|allowdisplaybreaks\\)"
     nil error)
      
      ("\\\\\\(\\(\\(text\\)?\\(rm\\|sf\\|tt\\|bf\\|md\\|it\\|sl\\|sc\\|up\\|em\\|emph\\)\\(series\\|family\\|shape\\)?\\)\\|\\(appendix\\|tableofcontents\\|listoffigures\\|listoftables\\|normalsize\\|small\\|footnotesize\\|scriptsize\\|tiny\\|large\\|Large\\|LARGE\\|huge\\|Huge\\|raggedright\\|makeindex\\|makeglossary\\|pmb\\|boldsymbol\\)\\)\\b" 
     nil decl)

    ;; various declarations/definitions
    ("\\\\\\(maketitle\\|setlength\\|settowidth\\|addtolength\\|setcounter\\|addtocounter\\)\\b" 
     nil define)
    ("\\\\\\([a-z]+box\\|text\\|intertext\\)\\b" nil keyword)))

   (t
    '(("\\\\\\(\\(no\\)?pagebreak\\|\\(new\\|clear\\(double\\)?\\)page\\|enlargethispage\\|\\(no\\)?linebreak\\|newline\\|-\\)"
     nil error)

      ("\\\\\\(\\(\\(text\\)?\\(rm\\|sf\\|tt\\|bf\\|md\\|it\\|sl\\|sc\\|up\\|em\\|emph\\)\\(series\\|family\\|shape\\)?\\)\\|\\(appendix\\|tableofcontents\\|listoffigures\\|listoftables\\|normalsize\\|small\\|footnotesize\\|scriptsize\\|tiny\\|large\\|Large\\|LARGE\\|huge\\|Huge\\|raggedright\\|makeindex\\|makeglossary\\)\\)\\b" 
       nil decl)

      ;; various declarations/definitions
      ("\\\\\\(maketitle\\|setlength\\|settowidth\\|addtolength\\|setcounter\\|addtocounter\\)\\b" 
       nil define)
      ("\\\\[a-z]+box\\b" nil keyword))))


  '(("``" "''" string))
  (and hilit-multilingual-strings
       '(("\"<" "\">" string)
         ("\"`" "\"'" string)
         ("<<" ">>" string)
         ("+" ";" string)))
    
  '(("\\\\\\(item\\(\\[.*\\]\\)?\\|\\\\\\(\*\\)?\\)" nil label) ;label, \\
    ("\\(^\\|[^\\\\]\\)\\(&+\\)" 2 label)     ; & within tables and such

    ;; "wysiwyg" emphasis
    (hilit-bracket-wysiwyg 
     "{\\\\\\(text\\)?\\(em\\|it\\|sl\\)\\(shape\\|family\\|series\\)?\\b" italic)
    (hilit-bracket-wysiwyg              ;Removed rm from list
     "{\\\\\\(text\\)?\\(bf\\|md\\|sc\\|up\\|tt\\|sf\\)\\(shape\\|family\\|series\\)?\\b" 
     bold))
    
  (cond 
   (hilit-AmSLaTeX-commands
    '((hilit-inside-bracket-region         ;also \boldsymbol{<>}, \pmb{<>},
       "\\\\\\(boldsymbol\\|pmb\\|text\\(bf\\|md\\|rm\\|sf\\|tt\\|sc\\|up\\)\\){" 
       bold)

      (hilit-bracket-region 
       "\\\\\\(\\(page\\|v\\|eq\\)?ref\\|tag\\|eqref\\|label\\|index\\|glossary\\|[A-Za-z]*cite[A-Za-z]*\\(\\[.*\\]\\)?\\){" 
       crossref)                          ; added \tag{} \eqref{}
      ("\\\\notag\\b" nil crossref)       ; and \notag

      (hilit-inside-environment
       "\\\\begin{\\(equation\\|eqnarray\\|gather\\|multline\\|align\\|x*alignat\\)\\(\*\\)?}" 
       glob-struct)))
   (t
    '((hilit-inside-bracket-region      ; Removed rm from list
       "\\\\\\(text\\(bf\\|md\\|sf\\|tt\\|sc\\|up\\)\\){" 
       bold)

      (hilit-bracket-region 
       "\\\\\\(\\(page\\|v\\)?ref\\|label\\|index\\|glossary\\|[A-Za-z]*cite[A-Za-z]*\\(\\[.*\\]\\)?\\){" 
       crossref)                            ; things that cross-reference

      (hilit-inside-environment "\\\\begin{\\(equation\\|eqnarray\\)\\(\*\\)?}"
                                glob-struct))))


  ;; FIXME: the following doesn't work.  Tried with nil and default. 
   ;(hilit-inside-bracket-region "\\\\\\(intertext\\|text\\|mbox\\){" default)
  ;; \intertext{<arbitrary text>} will set normal text. 
  ;;  And within any math mode \text{<>} acts like a 'smart' \mbox{}.

  '((hilit-inside-bracket-region "\\\\\\(text\\(it\\|sl\\)\\|emph\\){" italic)

    ("\\\\("  "\\\\)" glob-struct)           ; \( \)
    ("[^\\\\\\(\\\\begin{avm}\\)]\\\\\\[" "\\\\\\]" 
     glob-struct) ; \[ \] but not \\[len] or \begin{avm}\[

;;; ("[^$\\]\\($\\($[^$]*\\$\\|[^$]*\\)\\$\\)" 1 formula); '$...$' or '$$...$$'
   ;;   ^^ but not \$102.54 is expensive for a car wash.
   ;;      Above formula breaks when excaped dollar sign are presnt within
   ;;      equation.  $ t = \$45.50 $.  I think the one below works ok:
    ("\\(^\\|[^$\\]\\)\\($\\($\\([^$]\\|\\\\\\$\\)*\\$\\|\\([^$]\\|\\\\\\$\\)*\\)\\$\\)" 
  ;;; ^^^^^^^      ^^^ -> $ test $ will be highlighted on-the-fly at line-begin
     2 glob-struct) ; '$...$' or '$$...$$'

    ;; things that bring in external files
    (hilit-bracket-region  "\\\\\\(include\\|input\\|bibliography\\){" include)
    ;; (re-)define new commands/environments/counters

    (hilit-bracket-region 
    "\\\\\\(re\\)?new\\(environment\\|command\\|length\\|theorem\\|counter\\){"
     defun)

    (hilit-bracket-region 
     "\\\\\\(\\(v\\|h\\)space\\|footnote\\(mark\\|text\\)?\\|\\(sub\\)*\\(paragraph\\|section\\)\\|chapter\\|part\\)\\(\*\\)?\\(\\[.*\\]\\)?{"
     keyword)

    (hilit-bracket-region 
     "\\\\\\(title\\|author\\|date\\|thanks\\|address\\)\\(\\[.*\\]\\)?{" 
     define)

    (hilit-bracket-region 
     "\\\\\\(\\(\\this\\)?pagestyle\\|pagenumbering\\|numberwithin\\|begin\\|end\\|nofiles\\|includeonly\\|bibliographystyle\\|document\\(style\\|class\\)\\|usepackage\\)\\(\\[.*\\]\\)?{" 
     decl)
   
    (hilit-bracket-region "\\\\caption\\(\\[[^]]*\\]\\)?{" warning))
  
  (and hilit-LaTeX-commands
       '(("\\(^\\|[^\\\\]\\)\\(\\\\[a-zA-Z\\\\]+\\)" 2 summary-killed)))))

(provide 'hilit-LaTeX)
;;; hilit-LaTeX.el ends here

