#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <netdb.h>
#include <dirent.h>
#include <string.h>
#include <sys/stat.h>
#include <math.h>
#include "config.h"
#include "both.h"
#include "suck.h"
#include "suckutils.h"

#ifdef KILLFILE
#include "killfile.h"
#endif

#ifdef TIMER
#include "timer.h"
#endif

#ifdef MYSIGNAL 
#include <signal.h>
#endif

#ifdef CHECK_HISTORY
#include "chkhistory.h"
#endif

/* function prototypes */
int get_message_index(PMaster);
int get_articles(PMaster);
int get_one_article(PMaster, int);
int get_announcements(PMaster);
int do_supplemental(PMaster);
int do_innbatch(int, char *);
int do_rnewsbatch(int, char *, long);
PList allocnode(char *, int);
int build_list(PMaster, char *, int *, int);
int restart_yn(PMaster);

#ifdef MYSIGNAL
void sighandler(int);

/*------------------------------------------*/
int GotSignal = FALSE;		
/* the only static variable allowed, it's   */
/* the only graceful way to handle a signal */
/*------------------------------------------*/
#endif

enum { BATCH_FALSE, BATCH_INNXMIT, BATCH_RNEWS };		/* poss values for batch variable */
enum { MANDATORY_YES = 'M' , MANDATORY_OPTIONAL = 'O' };	/* do we have to download this article */
enum { STATUS_STDOUT, STATUS_STDERR };
enum { RESTART_YES, RESTART_NO, RESTART_ERROR };
/*------------------------------------------------------------------*/
void main(int argc, char *argv[]) {
	
	struct hostent *hi;
	Master master;
	PList temp;
	char *lockfile;

	char *host = NULL, *batchfile = NULL, *status_file = NULL, *ptr;
	int loop, batch = BATCH_FALSE, retval = RETVAL_OK, always_batch = FALSE;
	long rnews_size = 0L;

	master.head = master.curr = NULL;
	master.nritems =  master.itemon = master.nrgot = 0;
	master.MultiFile = FALSE;
	master.msgs = stdout;
	master.sockfd = -1;
	master.status_file = FALSE;	/* are status messages going to a file */
	master.do_killfile = TRUE;
	master.do_chkhistory = TRUE;

	/* allow no args, only the hostname, or hostname as first arg */
	host = getenv("NNTPSERVER");		/* the default */
	switch(argc) {
	  case 1:
		break;
	  case 2:   
		host = argv[1];
		break;
	  default:
		loop = 1;
		if(argv[1][0] != '-') {
			host = 	argv[loop++];
		}			
		for(;loop<argc;loop++) {
			if(argv[loop][0] == '-') {
				switch(argv[loop][1]) {
				  case 'a':	/* if we have downloaded at least one article, then batch up */
						/* even on errors */
					always_batch = TRUE;
					break;
				  case 'h':
					host = (loop+1 == argc) ? getenv("NNTPSERVER") : argv[++loop];
					break;	
				  case 'm':
					master.MultiFile = TRUE;
					break;
				  case 'r':
					if(loop+1 == argc) {
						error_log(ERRLOG_REPORT, "No rnews batch file size provided\n");
						retval = RETVAL_ERROR;
					}
					else {
						rnews_size = atol(argv[++loop]);
					}
					break;
				  case 'p':
					if(loop+1 == argc) {
						error_log(ERRLOG_REPORT, "No postfix provided\n");
						retval = RETVAL_ERROR;
					}
					else {
						full_path(FP_SET_POSTFIX, FP_NONE, argv[++loop]);
					}
					break;
				  case 'd':	/* change one of the directory names */
					if(loop+1 == argc) { 
						error_log(ERRLOG_REPORT, "No directory name provided\n");
						retval = RETVAL_ERROR;
					}
					else {
						ptr = NULL;
						switch (argv[loop][2]) {
						  case 't':
							ptr = full_path(FP_SET, FP_TMPDIR, argv[++loop]);
							break;
						  case 'd':
							ptr = full_path(FP_SET, FP_DATADIR, argv[++loop]);
							break;
						  case 'm':
							ptr = full_path(FP_SET, FP_MSGDIR, argv[++loop]);
							break;
						}
						if(ptr == NULL) { 
							error_log(ERRLOG_REPORT, "Invalid directory\n");
							retval = RETVAL_ERROR;
							break;
						}
					}
					break;		 
				  case 'b':	/* batch file implies MultiFile Mode */
					switch (argv[loop][2]) {
					  case 'i':
						batch = BATCH_INNXMIT;
						master.MultiFile = TRUE;		
						break;
					  case 'r':
						batch = BATCH_RNEWS;
						master.MultiFile = TRUE;
						break;
					  default:
						error_log(ERRLOG_REPORT, "Invalid Batch arg: %s\n", argv[loop]);
						retval = RETVAL_ERROR;
						break;
				  	}
					if(retval == RETVAL_OK && loop+1 == argc) {
						error_log(ERRLOG_REPORT, "No Batch file name provided\n");
						retval = RETVAL_ERROR;
					}
					else {
						batchfile = argv[++loop];
					}
					break;
				  case 'e':	/* use default error log path */
					error_log(ERRLOG_SET_FILE, ERROR_LOG);
					break;
				  case 'E':	/* error log path */
					if(loop+1 == argc) { 
						error_log(ERRLOG_REPORT, "No Error Log name provided\n");
						retval = RETVAL_ERROR;
					}
					else {
						error_log(ERRLOG_SET_FILE, argv[++loop]);
					}
					break;
				  case 's':	/* use default status log name */
					status_file = STATUS_LOG;
					break;
				  case 'S':	/* status log path */
					if(loop+1 == argc) {
						error_log(ERRLOG_REPORT, "No Status Log name provided\n");
						retval = RETVAL_ERROR;
					}
					else {
						status_file = argv[++loop];
					}
					break;
				  case 'V': 	/* show version number */
				  	error_log(ERRLOG_REPORT, "Suck version %d.%d.%d\n",SUCKVER, SUCKVER_MINOR, SUCKVER_PATCH);
					retval = RETVAL_VERNR;	/* so we don't do anything else */
				  	break;
				  case 'K':
					master.do_killfile = FALSE;
					break;
				  case 'H':
					master.do_chkhistory = FALSE;
					break;
				  default:
					error_log(ERRLOG_REPORT, "Invalid argument: %s\n", argv[loop]);
					retval = RETVAL_ERROR;
					break;
				}			
			}
			else {
				error_log(ERRLOG_REPORT, "Invalid argument: %s\n", argv[loop]);
				retval = RETVAL_ERROR;
			}	
		}
		break;
	}
	if(retval == RETVAL_OK) {
		
#ifdef LOCKFILE
		/* this has to be here since we use full_path() to get path for lock file */
		/* and it isn't set up until we process the args above. */
		if(do_lockfile() != RETVAL_OK) {
			exit(-1);
		}
#endif

#ifdef MYSIGNAL
		signal(MYSIGNAL, sighandler); 	/* set up signal handler */
		signal_block(MYSIGNAL_SETUP);	/* set up sgetline() to block signal */ 
#endif

		/* set up status log, if none specified or unable to open status log */
		/* then  use stdout or stderr */

		if(status_file != NULL) {
			/* okay attempt to open it up */
			if((master.msgs = fopen(status_file, "a")) == NULL) {
				MyPerror("suck: status log");
			}
			else {
				master.status_file = TRUE;
			}
		}
		if(master.status_file == FALSE) {
			/* if not in multifile mode, all status msgs MUST go to stderr to not mess up articles */
			master.msgs = ( master.MultiFile == FALSE) ? stderr : stdout ;
		}
					
		master.sockfd = connect_to_nntphost( host, &hi, master.msgs);
		if(master.sockfd < 0 ) {
			retval = RETVAL_ERROR;
		}
		else if(get_announcements(&master) < RETVAL_OK) {
			retval = RETVAL_ERROR;
		}
		else {
			if((loop = restart_yn(&master)) == RESTART_YES) { 
  				fprintf(master.msgs, "Initiating restart at article %d of %d\n", master.itemon, master.nritems);
				retval = get_articles(&master);
			}
			else if(loop == RESTART_ERROR) {
				retval = RETVAL_ERROR;
			}
			else if(get_message_index(&master) < RETVAL_OK) {
				retval = RETVAL_ERROR;
			}
			else if(master.nritems == 0) {
 				fprintf(master.msgs,"No articles.\r\n");
				retval = RETVAL_NOARTICLES;
			}
			else {
				retval = get_articles(&master);
			}	
			sputline( master.sockfd, "quit\r\n" );
		}
		if(master.sockfd >= 0) {
			close(master.sockfd);
			fprintf(master.msgs,"Closed connection to %s\n",hi->h_name);
		}
		if((retval == RETVAL_OK || always_batch == TRUE) && master.nrgot > 0) {
			switch(batch) {
			  case BATCH_INNXMIT:
				fprintf(master.msgs,"Building Inn Batch File\n");
				do_innbatch(master.nritems, batchfile);
				break;
			  case BATCH_RNEWS:
				fprintf(master.msgs,"Building Rnews Batch File(s)\n");
				do_rnewsbatch(master.nritems, batchfile, rnews_size);
				break;
			  default:
				break;
			}
		}
		/* close out status log */
		if(master.msgs != NULL && master.msgs != stdout && master.msgs != stderr) {
			fclose(master.msgs);
		}
		if(master.head != NULL) {
			/* clean up memory */
			master.curr = master.head;
			while(master.curr != NULL) {
				temp = (master.curr)->next;
				free_one_node(master.curr);
				master.curr = temp;
			}
		}		
#ifdef LOCKFILE
		lockfile = full_path(FP_GET, FP_TMPDIR, N_LOCKFILE);	
		if(lockfile != NULL) {
			unlink(lockfile);
		}
#endif	
	}
	exit(retval);
}
/*--------------------------------------------------------------------*/
int get_message_index(PMaster master) {

	int count,low,high,response,lastread,retval,i,removed;
	char *sp,*inbuf,buf[MAXLINLEN],group[512];
	FILE *ifp,*tmpfp;

	retval = RETVAL_OK;
	ifp = tmpfp = NULL;
	removed = 0;
	if((ifp = fopen(full_path(FP_GET, FP_DATADIR, N_OLDRC), "r" )) == NULL) {
		MyPerror(full_path(FP_GET, FP_DATADIR, N_OLDRC));
		retval = RETVAL_ERROR;
	}
	else if((tmpfp = fopen(full_path(FP_GET, FP_TMPDIR, N_NEWRC), "w" )) == NULL) {
		MyPerror(full_path(FP_GET, FP_TMPDIR, N_NEWRC));
		retval = RETVAL_ERROR;
	}
	while(retval == RETVAL_OK && fgets(buf, MAXLINLEN-1, ifp) != NULL) {
		if(buf[0] == SUCKNEWSRC_COMMENT_CHAR) {
			/* skip this group */
			fputs(buf, tmpfp);
			fprintf(master->msgs, "Skipping Line: %s", buf);
			continue;
		}
		if(sscanf(buf, "%s %d\n", group, &lastread) != 2) {
			error_log(ERRLOG_REPORT, "Invalid Line: %s", buf);
			continue;
		}
  		sprintf(buf,"group %s\r\n",group);
		sputline(master->sockfd,buf);
		i = sgetline(master->sockfd,&inbuf);
#ifdef DEBUG2
		do_debug("\ngroup %s\n%s",group,inbuf);
#endif
		if(i < 0) {
			retval = RETVAL_ERROR;
		}
		else if((sp = number(inbuf, &response)) == NULL) {
			error_log(ERRLOG_REPORT,"Protocol error!\r\n");
			retval = RETVAL_ERROR;
		}
		else {
			if(response != 211) {
				/* handle group not found - rewrite line in newrc */
				if(response == 411) {
					error_log(ERRLOG_REPORT,"GROUP <%s> not found on host\r\n", group);
					fprintf(tmpfp, "%s %d\n", group, lastread);
				}
				else {
					error_log(ERRLOG_REPORT,"GROUP <%s>, unexpected response %d\r\n", group, response );
				}
   				continue; /* next newsgroup */
			}
			sp = number(sp, &count);
  			sp = number(sp, &low);
			sp = number(sp, &high);
  			fprintf(tmpfp, "%s %d\n", group, high);
  			
  			/* add a sanity check in case remote host changes its numbering scheme */
			/* the > 0 is needed, since if a nnrp site has no article it will reset */
			/* the count to 0.  So not an error */
  			if(lastread > high && high > 0) {
  				fprintf(master->msgs,"%s...High Article Nr is low, did host reset its counter?",group);
  			}

			/* this has to be >= 0 since if there are no article on server high = 0 */
			/* so if we write out 0, we must be able to handle zero as valid lastread */
			if (lastread < high && lastread >= 0) {
   				if(lastread < low) {
					lastread = low - 1;
				}
   
   				sprintf(buf, "xhdr Message-ID %d-\r\n", lastread+1);
   				fprintf(master->msgs,"%s...articles %d-%d\r\n", group, lastread+1, high);
   				sputline(master->sockfd, buf);
				i = sgetline(master->sockfd, &inbuf);
				number(inbuf, &response);
				if(i < 0) {
					retval = RETVAL_ERROR;
				}
				else if (response == 221) {
					do {
     						if(sgetline(master->sockfd, &inbuf) < 0) {
							retval = RETVAL_ERROR;
						}
						else if (*inbuf != '.' ) {
      							for(sp = inbuf; *sp != '\0' && *sp != '<'; sp++) {
								 /* find begin of index */
								if (sp== inbuf+(MAXLINLEN-4)) {
									continue; /* the NNTP server tried to pull your leg */
								}
							}
							retval = build_list(master, inbuf, &removed, MANDATORY_OPTIONAL);
						}
					} while (retval == RETVAL_OK && strcmp(inbuf, ".\n")!=0);
				} /* end if response */
   				else {
					error_log(ERRLOG_REPORT,"Unexpected response %d getting IDs\r\n", response);
					retval = RETVAL_ERROR;
				}
			} /* end if lastread */
		} /* end else */
	} /* end while */
	if(retval == RETVAL_OK) {
		fprintf(master->msgs, "%d Articles to download, %d dupes removed\n", master->nritems, removed);
		retval = do_supplemental(master);
	}
	if(tmpfp != NULL) {
		fclose(tmpfp);
	}
	if(ifp != NULL) {
		fclose(ifp);
	}
	return retval;
}
/*-----------------------------------------------------------*/
int get_articles(PMaster master) {

	int retval, logcount, i;
	FILE *logfp,*ifp;

#ifdef KILLFILE
	PKillStruct killp = NULL;
#endif

	retval = RETVAL_OK;
	logfp = ifp = NULL;

	logcount = log10((double)master->nritems) + 1;	/* how many digits wide is the articleCount */


#ifdef KILLFILE
	if(master->do_killfile == TRUE) {
		killp = parse_killfile();
	}
#endif		
	if(master->MultiFile == TRUE && checkdir(full_path(FP_GET, FP_MSGDIR, NULL)) == FALSE) {
		retval = RETVAL_ERROR;
	}
	else {
	 	if((logfp = fopen(full_path(FP_GET, FP_TMPDIR, N_RESTART), "w" )) == NULL) {
			MyPerror(full_path(FP_GET, FP_TMPDIR, N_RESTART));
			retval = RETVAL_ERROR;
		}
 		else {
			/* do this here so we write out the restart for all articles */
			/* already downloaded 	*/
			master->curr = master->head;
			for(i=0;i<master->itemon;i++) {
				fprintf(logfp,"1");
				master->curr = (master->curr)->next;
			}
			fflush(logfp);

#ifdef TIMER
			TimerFunc(TIMER_START, 0);
#endif
#ifdef MYSIGNAL
			while(retval == RETVAL_OK && master->curr != NULL && GotSignal == FALSE) {
#else
			while(retval == RETVAL_OK && master->curr != NULL) {
#endif
#ifdef DEBUG2
				do_debug("Article nr = %s mandatory = %c\n", (master->curr)->msgnr, (master->curr)->mandatory);
#endif
				/* write out restart */
				fputc('1',logfp);
  				fflush(logfp);
				master->itemon++;
				if(master->status_file == FALSE) {
					/* if we are going to a file, we don't want all of these articles printed */
#ifndef TIMER
  					fprintf(master->msgs, "%5d\r",master->nritems  - master->itemon);
#else 
					fprintf(master->msgs, "%5d %s\r", master->nritems - master->itemon, TimerFunc(TIMER_DISPLAY, 0));
#endif
					fflush(master->msgs);	/* so message gets printed now */
				}
#ifdef KILLFILE
				/* do we check for kill or not */
				retval = ((master->curr)->mandatory == MANDATORY_YES || killp == NULL) 
					? get_one_article(master, logcount) : get_one_article_kill(master, logcount, killp);
#else
				retval = get_one_article(master, logcount);
#endif
				master->curr = (master->curr)->next; 	/* get next article */
 
		 	} /* end while */
			fclose(logfp);
			if(retval == RETVAL_OK && master->nritems == master->itemon) {
				unlink(full_path(FP_GET, FP_TMPDIR, N_RESTART));
			}
#ifdef TIMER
			if(retval == RETVAL_OK) {
				fprintf(master->msgs,"%s\n", TimerFunc(TIMER_TOTALS, 0));
			}
#endif				
		}
		fclose(ifp);
	}
#ifdef KILLFILE
	free_killfile(killp);
#endif
	return retval;
}
/*------------------------------------------------------------*/
int get_announcements(PMaster master) {
	char *inbuf;
	int retval = RETVAL_OK;
		
	/* Get the announcement line */
	if(sgetline(master->sockfd, &inbuf) < 0) {
		retval = RETVAL_ERROR;
	}
	else {
		fprintf(master->msgs,"%s", inbuf );
#ifndef NNRP
		sputline(master->sockfd,"mode reader\r\n");	
		if( sgetline(master->sockfd,&inbuf) < 0) {	
			retval = RETVAL_ERROR;		  	
		}					
		else {					
			/* Again the announcement */
			fprintf(master->msgs,"%s",inbuf);
		}
#endif
	}
	return retval;
}
/*----------------------------------------------------------------------*/
/* add items from supplemental list to link list then write it out	*/
/*----------------------------------------------------------------------*/ 
int do_supplemental(PMaster master) {

	int retval, removed, oldkept;
	FILE *fp;
	PList curr; 
	char linein[MAXLINLEN+1];

	retval = RETVAL_OK;
	removed = 0;
	oldkept = master->nritems;		

	if((fp = fopen(full_path(FP_GET, FP_DATADIR, N_SUPPLEMENTAL), "r")) != NULL) {
		fprintf(master->msgs, "Processing Supplemental List\n");
		while(retval == RETVAL_OK && fgets(linein, MAXLINLEN, fp) != NULL) {
			if(linein[0] != '<') {
				error_log(ERRLOG_REPORT, "Supplemental Invalid line: %s, ignoring\n", linein);
			}
			else {
				retval = build_list(master, linein, &removed, MANDATORY_YES);
			}
		}
		fprintf(master->msgs,"Supplemental List Processed - %d articles added, %d articles removed\n", 
				master->nritems-oldkept, removed);
		fclose(fp);
	}
#ifdef CHECK_HISTORY
	if(master->do_chkhistory == TRUE) {
		chkhistory(master);
	}
#endif
	fprintf(master->msgs,"Total Articles to Download: %d\n", master->nritems); 

	if(master->head != NULL && master->nritems > 0) {
		/* now write out whatever list we have*/
		if((fp = fopen(full_path(FP_GET, FP_TMPDIR, N_SORTED), "w")) == NULL) {
			retval = RETVAL_ERROR;
			MyPerror(full_path(FP_GET, FP_TMPDIR, N_SORTED));
		}
		else {
			curr = master->head;
			while(curr != NULL && retval == RETVAL_OK) {
				if(fprintf(fp, "%s %c\n", curr->msgnr, curr->mandatory) == 0) {
					retval = RETVAL_ERROR;
					MyPerror(full_path(FP_GET, FP_TMPDIR, N_SORTED));
				}
				curr = curr->next;
			}
			fclose(fp);
		}
	}
	return retval;
}
/*--------------------------------------------------------------------------*/
int build_list(PMaster master, char *linein,  int *removed, int mandatory) {

	PList curr, prev;
	int match, retval;
	char *st_ptr, *ptr;

	retval = RETVAL_OK;
	st_ptr = linein;
	while(*st_ptr != '<' && *st_ptr != '\0') {
		st_ptr++;
	}
	ptr = st_ptr;
	while(*ptr != '>' && *ptr != '\0') {
		ptr++;
	}
	
	if(*st_ptr != '<' && *ptr != '>') {
		error_log(ERRLOG_REPORT, "Invalid Article NR, skipping: %s\n", linein);
	}
	else {
		*(ptr+1) = '\0';	/* ensure null termination */
		
		/* now see if it's in the linked list */
		curr = master->head;
		if(curr == NULL) {
			/* special case, 1st node */
			if((master->head = allocnode(st_ptr, mandatory)) == NULL) {
				retval = RETVAL_ERROR;
			}
			else {
				master->nritems++;
			}
		}
		else {	
			do {
				/* do a prelim 1st test so that we don't have to always call strcmp() */
				/* can't use first char, since its a <, so try second character */
				/* hopefully this will speed things up a bit. */
				match = (st_ptr[1] != curr->msgnr[1]) ? 1 : strcmp(st_ptr, curr->msgnr);
				prev = curr;
				curr = curr->next;			
			}
			while(match != 0 && curr != NULL);
			if(match == 0) {
				if(removed != NULL) {
					(*removed)++;
				}
				prev->mandatory = (char) mandatory;			
				/* this is needed in case we are in mandatory list */
				/* and have a dupe of one in optional list */
			}
			else {
				/* reached end of list */
				if((prev->next = allocnode(st_ptr, mandatory)) == NULL) {
					retval = RETVAL_ERROR;
				}
				else {
					master->nritems++;
				}
			}
		}
	}
	return retval;
}
/*-----------------------------------------------------------------------*/
PList allocnode(char *linein, int mandatory) {
	/* if allocate memory here, must free in free_one_node */
	PList ptr;

	if((ptr = malloc(sizeof(List))) == NULL) {
		error_log(ERRLOG_REPORT, "Out of Memory, aborting");
	}
	else if((ptr->msgnr = calloc(strlen(linein)+1, sizeof(char))) == NULL) {
		error_log(ERRLOG_REPORT, "Out of Memory, aborting");
		free(ptr);
	}
	else {
		strcpy(ptr->msgnr, linein);
		ptr->next = NULL;
		ptr->mandatory = (char) mandatory;
	}
	return ptr;
}
/*------------------------------------------------------------------------*/
void free_one_node(PList node) {

	free(node->msgnr);
	free(node);
}
/*----------------------------------------------------------*/
int do_innbatch(int articleCount, char *batchfile) {
	/* build batch file that contains article listing */
	/* needed by innxmit */
	/* this is done by searching thru MSGDIR to find files */
	/* which match our naming convention */

	int retval = RETVAL_OK;
	FILE *fptr;
	char tmp[20];
	DIR *dptr;
	struct dirent *entry;

#ifdef DEBUG2
	do_debug("BATCH FILE: %s\n", batchfile);
#endif
	if((fptr = fopen(batchfile, "w")) == NULL) {
		MyPerror(batchfile);
		retval = RETVAL_ERROR;
	}
	else if((dptr = opendir(full_path(FP_GET, FP_MSGDIR, ""))) == NULL) {
		MyPerror(full_path(FP_GET, FP_MSGDIR, ""));
		retval = RETVAL_ERROR;
		fclose(fptr);
	}
	else {
		sprintf(tmp, "-%d",articleCount);	/* this will be string we search for */
		/* look for entries which have our file name ...-xxxxx where xxxx is articleCount */
		while((entry = readdir(dptr)) != NULL) {
			if(strstr(entry->d_name, tmp) != NULL) {
				fprintf(fptr, "%s\n", full_path(FP_GET_NOPOSTFIX, FP_MSGDIR, entry->d_name));
			}
		}
		fclose(fptr);
		closedir(dptr);
	}
	return retval;
}
/*----------------------------------------------------------*/
int do_rnewsbatch(int articleCount, char *batchfile, long max_file_size) {
	/* build rnews formated file of articles */
	/* this is done by searching thru MSGDIR to find files */
	/* which match our naming convention */

	/* if max_file_size > 0, then create multiple files up to  max file size */

	int i, batchnr = 0, retval = RETVAL_OK;
	FILE *fptr = NULL, *fpin;
	char tmp[20], *tptr, buf[MAXLINLEN];
	DIR *dptr;
	struct dirent *entry;
	struct stat sbuf;
	long cursize = 0L;

	if((dptr = opendir(full_path(FP_GET, FP_MSGDIR, ""))) == NULL) {
		MyPerror(full_path(FP_GET, FP_MSGDIR, ""));
		retval = RETVAL_ERROR;
		fclose(fptr);
	}
	else {
		sprintf(tmp, "-%d",articleCount);	/* this will be string we search for */
		/* look for entries which have our file name ...-xxxxx where xxxx is articleCount */
		while(retval == RETVAL_OK && (entry = readdir(dptr))) {
			if(strstr(entry->d_name, tmp) != NULL) {
#ifdef DEBUG2
				do_debug("Adding File %s to batch\n", entry->d_name);
#endif
				tptr = full_path(FP_GET_NOPOSTFIX, FP_MSGDIR, entry->d_name);
				if(stat(tptr, &sbuf) != 0 || (fpin = fopen(tptr, "r")) == NULL) {
					MyPerror(tptr);
					retval = RETVAL_ERROR;
				}
				else {
					if( cursize == 0 ) {
						if(fptr != NULL) {
							/* close old file */
							fclose(fptr);
							batchnr++;
						}
						/* have to open file */
						if(batchnr == 0) {
							strcpy(buf, batchfile);
						}
						else {
							sprintf(buf, "%s%d", batchfile, batchnr);
						}
#ifdef DEBUG2
						do_debug("BATCH FILE: %s\n", buf);
#endif
						if((fptr = fopen(buf, "w")) == NULL) {
							MyPerror(buf);
							retval = RETVAL_ERROR;
						}
					}
					if(retval == RETVAL_OK) {
						/* first put #! rnews size */
						fprintf(fptr, "#! rnews %ld\n", sbuf.st_size);
		
						/* use fread/fwrite in case lines are longer than MAXLINLEN */
						while((i = fread(buf, 1, MAXLINLEN, fpin)) > 0) {
							fwrite(buf, 1, i, fptr);
						}
						fclose(fpin);
						unlink(tptr);	/* we are done with it, nuke it */
						cursize += sbuf.st_size;
						/* keep track of current file size, we can ignore the #! rnews */
						/* size, since it adds so little to the overall size */
					}
					if(max_file_size > 0L && cursize > max_file_size) {
						/* reached file size length */
						cursize = 0L;	/* this will force a close and open on next article */
					}
				}
			}
		}
		fclose(fptr);
		closedir(dptr);
	}
	return retval;
}
/*-----------------------------------------------------------------------------------------*/
int get_one_article(PMaster master, int logcount) {

	char buf[MAXLINLEN+1], *fname;
	int retval;
	FILE *fptr;

	retval = RETVAL_OK;

	/* build command to get article header*/
	sprintf(buf, "article %s\r\n", (master->curr)->msgnr);
	switch(send_a_command(master->sockfd, buf, 220)) {
	  case RETVAL_OK:
		if(master->MultiFile == TRUE) {
			/* open file */
			/* file name will be ####-#### ex 001-166 (nron,total) */
			sprintf(buf,"%0*d-%d", logcount, master->itemon ,master->nritems);
			fname = full_path(FP_GET, FP_MSGDIR, buf);
#ifdef DEBUG2
			do_debug("File name = \"%s\"", fname);
#endif
			if((fptr = fopen(fname, "w")) == NULL) {
				MyPerror(fname);
				retval = RETVAL_ERROR;
			}
			else {
				retval = get_a_chunk(master->sockfd, fptr);
				fclose(fptr);
			}
			/* if we don't get the whole article, nuke the sucker */
			if(retval != RETVAL_OK) {
				unlink(fname);
			}
		}
		else {
			retval = get_a_chunk(master->sockfd, stdout);
			fputs(".\n", stdout);	/* needed */
		}
		if(retval == RETVAL_OK) { 
			master->nrgot++;
		}
		break;
	  case RETVAL_ERROR:
		retval = RETVAL_ERROR;
		break;
	  case RETVAL_UNEXPECTEDANS:	/* just skip to next article */
		break;
	}
	return retval;
}
/*---------------------------------------------------------------------------*/
int get_a_chunk(int sockfd, FILE *fptr) {

	int done, partial, len, retval;
	char *inbuf;

	retval = RETVAL_OK;
	done = FALSE;
	partial = FALSE;

	/* partial = was the previous line a complete line or not */
	/* this is needed to avoid a scenario where the line is MAXLINLEN+1 */
	/* long and the last character is a ., which would make us think */
	/* that we are at the end of the article when we actually aren't */
	
	while(done == FALSE && (len = sgetline(sockfd, &inbuf)) >= 0) {
#ifdef TIMER
		TimerFunc(TIMER_ADDBYTES, len);
#endif			
		if(inbuf[0] == '.' && partial == FALSE) {
			if(len == 2 && inbuf[1] == '\n') {
				done = TRUE;
			}
			else if(fptr != stdout) {
				/* handle double dots IAW RFC977 2.4.1*/
				/* if we are going to stdout, we have to leave double dots alone */
				/* since it might mess up the .\n for EOM */
				inbuf++;	/* move past first dot */
				len--;
			}
		}
		if(done == FALSE) {
			fputs(inbuf, fptr);
			fflush(fptr);
			partial= (len==MAXLINLEN&&inbuf[len-1]!='\n') ? TRUE : FALSE;
		} 
	}
	if(len < 0) {
		retval = RETVAL_ERROR;
	}
	return retval;
}
/*---------------------------------------------------------------------------*/
int send_a_command(int sockfd, char *cmd, int good_response) {

	char *inbuf;
	int response, retval;
#ifdef TIMER
	int len;
#endif
	retval = RETVAL_OK;

#ifdef DEBUG2
	do_debug("\nSENDING REQUEST\n%s",cmd);
#endif
 	sputline(sockfd,cmd);
#ifndef TIMER
 	if(sgetline(sockfd,&inbuf) < 0) {
#else
	if((len = sgetline(sockfd, &inbuf)) < 0) {
#endif
		retval = RETVAL_ERROR;
	}
	else {
#ifdef TIMER
		TimerFunc(TIMER_ADDBYTES, len);
#endif
		number(inbuf, &response);
		if (response != good_response) {
			error_log(ERRLOG_REPORT,"*** Unexpected response to command: %s\n%s\n", cmd,inbuf);
			retval = RETVAL_UNEXPECTEDANS;
		}
	}
	return retval;
}		
/*-----------------------------------------------------------------------------*/
int restart_yn(PMaster master) {
	/* the restart file consists of 1 byte for each article prior to its */
	/* being written.  So if its 2 bytes long, we've actually only got */
	/* the first article.  If its zero bytes long, we aborted before we */
	/* even got a chance to write the first byte */

	struct stat buf;
	int i, retval = RESTART_NO;
	FILE *fp;
	char msgnr[MAXLINLEN];
	char optional;

	if(stat(full_path(FP_GET, FP_TMPDIR, N_RESTART), &buf) == 0) {
		/* If suck.restart exists something went wrong */
		retval = RESTART_YES;	
		master->itemon = (buf.st_size == 0) ? 0 : buf.st_size - 1;
		/* the above is done since we write the byte before hand */
		/* now build the link list in master */

		i = RETVAL_OK;

		if((fp = fopen(full_path(FP_GET, FP_TMPDIR, N_SORTED), "r")) != NULL) {
			while(i == RETVAL_OK && fscanf(fp, "%s %c", msgnr, &optional) == 2) {
				i = build_list(master, msgnr, NULL, optional);
			}
			fclose(fp);
		}
		else {
			i = RETVAL_ERROR;
		}
		if(i == RETVAL_ERROR) {
			retval = RESTART_ERROR;
		}
	}
	return retval;
}
/*-----------------------------------------------------------------*/
#ifdef MYSIGNAL
void sighandler(int what) {

	error_log(ERRLOG_REPORT, "Signal received, will finish downloading article.\n");
	GotSignal = TRUE;

}
#endif
