/************************************************************************
 * cmds.c -- part of rpncalc.c						*
 *									*
 * A little RPN (Reverse Polish Notation) calculator,                   *
 * rudimentary emulating a HP 28S. 					*
 * 								        *
 * rpncalc is (c) David Frey, 1993, 1994, 1995				*
 *								        * 
 * This program is free software; you can redistribute it and/or modify *
 * it under the terms of the GNU General Public License as published by *
 * the Free Software Foundation; either version 2 of the License, or    *
 * (at your option) any later version.                                  *
 *									* 
 * This program is distributed in the hope that it will be useful,      *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 * GNU General Public License for more details.                         *
 *									* 
 * You should have received a copy of the GNU General Public License    *
 * along with this program; if not, write to the Free Software          *
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.            *
 *									* 
 ************************************************************************/

/* $Id: cmds.c,v 1.0 1995/12/31 18:21:07 david Rel $
 * $Log: cmds.c,v $
 * Revision 1.0  1995/12/31 18:21:07  david
 * Initial revision
 * */

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <errno.h>
#include <limits.h>

#include "cmds.h"

extern int errno;
extern int pushtostack;
extern int digits;
extern enum BASE base;

double res;

/* -------------------------------------------------------- constants */

double pi(void)
{
  return M_PI;
}

double e(void)
{
  return M_E;
}

/* -------------------------------------------------------- unary operators */

double chs(double f)
{
  return -f;
}

double sqr(double f)
{
  return f * f;
}

double inv(double f)
{
  if (f != 0)   res = 1 / f;
  else          { res = HUGE_VAL; errno = ERANGE; }	    
  return res;
}

double log2(double f)
{
  return log(f) / M_LN2;
}

double fact(double n)
{
  int i;

  if (n < 0)
  {
    fprintf(stderr, "defined only for arguments >= 0.\n");
    pushtostack=FALSE;
  }
  else 
  {
    if ((n == 0) || (n == 1)) res = 1;
    else
    {
      res = 1;
      for (i = n; i > 1; i--) res *= i;
    }
  }
  return res;
}

double prec(double p)
{
  digits=p; pushtostack=FALSE;
  return 0.0; /* dummy value */
}

double not(double l)
{
  return (double)(~(long int)l);
}

/* ------------------------------------------------------- binary operators */

double plus(double s1, double s2)
{
  return s1 + s2;
}

double minus(double s, double m)
{
  return  s-m;
}

double times(double f1, double f2)
{
  return f1 * f2;
}

double divide(double n, double d)
{
  if (d != 0) res = n / d;
  else      { res = HUGE_VAL; errno = ERANGE; }	    
  return res;
}

ldiv_t quotrem;

double idiv(double n, double d)
{
  if ((n <= LONG_MAX) && (d <= LONG_MAX))
  {
    if (d != 0)
    {
      quotrem = ldiv((long int)n,(long int)d); 
      res = quotrem.quot;
    }
    else
    { 
      res = HUGE_VAL; errno = ERANGE; 
    }	 
  }
  else
  { 
    errno = EOVERFLOW; pushtostack=FALSE;
  }	    
  return res;
}

double mod(double n, double d)
{
 if ((n <= LONG_MAX) && (d <= LONG_MAX))
 {
   if (d != 0)
   {
     quotrem = ldiv((long int)n,(long int)d);
     res = quotrem.rem;  
   }
   else
   { 
     res = HUGE_VAL; errno = ERANGE; 
   }	    
 }
 else
 {	
   errno = EOVERFLOW; pushtostack=FALSE;
 }	    
 return res;
}

/* Stein's Greatest Common Divisor (GCD) algorithm */
long int stein(double d1, double d2)
{
  long int n1, n2;
  long int d;				 /* difference */
  long int c;				 /* c: shift-count (see below) */
  long int t;				 /* temporary */

  c = 0; n1=(long int)d1; n2=(long int)d2;
  /* both integers are even; shift them until one gets odd. */
  while (((n1 & 1) == 0) && ((n2 & 1) == 0))
  {
    n1 >>= 1; n2 >>= 1;  c++;
  }

  do
  {
    if ((n2 & 1) == 0) { t = n1; n1 = n2; n2 = t; }
    
    while ((n1 & 1) == 0)
      n1 >>= 1;

    /*
     * shift n1 until it gets odd.
     */
    d = n1 - n2;
    if (d < 0) { n2 = n1; d = -d; }
    n1 = d >> 1;
  }
  while (n1 > 0);

  return (n2 * (1 << c));
}

double gcd(double n, double d)
{
  if ((n <= LONG_MAX) && (d <= LONG_MAX))
  {
    if (d != 0) res = stein(n,d);
    else      { res = HUGE_VAL; errno = ERANGE; }
  }
  else
  {	      
    errno = EOVERFLOW;	pushtostack = FALSE;
  }	    
  return res;
}

double and(double l1, double l2)
{
  return (double)((long int)l1 & (long int)l2);
}

double or(double l1, double l2)
{
  return (double)((long int)l1 | (long int)l2);
}

double xor(double l1, double l2)
{
  return (double)((long int)l1 ^ (long int)l2);
}

/* ------------------------------------------------------- n-ary operators */
int i,n;

double sum(void)
{
  n = depth(); res = 0;
  for(i=1;i<=n;i++) res += pick(i);
  return res;
}

double prod(void)
{
  n = depth(); res = 1;
  for(i=1;i<=n;i++) res *= pick(i);
  return res;
}
