/*
 *                            COPYRIGHT
 *
 *  PCB, interactive printed circuit board design
 *  Copyright (C) 1994 Thomas Nau
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Contact addresses for paper mail and Email:
 *  Thomas Nau, Schlehenweg 15, 88471 Baustetten, Germany
 *  Thomas.Nau@medizin.uni-ulm.de
 *
 */

static	char	*rcsid = "$Header: dialog.c,v 1.4 94/07/19 14:08:19 nau Exp $";

/* dialog routines
 */

#include <stdio.h>
#include <stdlib.h>

#include "global.h"

#include "data.h"
#include "dialog.h"
#include "error.h"
#include "memory.h"
#include "misc.h"

#include <X11/Shell.h>
#include <X11/Xaw/AsciiText.h>
#include <X11/Xaw/Command.h>
#include <X11/Xaw/Dialog.h>
#include <X11/Xaw/Form.h>
#include <X11/Xaw/Label.h>
#include <X11/Xaw/Toggle.h>

/* ---------------------------------------------------------------------------
 * some useful types used for dialog boxes ...
 */
typedef struct						/* a dialogs buttons */
{
	char			*Name,			/* the widgets name */
					*Label;			/* the buttons text */
	XtCallbackProc	Callback;		/* the buttons select-handler */
	XtPointer		ClientData;		/* data passed to the handler */
	Widget			W;				/* the button widget itself */
} DialogButtonType, *DialogButtonTypePtr;

/* ---------------------------------------------------------------------------
 * some local identifiers
 */
static	Widget	AbortPopup;
static	int		ReturnCodeOK,			/* used for 'GetUserOK' dialog */
				ReturnCodeWarning,		/* 'WarningDialog' */
				ReturnCodeUserString,	/* 'GetUserInput' dialog */
				ReturnCodeAbort;		/* 'Abort' dialog */

/* ---------------------------------------------------------------------------
 * set all keys which generate a new line not 'no operation'.
 * 'Return' and 'Escape' are define for a action routine
 */
static	String	Translations = 
					"<Key>Linefeed: no-op()\n "
					"Ctrl<Key>j:    no-op()\n "
					"Ctrl<Key>m:    no-op()\n "
					"Ctrl<Key>o:    no-op()\n ";

/* ---------------------------------------------------------------------------
 * some local prototypes
 */
static	int		DialogEventLoop(int *);
static	Widget	AddButtons(Widget, Widget, DialogButtonTypePtr, size_t);
static	Widget	CreateDialogBox(Widget, char *, DialogButtonTypePtr, size_t);
static	void	CB_GetUserOK(Widget, XtPointer, XtPointer);
static	void	CB_Warning(Widget, XtPointer, XtPointer);
static	void	StartDialog(Widget);
static	void	EndDialog(Widget);
static	void	CB_Abort(Widget, XtPointer, XtPointer);

/* ---------------------------------------------------------------------------
 * waits until returncode is differnt from -1
 */
static int DialogEventLoop(int *Code)
{
	XEvent	event;

	for(*Code = -1; *Code == -1;)
	{
		XtAppNextEvent(Context, &event);
		XtDispatchEvent(&event);
	}
	return(*Code);
}

/* ---------------------------------------------------------------------------
 * adds the buttons to a dialog
 */
static Widget AddButtons(Widget Parent, Widget Top, DialogButtonTypePtr Buttons, size_t Count)
{
	int		i;

	for (i = 0; i < Count; i++)
	{
			/* skip button if there's no label,
			 * used for dialogs without default button
			 */
		if (!Buttons[i].Label)
			continue;
		Buttons[i].W  = XtVaCreateManagedWidget(Buttons[i].Name,
			commandWidgetClass,
			Parent,
			XtNlabel, Buttons[i].Label,
			XtNfromHoriz, i ? Buttons[i-1].W : NULL,
			XtNfromVert, Top,
			XtNresizable, True,
			NULL);

		XtAddCallback(Buttons[i].W, XtNcallback, Buttons[i].Callback, Buttons[i].ClientData);
	}
	return(Buttons[Count-1].W);
}

/* ---------------------------------------------------------------------------
 * thanks to Ellen M. Sentovich and Rick L. Spickelm for their hint in
 * xrn
 *
 * creates a dialog box as a child of ... (see arguments)
 */
static Widget CreateDialogBox(Widget Parent, char *Title, DialogButtonTypePtr Buttons, size_t Count)
{
	Widget	dialog,
			popup,
			ancestorshell;
	XEvent	event;

		/* find closest ancestor which is a shell */
	ancestorshell = Parent;
	while (ancestorshell && (! XtIsShell(ancestorshell)))
		ancestorshell = XtParent(ancestorshell);

		/* creating the popup shell */
	popup = XtVaCreatePopupShell("popup", transientShellWidgetClass,
		Parent,
		XtNtransientFor, ancestorshell,
		XtNsaveUnder, True,
		XtNallowShellResize, True,
		NULL);
	dialog = XtVaCreateManagedWidget("dialog", dialogWidgetClass,
		popup,
		XtNlabel, Title,
		NULL);

		/* now we add the buttons, the first one is always default */
	AddButtons(dialog, NULL, Buttons, Count);

		/* install the accelerators to the default button,
		 * (we assume that there's at least one button)
		 * the cancel button and the input field
		 */
	XtInstallAccelerators(dialog, Buttons[0].W);
	if (Count >= 2)
		XtInstallAccelerators(dialog, Buttons[1].W);

	return(popup);
}

/* ---------------------------------------------------------------------------
 * centers the widget around the current cursor position
 */
void CenterWidgetAtPointerLocation(Widget Popup)
{
	Window			root, child;
	int				root_x, root_y,
					child_x, child_y;
	unsigned int	mask;
	Dimension		width, height;

		/* get current pointer position relativ to it's parent */
	XQueryPointer(Dpy, XtWindow(Popup),
		&root, &child,
		&root_x, &root_y,
		&child_x, &child_y,
		&mask);

		/* get the dialogs size */
	XtVaGetValues(Popup,
		XtNheight, &height,
		XtNwidth, &width,
		NULL);	
	
		/* make sure position is inside our screen */
	if ((root_x -= (width/2)) < 0)
		root_x = 0;
	else
		if (root_x > WidthOfScreen(XtScreen(Popup)) -width)
			root_x =  WidthOfScreen(XtScreen(Popup)) -width;
	if ((root_y -= (height/2)) < 0)
		root_y = 0;
	else
		if (root_y > HeightOfScreen(XtScreen(Popup)) -height)
			root_y =  HeightOfScreen(XtScreen(Popup)) -height;
	XtVaSetValues(Popup,
		XtNx, root_x,
		XtNy, root_y,
		NULL);
}

/* ---------------------------------------------------------------------------
 * pops up a dialog
 * first realize the dialog and
 * focus keyboard events from the application to it
 */
static void StartDialog(Widget Popup)
{
	XtRealizeWidget(Popup);
	CenterWidgetAtPointerLocation(Popup);
	XtPopup(Popup, XtGrabExclusive);
}

/* ---------------------------------------------------------------------------
 * removes a dialog from screen and from memory
 */
static void EndDialog(Widget Popup)
{
	XtPopdown(Popup);
	XtDestroyWidget(Popup);
}

/* ---------------------------------------------------------------------------
 * callback for 'GetUserOK' dialog
 */
static void CB_GetUserOK(Widget W, XtPointer ClientData, XtPointer CallData)
{
	ReturnCodeOK = (int) ClientData;
}

/* ---------------------------------------------------------------------------
 * gets users OK for something that is passed as 'Message'
 */
int GetUserOK(Widget Parent, char *Message)
{
			Widget				popup;
	static	DialogButtonType	buttons[] = {
		{ "defaultButton", "   OK   ", CB_GetUserOK, (XtPointer) 1, NULL },
		{ "cancelButton", "No/Cancel", CB_GetUserOK, (XtPointer) 0, NULL }};

		/* create dialog box */
	popup = CreateDialogBox(Parent, Message, buttons, ENTRIES(buttons));
	StartDialog(popup);

		/* wait for dialog to complete */
	DialogEventLoop(&ReturnCodeOK);
	EndDialog(popup);
	return(ReturnCodeOK);
}

/* ---------------------------------------------------------------------------
 * callback for 'WarningDialog' dialog
 */
static void CB_Warning(Widget W, XtPointer ClientData, XtPointer CallData)
{
	ReturnCodeWarning = (int) ClientData;
}

/* ---------------------------------------------------------------------------
 * prints out a warning
 */
void WarningDialog(Widget Parent, char *Message)
{
			Widget				popup;
	static	DialogButtonType	buttons[] = {
		{ "defaultButton", "  OK  ", CB_Warning, (XtPointer) 1, NULL }};

		/* create dialog box */
	popup = CreateDialogBox(Parent, Message, buttons, ENTRIES(buttons));
	StartDialog(popup);

		/* wait for dialog to complete */
	DialogEventLoop(&ReturnCodeWarning);
	EndDialog(popup);
}

/* ---------------------------------------------------------------------------
 * called from action routine ActionUserInput()
 * which passes True for OK selection and False for CANCEL
 */
void EndCommand(Boolean OK)
{
	ReturnCodeUserString = OK ? 1 : 0;
}

/* ---------------------------------------------------------------------------
 * gets a string from user, memory is allocated for the string,
 * the user is responsible for releasing the allocated memory
 * string might be empty if flag is set
 */
char *GetUserInput(Widget Parent, Boolean EmptyOK, char *Message, char *OutputString)
{
	char	*string = NULL;

		/* set some reources for the input fields or create them */
	XtVaSetValues(Output.InputText, XtNlabel, EMPTY(Message), NULL);

		/* the input field itself */
	Output.InputField = XtVaCreateManagedWidget("inputField", asciiTextWidgetClass,
		Output.MasterForm,
		XtNresizable, True,
		XtNresize, XawtextResizeWidth,
		XtNstring, EMPTY(OutputString),
		XtNwrap, XawtextWrapNever,
		XtNeditType, XawtextEdit,
		XtNfromHoriz, Output.InputText,
		XtNfromVert, NULL,
		XtNleft, XtChainLeft,
		XtNright, XtChainLeft,
		XtNtop, XtChainTop,
		XtNbottom, XtChainTop,
		NULL);

		/* set focus to input field, override default translations
		 * and install accelerators
		 * grap all events --> make widget modal
		 */
	XtSetKeyboardFocus(Output.Toplevel, Output.InputField);
	XtOverrideTranslations(Output.InputField,
		XtParseTranslationTable(Translations));
	XtInstallAccelerators(Output.InputField, Output.MasterForm);
	XtAddGrab(Output.InputField, True, False);
	XMapWindow(Dpy, XtWindow(Output.InputText));
	XtRealizeWidget(Output.InputField);

		/* wait for input to complete and allocate memory if necessary */
	while (DialogEventLoop(&ReturnCodeUserString))
	{
			/* strip white space and return string */
		XtVaGetValues(Output.InputField, XtNstring, &string, NULL);
		if ((string = StripWhiteSpaceAndDup(string)) == NULL)
			if (EmptyOK)
				string = MyStrdup("", "GetUserInput()");
			else
			{
				MyWarningDialog("I don't like empty input fields\n");
				continue;
			}
		break;
	}

		/* restore normal outfit */
	XtRemoveGrab(Output.InputField);
	XtDestroyWidget(Output.InputField);
	XUnmapWindow(Dpy, XtWindow(Output.InputText));
	return(string);
}

/* ---------------------------------------------------------------------------
 * callback for 'AbortDialog' dialog
 */
static void CB_Abort(Widget W, XtPointer ClientData, XtPointer CallData)
{
	ReturnCodeAbort = (int) ClientData;
}

/* ---------------------------------------------------------------------------
 * creates an 'Abort' dialog
 */
void SetupAbortDialog(Widget Parent, char *Message)
{
	static	DialogButtonType	buttons[] = {
		{ "defaultButton", " Abort ", CB_Abort, (XtPointer) 1, NULL }};

		/* create dialog box */
	AbortPopup = CreateDialogBox(Parent, Message, buttons, ENTRIES(buttons));
	StartDialog(AbortPopup);
	ReturnCodeAbort = 0;
}

/* ---------------------------------------------------------------------------
 * destroys an 'Abort' dialog
 */
void EndAbortDialog(void)
{
	EndDialog(AbortPopup);
}

/* ---------------------------------------------------------------------------
 * checks if abort button was pressed, returns true in this case
 */
Boolean CheckAbort(void)
{
	XEvent	event;

	while (XtAppPending(Context))
	{
		XtAppNextEvent(Context, &event);
		XtDispatchEvent(&event);
	}
	return(ReturnCodeAbort == 1);
}

