/*	Copyright (C) 1995 Free Software Foundation, Inc.
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this software; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * As a special exception, the Free Software Foundation gives permission
 * for additional uses of the text contained in its release of GUILE.
 *
 * The exception is that, if you link the GUILE library with other files
 * to produce an executable, this does not by itself cause the
 * resulting executable to be covered by the GNU General Public License.
 * Your use of that executable is in no way restricted on account of
 * linking the GUILE library code into it.
 *
 * This exception does not however invalidate any other reasons why
 * the executable file might be covered by the GNU General Public License.
 *
 * This exception applies only to the code released by the
 * Free Software Foundation under the name GUILE.  If you copy
 * code from other Free Software Foundation releases into a copy of
 * GUILE, as the General Public License permits, the exception does
 * not apply to the code that you add in this way.  To avoid misleading
 * anyone as to the status of such modified files, you must delete
 * this exception notice from them.
 *
 * If you write modifications of your own for GUILE, it is your choice
 * whether to permit this exception to apply to your modifications.
 * If you do not wish that, delete this exception notice.  
 */


#include <stdio.h>
#include "_scm.h"



/* NUM_HASH_BUCKETS is the number of symbol scm_hash table buckets. 
 */
#define NUM_HASH_BUCKETS 137




/* {Symbols}
 */

#ifdef __STDC__
unsigned long 
scm_strhash (unsigned char *str, sizet len, unsigned long n)
#else
unsigned long 
scm_strhash (str, len, n)
     unsigned char *str;
     sizet len;
     unsigned long n;
#endif
{
  if (len > 5)
    {
      sizet i = 5;
      unsigned long h = 264 % n;
      while (i--)
	h = ((h << 8) + ((unsigned) (scm_downcase[str[h % len]]))) % n;
      return h;
    }
  else
    {
      sizet i = len;
      unsigned long h = 0;
      while (i)
	h = ((h << 8) + ((unsigned) (scm_downcase[str[--i]]))) % n;
      return h;
    }
}

int scm_symhash_dim = NUM_HASH_BUCKETS;


/* scm_sym2vcell
 * looks up the symbol in the symhash table. 
 */
#ifdef __STDC__
SCM 
scm_sym2vcell (SCM sym, SCM thunk, SCM definep)
#else
SCM 
scm_sym2vcell (sym, thunk, definep)
     SCM sym;
     SCM thunk;
     SCM definep;
#endif
{
  if (NIMP(thunk))
    {
      SCM var = scm_apply (thunk, sym, scm_cons(definep, listofnull));

      if (var == BOOL_F)
	return BOOL_F;
      else
	{
	  if (IMP(var) || !VARIABLEP (var))
	    scm_wta (sym, "strangely interned symbol? ", "");
	  return VARVCELL (var);
	}
    }
  else
    {
      SCM lsym, z;
      sizet scm_hash = scm_strhash (UCHARS (sym), (sizet) LENGTH (sym),
				    (unsigned long) scm_symhash_dim);
      for (lsym = VELTS (symhash)[scm_hash]; NIMP (lsym); lsym = CDR (lsym))
	{
	  z = CAR (lsym);
	  if (CAR (z) == sym)
	    return z;
	}
      /* DEFINEP is ignored here on the grounds that only 
       * symbols interned normally (on creation) in the symhash table
       * ought to be used for definitions in the symhash table.  
       * Therefore, SYM ought to already be interned and should have been
       * found by the preceeding for loop.  If it wasn't, it can only
       * be an error.
       *
       * Why not allow strange symbols in the symbol table?  Because
       * the SYMBOL_HASH table contains special variables that parts
       * of the C code look up by name.   Referential transparency of 
       * variables names is therefore very important;  strange symbols
       * break referential transparency of variable names.
       */
      return scm_wta (sym, "uninterned symbol? ", "");
    }
}

/* scm_sym2ovcell
 * looks up the symbol in an arbitrary obarray (defaulting to symhash).
 */
#ifdef __STDC__
SCM 
scm_sym2ovcell_soft (SCM sym, SCM obarray)
#else
SCM 
scm_sym2ovcell_soft (sym, obarray)
     SCM sym;
     SCM obarray;
#endif
{
  SCM lsym, z;
  sizet scm_hash;

  scm_hash = scm_strhash (UCHARS (sym),
			  (sizet) LENGTH (sym),
			  LENGTH (obarray));
  for (lsym = VELTS (obarray)[scm_hash];
       NIMP (lsym);
       lsym = CDR (lsym))
    {
      z = CAR (lsym);
      if (CAR (z) == sym)
	return z;
    }
  return BOOL_F;
}

#ifdef __STDC__
SCM 
scm_sym2ovcell (SCM sym, SCM obarray)
#else
SCM 
scm_sym2ovcell (sym, obarray)
     SCM sym;
     SCM obarray;
#endif
{
  SCM answer;
  answer = scm_sym2ovcell_soft (sym, obarray);
  if (answer != BOOL_F)
    return answer;
  scm_wta (sym, "uninterned symbol? ", "");
  return UNSPECIFIED;		/* not reached */
}

#ifdef __STDC__
SCM 
scm_intern_obarray_soft (char *name, sizet len, SCM obarray, int softness)
#else
SCM 
scm_intern_obarray_soft (name, len, obarray, softness)
     char *name;
     sizet len;
     SCM obarray;
     int softness;
#endif
{
  SCM lsym;
  SCM z;
  register sizet i;
  register unsigned char *tmp;
  sizet scm_hash;

  i = len;
  tmp = (unsigned char *) name;

  if (obarray == BOOL_F)
    {
      scm_hash = scm_strhash (tmp, i, 1019);
      goto uninterned_symbol;
    }

  scm_hash = scm_strhash (tmp, i, LENGTH(obarray));

  if (softness == -1)
    goto mustintern_symbol;

  for (lsym = VELTS (obarray)[scm_hash]; NIMP (lsym); lsym = CDR (lsym))
    {
      z = CAR (lsym);
      z = CAR (z);
      tmp = UCHARS (z);
      if (LENGTH (z) != len)
	goto trynext;
      for (i = len; i--;)
	if (((unsigned char *) name)[i] != tmp[i])
	  goto trynext;
      return CAR (lsym);
    trynext:;
    }

 uninterned_symbol:
  if (softness)
    return BOOL_F;

 mustintern_symbol:
  lsym = scm_makfromstr (name, len, SYMBOL_SLOTS);
  DEFER_INTS;
  SETLENGTH (lsym, (long) len, tc7_msymbol);
  SYMBOL_HASH (lsym) = scm_hash;
  ALLOW_INTS;
  if (obarray == BOOL_F)
    {
      SCM answer;
      NEWCELL (answer);
      DEFER_INTS;
      CAR (answer) = lsym;
      CDR (answer) = SCM_UNDEFINED;
      ALLOW_INTS;
      return answer;
    }
  else
    return CAR (VELTS (obarray)[scm_hash] =
		scm_acons (lsym, SCM_UNDEFINED, VELTS (obarray)[scm_hash]));
}

#ifdef __STDC__
SCM
scm_intern_obarray (char *name, sizet len, SCM obarray)
#else
SCM
scm_intern_obarray (name, len, obarray)
     char *name;
     sizet len;
     SCM obarray;
#endif
{
  return scm_intern_obarray_soft (name, len, obarray, 0);
}


#ifdef __STDC__
SCM 
scm_intern (char *name, sizet len)
#else
SCM 
scm_intern (name, len)
     char *name;
     sizet len;
#endif
{
  return scm_intern_obarray (name, len, symhash);
}

#ifdef __STDC__
SCM
scm_intern0 (char * name)
#else
SCM
scm_intern0 (name)
     char * name;
#endif
{
  return scm_intern (name, strlen (name));
}


#ifdef __STDC__
SCM 
scm_sysintern (char *name, SCM val)
#else
SCM 
scm_sysintern (name, val)
     char *name;
     SCM val;
#endif
{
  SCM easy_answer;
  easy_answer = scm_intern_obarray_soft (name, strlen (name), symhash, 1);
  if (NIMP (easy_answer))
    {
      CDR (easy_answer) = val;
      return easy_answer;
    }
  else
    {
      SCM lsym;
      sizet len = strlen (name);
      register unsigned char *tmp = (unsigned char *) name;
      sizet scm_hash = scm_strhash (tmp, len, (unsigned long) scm_symhash_dim);
      NEWCELL (lsym);
      SETLENGTH (lsym, (long) len, tc7_ssymbol);
      SETCHARS (lsym, name);
      lsym = scm_cons (lsym, val);
      VELTS (symhash)[scm_hash] = scm_cons (lsym, VELTS (symhash)[scm_hash]);
      return lsym;
    }
}


PROC (s_symbol_p, "symbol?", 1, 0, 0, scm_symbol_p);
#ifdef __STDC__
SCM
scm_symbol_p(SCM x)
#else
SCM
scm_symbol_p(x)
     SCM x;
#endif
{
	if IMP(x) return BOOL_F;
	return SYMBOLP(x) ? BOOL_T : BOOL_F;
}

PROC (s_symbol_to_string, "symbol->string", 1, 0, 0, scm_symbol_to_string);
#ifdef __STDC__
SCM
scm_symbol_to_string(SCM s)
#else
SCM
scm_symbol_to_string(s)
     SCM s;
#endif
{
	ASSERT(NIMP(s) && SYMBOLP(s), s, ARG1, s_symbol_to_string);
	return scm_makfromstr(CHARS(s), (sizet)LENGTH(s), 0);
}

PROC (s_string_to_symbol, "string->symbol", 1, 0, 0, scm_string_to_symbol);
#ifdef __STDC__
SCM
scm_string_to_symbol(SCM s)
#else
SCM
scm_string_to_symbol(s)
     SCM s;
#endif
{
	ASSERT(NIMP(s) && ROSTRINGP(s), s, ARG1, s_string_to_symbol);
	s = scm_intern(CHARS(s), (sizet)LENGTH(s));
	return CAR(s);
}


PROC (s_string_to_obarray_symbol, "string->obarray-symbol", 2, 0, 0, scm_string_to_obarray_symbol);
#ifdef __STDC__
SCM
scm_string_to_obarray_symbol(SCM o, SCM s)
#else
SCM
scm_string_to_obarray_symbol(o, s)
     SCM o;
     SCM s;
#endif
{
  ASSERT(NIMP(s) && ROSTRINGP(s), s, ARG2, s_string_to_obarray_symbol);
  ASSERT((o == BOOL_F) || (NIMP(s) && VECTORP(o)),
	 o, ARG1, s_string_to_obarray_symbol);
  s = scm_intern_obarray (CHARS(s), (sizet)LENGTH(s), o);
  return CAR(s);
}

PROC (s_intern_symbol, "intern-symbol", 2, 0, 0, scm_intern_symbol);
#ifdef __STDC__
SCM
scm_intern_symbol(SCM o, SCM s)
#else
SCM
scm_intern_symbol(o, s)
     SCM o;
     SCM s;
#endif
{
        sizet hval;
	ASSERT(NIMP(s) && SYMBOLP(s), s, ARG2, s_intern_symbol);
	if (o == BOOL_F)
	  o = symhash;
	ASSERT(NIMP(o) && VECTORP(o), o, ARG1, s_intern_symbol);
	hval = scm_strhash (UCHARS (s), LENGTH (s), LENGTH(o));
	/* If the symbol is already interned, simply return. */
	{
	  SCM lsym;
	  SCM sym;
	  for (lsym = VELTS (o)[hval];
	       NIMP (lsym);
	       lsym = CDR (lsym))
	    {
	      sym = CAR (lsym);
	      if (CAR (sym) == s)
		return UNSPECIFIED;
	    }
	  VELTS (o)[hval] =
	    scm_acons (s, SCM_UNDEFINED, VELTS (o)[hval]);
	}
	return UNSPECIFIED;
}

PROC (s_unintern_symbol, "unintern-symbol", 2, 0, 0, scm_unintern_symbol);
#ifdef __STDC__
SCM
scm_unintern_symbol(SCM o, SCM s)
#else
SCM
scm_unintern_symbol(o, s)
     SCM o;
     SCM s;
#endif
{
        sizet hval;
	ASSERT(NIMP(s) && SYMBOLP(s), s, ARG2, s_unintern_symbol);
	if (o == BOOL_F)
	  o = symhash;
	ASSERT(NIMP(o) && VECTORP(o), o, ARG1, s_unintern_symbol);
	hval = scm_strhash (UCHARS (s), LENGTH (s), LENGTH(o));
	{
	  SCM lsym_follow;
	  SCM lsym;
	  SCM sym;
	  for (lsym = VELTS (o)[hval], lsym_follow = BOOL_F;
	       NIMP (lsym);
	       lsym_follow = lsym, lsym = CDR (lsym))
	    {
	      sym = CAR (lsym);
	      if (CAR (sym) == s)
		{
		  /* Found the symbol to unintern. */
		  if (lsym_follow == BOOL_F)
		    VELTS(o)[hval] = lsym;
		  else
		    CDR(lsym_follow) = CDR(lsym);
		  return BOOL_T;
		}
	    }
	}
	return BOOL_F;
}

PROC (s_symbol_binding, "symbol-binding", 2, 0, 0, scm_symbol_binding);
#ifdef __STDC__
SCM
scm_symbol_binding (SCM o, SCM s)
#else
SCM
scm_symbol_binding (o, s)
     SCM o;
     SCM s;
#endif
{
  SCM vcell;
  ASSERT(NIMP(s) && SYMBOLP(s), s, ARG2, s_symbol_binding);
  if (o == BOOL_F)
    o = symhash;
  ASSERT(NIMP(o) && VECTORP(o), o, ARG1, s_symbol_binding);
  vcell = scm_sym2ovcell (s, o);
  return CDR(vcell);
}


PROC (s_symbol_interned_p, "symbol-interned?", 2, 0, 0, scm_symbol_interned_p);
#ifdef __STDC__
SCM
scm_symbol_interned_p (SCM o, SCM s)
#else
SCM
scm_symbol_interned_p (o, s)
     SCM o;
     SCM s;
#endif
{
  SCM vcell;
  ASSERT(NIMP(s) && SYMBOLP(s), s, ARG2, s_symbol_interned_p);
  if (o == BOOL_F)
    o = symhash;
  ASSERT(NIMP(o) && VECTORP(o), o, ARG1, s_symbol_interned_p);
  vcell = scm_sym2ovcell_soft (s, o);
  return (NIMP(vcell)
	  ? BOOL_T
	  : BOOL_F);
}


PROC (s_symbol_bound_p, "symbol-bound?", 2, 0, 0, scm_symbol_bound_p);
#ifdef __STDC__
SCM 
scm_symbol_bound_p (SCM o, SCM s)
#else
SCM 
scm_symbol_bound_p (o, s)
     SCM o;
     SCM s;
#endif
{
  SCM vcell;
  ASSERT(NIMP(s) && SYMBOLP(s), s, ARG2, s_symbol_bound_p);
  if (o == BOOL_F)
    o = symhash;
  ASSERT(NIMP(o) && VECTORP(o), o, ARG1, s_symbol_bound_p);
  vcell = scm_sym2ovcell_soft (s, o);
  return ((  NIMP(vcell)
	   && (CDR(vcell) != SCM_UNDEFINED))
	  ? BOOL_T
	  : BOOL_F);
}


PROC (s_symbol_set_x, "symbol-set!", 3, 0, 0, scm_symbol_set_x);
#ifdef __STDC__
SCM 
scm_symbol_set_x (SCM o, SCM s, SCM v)
#else
SCM 
scm_symbol_set_x (o, s, v)
     SCM o;
     SCM s;
     SCM v;
#endif
{
  SCM vcell;
  ASSERT(NIMP(s) && SYMBOLP(s), s, ARG2, s_symbol_set_x);
  if (o == BOOL_F)
    o = symhash;
  ASSERT(NIMP(o) && VECTORP(o), o, ARG1, s_symbol_set_x);
  vcell = scm_sym2ovcell (s, o);
  CDR(vcell) = v;
  return UNSPECIFIED;
}

static void
msymbolize (s)
     SCM s;
{
  SCM string;
  string = scm_makfromstr (CHARS (s), LENGTH (s), SYMBOL_SLOTS);
  DEFER_INTS;
  CHARS (s) = CHARS (string);
  SETLENGTH (s, LENGTH (s), tc7_msymbol);
  CDR (string) = EOL;
  CAR (string) = EOL;
  ALLOW_INTS;
}


PROC (s_symbol_fref, "symbol-fref", 1, 0, 0, scm_symbol_fref);
#ifdef __STDC__
SCM
scm_symbol_fref (SCM s)
#else
SCM
scm_symbol_fref (s)
     SCM s;
#endif
{
  ASSERT(NIMP(s) && SYMBOLP(s), s, ARG1, s_symbol_fref);
  if (TYP7(s) == tc7_ssymbol)
    msymbolize (s);
  return SYMBOL_FUNC (s);
}


PROC (s_symbol_pref, "symbol-pref", 1, 0, 0, scm_symbol_pref);
#ifdef __STDC__
SCM
scm_symbol_pref (SCM s)
#else
SCM
scm_symbol_pref (s)
     SCM s;
#endif
{
  ASSERT(NIMP(s) && SYMBOLP(s), s, ARG1, s_symbol_pref);
  if (TYP7(s) == tc7_ssymbol)
    msymbolize (s);
  return SYMBOL_PROPS (s);
}


PROC (s_symbol_fset_x, "symbol-fset!", 2, 0, 0, scm_symbol_fset_x);
#ifdef __STDC__
SCM
scm_symbol_fset_x (SCM s, SCM val)
#else
SCM
scm_symbol_fset_x (s, val)
     SCM s;
     SCM val;
#endif
{
  ASSERT(NIMP(s) && SYMBOLP(s), s, ARG1, s_symbol_fset_x);
  if (TYP7(s) == tc7_ssymbol)
    msymbolize (s);
  SYMBOL_FUNC (s) = val;
  return UNSPECIFIED;
}


PROC (s_symbol_pset_x, "symbol-pset!", 2, 0, 0, scm_symbol_pset_x);
#ifdef __STDC__
SCM
scm_symbol_pset_x (SCM s, SCM val)
#else
SCM
scm_symbol_pset_x (s, val)
     SCM s;
     SCM val;
#endif
{
  ASSERT(NIMP(s) && SYMBOLP(s), s, ARG1, s_symbol_pset_x);
  if (TYP7(s) == tc7_ssymbol)
    msymbolize (s);
  SYMBOL_PROPS (s) = val;
  return UNSPECIFIED;
}


PROC (s_symbol_hash, "symbol-hash", 1, 0, 0, scm_symbol_hash);
#ifdef __STDC__
SCM
scm_symbol_hash (SCM s)
#else
SCM
scm_symbol_hash (s)
     SCM s;
#endif
{
  ASSERT(NIMP(s) && SYMBOLP(s), s, ARG1, s_symbol_hash);
  return MAKINUM ((unsigned long)s ^ SYMBOL_HASH (s));
}


#ifdef __STDC__
void
scm_init_symbols (void)
#else
void
scm_init_symbols ()
#endif
{
#include "symbols.x"
}

