/*
 *  CU sudo version 1.4
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 1, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Please send bugs, changes, problems to sudo-bugs@cs.colorado.edu
 *
 *******************************************************************
 *
 *  This module contains tgetpass(), getpass(3) with a timeout.
 *  It should work on any OS that supports sgtty (4BSD), termio (SYSV),
 *  or termios (POSIX) line disciplines.
 *
 *  Todd C. Miller  Sun Jun  5 17:22:31 MDT 1994
 */

#ifndef lint
static char rcsid[] = "$Id: tgetpass.c,v 1.37 1996/04/02 21:42:59 millert Exp $";
#endif /* lint */

#include "config.h"

#include <stdio.h>
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif /* HAVE_UNISTD_H */
#ifdef HAVE_STRING_H
#include <string.h>
#endif /* HAVE_STRING_H */
#ifdef HAVE_STRINGS_H
#include <strings.h>
#endif /* HAVE_STRINGS_H */
#include <limits.h>
#include <pwd.h>
#include <sys/types.h>
#ifdef HAVE_SYS_BSDTYPES_H
#include <sys/bsdtypes.h>
#endif /* HAVE_SYS_BSDTYPES_H */
#ifdef _AIX
#include <sys/select.h>
#endif /* _AIX */
#include <sys/time.h>
#include <signal.h>
#include <fcntl.h>
#ifdef HAVE_TERMIOS_H
#include <termios.h>
#else
#ifdef HAVE_TERMIO_H
#include <termio.h>
#else
#include <sgtty.h>
#include <sys/ioctl.h>
#endif /* HAVE_TERMIO_H */
#endif /* HAVE_TERMIOS_H */

#include <pathnames.h>
#include "compat.h"

#ifndef TCSASOFT
#define TCSASOFT	0
#endif /* TCSASOFT */


/******************************************************************
 *
 *  tgetpass()
 *
 *  this function prints a prompt and gets a password from /dev/tty
 *  or stdin.  Echo is turned off (if possible) during password entry
 *  and input will time out based on the value of timeout.
 */

char * tgetpass(prompt, timeout)
    const char *prompt;
    int timeout;
{
#ifdef HAVE_TERMIOS_H
    struct termios term;
    tcflag_t svflagval;
#else
#ifdef HAVE_TERMIO_H
    struct termio term;
    unsigned short svflagval;
#else
    struct sgttyb ttyb;
    int svflagval;
#endif /* HAVE_TERMIO_H */
#endif /* HAVE_TERMIOS_H */
#ifdef POSIX_SIGNALS
    sigset_t oldmask;
    sigset_t mask;
#else
    int oldmask;
#endif /* POSIX_SIGNALS */
    int input, output, n;
    static char buf[_PASSWD_LEN + 1];
    fd_set readfds;
    struct timeval tv;

    /*
     * mask out SIGINT and SIGTSTP, should probably just catch and deal.
     */
#ifdef POSIX_SIGNALS
    (void) memset((VOID *)&mask, 0, sizeof(mask));
    (void) sigaddset(&mask, SIGINT);
    (void) sigaddset(&mask, SIGTSTP);
    (void) sigprocmask(SIG_BLOCK, &mask, &oldmask);
#else
    oldmask = sigblock(sigmask(SIGINT)|sigmask(SIGTSTP));
#endif

    /*
     * open /dev/tty for reading/writing if possible or use
     * stdin and stderr instead.
     */
    if ((input = open(_PATH_TTY, O_RDWR)) < 0) {
	input = fileno(stdin);
	output = fileno(stderr);
    } else {
	output = input;
    }

    /*
     * turn off echo
     */
#ifdef HAVE_TERMIOS_H
    (void) tcgetattr(input, &term);
    svflagval = term.c_lflag;
    term.c_lflag &= ~ECHO;
    (void) tcsetattr(input, TCSAFLUSH|TCSASOFT, &term);
#else
#ifdef HAVE_TERMIO_H
    (void) ioctl(input, TCGETA, &term);
    svflagval = term.c_lflag;
    term.c_lflag &= ~ECHO;
    (void) ioctl(input, TCSETA, &term);
#else
    (void) ioctl(input, TIOCGETP, &ttyb);
    svflagval = ttyb.sg_flags;
    ttyb.sg_flags &= ~ECHO;
    (void) ioctl(input, TIOCSETP, &ttyb);
#endif /* HAVE_TERMIO_H */
#endif /* HAVE_TERMIOS_H */

    /* print the prompt */
    (void) write(output, prompt, strlen(prompt));

    /* setup for select(2) */
    FD_ZERO(&readfds);
    FD_SET(input, &readfds);

    /* set timeout for select */
    tv.tv_sec = timeout;
    tv.tv_usec = 0;

    /* how many file descriptors may we have? */
#ifdef HAVE_SYSCONF
    n = sysconf(_SC_OPEN_MAX);
#else
    n = getdtablesize();
#endif /* HAVE_SYSCONF */

    /*
     * get password or return empty string if nothing to read by timeout
     */
    buf[0] = '\0';
    if (select(n, &readfds, NULL, NULL, &tv) > 0 &&
	(n = read(input, buf, sizeof(buf))) > 0) {
	buf[n - 1] = '\0';
    }

     /* turn on echo */
#ifdef HAVE_TERMIOS_H
    term.c_lflag = svflagval;
    (void) tcsetattr(input, TCSAFLUSH|TCSASOFT, &term);
#else
#ifdef HAVE_TERMIO_H
    term.c_lflag = svflagval;
    (void) ioctl(input, TCSETA, &term);
#else
    ttyb.sg_flags = svflagval;
    (void) ioctl(input, TIOCSETP, &ttyb);
#endif /* HAVE_TERMIO_H */
#endif /* HAVE_TERMIOS_H */

    /* print a newline since echo is turned off */
    (void) write(output, "\n", 1);

    /* restore old signal mask */
#ifdef POSIX_SIGNALS
    (void) sigprocmask(SIG_SETMASK, &oldmask, NULL);
#else
    (void) sigsetmask(oldmask);
#endif

    /* close /dev/tty if that's what we opened */
    if (input != fileno(stdin))
	(void) close(input);

    return(buf);
}
